/* This file is part of
 * ======================================================
 *
 *           LyX, The Document Processor
 *
 *	    Copyright 1995 Matthias Ettrich
 *          Copyright 1995-2001 The LyX Team.
 *
 * ====================================================== */

#include <config.h>

#ifdef __GNUG__
#pragma implementation
#endif

#include "lyx_gui.h"
#include FORMS_H_LOCATION
#include "tex-strings.h"
#include "lyx_main.h"
#include "debug.h"
#include "version.h"
#include "LyXView.h"
#include "buffer.h"
#include "BufferView.h"
#include "lyxserver.h"
#include "lyxrc.h"
#include "gettext.h"
#if FL_REVISION < 89 || (FL_REVISION == 89 && FL_FIXLEVEL < 5)
#include "lyxlookup.h"
#endif
#include "bufferlist.h"
#include "ColorHandler.h"

#include "frontends/GUIRunTime.h"
#include "frontends/xforms/xforms_helpers.h" // for XformColor

#include "support/filetools.h"
#include "support/os.h"
#include "support/lyxlib.h"

#include <cstdlib>
#include <fcntl.h>

using std::endl;

extern LyXServer * lyxserver;
extern bool finished;	// flag, that we are quitting the program
extern BufferList bufferlist;
extern string user_lyxdir;

FL_CMD_OPT cmdopt[] =
{
	{"-geometry", "*.geometry", XrmoptionSepArg, "690x510"}
};

namespace {

int width  = 690;
int height = 510;
int xpos   = -1;
int ypos   = -1;
char geometry[40];

} // namespace anon


FL_resource res[] =
{
	{"geometry", "geometryClass", FL_STRING, geometry, "", 40}
};


extern "C" {

static
int LyX_XErrHandler(Display * display, XErrorEvent * xeev)
{
	if (xeev->error_code == BadWindow) {
		// We don't abort on BadWindow
		return 0;
	}

	// emergency cleanup
	LyX::emergencyCleanup();

	// Get the reason for the crash.
	char etxt[513];
	XGetErrorText(display, xeev->error_code, etxt, 512);
	lyxerr << etxt << " id: " << xeev->resourceid << endl;
	// By doing an abort we get a nice backtrace. (hopefully)
	lyx::abort();
	return 0; // Solaris CC wants us to return something
}

}


LyXGUI::LyXGUI(LyX * owner, int * argc, char * argv[], bool GUI)
	: _owner(owner), lyxViews(0)
{
	gui = GUI;
	if (!gui)
		return;

	//
	setDefaults();

	static const int num_res = sizeof(res)/sizeof(FL_resource);
	fl_initialize(argc, argv, "LyX", cmdopt, num_res);
	// It appears that, in xforms >=0.89.5, fl_initialize()
	// calls setlocale() and ruins our LC_NUMERIC setting.
	locale_init();
	fl_get_app_resources(res, num_res);

	static const int geometryBitmask =
		XParseGeometry(geometry,
				&xpos,
				&ypos,
				reinterpret_cast<unsigned int *>(&width),
				reinterpret_cast<unsigned int *>(&height));

	Display * display = fl_get_display();
	if (!display) {
		lyxerr << "LyX: unable to access X display, exiting" << endl;
		os::warn("Unable to access X display, exiting");
		exit(1);
	}
	fcntl(ConnectionNumber(display), F_SETFD, FD_CLOEXEC);
	// X Error handler install goes here
	XSetErrorHandler(LyX_XErrHandler);

	// A width less than 590 pops up an awkward main window
	// The minimal values of width/height (590/400) are defined in
	// src/lyx.C
	if (width < 590) width = 590;
	if (height < 400) height = 400;

	// If width is not set by geometry, check it against monitor width
	if (!(geometryBitmask & 4)) {
		Screen * scr = ScreenOfDisplay(fl_get_display(), fl_screen);
		if (WidthOfScreen(scr) - 8 < width)
			width = WidthOfScreen(scr) - 8;
	}

	// If height is not set by geometry, check it against monitor height
	if (!(geometryBitmask & 8)) {
		Screen * scr = ScreenOfDisplay(fl_get_display(), fl_screen);
		if (HeightOfScreen(scr) - 24 < height)
			height = HeightOfScreen(scr) - 24;
	}

	// Recalculate xpos if it's negative
	if (geometryBitmask & 16)
		xpos += WidthOfScreen(ScreenOfDisplay(fl_get_display(),
						      fl_screen)) - width;

	// Recalculate ypos if it's negative
	if (geometryBitmask & 32)
		ypos += HeightOfScreen(ScreenOfDisplay(fl_get_display(),
						       fl_screen)) - height;

	// Initialize the LyXColorHandler
	lyxColorHandler.reset(new LyXColorHandler);
}


// A destructor is always necessary  (asierra-970604)
LyXGUI::~LyXGUI()
{
	// Lyxserver was created in this class so should be destroyed
	// here.  asierra-970604
	delete lyxserver;
	lyxserver = 0;
	delete lyxViews;
#if FL_REVISION < 89 || (FL_REVISION == 89 && FL_FIXLEVEL < 5)
	CloseLyXLookup();
#endif
}


void LyXGUI::setDefaults()
{
	GUIRunTime::setDefaults();
}


// This is called after we have parsed lyxrc
void LyXGUI::init()
{
	if (!gui)
		return;

	create_forms();

	if (lyxrc.popup_font_encoding.empty())
		lyxrc.popup_font_encoding = lyxrc.font_norm;
	// Set the font name for popups and menus
	string boldfontname = lyxrc.popup_bold_font
			       + "-*-*-*-?-*-*-*-*-"
			       + lyxrc.popup_font_encoding;
		// "?" means "scale that font"
	string fontname = lyxrc.popup_normal_font
			       + "-*-*-*-?-*-*-*-*-"
			       + lyxrc.popup_font_encoding;

	int bold = fl_set_font_name(FL_BOLD_STYLE, boldfontname.c_str());
	int normal = fl_set_font_name(FL_NORMAL_STYLE, fontname.c_str());
	if (bold < 0)
		lyxerr << "Could not set menu font to "
		       << boldfontname << endl;

	if (normal < 0)
		lyxerr << "Could not set popup font to "
		       << fontname << endl;

	if (bold < 0 && normal < 0) {
		lyxerr << "Using 'helvetica' font for menus" << endl;
		boldfontname = "-*-helvetica-bold-r-*-*-*-?-*-*-*-*-iso8859-1";
		fontname = "-*-helvetica-medium-r-*-*-*-?-*-*-*-*-iso8859-1";
		bold = fl_set_font_name(FL_BOLD_STYLE, boldfontname.c_str());
		normal = fl_set_font_name(FL_NORMAL_STYLE, fontname.c_str());

		if (bold < 0 && normal < 0) {
			lyxerr << "Could not find helvetica font. Using 'fixed'." << endl;
			fl_set_font_name(FL_NORMAL_STYLE, "fixed");
			normal = bold = 0;
		}
	}
	if (bold < 0)
		fl_set_font_name(FL_BOLD_STYLE, fontname.c_str());
	else if (normal < 0)
		fl_set_font_name(FL_NORMAL_STYLE, boldfontname.c_str());

	// put here (after fl_initialize) to avoid segfault. Cannot be done
	// in setDefaults() (Matthias 140496)
	// Moved from ::LyXGUI to ::init to allow popup font customization
	// (petr 120997).
	fl_setpup_fontstyle(FL_NORMAL_STYLE);
	fl_setpup_fontsize(FL_NORMAL_SIZE);
	fl_setpup_color(FL_MCOL, FL_BLACK);
	fl_set_goodies_font(FL_NORMAL_STYLE, FL_NORMAL_SIZE);
#if FL_REVISION < 89
	fl_set_oneliner_font(FL_NORMAL_STYLE, FL_NORMAL_SIZE);
#else
	fl_set_tooltip_font(FL_NORMAL_STYLE, FL_NORMAL_SIZE);
#endif

	// all lyxrc settings has to be done here as lyxrc has not yet
	// been read when the GUI is created (Jug)

	// Update parameters.
	lyxViews->redraw();

	// Initialize the views.
	lyxViews->init();

	// this should be moved ...
	lyxserver = new LyXServer(lyxViews->getLyXFunc(), lyxrc.lyxpipes);
}


void LyXGUI::create_forms()
{
	lyxerr[Debug::INIT] << "Initializing LyXView..." << endl;
	lyxViews = GUIRunTime::createMainView(width, height);
	lyxerr[Debug::INIT] << "Initializing LyXView...done" << endl;

	// From here down should be done by somebody else. (Lgb)

	// This is probably as good a time as any to map the xform colours,
	// should a mapping exist.
	string const filename = AddName(user_lyxdir, "preferences.xform");
	XformsColor::read(filename);

	// Show the main & title form
	int main_placement = FL_PLACE_CENTER | FL_FREE_SIZE;
	// Did we get a valid position?
	if (xpos >= 0 && ypos >= 0) {
		lyxViews->setPosition(xpos, ypos);
		main_placement = FL_PLACE_POSITION;
	}

	lyxViews->show(main_placement, FL_FULLBORDER, "LyX");
}


void LyXGUI::runTime()
{
	if (!gui) return;

	GUIRunTime::runTime();
}


void LyXGUI::regBuf(Buffer * b)
{
	lyxViews->view()->buffer(b);
}


LyXView * LyXGUI::getLyXView() const
{
	return lyxViews;
}
