/***************************************************************************
 * LPRng - An Extended Print Spooler System
 *
 * Copyright 1988-1995 Patrick Powell, San Diego State University
 *     papowell@sdsu.edu
 * See LICENSE for conditions of use.
 *
 ***************************************************************************
 * MODULE: utils.c
 * PURPOSE: BSD UNIX compatibility routines
 **************************************************************************/

static const char *const _id =
"utils.c,v 1.3 1998/03/29 23:49:29 papowell Exp";

/*******************************************************************
 * Some stuff for Solaris and other SVR4 impls; emulate BSD sm'tics.
 * feel free to delete the UCB stuff if we need to tighten up the
 * copyright -- it's just there to help porting.
 *
 * Patrick Powell Thu Apr  6 20:08:56 PDT 1995
 * Taken from the PLP Version 4 Software Distribution
 ******************************************************************/

#include "psfilter.h"
#include "timeout.h"
#include "errorcodes.h"

int udp_open( char *device );

/**************************************************************
 * 
 * signal handling:
 * SIGALRM should be the only signal that terminates system calls;
 * all other signals should NOT terminate them.
 * This signal() emulation function attepts to do just that.
 * (Derived from Advanced Programming in the UNIX Environment, Stevens, 1992)
 *
 **************************************************************/

void summary( char *s );

#ifdef HAVE_SIGACTION

/* solaris 2.3 note: don't compile this with "gcc -ansi -pedantic";
 * due to a bug in the header file, struct sigaction doesn't
 * get declared. :(
 */

plp_sigfunc_t plp_signal (int signo, plp_sigfunc_t func)
{
	struct sigaction act, oact;

	act.sa_handler = func;
	(void) sigemptyset (&act.sa_mask);
	act.sa_flags = 0;
	if (signo == SIGALRM) {
#ifdef SA_INTERRUPT
		act.sa_flags |= SA_INTERRUPT;           /* SunOS */
#endif
	} else {
#ifdef SA_RESTART
		act.sa_flags |= SA_RESTART;             /* SVR4, 4.3+BSD */
#endif
	}
	if (sigaction (signo, &act, &oact) < 0) {
		return (SIG_ERR);
	}
	return (plp_sigfunc_t) oact.sa_handler;
}

#else /* HAVE_SIGACTION */

plp_sigfunc_t plp_signal (int signo, plp_sigfunc_t func)
{
	/* sigaction is not supported. Just set the signals. */
	return (plp_sigfunc_t)signal (signo, func); 
}
#endif

/*
 * Time_str: return "cleaned up" ctime() string...
 *
 * Thu Aug 4 12:34:17 BST 1994 -> Aug  4 12:34:17
 * Thu Aug 4 12:34:17 BST 1994 -> 12:34:17
 */

char *Time_str()
{
    time_t tvec;
    static char s[99];
	char *t;

	tvec = time( (void *) 0 );
	t = ctime (&tvec);
    (void)strcpy(s, t);
	t = &s[4];
	s[19] = 0;
	return(t);
}

/***************************************************************************
 * Utility functions: write a string to a fd (bombproof)
 *   write a char array to a fd (fairly bombproof)
 * Note that there is a race condition here that is unavoidable;
 * The problem is that there is no portable way to disable signals;
 *  post an alarm;  <enable signals and do a read simultaneously>
 * There is a solution that involves forking a subprocess, but this
 *  is so painful as to be not worth it.  Most of the timeouts
 *  in the LPR stuff are in the order of minutes, so this is not a problem.
 *
 * Note: we do the write first and then check for timeout.
 ***************************************************************************/

int Write_fd_str( int fd, char *msg )
{
	int len, i;

	i = len = strlen(msg);
	while( len > 0 && (i = write( fd, msg, len ) ) > 0 ){
		len -= i, msg += i;
		if( Timeout_pending && Alarm_timed_out){
			i = -1;
			break;
		}
	}
	return( i );
}

int Write_fd_len( int fd, char *msg, int len )
{
	int i;

	i = len;
	while( len > 0 && (i = write( fd, msg, len ) ) > 0 ){
		len -= i, msg += i;
		if( Timeout_pending && Alarm_timed_out){
			i = -1;
			break;
		}
	}
	return( i );
}

/* setstatus: put a message in the "status" file, if available. */

int Max_buffer = 10;
static int size;
static int min_size;
static char *save;
static int active;

/* VARARGS2 */
#ifdef HAVE_STDARGS
void setstatus (char *msg,...)
#else
void setstatus (va_alist) va_dcl
#endif
{
#ifndef HAVE_STDARGS
	char *msg;
#endif
	int i, len;
	static int fd;
	static int first_call;
	int err;
	struct stat statb;
	char buf[512];
	char *s;
	VA_LOCAL_DECL
	VA_START (msg);
	VA_SHIFT (msg, char *);

	if( active ) return;
	++active;
	plp_snprintf( buf, sizeof(buf), "%s ", name );
	if(debug>2){
		len = strlen(buf);
		plp_snprintf( buf+len, sizeof(buf)-len, " [%d] ", getpid() );
	}
	len = strlen(buf);
	(void) vplp_snprintf( buf+len, sizeof(buf)-2-len, msg, ap);
	if( strchr(buf,'\n') == 0 ){
		safestrncat( buf, "\n" );
	}
	if( first_call == 0 ){
		if( Max_buffer == 0 ) Max_buffer = 10;
		size = Max_buffer * 1024;
		min_size = size/4;
		save = malloc(min_size+1);
		if( save == 0 ){
			logerr_die("malloc %d failed", min_size + 1);
			exit( JABORT );
		}
		if( statusfile && *statusfile){
			fd = open( statusfile, O_RDWR | O_APPEND );
			err = errno;
			if( fd < 0 ){
				fprintf(stderr, "couldn't open '%s': %s\n",
					statusfile, Errormsg(err) );
			}
		}
		first_call = 1;
		if( fstat( fd, &statb ) < 0 ){
			fprintf( stderr, "setstatus: fstat failed - '%s'\n",
				Errormsg(errno) );
			exit( JABORT );
		}
		if( statb.st_size > size ){
			if( lseek( fd, statb.st_size - min_size, SEEK_SET ) < 0  ){
				logerr_die( "setstatus: lseek failed" );
			}
			for( s = save, len = min_size;
				len > 0 && (i = read( fd, s, len )) > 0;
					s += i, len -= i );
			*s = 0;
			if( (s = strchr( save, '\n' )) ){
				++s;
			} else {
				s = save;
			}
			/* fprintf( stderr, "setstatus: truncating\n" ); */
			if( lseek(fd, 0, 0 ) < 0 ){
				logerr_die( "setstatus: lseek failed" );
			}
			if( ftruncate(fd, 0 ) < 0 ){
				logerr_die( "setstatus: truncate failed" );
			}
			Write_fd_str( fd, s );
		}
	}
	if( fd > 0 ){
		Write_fd_str( fd, buf );
	}
	if(debug>2){
		Write_fd_str( 2, buf );
	}
	if( summaryfile ){
		summary( buf );
	}
	--active;
}

static int summary_fd = -1;
void summary( char *s )
{
	char *str;
	int err;
	if( summary_fd < 0 ){
		if( summary_fd == -2 ) return;
		if( strpbrk( summaryfile, "%@" ) ){
			summary_fd = udp_open( summaryfile );
		} else {
			summary_fd = open( summaryfile, O_RDWR|O_CREAT|O_APPEND, 0644 );
			err = errno;
			if( summary_fd < 0 ){
				fprintf( stderr, "could not open '%s' - %s", summaryfile,
					Errormsg(err) );
				return;
			}
		}
		if( summary_fd < 0 ){
			summary_fd = -2;
			return;
		}
	}
	/* truncate the file - note - must be open R/W on some systems... */
	(void)ftruncate(summary_fd,0);
	if( (str = strchr( s, ':' )) ){
		++str;
		while( isspace( *str ) ) ++str;
		s = str;
	}
	Write_fd_str( summary_fd, s );
}

int udp_open( char *device )
{
	int port, i, fd, err;
	struct hostent *hostent;
	struct sockaddr_in sin;
	struct servent *servent;
	char *s;

	if(debug>2)setstatus( "udp_open: '%s'\n",device );
	if( (s = strpbrk( device, "@%" )) == 0 ){
		setstatus( "udp_open: missing port number '%s'\n",device );
		return( -1 );
	}
	if( strpbrk( s+1, "@%" ) ){
		setstatus( "udp_open: two '@' or '%' in name '%s'\n",
			device );
		return( -1 );
	}
	port = atoi( s+1 );
	if( port <= 0 ){
		servent = getservbyname( s+1, "udp" );
		if( servent ){
			port = ntohs( servent->s_port );
		}
	}
	if( port <= 0 ){
		setstatus( "udp_open: bad port number '%s'\n",s+1 );
		return( -1 );
	}
	i = *s;
	*s = 0;
	sin.sin_family = AF_INET;
	sin.sin_addr.s_addr = -1;

	if( (hostent = gethostbyname(device)) ){
		/*
		 * set up the address information
		 */
		if( hostent->h_addrtype != AF_INET ){
			setstatus( "udp_open: bad address type for host '%s'\n",
				device);
		}
		memcpy( &sin.sin_addr, hostent->h_addr, hostent->h_length );
	} else {
		sin.sin_addr.s_addr = inet_addr(device);
	}
	*s = i;
	if( sin.sin_addr.s_addr == -1){
		setstatus("udp_open: unknown host '%s'\n", device);
		return(-1);
	}
	sin.sin_port = htons( port );
	if(debug>2) setstatus( "udp_open: destination '%s' port %d\n",
		inet_ntoa( sin.sin_addr ), ntohs( sin.sin_port ) );
	fd = socket (AF_INET, SOCK_DGRAM, 0);
	err = errno;
	if (fd < 0) {
		setstatus("udp_open: socket call failed - %s\n", Errormsg(err) );
		return( -1 );
	}
	i = connect (fd, (struct sockaddr *) & sin, sizeof (sin));
	err = errno;

	if( i < 0 ){
		setstatus("udp_open: connect to '%s port %d' failed - %s\n",
			inet_ntoa( sin.sin_addr ), ntohs( sin.sin_port ),
			Errormsg(errno) );
		close(fd);
		fd = -1;
	}
	return( fd );
}
