/***************************************************************************
 * LPRng - An Extended Print Spooler System
 *
 * Copyright 1988-1995 Patrick Powell, San Diego State University
 *     papowell@sdsu.edu
 * See LICENSE for conditions of use.
 * From the original PLP Software distribution
 *
 ***************************************************************************
 * MODULE: sync.c
 * PURPOSE:  synchronize  the Postscript Filter
 **************************************************************************/

#ifndef lint
static const char *const _id =
	"sync.c,v 1.7 1997/12/18 09:45:47 papowell Exp";
#endif

#include "psfilter.h"
#include "timeout.h"
#include "errorcodes.h"
#if !defined(ultrix) && defined(HAVE_SYSLOG_H)
# include <syslog.h>
#endif
#if defined(HAVE_SYS_SYSLOG_H)
# include <sys/syslog.h>
#endif

#include <syslog.h>

static char lastprintererr[128];

/***************************************************************************
 * printer status:
 *  This has the format  '%%[ key: value; key:value ]%%'
 *  We parse a return status and then look at the fields
 ***************************************************************************/

/**************************************************************************
 *  key tags
 *   i.e. - key: value
 **************************************************************************/

#define PAIR(X) {#X,X}
struct errorkey msg_key[] = {
	PAIR(BAD_MSG),
	PAIR(STATUS),
	PAIR(ERROR),
	PAIR(EXITSERVER),
	PAIR(FLUSHING),
	PAIR(JOB),
	PAIR(PRINTERERROR),
	PAIR(SOURCE),
	PAIR(PAGECOUNT),
	{0}
};

/**************************************************************************
 * Status values
 **************************************************************************/
struct errorkey status_key[] = {
	PAIR(BAD_STATUS),
	PAIR(IDLE),
	PAIR(BUSY),
	PAIR(PRINTING),
	PAIR(WAITING),
	PAIR(PRINTERROR),
	PAIR(INITIALIZING),
	{0}
};


/*
 * read_pr.  This is more or less a non-blocking line-buffered version of read.
 *   In theory we could use stdio, except that nobody is quite sure how it
 *   interacts with non-blocking I/O and select.
 * buf must be at least as long as statusbuflen.  This keeps us from having
 *   to return only partial data, which would complicate the coding and
 *   testing beyond what I'm willing to accept at the moment.
 * return: 
 *   -1 if error, errno == EAGAIN if there's nothing to read, which typically
 *       means we got something but it wasn't a full line.  Caller should
 *       check for EAGAIN and EINTR if you want to be portable.  (They're
 *       the same for some systems.)
 *   0 if select timeout
 *   >0 for data
 */

 /* in will not be allowed to catch up with out, so if they're
  * equal, it means the buffer is empty
  */

static char prbuf[STATUSBUFLEN+1];  /* +1 is to get us a null at the end */

int read_pr( char *buf, int max_buf, int timeout)
{
	int len, fd, i, add_lf;
	char *cp;
	int first;
	fd_set readfds;        /* input ready ? */
	struct timeval delay;   /* timeout if needed */
	int status;                           /* status of last try */
	int err;                              /* last error status */

	first = 1;

	/* see if we have a line */
	while( 1 ){
		while( (cp = strpbrk(prbuf, "\015\012\004")) ){
			*cp++ = 0; /* convert end of line to null  */
			/* copy to buffer */
			strncpy(buf, prbuf, max_buf);
			/* move to start of buffer */
			for( i = 0; (prbuf[i] = cp[i]); ++i );
			/* check for empty string */
			if( *buf ) return(1);
		}
		/* throw away long strings */
		if( strlen( prbuf ) >= max_buf ){
			prbuf[0] = 0;
		}

		FD_ZERO( &readfds );
		FD_SET( 1, &readfds );
		FD_SET( Udp_status_sock, &readfds );

		/* clear timeout */
		memset( &delay, 0, sizeof(delay) );
		delay.tv_sec = timeout;  /* 10 seconds wait */

		status = select( 20, (void *)&readfds, (void *)0, (void *)0, &delay );
		err = errno;
		if(debug>2)setstatus( "read_pr: select status %d at %s\n", status, Time_str() );
		if( status < 0 ){
			if(debug>2)setstatus( "read_pr: select failed '%s' at %s\n",
				Errormsg(err), Time_str() );
			errno = err;
			return ( -1 );
		} else if( status == 0 ){ /* select timed out */
			  return 0;
		}

		len = strlen(prbuf);
		cp = prbuf+len;

		add_lf = 0;
		if( FD_ISSET( Udp_status_sock, &readfds ) ){
			add_lf = 1;
			fd = Udp_status_sock;
		} else {
			fd = 1;
		}
		i = read( fd, cp, sizeof(prbuf)-1-len );
		err = errno;

		if( i < 0  ) {
			setstatus( "read_pr: read failed '%s' at %s\n",
			Errormsg(err), Time_str() );
			errno = err;
			return -1;
		}
		cp[i] = 0;
		if(debug>2)setstatus( "read_pr: read %d, '%s' at %s\n", i, cp, Time_str());
		if(i == 0){ /* probably ^D. */
			if( !isatty( fd ) ){
				setstatus( "read_pr: possible eof from printer '%s'\n",
					Time_str() );
				errno = EIO;
				/* not sure what error we should give here, but the
				 * original code treats it as an error */
				return( -1 );
			 }
			 cp[0] = '\n';
			 cp[1] = 0;
			/* original code ignores it, presumably considering it
			 * as a blank line.  I think it's safer to make it an
			 * explicit end of line, since the caller throws away
			 * blank lines. */
			 i = 1;
		} else if( add_lf ){
			cp[i++] = '\n';
			cp[i] = 0;
		}
	}
}

/**************************************************************************
 * char *key_str( struct errorkey *table, int value )
 * translate from value to a key
 *  input: table to use and value to find
 *  output: string
 **************************************************************************/
char *key_str( struct errorkey *table, int value )
{
	char *s;
	static char msg[40];
	while( (s = table->key) && value != table->value ) ++table; 
	if( s == 0 ){
		sprintf( msg, "value %d not key", value );
		s = msg;
	}
	return( s );
}

/***************************************************************************
 *int sync_pr(
 *  int timeout,    - timeout between queries
 *  int max_try,    - max attempts
 *  int sync,       - send sync (^T)
 *  int end_job,    - send endjob (^D)
 *  char *msg,      - send this message
 *	int maxresponse, - max time to wait for a response
 *  int clear_no_response - send a ^C if no reponse
 *  int desired_type - type of message wanted
 *  int idle         - wait for idle
 *  int force_ps     - force PostScript actions by sending %! and flush
 *
 *  We query the printer by sending a ^T
 *  if the end_job flag is set we also send a ^D
 *  until we get status echoed back.
 *
 * returns: -1 on error, 0 on success
 *
 ***************************************************************************/

int Write_with_timeout( int fd, char *buffer, int len, int timeout )
{
	static int i;
	i = len;
	if( len > 0 ){
		if( Set_timeout() ){
			Set_timeout_alarm( timeout, 0 );
			i = Write_fd_len( fd, buffer, len );
		}
	}
	Clear_timeout();
	return( i );
}

int sync_pr( int timeout, int max_try, int sync, int end_job_parm, char *msg,
	int maxresponse, int clear_no_response, int desired_type, int idle,
	int force_ps )
{
	int count;		/* number of times we are doing this */
	int status;				/* status of last try */
	struct timeval delay;   /* timeout if needed */
	int i, len;
	int err;				/* last error status */
	int decoded_status;		/* last decoded status */
	int response;			/* interval between last decoded status */
	int clearit;		/* clear if no response */
	int done;				/* we are done */
	int pause;				/* need to pause */
	long start_time;		/* start time */
	long current_time;		/* current time */
	int threshold;
	int end_job;
      char saved_input[STATUSBUFLEN];  /* save incoming PS messages for possible reporting */

	end_job = end_job_parm;
	clearit = 0;

	statusbuf[sizeof(statusbuf)-1] = 0;	/* make sure end flag on string */
	status = -1;

	response = 0;
	if(debug>2)setstatus("sync_pr: timeout %d, max_try %d\n", timeout, max_try );

	/* retry the main loop until we get error or run out of attempts */
	start_time = time( (void *)0 );
	threshold = 60;
	done = 0;
	decoded_status = 0;
	for( count = 0; done == 0 && (max_try == 0 || count < max_try); ++count ){
		if(debug>0)setstatus( "sync_pr: trying '%s' '%s' '%s' '%s' at %s\n",
			sync?"sync":"no sync", msg?"msg":"no msg",
			end_job?"end":"no end", clearit?"clearit":"no clear",
			Time_str() );

		/* set up outgoing message */
		len = 0;
		if( Udp_status_port == 0 ){
			if( clearit ) statusbuf[len++] = CTRL_C;
			if( clearit || end_job ) statusbuf[len++] = CTRL_D;
		}
		clearit = 0;
		end_job = 0;
		pause = 0;
		if( force_ps ){
			strncpy( statusbuf+len, "%!\n", sizeof(statusbuf)-len );
			len += strlen( statusbuf+len );
			strncpy( statusbuf+len, " flush flush\n", sizeof(statusbuf)-len );
			len += strlen( statusbuf+len );
			if( Udp_status_port == 0 ) statusbuf[len++] = CTRL_D;
		}
		if( sync && Udp_status_port == 0 ) statusbuf[len++] = CTRL_T;
		if( msg ){
			strncpy( statusbuf+len, msg, sizeof(statusbuf)-len );
			len = strlen( statusbuf );
		}
		statusbuf[len] = 0;
		if(debug>2) setstatus( "sync_pr: writing %d '%s' at %s\n",
			len, statusbuf, Time_str() );

		/* do write to stdout */
		if( len > 0 ){
			i = Write_with_timeout( 1, statusbuf, len, timeout );
			err = errno;
			if( Alarm_timed_out ){
				if(debug>2) setstatus( "sync_pr: write stdout time out at %s\n",
					Time_str() );
			} else if( i <= 0  ){
				if(debug>2) setstatus( "sync_pr: write stdout failed '%s' at %s\n",
					Errormsg(err), Time_str() );
				if( err != EINTR && err != EAGAIN ){
					fatal( "sync_pr: write failed '%s' at %s\n",
						Errormsg(err), Time_str() );
				}
			}
		}
		/* do write to status port */
		if( Udp_status_port && sync ){
			i = Write_with_timeout( Udp_status_sock, "\n", 1, timeout );
			err = errno;
			if( Alarm_timed_out ){
				if(debug>2) setstatus( "sync_pr: write statusport time out at %s\n",
					Time_str() );
			} else if( i <= 0  ){
				if(debug>2) setstatus( "sync_pr: write statusport failed '%s' at %s\n",
					Errormsg(err), Time_str() );
				if( err != EINTR && err != EAGAIN ){
					fatal( "sync_pr: write failed '%s' at %s\n",
						Errormsg(err), Time_str() );
				}
			}
		}
		response = 0;
		while(done == 0){
			if(debug>2)setstatus(
"sync_pr: done %d, timeout %d, decoded_status %s, response %d, maxresponse %d at %s\n",
			done, timeout, key_str(msg_key,decoded_status),
			response, maxresponse, Time_str() );

			i = read_pr( statusbuf, sizeof(statusbuf)-1, timeout);
			if (i < 0) {
				err = errno;
				if (err == EAGAIN || err == EINTR)
						continue;
				setstatus( "sync_pr: error '%s' at %s\n",
						Errormsg(err), Time_str() );
			  	fatal("sync_pr: read error - '%s'\n", Errormsg(err));
			} else if (i == 0) {  /* select timeout */
				if(debug>0)setstatus( "sync_pr: select timeout at %s\n",
					Time_str() );
				break;
			}

			i = strlen(statusbuf);
			while( i-- > 0 && isspace( statusbuf[i] ) ) statusbuf[i] = 0;

			/* Save the original status message since decode_status() destroys it */
			strncpy(saved_input, statusbuf, (sizeof(saved_input)-1));

			/* decode the status */
			if(debug>2) setstatus( "sync_pr: decoding '%s', strlen %d at %s\n",
				  statusbuf, strlen(statusbuf), Time_str() );
			/* ignore blank lines and non-status lines */
			if( statusbuf[0] ){
				decoded_status = decode_status(statusbuf, desired_type );

                                if (ReportPSOutput) {
                                        if ( (decoded_status != STATUS) &&
                                                (decoded_status != PAGECOUNT)) {
 
                                                fprintf(stderr, "From printer: %s\n",
                                                        saved_input);
                                        }
                                }

				if(debug>2) setstatus(
			"sync_pr: decode_status result '%s', wanted %s, new_status '%s'\n",
					key_str(msg_key,decoded_status),
					key_str(msg_key,desired_type),
					key_str(status_key,New_status) );
				done = pause = 0;
				/* check to see if the status was the correct one */
				if( decoded_status > 0 ){
					response = 1;
					if( decoded_status == desired_type ){
						done = 1;
						/* if we have a status request, check for idle
							or waiting */
						if( decoded_status == STATUS ){
							switch( New_status ){
							case IDLE:
							case WAITING:
								break;
							default:
								if( idle ){
									pause = 1;
									done = 0;
	/* clh - clear end of job.  One is enough.  Some printers are busy briefly
	 * after an endjob, so we need ^D ^T ^T.  If we do ^D each time we'll
	 * never get idle
	 */
								}
								break;
							}
						}
					}
				}
			}
		}
		if( done == 0 && response == 0 ){
			/* clear timeout */
			current_time = time( (void *)0 );
			i = (current_time - start_time);
			if( i > threshold ){
				if (lastprintererr[0]){
					setstatus("sync_pr: no response in %d secs; last printer error: %s\n",
						 i, lastprintererr);
				} else {
					setstatus("sync_pr: no response in %d secs\n", i);
				}
				threshold = ((i/60)+1)*60;
			}
			if( maxresponse && i > maxresponse ){
				setstatus( "no reponse from '%s' in %d secs\n",
					printer, response );
#ifdef CLH
				syslog( LOG_CRIT|LOG_LPR, "no reponse from '%s' in %d secs",
					printer?printer:"psfilter", response );
#endif
				if( clear_no_response ) clearit = 1;
				end_job = 1;
				response = 0;
			}
			if( pause ){
				/* now we do a delay */
				memset( &delay, 0, sizeof(delay) );
				delay.tv_sec = timeout;  /* wait */
				if(debug>2)setstatus("sync_pr: pausing %d\n",
					delay.tv_sec);
				select( 20, (void *)0,
					(void *)0, (void *)0, &delay );
				response += timeout;
			}
		}
	}
	if(debug>2)setstatus( "sync_pr: done (return %d) at %s\n",
		done, Time_str() );
	return( done );
}

/***************************************************************************
 * int decode_status( char *buf )
 *  split the status line up into fields
 *  return: -1 if failure, 0 if no status, type of status if success
 ***************************************************************************/
int decode_status( char *s, int desired_type )
{
	char *field, *end, *endkey;
	int i, j, c;
	int status = 0;		/* status to find */
	/* first we break into fields */

	fieldcount = 0;
	if(debug>2)setstatus("decode_status: '%s', len %d\n",
		s, strlen(s) );
	if( (endkey = strstr( s, "%%[" )) ){
		s = endkey+3;
	}
	/* find the end of the status */
	endkey = strstr( s, "]%%" );
	if( endkey ){
		*endkey = 0;
	}

	lastprintererr[0] = '\0';
	
	/* zero out returned status */
	memset( &fields, 0, sizeof(fields) );
	/* work down through the status */
	for(fieldcount = 0;
		s && *s && fieldcount < sizeof(fields)/sizeof(fields[0]);
		s = end, ++fieldcount ){
		end = strchr( s, ';' );
		if( end ){
			*end++ = 0;
		}
		while( isspace(*s) ) ++s;
		endkey = strchr( s, ':' );
		if( endkey ){
			*endkey++ = 0;
			while( isspace(*endkey) ) ++endkey;
		}
		fields[fieldcount].key = s;
		fields[fieldcount].value = endkey;
		if( endkey ){
			c = strlen(endkey);
			while( c-->0 && isspace( endkey[c] ) ) endkey[c] = 0;
		}
	}

	if(debug>2) for( i = 0; i < fieldcount; ++i ){
		setstatus("decode_status: [%d] key '%s' value '%s'\n", 
			i, fields[i].key, fields[i].value );
	}

	New_status = 0;
	New_type = 0;
	for( i = 0; i < fieldcount; ++i ){
		field = fields[i].key;
		/* now we try to find the message type */
		for( j = 0; (s = msg_key[j].key) && strcasecmp(s, field); ++j );
		if( s ){
			c = fields[i].type = msg_key[j].value; 
			field = fields[i].value;
			if(debug>2)setstatus("decode_status: key '%s'->'%s'\n",
				s, key_str(msg_key,c) );

			if ( c == desired_type ) {
				New_type = c;
			}

			switch( c ){
			case STATUS:
				if (strncasecmp (field, "PrinterError:", 13) == 0) {
				  char *errmsg;

				  errmsg = field + 13;
				  while (isspace(*errmsg)) errmsg++;
				  setstatus("decode_status: printer '%s' error '%s'\n",
					printer, errmsg );
				  strncpy(lastprintererr, errmsg, 
					sizeof(lastprintererr) - 1);
				}
				
				for( j = 0; (s = status_key[j].key) && strcasecmp(s, field);
					++j );
				if( s ){
					status = fields[i].status = status_key[j].value; 
					if(debug>2)setstatus("decode_status: new_status '%s'->'%s'\n",
						s, key_str(status_key,status) );
					New_status = Last_status = status;
				}
				break;
			case PRINTERERROR:
				setstatus("decode_status: printer '%s' error '%s'\n",
					printer, field );
				strncpy(lastprintererr, field, 
					sizeof(lastprintererr) - 1);

#if !defined(CLH)
				syslog( LOG_CRIT|LOG_LPR, "PRINTER '%s' ERROR '%s'",
					printer, field );
#endif
				break;
			case PAGECOUNT:
				npages = atoi(field);
				setstatus("decode_status: PAGECOUNT '%s'=%d\n",
					field, npages );
				break;
			}
		}
	}

	/* If desired_type (above) was not found, then New_type
		is still 0.  Return type of first field instead. */
	if( (!New_type) && (fieldcount > 0) ){
		New_type = fields[0].type;
	}
	return( New_type );
}

int find_status()
{
	int i;
	for( i = 0; i < fieldcount; ++i ){
		if( fields[i].type == STATUS ) return( fields[i].status ); 
	}
	return( 0 );
}


/***************************************************************************
 * int getpagecount()
 * send a get pagecount to the printer
 * returns: -1 if no count
 *          page count otherwise
 ***************************************************************************/

static char pagecount[] = {
"%!\n"	/* defeats auto sensing printers */
"(%%[ pagecount: )print statusdict /pagecount get exec "
"(                )cvs print "
"( ]%%) = flush\n"
};

int getpagecount( int timeout, int max_try, int max_response, int endpause,
		int nosync, int force_ps )
{
	int i, attempt;
	int opages = -1;

	npages = -1;
	if(debug>0)setstatus(
		"getpagecount: timeout %d, max_try %d, max_resp %d, endpause %d, nosync %d, force_ps %d",
			timeout, max_try, max_response, endpause, nosync, force_ps );
	for( attempt = 0; npages < 0
		&& (max_try == 0 || attempt < max_try); ++attempt ){
		New_status = -1;
		if(debug>0)setstatus( "getpagecount: attempt %d\n", attempt+1 );
		/* we wait until the printer is idle and/or force it idle */
		
		if( nosync == 0 ){				
			i = sync_pr( timeout,0,1,1,(char *)0,max_response,1,STATUS,1,force_ps); 
			if(debug>0)setstatus(
				"getpagecount: check for idle status returned %d, status '%s'\n",
					i, key_str(status_key,New_status) );
			/* check to see if we have finished the job */
			if( New_status == IDLE || New_status == WAITING ){
				if( endpause > 0 ){
					if(debug>0)setstatus(
				"getpagecount: pausing %d secs for job flush\n", endpause );
					sleep(endpause);
				}
			} else {
				continue;
			}
		}
		/* now we send out a getpagecount */
		if(debug>0)setstatus( "getpagecount: doing pagecount query\n" );
		i = sync_pr( timeout, 0, 0, 0, pagecount,max_response,
			0,PAGECOUNT, 0, force_ps);
		if(debug>0)setstatus("getpagecount: pagecount query status %d\n", i);
		if(debug>2)setstatus("getpagecount: old %d, new %d\n", opages, npages );
		if( npages > 0 && endpause > 0 && opages != npages ){
			sleep( endpause );
			if(debug>2)setstatus("getpagecount: end pause %d at %s\n", endpause,
				Time_str() );
			opages = npages;
		}
	}
	return( npages );
}

/***************************************************************************
 * void do_status( char *stop )
 *  set up a timer (10 secs) and periodically send a status request
 *  to the printer.  Wait for the request.
 ***************************************************************************/

void do_status( int interval, int timeout )
{
	fd_set readfds;	/* input ready ? */
	struct timeval delay;   /* timeout if needed */
	int status;
	int err, ppid, maxport, i;
	int outport = 1;
	int decoded_status;		/* last decoded status */

	if(debug>1)setstatus(
		"do_status: starting, interval %d, timeout %d at %s\n",
		interval, timeout, Time_str() );
	if( (Statuspid = fork()) < 0 ){
		err = errno;
		setstatus("filter: fork failed - %s\n", Errormsg(err) );
		cleanup_job(JABORT);
	} else if( Statuspid ){
		/* get the status from the printer */
		return;
	}
	if( Udp_status_port ){
		outport = Udp_status_sock;
	}
	maxport = outport+1;
	ppid = getppid();
	while(1){
		if(debug>2) setstatus( "do_status: waiting for status at %s\n",
			Time_str() );
		/* check to see if the parent was killed off */
		if( kill( ppid, 0 ) < 0 ){
			if(debug>2) setstatus( "do_status: parent died at %s\n",
				Time_str() );
			exit(0);
		}
		/* set up wait between queries */
		FD_ZERO( &readfds );
		FD_SET( outport, &readfds );
		memset( &delay, 0, sizeof(delay) );
		delay.tv_sec = interval;  /* wait */
		status = select( 20, (void *)&readfds,
			(void *)0, (void *)0, &delay );
		err = errno;
		/* check for input or timeout */
		if( status < 0 ){
			if( err != EINTR ){
				setstatus( "do_status: select error - %s\n",
					Errormsg(err) );
				exit(0);
			}
		} else if( status == 0 ){
			/* do write to stdout */
			statusbuf[0] = CTRL_T;
			i = Write_with_timeout( outport, statusbuf, 1, timeout );
			err = errno;
			if( Alarm_timed_out ){
				if(debug>2) setstatus( "do_status: write stdout time out at %s\n",
					Time_str() );
			} else if( i <= 0  ){
				if(debug>2) setstatus( "do_status: write stdout failed '%s' at %s\n",
					Errormsg(err), Time_str() );
				if( err != EINTR && err != EAGAIN ){
					logerr( "do_status: write failed '%s' at %s\n",
						Errormsg(err), Time_str() );
					exit(0);
				}
			}
		} else {
			/* see what status we have out there */
			do{
				i = read_pr( statusbuf, sizeof(statusbuf)-1, timeout);
				if (i < 0) {
					err = errno;

					if (err == EAGAIN || err == EINTR) continue;

					setstatus( "do_status: error '%s' at %s\n",
							Errormsg(err), Time_str() );
					fatal("do_status: read error - '%s'\n", Errormsg(err));
				} else if (i == 0) {  /* select timeout */
					if(debug>0)setstatus( "do_status: select timeout at %s\n",
						Time_str() );
				}
				if(debug>2) setstatus( "do_status: decoding '%s', strlen %d at %s\n",
					  statusbuf, strlen(statusbuf), Time_str() );
				/* ignore blank lines and non-status lines */
				decoded_status = decode_status(statusbuf, 0 );
				if(debug>2) setstatus(
					"do_status: decode_status result '%s'\n",
						key_str(msg_key,decoded_status) );
			} while( i > 0 );
		}
	}
}
