/***************************************************************************
 * LPRng - An Extended Print Spooler System
 *
 * Copyright 1988-1995 Patrick Powell, San Diego State University
 *     papowell@sdsu.edu
 * See LICENSE for conditions of use.
 * From the original PLP Software distribution
 *
 ***************************************************************************
 * MODULE: psfilter.c
 * PURPOSE:  setup a filter environment
 **************************************************************************/

/***************************************************************************
 * MODULE: main.c for filters
 ***************************************************************************
 * Revision History: Baseline Fri Jul 14 16:16:34 PDT 1995
 * 
 * Patrick Powell Fri Jul 14 16:16:13 PDT 1995 
 *
 ***************************************************************************/
#ifndef lint
static const char *const _id =
	"psfilter.c,v 1.8 1998/03/29 23:49:29 papowell Exp";
#endif

/***************************************************************************
 *  Filter template and frontend.
 *
 *	A filter is invoked with the following parameters,
 *  which can be in any order, and perhaps some missing.
 *
 *  filtername
 *    -T[debug=level][,dev=device][,stty=stty options][,reverse][summary=addr]*
 *    [-c] [-?option] [arguments]
 * dev=/device   - local device
 * dev=host%port - remote connection
 * summary=filename - file name
 * summary=host%port or host@port - UDP connection
 * endpause=secs   - time needed for job to be completed before
 *                   getting pagecount from printer
 * 
 *  1. Parameters can be in different order than the above.
 *  2. Optional parameters can be missing
 *  3. Values specified for the width, length, etc., are from PRINTCAP
 *     or from the overridding user specified options.
 *
 * VARIABLE  FLAG          TYPE    PURPOSE / PRINTCAP ENTRTY
 * name     name of filter char*    argv[0], program identification
 * width    -wwidth	       int      PW, width in chars
 * length   -llength	   int      PL, length in lines
 * xwidth   -xwidth        int      PX, width in pixels
 * xlength  -xlength       int      PY, length in pixels
 * literal  -c	           int      if set, ignore control chars
 * controlfile -kcontrolfile char*  control file name
 * bnrname  -Lbnrname      char*    banner name
 * indent   -iindent       int      indent amount (depends on device)
 * zopts    -Zoptions      char*    extra options for printer
 * comment  -Scomment      char*    printer name in comment field
 * class    -Cclass        char*    classname
 * job      -Jjob          char*    jobname
 * accntname -Raccntname   char*    account for billing purposes
 * login    -nlogin        char*    login name
 * host     -hhost         char*    host name
 * format   -Fformat       char*    format
 * statusfile  -sstatusfile   char*    statusfile
 * accntfile file          char*    AF, accounting file
 *
 * npages    - number of pages for accounting
 * debug     - sets debug level
 * verbose   - echo to a log file
 *
 *	The functions fatal(), logerr(), and logerr_die() can be used to report
 *	status. The variable errorcode can be set by the user before calling
 *	these functions, and will be the exit value of the program. Its default
 *	value will be 2 (abort status).
 *	fatal() reports a fatal message, and terminates.
 *	logerr() reports a message, appends information indicated by errno
 *	(see perror(2) for details), and then returns.
 *	logerr_die() will call logerr(), and then will exit with errorcode
 *	status.
 *	Both fatal() and logerr_die() call the cleanup_job() function before exit.
 *
 * DEBUGGING:  a simple minded debugging version can be enabled by
 * compiling with the -DDEBUG option.
 */

#define EXTERN

#include "psfilter.h"
#include "timeout.h"
#include "errorcodes.h"

#if defined(HAVE_TERMIOS_H)
# include <termios.h>
#endif

void do_exit(int signal)
{
	if(debug>3)setstatus("** pid %d signal %d\n", getpid(), signal);
	cleanup_job(JABORT);
}

/*
 * default exit status, causes abort
 */

void getargs( int argc, char *argv[], char *envp[] );

int main( int argc, char *argv[], char *envp[] )
{
	Argv_p = argv;
	Envp_p = envp;

	Accounting_fd = dup(0);
	if( Accounting_fd == 4 ){
		close( Accounting_fd );
		Accounting_fd = 3;
	} else {
		close( Accounting_fd);
		Accounting_fd = 0;
	}
    debug = 2;
	getargs( argc, argv, envp );
	/*
	 * Turn off SIGPIPE
	 */
	(void)signal( SIGPIPE, SIG_IGN );
	(void)signal( SIGINT, do_exit );
	if( printer == 0 ) printer = device;
	if( printer == 0 ) printer = "?????";
	fflush(stdout); fflush(stderr);
	if( OFfilter || (format && format[0] == 'o') ){
		filter( filter_stop, Timeout, Max_try, Max_response );
	} else {
		filter( (char *)0, Timeout, Max_try, Max_response );
	}
	return(0);
}

/****************************************************************************
 * Extract the necessary definitions for error message reporting
 ****************************************************************************/

#if !defined(HAVE_STRERROR)
# if defined(HAVE_SYS_NERR)
#   if !defined(HAVE_SYS_NERR_DEF)
      extern int sys_nerr;
#   endif
#   define num_errors    (sys_nerr)
# else
#  	define num_errors    (-1)            /* always use "errno=%d" */
# endif
# if defined(HAVE_SYS_ERRLIST)
#  if !defined(HAVE_SYS_ERRLIST_DEF)
    extern const char *const sys_errlist[];
#  endif
# else
#  undef  num_errors
#  define num_errors   (-1)            /* always use "errno=%d" */
# endif
#endif

const char * Errormsg ( int err )
{
    const char *cp;

#if defined(HAVE_STRERROR)
	cp = strerror(err);
#else
# if defined(HAVE_SYS_ERRLIST)
    if (err >= 0 && err <= num_errors) {
		cp = sys_errlist[err];
    } else
# endif
	{
		static char msgbuf[32];     /* holds "errno=%d". */
		/* SAFE use of sprintf */
		(void) sprintf (msgbuf, "errno=%d", err);
		cp = msgbuf;
    }
#endif
    return (cp);
}

static char errmsg[512];
#ifdef HAVE_STDARGS
void log(char *msg, ...)
#else
void log( va_alist ) va_dcl
#endif
{
#ifndef HAVE_STDARGS
	char *msg;
#endif
	int len;
	VA_LOCAL_DECL
	VA_START(msg);
	VA_SHIFT(msg, char *);
	(void)plp_snprintf(errmsg, sizeof(errmsg), "%s: ", name);
	len = strlen(errmsg);
	(void)vplp_snprintf(errmsg+len, sizeof(errmsg)-len-2, msg, ap);
	len = strlen(errmsg) - 1;
	if( errmsg[len++] != '\n' ){
		errmsg[len++] = '\n';
		errmsg[len] = 0;
	}
	Write_fd_str(2, errmsg);
	VA_END;
}

#ifdef HAVE_STDARGS
void fatal(char *msg, ...)
#else
void fatal( va_alist ) va_dcl
#endif
{
#ifndef HAVE_STDARGS
	char *msg;
#endif
	int len;
	VA_LOCAL_DECL
	VA_START(msg);
	VA_SHIFT(msg, char *);
	fprintf(stderr, "FATAL\n"); fflush(stderr);
	(void)plp_snprintf(errmsg, sizeof(errmsg), "%s: ", name);
	len = strlen(errmsg);
	(void)vplp_snprintf(errmsg+len, sizeof(errmsg)-len-2, msg, ap);
	len = strlen(errmsg) - 1;
	if( errmsg[len++] != '\n' ){
		errmsg[len++] = '\n';
		errmsg[len] = 0;
	}
	Write_fd_str(2, errmsg);
	VA_END;
	cleanup_job(errorcode);
	exit(errorcode);
}

#ifdef HAVE_STDARGS
void logerr(char *msg, ...)
#else
void logerr( va_alist ) va_dcl
#endif
{
#ifndef HAVE_STDARGS
	char *msg;
#endif
	int err = errno;
	int len;
	VA_LOCAL_DECL
	VA_START(msg);
	VA_SHIFT(msg, char *);
	(void)plp_snprintf(errmsg, sizeof(errmsg), "%s: ", name);
	len = strlen(errmsg);
	(void)vplp_snprintf(errmsg+len, sizeof(errmsg)-len-2, msg, ap);
	len = strlen(errmsg);
	(void)plp_snprintf(errmsg+len, sizeof(errmsg)-len-2, " - %s", Errormsg(err) );
	len = strlen(errmsg) - 1;
	if( errmsg[len++] != '\n' ){
		errmsg[len++] = '\n';
		errmsg[len] = 0;
	}
	Write_fd_str(2, errmsg);
	VA_END;
}

#ifdef HAVE_STDARGS
void logerr_die(char *msg, ...)
#else
void logerr_die( va_alist ) va_dcl
#endif
{
#ifndef HAVE_STDARGS
	char *msg;
#endif
	int err = errno;
	int len;
	VA_LOCAL_DECL
	VA_START(msg);
	VA_SHIFT(msg, char *);
	(void)plp_snprintf(errmsg, sizeof(errmsg), "%s: ", name);
	len = strlen(errmsg);
	(void)vplp_snprintf(errmsg+len, sizeof(errmsg)-len-2, msg, ap);
	len = strlen(errmsg);
	(void)plp_snprintf(errmsg+len, sizeof(errmsg)-len-2, " - %s", Errormsg(err) );
	len = strlen(errmsg) - 1;
	if( errmsg[len++] != '\n' ){
		errmsg[len++] = '\n';
		errmsg[len] = 0;
	}
	Write_fd_str(2, errmsg);
	VA_END;
	cleanup_job(errorcode);
}

#define STRV 0
#define INTV 1
#define FLGV 2
struct values {
	char *key;
	char **variable;
	int type;
} device_keys[] = {
	{ "accounting", &Accounting_script, STRV },
	{ "banner", (char **)&Use_banner, FLGV },
	{ "debug", (char **)&debug, INTV },
	{ "dev", &device, STRV },
	{ "reportpsoutput", (char **)&ReportPSOutput, FLGV },
	{ "endpause", (char **)&End_pause, INTV },
	{ "forcepagecount", (char **)&Force_pagecount, FLGV },
	{ "forceps", (char **)&Force_ps, FLGV },
	{ "maxresponse", (char **)&Max_response, INTV },
	{ "nosync", (char **)&No_sync, FLGV },
	{ "pagecount", (char **)&Page_count, FLGV },
	{ "reverse", (char **)&reverse, FLGV },
	{ "status", (char **)&Get_status, FLGV },
	{ "stty", &stty_str, STRV },
	{ "summary", &summaryfile, STRV },
	{ "tbcp", (char **)&tbcp, FLGV },
	{ "udp_status_port", &Udp_status_port, STRV },
	{ 0 }
};

void errorlist()
{
	struct values *key;
	char *s;
	int i = 0;
	fprintf( stderr, "psfilter: -T options are: " );
	for( key = device_keys; key->key; ++key ){
		if( i ){
			if( !(i %4 ) ){
				fprintf( stderr, "\n   " );
			} else {
				fprintf( stderr, ", " );
			}
		}
		switch( key->type ){
			case STRV: s="=str"; break;
			case INTV: s="=num"; break;
			case FLGV: s="=[on|off]"; break;
			default: s="??"; break;
		}
		fprintf(stderr,"%s%s", key->key,s );
		++i;
	}
	fprintf( stderr, "\n" );
}

void fix_keys( char *s, struct values *keys )
{
	char *end, *value;
	struct values *key;
	for( ; s ; s = end ){
		if( (end = strchr(s, ',') ) ){
			*end++ = 0;
		}
		if( (value = strchr(s,'=')) ){
			*value++ = 0;
		}
		for( key = keys; key->key; ++key ){
			if( !strcasecmp(key->key,s) ){
				if(debug>3)fprintf(stderr,"fix_keys: found '%s' value '%s'\n",
					s, value?value:"<NULL>" );
				switch( key->type ){
				case STRV: *key->variable = value; break;
				case INTV: *(int *)key->variable = value?atoi(value):0; break;
				case FLGV: *(int *)key->variable = 
					( value == 0 || !strcasecmp( value, "yes" )
						|| !strcasecmp( value, "on" ) );
					break;
				}
				break;
			}
		} 
		if( key->key == 0 ){
			errorlist();
		}
	}
}

extern char *Copyright[];

void getargs( int argc, char *argv[], char *envp[] )
{
	int i, c;		/* argument index */
	char *arg, *optarg;	/* argument */


	if( (name = argv[0]) == 0 ) name = "FILTER";
	if( (arg = strrchr( name, '/' )) ) name = arg+1;
	if( strcmp( "psof", name ) == 0 ) OFfilter = 1;
	if( strcmp( "psif", name ) == 0 ) IFfilter = 1;
	for( i = 1; i < argc && (arg = argv[i])[0] == '-'; ++i ){
		if( (c = arg[1]) == 0 ){
			fprintf( stderr, "missing option flag");
			i = argc;
			break;
		}
		optarg = 0;
		/* Check for arg unless option has no args */
		if (c != 'v' && c != 'c') {
			if (arg[2] == '\0') {			/* -X arg */
				if ((i + 1) < argc) {
					optarg = argv[++i];
				} else {
					fprintf(stderr,
						"missing option argument");
					break;
				}
			} else {				/* -Xarg */
				optarg = &arg[2];
			}
		}

		switch(c){
			case 'C': class = optarg; 			break; 
			case 'E': errorfile = optarg;		break;
			case 'F': format = optarg;			break; 
			case 'J': job = optarg;				break; 
			case 'L': bnrname = optarg;			break; 
			case 'P': printer = optarg;			break; 
			case 'Q': queuename = optarg;		break; 
			case 'R': accntname = optarg;		break; 
			case 'S': comment = optarg;			break; 
			case 'T': fix_keys(optarg,device_keys);	break; 
			case 'Z': zopts = optarg;			break; 
			case 'a': accntfile = optarg;		break;
			case 'c': literal = 1;				break;
			case 'e': datafile = optarg;		break;
			case 'h': host = optarg;			break; 
			case 'i': indent = atoi( optarg );	break; 
			case 'j': jobnumber = optarg;		break; 
			case 'k': controlfile = optarg;		break; 
			case 'l': length = atoi( optarg );	break; 
			case 'n': login = optarg;			break; 
			case 'p': printer = optarg;			break; 
			case 's': statusfile = optarg;		break; 
			case 'v': verbose = 1;				break; 
			case 'w': width = atoi( optarg );	break; 
			case 'x': xwidth = atoi( optarg );	break; 
			case 'y': ylength = atoi( optarg );	break;
			default: break;
		}
	}
	if( verbose ){
		for( i = 0; Copyright[i]; ++i ){
			fprintf( stderr, "%s\n", Copyright[i] );
		} 
	}
	if( i < argc ){
		accntfile = argv[i];
	}
	/* check for IF filter behaviour */
	if( format && *format == 'f' ){
		IFfilter = 1;
	}
	if( errorfile ){
		int fd;
		fd = open( errorfile, O_APPEND | O_WRONLY );
		if( fd < 0 ){
			fprintf( stderr, "cannot open error log file '%s'", errorfile );
		} else {
			fprintf( stderr, "using error log file '%s'", errorfile );
			if( fd != 2 ){
				dup2(fd, 2 );
				close(fd);
			}
		}
	}
	if( zopts && strstr( zopts, "tbcp" ) ){
		tbcp = 1;
	}
	if( verbose || debug > 2 ){
		fprintf(stderr, "%s command:\n", name );
		for( i = 0; i < argc; ++i ){
			fprintf(stderr, "argv[%d] = %s\n", i, argv[i] );
		}
		fflush(stderr);
	} 
	if(debug>2 && OFfilter)fprintf(stderr,"PSOF: banner '%s'\n", Banner ); 
	if(debug>2){
		fprintf(stderr,"FILTER decoded options:\n" );
		fprintf(stderr,"login '%s'\n", login? login : "null" );
		fprintf(stderr,"host '%s'\n", host? host : "null" );
		fprintf(stderr,"class '%s'\n", class? class : "null" );
		fprintf(stderr,"format '%s'\n", format? format : "null" );
		fprintf(stderr,"job '%s'\n", job? job : "null" );
		fprintf(stderr,"printer '%s'\n", printer? printer : "null" );
		fprintf(stderr,"controlfile '%s'\n", controlfile? controlfile:"null");
		fprintf(stderr,"queuename '%s'\n", queuename? queuename : "null" );
		fprintf(stderr,"accntname '%s'\n", accntname? accntname : "null" );
		fprintf(stderr,"zopts '%s'\n", zopts? zopts : "null" );
		fprintf(stderr,"literal, %d\n", literal);
		fprintf(stderr,"indent, %d\n", indent);
		fprintf(stderr,"length, %d\n", length);
		fprintf(stderr,"width, %d\n", width);
		fprintf(stderr,"xwidth, %d\n", xwidth);
		fprintf(stderr,"ylength, %d\n", ylength);
		fprintf(stderr,"accntfile '%s'\n", accntfile? accntfile : "null" );
		fprintf(stderr,"errorfile '%s'\n", errorfile? errorfile : "null" );
		fprintf(stderr,"statusfile '%s'\n", statusfile? statusfile : "null" );
		fprintf(stderr, "FILTER environment: " );
		for( i = 0; (arg = envp[i]); ++i ){
			fprintf(stderr,"%s\n", arg );
		}
		fprintf(stderr, "RUID: %d, EUID: %d\n", (int)getuid(), (int)geteuid() );
		fflush(stderr);
	}
}

/*
 * suspend():  suspends the output filter, waits for a signal
 */

void suspend()
{
	if(debug>2)fprintf(stderr,"FILTER suspending\n");
	fflush(stderr);
	kill(getpid(), SIGSTOP);
	if(debug>2)fprintf(stderr,"FILTER awake\n");
}

/******************************************
 * prototype filter() and cleanup_job() functions;
 * filter will scan the input looking for the suspend string
 * if any.
 ******************************************/
void cleanup_job( int code )
{
	int pid;
	pid = getpid();
	if(debug>2)setstatus(
	"cleanup_job: pid %d, code %d, Statuspid %d, TextPid %d, ReversePid %d\n",
			pid, code, Statuspid, TextPid, ReversePid );
	/* signal the filterpid if you are not it */
	if( pid == Filter_pid ){
		if( Statuspid > 0 ){
			kill( Statuspid, SIGINT );
			kill( Statuspid, SIGCONT );
		}
		if( TextPid > 0 ){
			kill( TextPid, SIGINT );
			kill( TextPid, SIGCONT );
		}
		if( ReversePid > 0 ){
			kill( ReversePid, SIGINT );
			kill( ReversePid, SIGCONT );
		}
	}
	exit( code );
}
