/*
 * Monitor for PSFILTER program
 *  Opens a UDP socket and waits for data to be sent to it.
 *
 *  monitor port
 *   port is an integer number or a service name in the services database
 */

#include "psfilter.h"
#include "errorcodes.h"
#if 0
#include <stdio.h>
#include <sys/types.h>
#include <sys/time.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <netdb.h>
#include <errno.h>
#endif

extern int errno;

int udp_open( char *portname );
int tcp_open( char *portname );

char buffer[1024];

void usage()
{
	fprintf( stderr, "usage: monitor port\n" );
	exit(1);
}

int main( int argc, char *argv[] )
{
	int fd1, fd2, fd3;
	int fds, n, i, cnt;
	fd_set readfds;
	fd_set acceptfds;
	struct sockaddr addr;
	int len;

	if( argc != 2 ){
		fprintf( stderr, "usage: monitor port\n" );
		exit(1);
	}
	fd1 = udp_open( argv[1] );
	fd2 = tcp_open( argv[1] );
	if( fd1 < 0 || fd2 < 0 ){
		usage();
	}
	fds = fd1;
	if( fd2 > fds ) fds = fd2;
	++fds;
	/* printf("fd1 %d, fd2 %d, fds %d\n", fd1, fd2, fds ); */
	while(1){
		FD_ZERO( &readfds );
		FD_ZERO( &acceptfds );
		FD_SET( fd1, &readfds );
		FD_SET( fd2, &readfds );
		FD_SET( fd1, &acceptfds );
		FD_SET( fd2, &acceptfds );
		n = select( fds, &readfds, (fd_set *)0, &acceptfds, (struct timeval *)0 );
		/* printf("select returned %d\n", n); */
		if( n < 0 ) continue;
		/* printf("read set fd1 %d fd2 %d\n",
			FD_ISSET( fd1, &readfds ), FD_ISSET( fd2, &readfds ) ); */
		/* printf("expect set fd1 %d fd2 %d\n",
			FD_ISSET( fd1, &acceptfds ), FD_ISSET( fd2, &acceptfds ) ); */
		if( FD_ISSET( fd1, &readfds ) ){
			n = read( fd1, buffer, sizeof(buffer)-1);
			for( i = 0; i < n; i += cnt ){
				cnt = write(1, buffer+i, n-i );
			}
		}
		if( FD_ISSET( fd2, &readfds ) || FD_ISSET( fd2, &acceptfds ) ){
			len = sizeof(addr);
			fd3 = accept( fd2, &addr, &len );
			if( fd3 < 0 ){
				perror( "accept failed" );
			}
			len = 1;
			/* printf( "accepted %d\n", fd3 ); */
			while( (n = read( fd3, buffer, sizeof(buffer)-1) ) > 0 ){
				for( i = 0; i < n; i += cnt ){
					cnt = write(1, buffer+i, n-i );
				}
			}
			close(fd3);
		}
	}
	return(0);
}

int udp_open( char *portname )
{
	int port, i, fd, err, len;
	struct sockaddr_in sin;
	struct servent *servent, *getservbyname();

	port = atoi( portname );
	if( port <= 0 ){
		servent = getservbyname( portname, "udp" );
		if( servent ){
			port = ntohs( servent->s_port );
		}
	}
	if( port <= 0 ){
		fprintf( stderr, "udp_open: bad port number '%s'\n",portname );
		return( -1 );
	}
	sin.sin_family = AF_INET;
	sin.sin_addr.s_addr = INADDR_ANY;
	sin.sin_port = htons( port );

	fd = socket( AF_INET, SOCK_DGRAM, 0 );
	err = errno;
	if( fd < 0 ){
		fprintf(stderr,"udp_open: socket call failed - %s\n", Errormsg(err) );
		return( -1 );
	}
	i = -1;
	i = bind( fd, (struct sockaddr *) & sin, sizeof (sin) );
	err = errno;
	len = 1;
	setsockopt( fd, SOL_SOCKET, SO_REUSEADDR,
		(char *)&len, sizeof( len ) );

	if( i < 0 ){
		fprintf(stderr,"udp_open: connect to '%s port %d' failed - %s\n",
			inet_ntoa( sin.sin_addr ), ntohs( sin.sin_port ),
			Errormsg(errno) );
		close(fd);
		fd = -1;
	}
	return( fd );
}

int tcp_open( char *portname )
{
	int port, i, fd, err, len;
	struct sockaddr_in sin;
	struct servent *servent, *getservbyname();

	port = atoi( portname );
	if( port <= 0 ){
		servent = getservbyname( portname, "tcp" );
		if( servent ){
			port = ntohs( servent->s_port );
		}
	}
	if( port <= 0 ){
		fprintf( stderr, "tcp_open: bad port number '%s'\n",portname );
		return( -1 );
	}
	sin.sin_family = AF_INET;
	sin.sin_addr.s_addr = INADDR_ANY;
	sin.sin_port = htons( port );

	fd = socket( AF_INET, SOCK_STREAM, 0 );
	err = errno;
	if( fd < 0 ){
		fprintf(stderr,"tcp_open: socket call failed - %s\n", Errormsg(err) );
		return( -1 );
	}
	i = -1;
	i = bind( fd, (struct sockaddr *) & sin, sizeof (sin) );
	len = 1;
	setsockopt( fd, SOL_SOCKET, SO_REUSEADDR,
		(char *)&len, sizeof( len ) );
	if( i != -1 ) i = listen( fd, 10 );
	err = errno;

	if( i < 0 ){
		fprintf(stderr,"tcp_open: connect to '%s port %d' failed - %s\n",
			inet_ntoa( sin.sin_addr ), ntohs( sin.sin_port ),
			Errormsg(errno) );
		close(fd);
		fd = -1;
	}
	return( fd );
}


/****************************************************************************
 * Extract the necessary definitions for error message reporting
 ****************************************************************************/

#if !defined(HAVE_STRERROR)
# if defined(HAVE_SYS_NERR)
#   if !defined(HAVE_SYS_NERR_DEF)
      extern int sys_nerr;
#   endif
#   define num_errors    (sys_nerr)
# else
#  	define num_errors    (-1)            /* always use "errno=%d" */
# endif
# if defined(HAVE_SYS_ERRLIST)
#  if !defined(HAVE_SYS_ERRLIST_DEF)
    extern const char *const sys_errlist[];
#  endif
# else
#  undef  num_errors
#  define num_errors   (-1)            /* always use "errno=%d" */
# endif
#endif

const char * Errormsg ( int err )
{
    const char *cp;

#if defined(HAVE_STRERROR)
	cp = strerror(err);
#else
# if defined(HAVE_SYS_ERRLIST)
    if (err >= 0 && err <= num_errors) {
		cp = sys_errlist[err];
    } else
# endif
	{
		static char msgbuf[32];     /* holds "errno=%d". */
		/* SAFE use of sprintf */
		(void) sprintf (msgbuf, "errno=%d", err);
		cp = msgbuf;
    }
#endif
    return (cp);
}
