/***************************************************************************
 * LPRng - An Extended Print Spooler System
 *
 * Copyright 1988-1995 Patrick Powell, San Diego State University
 *     papowell@sdsu.edu
 * See LICENSE for conditions of use.
 * From the original PLP Software distribution
 *
 * Note: changes, bugfixes and modifications also by
 *  Charles Hedrick <hedrick@klinzhai.rutgers.edu>
 *
 ***************************************************************************
 * MODULE: filter.c for filter
 * PURPOSE:  run the filter
 **************************************************************************/

#ifndef lint
static const char *const _id =
	"filter.c,v 1.10 1998/03/29 23:49:28 papowell Exp";
#endif

#include "psfilter.h"
#include "errorcodes.h"
#include "timeout.h"

/***************************************************************************
Commentary:
This is a fairly simple filter design.
1. We check to see if we have to open a device;
   - local (/dev/xx) or remote (host%port)
   - local device may require some stty commands as well
2. we then fork a child process to handle the stream stuff.
3. periodically we wake up and send a ^T status to the unit
4. If we do not get status back after a reasonable time,  we
   start to wonder.
 ***************************************************************************/

static char startstr[] = "\001M";
static char endstr[] = "\033%-12345X";
void do_of_stream( char *stop );
void do_if_stream( void );

void filter(char *stop, int timeout, int max_try, int max_response )
{
	static int i;
	static int fd = -1;
	int err;
	int pid, status;
	char *s, *t;
	int port;
	int orig_port = 0;
	struct hostent *hostent;
	struct sockaddr_in sin, src_sin;

	/*
	 * do whatever initializations are needed
	 */
	Filter_pid = getpid();

	if(debug>2)setstatus(
		"filter: timeout %d, max_try %d\n", timeout, max_try );
	if( device ){
		setstatus("filter: opening '%s' at %s\n", device, Time_str() );
		if( (device[0] == '/') ){
			if( Set_timeout()){
				Set_timeout_alarm( timeout, 0 );
				/*fd = open( device, O_RDWR ); */
				fd = open( device, O_RDWR );
			}
			Clear_timeout();
			err = errno;
			if( Alarm_timed_out ){
				setstatus( "filter: open '%s' timeout\n",device );
				cleanup_job( JABORT );
			} else if( fd < 0 ){
				setstatus( "filter: open '%s' failed - %s\n",device, Errormsg(err) );
				cleanup_job( JABORT );
			}
			if( isatty( fd ) && stty_str ){
				Do_stty( fd, stty_str );
			}
		} else {
			char buffer[64];
			buffer[sizeof(buffer)-1] = 0;
			strncpy(buffer, device, sizeof(buffer)-1);
			if( (s = strpbrk( buffer, "%" )) == 0 ){
				setstatus( "filter: missing port number '%s'\n",device );
				cleanup_job( JABORT );
			}
			*s++ = 0;
			port = strtol( s, &t, 0 );
			if( t == 0 || port <= 0){
				setstatus( "filter: bad port number '%s'\n",device );
				cleanup_job( JABORT );
			}
			if( *t ){
				if( *t == '+' ){
					orig_port = strtol( t+1, 0, 0 );
				}
			}
			setstatus("filter: host '%s', port %d, orig_port %d", buffer, port, orig_port );
			sin.sin_family = AF_INET;
			if( (hostent = gethostbyname(buffer)) ){
				/*
				 * set up the address information
				 */
				if( hostent->h_addrtype != AF_INET ){
					setstatus( "filter: bad address type for host '%s'\n",
						device);
					cleanup_job( JABORT );
				}
				if( hostent->h_length > sizeof( sin.sin_addr ) ){
					setstatus( "filter: bad address length for host '%s'\n",
						device);
					cleanup_job( JABORT );
				}
				memcpy( &sin.sin_addr, hostent->h_addr, hostent->h_length );
			} else {
				sin.sin_addr.s_addr = inet_addr(buffer);
				if( sin.sin_addr.s_addr == -1){
					setstatus("filter: unknown host '%s'\n", buffer);
					cleanup_job( JABORT );
				}
			}
			*s = '%';
			sin.sin_port = htons( port );
			if(debug>2) setstatus( "filter: destination '%s' port %d\n",
				inet_ntoa( sin.sin_addr ), ntohs( sin.sin_port ) );
			fd = socket (AF_INET, SOCK_STREAM, 0);
			err = errno;
			if (fd < 0) {
				setstatus("filter: socket call failed - %s\n", Errormsg(err) );
				cleanup_job( JABORT );
			}
			i = -1;
			if( orig_port ){
				memset( &src_sin, 0, sizeof( src_sin ) );
				src_sin.sin_family = AF_INET;
				src_sin.sin_port = htons( orig_port );
				if( bind( fd, (void *)&src_sin, sizeof( src_sin ) ) < 0 ){
					setstatus("filter: bind to %d failed - %s\n",
						orig_port, Errormsg(err) );
					cleanup_job( JABORT );
				}
			}
			if( Set_timeout() ){
				Set_timeout_alarm( timeout, 0 );
				i = connect (fd, (struct sockaddr *) & sin, sizeof (sin));
			}
			Clear_timeout();
			err = errno;

			if( Alarm_timed_out ){
				setstatus("filter: connect to '%s port %d' time out\n",
					inet_ntoa( sin.sin_addr ), ntohs( sin.sin_port ) );
				cleanup_job( JABORT );
			} else if( i < 0 ){
				setstatus("filter: connect to '%s port %d' failed - %s\n",
					inet_ntoa( sin.sin_addr ), ntohs( sin.sin_port ),
					Errormsg(errno) );
				cleanup_job( JABORT );
			}
		}
		if( fd >= 0 && fd != 1 ){
			if(debug>2)setstatus("filter: dup %d to %d\n", fd, 1 );
			if( dup2( fd, 1 ) < 0 ){
				setstatus("filter: dup %d to %d failed - %s\n",
					fd, 1, Errormsg(errno) );
				cleanup_job(JABORT);
			}
			close( fd );
		}
		setstatus( "filter: connection to printer\n" );
	}

#if defined(F_GETFL)
	if( Get_status ){
		int status;
		if(debug>2)setstatus("filter: trying fcntl" );
		status = fcntl( 1, F_GETFL );
		if(debug>2)setstatus("filter: fcntl returned 0x%x, O_RDWR = 0x%x", status, O_RDWR );
		if( (status & O_RDWR) == 0 ){
			setstatus("ifhp: cannot read stdout, not getting status" );
			Get_status = 0;
		}
	}
#endif

	if( Get_status ){
		/* get the page count */
		if( Udp_status_port ){
			if(debug>2)setstatus("filter: using UPD status '%s'",
				Udp_status_port );
			Udp_status_sock = udp_open(Udp_status_port);
			if( Udp_status_sock < 0 ){
				setstatus("filter: udp_open '%s' failed - %s\n",
					Udp_status_port, Errormsg(errno) );
				cleanup_job(JABORT);
			}
		}
		if( Page_count ){
			setstatus("filter: connected, getting page count at %s\n",
				Time_str() );
			npages = getpagecount( timeout, 0, max_response, 0, No_sync, Force_ps );
			if( npages < 0 ){
				setstatus("filter: pagecount '%s' failed\n", printer );
				cleanup_job( JABORT );
			}
			setstatus("filter: starting page counter value '%d'\n", npages );
			initialpagecount = npages;
		}
		fflush(stdout);
		/* start the status monitoring */
		do_status( 30, 5 );
		setstatus("filter: connected and synced to printer\n");
	} else if( Force_pagecount ){
		npages = getpagecount( timeout, 0, max_response, 0, No_sync, Force_ps );
		if( npages < 0 ){
			setstatus("filter: pagecount '%s' failed\n", printer );
			cleanup_job( JABORT );
		}
		setstatus("filter: starting page counter value '%d'\n", npages );
		initialpagecount = npages;
	}
	doaccnt(1);

	if( stop ){
		tbcp = 0;
		do_of_stream( stop );
	} else {
		do_if_stream();
	}

	if( Statuspid > 0 ){
		kill( Statuspid, SIGINT );
		kill( Statuspid, SIGCONT );
		pid = 0;
		do{
			errno = 0;
			status = 0;
			pid = waitpid(Statuspid, &status, 0);
			err = errno;
			if(debug>3)setstatus("filter: status pid %d status 0x%x\n",
				pid, status );
		} while( (pid != Statuspid) || (pid == -1 && err != ECHILD) );
		Statuspid = 0;
	}

	if( Get_status ){
		setstatus("filter: doing sync at end of job\n");
		if( Page_count ){
			npages = getpagecount( timeout, 0, max_response, End_pause,
				No_sync, Force_ps );
			if( npages < 0 ){
				setstatus("filter: pagecount '%s' failed\n", printer );
			}
		}
	} else if( Force_pagecount ){
		npages = getpagecount( timeout, 0, max_response, End_pause,
			No_sync, Force_ps );
		if( npages < 0 ){
			setstatus("filter: pagecount '%s' failed\n", printer );
		}
	}
	setstatus(
	"filter: end job - final page counter value '%d', total pages %d\n",
		npages, npages - initialpagecount );
	doaccnt(0);
	Write_fd_len( 1, "\004", 1 );
	if( tbcp ) Write_fd_len( 1, endstr, sizeof(endstr)-1 );
	setstatus("filter: job finished successfully\n");
	exit( JSUCC );
}

/***************************************************************************
 * void do_of_stream( char *stop )
 *  read stdin and dump to stdout until finished; look for the stop
 *  string
 ***************************************************************************/

void do_of_stream( char *stop )
{
	int c, pid, status;
	int state = 0;
	int linecount = 0;
	int piped[2];		/* pipe */
	int err = 0, i;

	setstatus("do_of_stream: starting transfer\n" );
	while( (c = getchar()) != EOF ){
		statusbuf[linecount++] = c;
		++Bytecount;
		if( (stop || state) && c == stop[state] ){
			++state;
			if( stop[state] == 0 ){
				Bytecount -= state;
				setstatus("do_stream: stopping, wrote %d so far\n",
					Bytecount );
				state = 0;
				if( Write_fd_len( 1, statusbuf, linecount-strlen(stop) ) < 0 ){
					err = errno;
					setstatus("do_stream: write failed - %s\n",
						Errormsg(err) );
					cleanup_job( JABORT );
				}
				linecount = 0;
				if( Statuspid > 0 ){
					kill( Statuspid, SIGINT );
					kill( Statuspid, SIGCONT );
					pid = 0;
					do{
						errno = 0;
						status = 0;
						pid = waitpid(Statuspid, &status, 0);
						err = errno;
						if(debug>3)setstatus("do_of_stream: status pid %d status 0x%x\n",
							pid, status );
					} while( (pid != Statuspid) || (pid == -1 && err != ECHILD) );
					Statuspid = 0;
				}

				/* we suspend */
				suspend();
				i = 0;
				statusbuf[i++] = CTRL_D;
				statusbuf[i++] = 0;
				if( Write_fd_str( 1, statusbuf ) < 0 ){
					err = errno;
					setstatus("do_stream: write failed - %s\n",
						Errormsg(err) );
					cleanup_job( JABORT );
				}
				if( Get_status ) do_status( 30, 5 );
			}
		} else {
			state = 0;
		}
		if( c == '\n' || linecount >= sizeof(statusbuf) ){ 
			for( i = 0; i < linecount && isspace( statusbuf[i] ); ++i );
			if( Use_banner && Banner && *Banner && i < linecount ){
				if( pipe(piped) < 0 ){
					err = errno;
					setstatus( "do_of_stream: pipe failed - %s\n",
						Errormsg(err) );
					cleanup_job(JFAIL);
				}
				if( (ReversePid = fork()) < 0 ){
					err = errno;
					setstatus( "do_of_stream: fork failed - %s\n",
						Errormsg(err) );
					cleanup_job(JFAIL);
				} else if( ReversePid == 0 ){
					if(debug>2)setstatus(
						"do_of_stream: daughter for banner\n" );
					dup2( piped[0], 0 );
					close( piped[0] );
					close( piped[1] );
					for( i = 3; i < 20; ++i ) close(i);
					Argv_p[0] = Banner;
					execve(Banner, Argv_p, Envp_p );
					setstatus(
						"do_of_stream: execve '%s' failed - %s\n", Banner,
						Errormsg(err) );
					exit(JABORT);
				}
				close( piped[0] );
				if( Write_fd_len( piped[1], statusbuf, linecount) < 0 ){
					err = errno;
					setstatus("write failed - %s\n",
						Errormsg(err) );
					setstatus("do_stream: write failed - %s\n",
						Errormsg(err) );
					cleanup_job( JABORT );
				}
				close( piped[1] );
				if(debug>2)setstatus( "do_of_stream: waiting for banner\n" );
				do{
					errno = 0;
					status = 0;
					pid = wait(&status);
					err = errno;
					if(debug>3)setstatus(
						"do_of_stream: status pid %d status 0x%x\n",
						pid, status );
					if( pid > 0 && status ){
						cleanup_job(WEXITSTATUS(status));
					}
				} while( pid < 0 && err != ECHILD );
				ReversePid = -1;
			} else {
				if( Write_fd_len( 1, statusbuf, linecount) < 0 ){
					err = errno;
					setstatus("do_of_stream: write failed - %s\n",
						Errormsg(err) );
					cleanup_job( JABORT );
				}
			}
			linecount = 0;
		}
	}
	if( ferror( stdin ) ){
		logerr( "do_of_stream: read error on stdin");
	}
	if( linecount > 0 ){
		if( Write_fd_len( 1, statusbuf, linecount) < 0 ){
			err = errno;
			setstatus("do_stream: write failed - %s\n",
				Errormsg(err) );
			cleanup_job( JABORT );
		}
		linecount = 0;
	}

	setstatus( "filter: finished transfer of %d bytes\n", Bytecount );
}


/***************************************************************************
 * void do_if_stream( char *stop )
 *  read stdin and dump to stdout until finished.
 * check to see if we have a postscript file or not
 * - if first block as 0 or 1 char we assume binary
 ***************************************************************************/


void do_if_stream()
{
	int len;			/* read length */
	int err, i, c, n;		/* ACME */
	int piped[2];		/* pipe */
	int ps = 0;			/* postscript file */
	char *s;			/* ACME */
	int pid;			/* status of process */
	int status = 0;		/* status to exit with */
	int outfd = 1;		/* parent writes to this */

	ReversePid = -1;
	TextPid = -1;
	setstatus("do_if_stream: starting transfer\n" );
	len = read(0,statusbuf,sizeof(statusbuf) );
	err = errno;
	if( len < 0 ){
		setstatus( "do_if_stream: input read error - %s\n", Errormsg(err) );
		status = JFAIL;
		goto error;
	}
	if( len == 0 ){
		setstatus( "do_if_stream: zero length file\n" );
		goto done;
	}
	if( literal == 0 && (memchr( statusbuf, 0x00, len )
	  ||  memchr( statusbuf, 0x01, len )) ){
		setstatus( "do_if_stream: binary file\n" );
		status = JREMOVE;
		goto error;
	}
	/* Check to see if there is a % followed by !
	 * Must be preceded by ^D, ESC and printable text; this covers
	 * HP job language (I think), unless they stick those ^G in
	 * to make life ugly.
	 */
	if( !literal
		&& (s = memchr( statusbuf, '%', len ))
		&& s[1] == '!' ){
		ps = 1;
		n = s - statusbuf;
		for( i = 0; ps && i < n ; ++i ){
			c = statusbuf[i];
			if( c!= CTRL_D && c != ESC && !isspace(c) && !isprint(c) ) ps = 0;
		}
	}
	if(debug>4)setstatus( "do_if_stream: input %s [%c%c%c]\n",
		literal?"literal":(ps?"PostScript":"Text"),
		statusbuf[0], statusbuf[1], statusbuf[2] );

	if( !literal && reverse ){
		if( pipe(piped) < 0 ){
			err = errno;
			setstatus( "do_if_stream: pipe failed - %s\n", Errormsg(err) );
			status = JFAIL;
			goto error;
		}
		if( (ReversePid = fork()) < 0 ){
			err = errno;
			setstatus( "do_if_stream: fork failed - %s\n", Errormsg(err) );
			status = JFAIL;
			goto error;
		} else if( ReversePid == 0 ){
			if(debug>2)setstatus( "do_if_stream: daughter for reverse\n");
			dup2( piped[0], 0 );
			close( piped[0] );
			close( piped[1] );
			for( i = 3; i < 20; ++i ) close( i );
			Argv_p[0] = Psrev;
			execve(Psrev, Argv_p, Envp_p );
			setstatus(
				"do_if_stream: execve '%s' failed - %s\n", Psrev, Errormsg(err) );
			exit(JABORT);
		} else {
			outfd = piped[1];
			close( piped[0] );
		}
	}

	/* convert to text if necessary */
	if( !literal && !ps ){
		if(debug>2)setstatus( "do_if_stream: input is plain text\n");
		if( pipe(piped) < 0 ){
			err = errno;
			setstatus( "do_if_stream: pipe failed - %s\n", Errormsg(err) );
			status = JFAIL;
			goto error;
		}
		if( (TextPid = fork()) < 0 ){
			err = errno;
			setstatus( "do_if_stream: fork failed - %s\n", Errormsg(err) );
			status = JFAIL;
			goto error;
		} else if( TextPid == 0 ){
			/* daughter - we are the first in line */
			if(debug>4)setstatus( "do_if_stream: daughter is converting text\n");
				/* we now set up the reading - piped[0] -> 0 */
			dup2( piped[0], 0 );
			close( piped[0] );
			close( piped[1] );
			for( i = 3; i < 20; ++i ) close( i );
			Argv_p[0] = Textps;
			Argv_p[1] = 0;
			execve(Textps, Argv_p, Envp_p );
			setstatus( "do_if_stream: execl '%s' failed - %s\n",
				Textps, Errormsg(err) );
			exit(JABORT);
		} else {
			if( outfd != 1 ) close( outfd );
			outfd = piped[1];
			close( piped[0] );
		}
	}

	/* if PostScript and TBCP requested, set it up */
	if( ps && tbcp ){
		if( Write_fd_len( outfd, startstr, sizeof(startstr)-1 ) < 0 ){
			err = errno;
			setstatus( "do_if_stream: write error to output - %s\n",
				Errormsg(errno) );
			status = JFAIL;
			goto shutdown;
		}
	} else {
		tbcp = 0;
	}

	/* send to the filters */
	do{
		Bytecount += len;
		if( tbcp ){
			/* copy to a buffer and then write the buffer */
			char buffer[256], *bf;
			int i, j, c;
			bf = statusbuf;
			while (len > 0){
				for( j = i = 0; i < len && j < sizeof(buffer)-2; ++i ){
					switch( (c = ((unsigned char *)bf)[i])  ){
					case 0x01: case 0x03: case 0x04: case 0x05:
					case 0x11: case 0x13: case 0x14: case 0x1B: case 0x1C:
						buffer[j++] = 0x01;
						c = c ^ 0x40;
						break;
					}
					buffer[j++] = c;
				}
				if( Write_fd_len( outfd, buffer, j ) < 0 ){
					err = errno;
					setstatus( "do_if_stream: write error to output - %s\n",
						Errormsg(errno) );
					status = JFAIL;
					goto shutdown;
				}
				len -= i;
				bf += i;
			}
		} else if( Write_fd_len( outfd, statusbuf, len ) < 0 ){
			err = errno;
			setstatus( "do_if_stream: write error to output - %s\n",
				Errormsg(errno) );
			status = JFAIL;
			goto shutdown;
		}
		len = read(0,statusbuf,sizeof(statusbuf) );
		if( len < 0 ){
			setstatus( "do_if_stream: read error from input - %s\n",
				Errormsg(errno) );
			status = JFAIL;
			goto shutdown;
		}
	}while( len > 0 );

done:
	setstatus( "do_if_stream: finished transfer of %d bytes\n", Bytecount );

shutdown:
	if( outfd != 1 ) close( outfd );

	/* wait for child */
	if( ReversePid >0 || TextPid > 0 ){
		if(debug>4)setstatus( "do_if_stream: waiting for child\n" );
		do{
			errno = 0;
			status = 0;
			pid = wait(&status);
			err = errno;
			if(debug>3)setstatus("do_if_stream: status pid %d status 0x%x\n",
				pid, status );
			if( pid > 0 && status ){
				cleanup_job(WEXITSTATUS(status));
			}
		} while( pid < 0 && err != ECHILD );
	}
	if( status ) cleanup_job( status );
	return;

error:
	cleanup_job(status);
}
