/* _NVRM_COPYRIGHT_BEGIN_
 *
 * Copyright 1999-2001 by NVIDIA Corporation.  All rights reserved.  All
 * information contained herein is proprietary and confidential to NVIDIA
 * Corporation.  Any use, reproduction, or disclosure without the written
 * permission of NVIDIA Corporation is prohibited.
 *
 * _NVRM_COPYRIGHT_END_
 */



/******************* Operating System Interface Routines *******************\
*                                                                           *
* Module: interface.c                                                       *
*   This is the OS interface module.  All operating system transactions     *
*   pass through these routines.  No other operating system specific code   *
*   or data should exist in the source.                                     *
*                                                                           *
\***************************************************************************/

#define  __NO_VERSION__
#include "nv-misc.h"

#include "os-interface.h"
#include "nv-linux.h"


static volatile int os_block_on_smp_barrier;

#ifdef CONFIG_SMP
static void ipi_handler(void *info)
{
    while(os_block_on_smp_barrier != 0) { barrier(); }
}
#endif

RM_STATUS os_raise_smp_barrier(VOID)
{
    os_block_on_smp_barrier = 1;
#ifdef CONFIG_SMP
    if (smp_call_function(ipi_handler, NULL, 1, 0) != 0)
        return RM_ERROR;
#endif
    return RM_OK;
}

RM_STATUS os_clear_smp_barrier(VOID)
{
    os_block_on_smp_barrier = 0;
    return RM_OK;
}

// return TRUE if the caller is the super-user
BOOL os_is_administrator(
    PHWINFO pDev
)
{
    return suser();
}

U032 os_get_page_size(VOID)
{
    return PAGE_SIZE;
}

ULONG os_get_page_mask(VOID)
{
    return PAGE_MASK;
}

//
// Some quick and dirty library functions.
// This is an OS function because some operating systems supply their
// own version of this function that they require you to use instead
// of the C library function.  And some OS code developers may decide to
// use the actual C library function instead of this code.  In this case,
// just replace the code within osStringCopy with a call to the C library
// function strcpy.
//
U008* os_string_copy(
    U008 *dst,
    const U008 *src
)
{
    return strcpy(dst, src);
}

RM_STATUS os_strncpy_from_user(
    U008 *dst,
    const U008 *src,
    U032 n
)
{
#if defined(NVCPU_IA64)
    return copy_from_user(dst, src, n) ? RM_ERR_BAD_ADDRESS : RM_OK;
#else
    return strncpy_from_user(dst, src, n) ? RM_ERR_BAD_ADDRESS : RM_OK;
#endif
}

S032 os_string_compare(
    const U008 *s1,
    const U008 *s2
)
{
    return strcmp(s1, s2);
}

U032 os_string_length(
    const U008* str
)
{
    return strlen(str);
}

U008* os_mem_copy(
    U008 *dst,
    const U008 *src,
    U032 length
)
{
    return memcpy(dst, src, length);
}

RM_STATUS os_memcpy_from_user(
    VOID *dst,
    const VOID* src,
    U032 length
)
{
    return copy_from_user(dst, src, length) ? RM_ERR_BAD_ADDRESS : RM_OK;
}

RM_STATUS os_memcpy_to_user(
    VOID *dst,
    const VOID* src,
    U032 length
)
{
    return copy_to_user(dst, src, length) ? RM_ERR_BAD_ADDRESS : RM_OK;
}

VOID* os_mem_set(
    VOID* dst,
    U008 c,
    U032 length
)
{
    return memset(dst, (int)c, length);
}

S032 os_mem_cmp(
    const U008 *buf0,
    const U008* buf1,
    U032 length
)
{
    return memcmp(buf0, buf1, length);
}

VOID* os_copy_in_ioctl_param(
    VOID *dst,
    VOID *src,
    U032 length
)
{
    if (copy_from_user(dst, src, length) > 0) {
        return NULL;
    } else {
        return dst;
    }
}

VOID* os_copy_out_ioctl_param(
    VOID *dst,
    VOID *src,
    U032 length
)
{
    if (copy_to_user(dst, src, length) > 0) {
        return NULL;
    } else {
        return (VOID *)1;
    }
}

/*
 * Operating System Memory Functions
 *
 * There are 2 interesting aspects of resource manager memory allocations
 * that need special consideration on Linux:
 *
 * 1. They are typically very large, (e.g. single allocations of 164KB)
 *
 * 2. The resource manager assumes that it can safely allocate memory in
 *    interrupt handlers.
 *
 * The first requires that we call vmalloc, the second kmalloc. We decide
 * which one to use at run time, based on the size of the request and the
 * context. Allocations larger than 128KB require vmalloc, in the context
 * of an ISR they fail.
 */

#define KMALLOC_LIMIT 131072

RM_STATUS os_alloc_mem(
    VOID **address,
    U032 size
)
{
    if (in_interrupt()) {
        if (size <= KMALLOC_LIMIT) {
            /*
             * The allocation request can be serviced with the Linux slab
             * allocator; the likelyhood of failure is still fairly high,
             * since kmalloc can't sleep.
             */
            *address = kmalloc(size, GFP_ATOMIC);
        } else {
            /*
             * If the requested amount of memory exceeds kmalloc's limit,
             * we can't fulfill the request, as vmalloc can not be called
             * from an interrupt context (bottom-half, ISR).
             */
            *address = NULL;
        }
    } else {
        if (size <= KMALLOC_LIMIT)
        {
            *address = kmalloc(size, GFP_KERNEL);

            // kmalloc may fail for larger allocations if memory is fragmented
            // in this case, vmalloc stands a better chance of making the 
            // allocation, so give it a shot.
            if (*address == NULL)
                NV_VMALLOC(*address, size);
        }
        else
        {
            NV_VMALLOC(*address, size);
        }
    }

    return *address ? RM_OK : RM_ERR_NO_FREE_MEM;
}

void os_free_mem(VOID *address)
{
    unsigned long va = (unsigned long) address;

    if (va >= VMALLOC_START && va < VMALLOC_END)
    {
        NV_VFREE(address);
    }
    else
    {
        NV_KFREE(address);
    }
}

/* note that since contiguous pages are allocated by order
 * we may allocate more pages than the caller really asked for.
 * we'll only lock down the number of pages the caller asked for.
 */
RM_STATUS os_alloc_contig_pages(
    VOID **address,
    U032 size
)
{
    int i;
    unsigned long va;

    /*
     * The size argument is assumed to be a multiple of OS_PAGE_SIZE, and
     * can thus be easily translated into a page count, and then into the
     * order argument expected by __get_free_pages - we may allocate more
     * memory than requested.
     */
    int count = size / PAGE_SIZE;

    va = __get_free_pages(NV_GFP_HW, calc_order(size));
    if (!va)
        return RM_ERR_NO_FREE_MEM;

    *address = (void *) va;

    for (i = 0; i < count; i++) {
        mem_map_reserve(GET_MAP_NR(__pa((va))));
        va += PAGE_SIZE;
    }

    return RM_OK;
}

/* we may have actually allocated more pages than the caller requested,
 * due to contiguous allocs being by order. Note though, that we only
 * locked down the number of pages the caller requested to be allocated.
 * we get the size the caller requested, so use that value to unlock
 * the same number of pages we locked, and to calculate the order we
 * allocated, so we properly free the allocation.
 */
VOID os_free_contig_pages(
    VOID *address,
    U032 size
)
{
    int i;
    int count = size / PAGE_SIZE;
    unsigned long va = (unsigned long) address;

    for (i = 0; i < count; i++) {
        mem_map_unreserve(GET_MAP_NR(__pa((va))));
        va += PAGE_SIZE;
    }

    va = (unsigned long) address;
    free_pages(va, calc_order(size));
}


/*****************************************************************************
*
*   Name: osGetCurrentTime
*
*****************************************************************************/

RM_STATUS os_get_current_time(
    U032 *seconds,
    U032 *useconds
)
{
    struct timeval tm;

    do_gettimeofday(&tm);

    *seconds = tm.tv_sec;
    *useconds = tm.tv_usec;

    return RM_OK;
}

//---------------------------------------------------------------------------
//
//  Misc services.
//
//---------------------------------------------------------------------------

#define NV_MAX_ISR_UDELAY           20000
#define NV_MAX_ISR_MDELAY           (NV_MAX_ISR_UDELAY / 1000)
#define NV_MSECS_PER_JIFFIE         (1000 / HZ)
#define NV_MSECS_TO_JIFFIES(msec)   ((msec) * HZ / 1000)
#define NV_USECS_PER_JIFFIE         (1000000 / HZ)
#define NV_USECS_TO_JIFFIES(usec)   ((usec) * HZ / 1000000)

// #define NV_CHECK_DELAY_ACCURACY 1

/*
 * It is generally a bad idea to use udelay() to wait for more than
 * a few milliseconds. Since the caller is most likely not aware of
 * this, we use mdelay() for any full millisecond to be safe.
 */

RM_STATUS os_delay_us(U032 MicroSeconds)
{
    unsigned long mdelay_safe_msec;
    unsigned long usec;

#ifdef NV_CHECK_DELAY_ACCURACY
    struct timeval tm1, tm2;

    do_gettimeofday(&tm1);
#endif

    if (in_irq() && MicroSeconds > NV_MAX_ISR_UDELAY)
        return RM_ERROR;
    
    mdelay_safe_msec = MicroSeconds / 1000;
    if (mdelay_safe_msec)
        mdelay(mdelay_safe_msec);

    usec = MicroSeconds % 1000;
    if (usec)
        udelay(usec);

#ifdef NV_CHECK_DELAY_ACCURACY
    do_gettimeofday(&tm2);
    nv_printf(NV_DBG_ERRORS, "osDelayUs %d: 0x%x 0x%x\n", MicroSeconds, tm2.tv_sec - tm1.tv_sec, tm2.tv_usec - tm1.tv_usec);
#endif

    return RM_OK;
}

/* 
 * On Linux, a jiffie represents the time passed in between two timer
 * interrupts. The number of jiffies per second (HZ) varies across the
 * supported platforms. On i386, where HZ is 100, a timer interrupt is
 * generated every 10ms; the resolution is a lot better on ia64, where
 * HZ is 1024. NV_MSECS_TO_JIFFIES should be accurate independent of
 * the actual value of HZ; any partial jiffies will be 'floor'ed, the
 * remainder will be accounted for with mdelay().
 */

RM_STATUS os_delay(U032 MilliSeconds)
{
    unsigned long MicroSeconds;
    unsigned long jiffies;
    unsigned long mdelay_safe_msec;

#ifdef NV_CHECK_DELAY_ACCURACY
    struct timeval tm1, tm2;

    do_gettimeofday(&tm1);
#endif
    
    if (in_irq() && MilliSeconds > NV_MAX_ISR_MDELAY)
        return RM_ERROR;

    if (in_interrupt()) 
    {
        mdelay(MilliSeconds);
        return RM_OK;
    }

    MicroSeconds = MilliSeconds * 1000;

    /* do we have a full jiffie to wait? */
    jiffies = NV_USECS_TO_JIFFIES(MilliSeconds);
    MicroSeconds -= jiffies * NV_USECS_PER_JIFFIE;

    // if we have at least 1 full jiffy to wait, give up the cpu
    // but first, make sure we haven't raised the irql level on
    // this cpu (most likely holding a lock). I'm seeing cases
    // where we give up the cpu with raised irql, and never get
    // rescheduled (due to not responding to a later interrupt?)
    if (jiffies) 
    {
        // make sure interrupts are enabled on the cpu
        if (!NV_IRQL_IS_RAISED())
        {
            /* give up the cpu */
            current->state = TASK_INTERRUPTIBLE;
            schedule_timeout(jiffies);
        } else {
            nv_printf(NV_DBG_ERRORS, "Trying to sleep during raised irql!!\n");
            nv_printf(NV_DBG_ERRORS, "  are we holding a lock?\n");
            nv_printf(NV_DBG_ERRORS, "  skipping os_delay\n");
            os_dbg_breakpoint();
            return RM_ERROR;
        }
    }

    /* catch the remainder, if any */
    if (MicroSeconds > 1000)
    {
        mdelay_safe_msec = MicroSeconds / 1000;
        mdelay(mdelay_safe_msec);
        MicroSeconds %= 1000;
    }
    if (MicroSeconds)
    {
        udelay(MicroSeconds);
    }

#ifdef NV_CHECK_DELAY_ACCURACY
    do_gettimeofday(&tm2);
    nv_printf(NV_DBG_ERRORS, "osDelay %d: 0x%x 0x%x\n", MilliSeconds, tm2.tv_sec - tm1.tv_sec, tm2.tv_usec - tm1.tv_usec);
#endif

    return RM_OK;
}

/* return CPU frequency in MHz */
U032 os_get_cpu_frequency(VOID)
{
    u64 tsc[2];
    u32 tsc_d;
    u32 cpu_mhz;

    tsc[0] = nv_rdtsc();
    mdelay(250);
    tsc[1] = nv_rdtsc();

    /* we timed 1/4th a second */
    tsc_d = (tsc[1] - tsc[0]) * 4;

    /* convert from Hz to MHz */
    cpu_mhz = tsc_d / 1000000;

    return cpu_mhz;
}

RM_STATUS os_get_current_process(U032 *pPid)
{
    *pPid = current->pid;
    return RM_OK;
}

RM_STATUS os_kill_process(
    U032 pid,
    U032 sig
)
{
    return kill_proc(pid, sig, 1) ? RM_ERR_OPERATING_SYSTEM : RM_OK;
}

/*******************************************************************************/
/*                                                                             */
/* Debug and logging utilities follow                                          */
/*                                                                             */
/*******************************************************************************/


// The current debug display level (default to maximum debug level)
int cur_debuglevel = 0xaaaaaaaa;

MODULE_PARM(silence_nvidia_output, "1i");
static int silence_nvidia_output = 0;


//
// this is what actually outputs the data.
//
inline void out_string(const char *str)
{
    printk("%d: %s", smp_processor_id(), str);
}    



#define MAX_ERROR_STRING 256

/*
 * nv_printf() prints to the "error" log for the driver.
 * Just like printf, adds no newlines or names
 * Returns the number of characters written.
 */

int nv_printf(
    int   debuglevel,
    const char *printf_format,
    ...
  )
{
    char nv_error_string[MAX_ERROR_STRING];
    char *p = nv_error_string;
    va_list arglist;
    int chars_written = 0;

    if (silence_nvidia_output)
        return 0;

    if (debuglevel >= ((cur_debuglevel>>4)&0x3))
    {
        va_start(arglist, printf_format);
        chars_written = vsprintf(p, printf_format, arglist);
        va_end(arglist);
        out_string(p);
    }

    return chars_written;
}

BOOL os_pci_device_present(
    U016 vendor,
    U016 device
)
{
    if (pci_find_device(vendor, device, NULL))
        return 1;
    return 0;
}

VOID* os_pci_init_handle(
    U008 bus,
    U008 slot,
    U008 function,
    U016 *vendor,
    U016 *device
)
{
    struct pci_dev *dev;
    dev = pci_find_slot(bus, PCI_DEVFN(slot, function));
    if (!dev)
        return NULL;
    if (vendor) *vendor = dev->vendor;
    if (device) *device = dev->device;
    return (VOID *) dev;
}

U008 os_pci_read_byte(
    VOID *handle,
    U008 offset
)
{
    U008 value;
    pci_read_config_byte( (struct pci_dev *) handle, offset, &value);
    return value;
}

U016 os_pci_read_word(
    VOID *handle,
    U008 offset
)
{
    U016 value;
    pci_read_config_word( (struct pci_dev *) handle, offset, &value);
    return value;
}

U032 os_pci_read_dword(
    VOID *handle,
    U008 offset
) 
{
    U032 value;
    pci_read_config_dword( (struct pci_dev *) handle, offset, (u32 *) &value);
    return value;
}

VOID os_pci_write_byte(
    VOID *handle,
    U008 offset,
    U008 value
)
{
    pci_write_config_byte( (struct pci_dev *) handle, offset, value);
}

VOID os_pci_write_word(
    VOID *handle,
    U008 offset,
    U016 value
)
{
    pci_write_config_word( (struct pci_dev *) handle, offset, value);
}

VOID os_pci_write_dword(
    VOID *handle,
    U008 offset,
    U032 value
)
{
    pci_write_config_dword( (struct pci_dev *) handle, offset, value);
}

VOID os_io_write_byte(
    PHWINFO pdev,
    U032 address,
    U008 value
)
{
    outb(value, address);
}

VOID os_io_write_word(
    PHWINFO pdev,
    U032 address,
    U016 value
)
{
    outw(value, address);
}

VOID os_io_write_dword(
    PHWINFO pdev,
    U032 address,
    U032 value
)
{
    outl(value, address);
}

U008 os_io_read_byte(
    PHWINFO pdev,
    U032 address
)
{
    return inb(address);
}

U016 os_io_read_word(
    PHWINFO pdev,
    U032 address
)
{
    return inw(address);
}

U032 os_io_read_dword(
    PHWINFO pdev,
    U032 address
)
{
    return inl(address);
}

ULONG os_cli(ULONG flags)
{
    save_flags(flags);
    cli();
    return flags;
}

ULONG os_sti(ULONG flags)
{
    restore_flags(flags);
    return flags;
}


typedef struct wb_mem_t {
    unsigned int start;
    unsigned int size;
    struct wb_mem_t *next;
} wb_mem;

static wb_mem *wb_list = NULL;

int save_wb(unsigned int start, unsigned int size)
{
#ifdef CONFIG_MTRR
    struct wb_mem_t *ptr;

    NV_KMALLOC(ptr, sizeof(struct wb_mem_t));
    if (ptr == NULL)
        return 0;

    ptr->start = start;
    ptr->size  = size;
    ptr->next  = wb_list;
    wb_list    = ptr;

    return 1;
#endif
    return 0;
}

int del_wb(unsigned int start, unsigned int size)
{
#ifdef CONFIG_MTRR
    struct wb_mem_t *ptr;
    struct wb_mem_t *prev;

    prev = ptr = wb_list;

    while (ptr != NULL) {
        if (ptr->start == start && ptr->size == size) {
            if (ptr == wb_list)
                wb_list = ptr->next;
            else
                prev->next = ptr->next;

            NV_KFREE(ptr);
            return 1;
        }
        prev = ptr;
        ptr = prev->next;
    }
#endif
    return 0;
}

/*
 * Don't assume MTRR-specific, as we could add support for PATs to 
 * achieve the same results on a PIII or higher
 */

RM_STATUS os_set_mem_range(
    U032 start,
    U032 size,
    U032 mode
)
{
#ifdef CONFIG_MTRR
    int err;

    /* for now, we only set memory to write-combined */
    if (mode != NV_MEMORY_WRITECOMBINED)
        return RM_ERROR;

    err = mtrr_add(start, size, MTRR_TYPE_WRCOMB, 0x0);
    if (err < 0)
    {
        nv_printf(NV_DBG_ERRORS, "nvidia: cannot write-combine 0x%08x, %dM\n",
            start, size / (1024 * 1024));
        return RM_ERROR;
    }

    save_wb(start, size);

    return RM_OK;
#endif
    return RM_ERROR;
}

RM_STATUS os_unset_mem_range(
    U032 start,
    U032 size
)
{
#ifdef CONFIG_MTRR
    if (del_wb(start, size))
    {
        mtrr_del(-1, start, size);
    }
    return RM_OK;
#endif
    return RM_ERROR;
}


/*
 * This needs to remap the AGP Aperture from IO space to kernel space,
 * or at least return a pointer to a kernel space mapping of the Aperture
 * should this also check for Write-Combining??
 */

VOID *os_map_kernel_space(
    U032 start,
    U032 size_bytes,
    U032 mode
)
{
    void *vaddr;

    if (in_interrupt())
    {
        nv_printf(NV_DBG_ERRORS, "trying to map 0x%x to kernel space in interrupt!\n", start);
        os_dbg_breakpoint();
        return NULL;
    }

    if (mode == NV_MEMORY_DEFAULT) {
        vaddr = ioremap(start, size_bytes);
    } else {
        vaddr = ioremap_nocache(start, size_bytes);
    }

#if defined (KERNEL_2_2)
    if ((vaddr == NULL)) // && (mode == NV_MEMORY_DEFAULT))
    {
        unsigned long map_nr = MAP_NR(__va(start));
        if (map_nr < max_mapnr) {
            vaddr = __va(start);
        }
    }
#endif

#ifdef DEBUG
    if (mode == NV_MEMORY_WRITECOMBINED) {
        nv_printf(NV_DBG_ERRORS, 
            "nvidia: request to write-combine remapped memory!");
        os_dbg_breakpoint();
    }
    else if (mode != NV_MEMORY_UNCACHED) {
        nv_printf(NV_DBG_ERRORS, "nvidia: request to cache remapped memory!");
        os_dbg_breakpoint();
    }
#endif 

    return vaddr;
}

VOID os_unmap_kernel_space(
    VOID *addr,
    U032 size_bytes
)
{
#if defined (KERNEL_2_2)
    if (MAP_NR(addr) < max_mapnr) {
        // if we didn't want the memory cached, this isn't necessary
        // but we shouldn't be in a timing critical piece of code.
        asm volatile("wbinvd":::"memory");
    } else 
#endif
    {
        iounmap(addr);
    }
}

VOID* os_map_user_space(
    VOID *kaddr,
    VOID **priv,
    U032 size_bytes,
    U032 mode
)
{
    os_dbg_breakpoint();
    return NULL;
}

VOID os_unmap_user_space(
    VOID *uaddr,
    VOID *priv
)
{
}

VOID* os_map_io_space(
    U032 start,
    U032 size_bytes,
    VOID **priv,
    U032 user,
    U032 mode
)
{
    struct vm_area_struct *vma;

    if (user == 0 || priv == NULL || *priv == NULL)
        return NULL;

    vma = (struct vm_area_struct *) *priv;

    if (NV_REMAP_PAGE_RANGE(vma->vm_start,
                start & PAGE_MASK, size_bytes, PAGE_SHARED))
        return NULL;

    return (void *)(NV_UINTPTR_T) vma->vm_start;
}

VOID os_unmap_io_space(
    VOID *addr,
    U032 size_bytes,
    VOID *priv,
    U032 user
)
{
}

// flush the cpu's cache, uni-processor version
RM_STATUS os_flush_cpu_cache()
{
    CACHE_FLUSH();
    return RM_OK;
}

// override initial debug level from registry
VOID os_dbg_init(void)
{
    U032 new_debuglevel;
    if (RM_OK == rm_read_registry_dword(0,
                                     "",
                                     "ResmanDebugLevel",
                                     &new_debuglevel))
    {
        if (new_debuglevel != ~0)
            cur_debuglevel = new_debuglevel;
    }
}

VOID os_dbg_set_level(U032 new_debuglevel)
{
    nv_printf(NV_DBG_SETUP, "Changing debuglevel from 0x%x to 0x%x\n", cur_debuglevel, new_debuglevel);
    cur_debuglevel = new_debuglevel;
}

/* we used to have a global lock that we had to drop so incoming interrupts
 * wouldn't hang the machine waiting for the lock. In the switch to finer
 * grained locking, I've had to remove that code. We don't know which card has
 * a lock or which threw the breakpoint. I should probably scan the list of
 * nv_state_t's and drop any held locks before throwing this breakpoint.
 */
VOID os_dbg_breakpoint(void)
{
#ifdef DEBUG
    out_string("Break\n");

#if defined(CONFIG_X86_REMOTE_DEBUG)
     __asm__ __volatile__ ("int $3");
#elif defined(CONFIG_KDB)
    KDB_ENTER();
#else
    out_string("Skipping INT 3, we don't like kernel panics\n");
#endif

#endif /* DEBUG */
}


U032 os_get_cpu_count()
{
    return smp_num_cpus;
}



// SGI SwapGroup/SwapBarrier extension
// This requires special hardware plus an external driver
// XXX - add code to allocate struct as needed, for each barrier id

// #define TESTING_SWAP 1

#if TESTING_SWAP
// testing code, showing how I tested things

#define ADD_BARRIER_FUNC    "sgitest_add_barrier"
#define REMOVE_BARRIER_FUNC "sgitest_remove_barrier"
#define SWAP_READY_FUNC     "sgitest_swap_ready"

// functions added to serial driver in kernel, from shrijeet
extern int sgi_add_barrier(int id);
extern int sgi_remove_barrier(int id);
extern int sgi_swap_ready(int id, void (*callback)(void *), void *data);

int
sgitest_add_barrier(int id)
{
    return sgi_add_barrier(id);
}

int 
sgitest_remove_barrier(int id)
{
    return sgi_remove_barrier(id);
}

int 
sgitest_swap_ready(int id, void (*callback)(void *), void *data)
{
    return sgi_swap_ready(id, callback, data);
}
#else

#define ADD_BARRIER_FUNC    "sgi_add_barrier"
#define REMOVE_BARRIER_FUNC "sgi_remove_barrier"
#define SWAP_READY_FUNC     "sgi_swap_ready"

#endif


struct barrier_funcs {
    int (*add_barrier)(int id);
    int (*remove_barrier)(int id);
    int (*swap_ready)(int id, void (*callback)(void *), void *data);
};

static struct barrier_funcs sgi_funcs = { NULL, NULL, NULL };

RM_STATUS os_init_swap_barrier(U032 id)
{
    // if we haven't been called before, initialize
    if (sgi_funcs.add_barrier == NULL)
    {
#if defined(TESTING_SWAP)
#if !defined (KERNEL_2_2)
        inter_module_register(ADD_BARRIER_FUNC,    THIS_MODULE, sgitest_add_barrier);
        inter_module_register(REMOVE_BARRIER_FUNC, THIS_MODULE, sgitest_remove_barrier);
        inter_module_register(SWAP_READY_FUNC,     THIS_MODULE, sgitest_swap_ready);
#endif
#endif
        sgi_funcs.add_barrier = GET_MODULE_SYMBOL(0, ADD_BARRIER_FUNC);
        sgi_funcs.remove_barrier = GET_MODULE_SYMBOL(0, REMOVE_BARRIER_FUNC);
        sgi_funcs.swap_ready = GET_MODULE_SYMBOL(0, SWAP_READY_FUNC);
    }

    // failed to initialize
    if (sgi_funcs.add_barrier == NULL)
    {
#if defined(TESTING_SWAP)
        nv_printf(NV_DBG_ERRORS, "FAILED TO INITIALIZE SWAP!!\n");
#endif
        return RM_ERROR;
    }

    return sgi_funcs.add_barrier(id);
}

RM_STATUS os_remove_swap_barrier(U032 id)
{
    RM_STATUS rmStatus;

    if (sgi_funcs.remove_barrier == NULL)
        return RM_ERROR;

    rmStatus = sgi_funcs.remove_barrier(id);

    if (sgi_funcs.add_barrier)
    {
        PUT_MODULE_SYMBOL(ADD_BARRIER_FUNC);
        sgi_funcs.add_barrier = NULL;
    }

    if (sgi_funcs.remove_barrier)
    {
        PUT_MODULE_SYMBOL(REMOVE_BARRIER_FUNC);
        sgi_funcs.remove_barrier = NULL;
    }

    if (sgi_funcs.swap_ready)
    {
        PUT_MODULE_SYMBOL(SWAP_READY_FUNC);
        sgi_funcs.swap_ready = NULL;
    }

    return rmStatus;
}

RM_STATUS os_swap_barrier(
    U032 id,
    VOID (*callback)(VOID *),
    VOID *data
)
{
    if (sgi_funcs.swap_ready == NULL)
        return RM_ERROR;

    return sgi_funcs.swap_ready(id, callback, data);
}



