/*
 * nvidia-installer: A tool for installing NVIDIA software packages on
 * Unix and Linux systems.
 *
 * Copyright (C) 2003 NVIDIA Corporation
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the:
 *
 *      Free Software Foundation, Inc.
 *      59 Temple Place - Suite 330
 *      Boston, MA 02111-1307, USA
 *
 *
 * files.c - this source file contains routines for manipulating
 * files and directories for the nv-instaler.
 */

#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <dirent.h>
#include <string.h>
#include <fcntl.h>
#include <errno.h>
#include <sys/mman.h>
#include <sys/utsname.h>
#include <stdlib.h>
#include <limits.h>
#include <libgen.h>
#include <utime.h>
#include <time.h>

#include "nvidia-installer.h"
#include "user-interface.h"
#include "files.h"
#include "misc.h"
#include "precompiled.h"

/*
 * remove_directory() - recursively delete a direcotry (`rm -rf`)
 */

int remove_directory(Options *op, const char *victim)
{
    struct stat stat_buf;
    DIR *dir;
    struct dirent *ent;
    char *filename;
    int len;
    
    if (lstat(victim, &stat_buf) == -1) {
        ui_error(op, "failure to open '%s'", victim);
        return FALSE;
    }
    
    if (S_ISDIR(stat_buf.st_mode) == 0) {
        ui_error(op, "%s is not a directory", victim);
        return FALSE;
    }
    
    if ((dir = opendir(victim)) == NULL) {
        ui_error(op, "Failure reading directory %s", victim);
        return FALSE;
    }
    
    while ((ent = readdir(dir)) != NULL) {

        if (((strcmp(ent->d_name, ".")) == 0) ||
            ((strcmp(ent->d_name, "..")) == 0)) continue;
        
        len = strlen(victim) + strlen(ent->d_name) + 2;
        filename = (char *) nvalloc(len);
        snprintf(filename, len, "%s/%s", victim, ent->d_name);
        
        if (lstat(filename, &stat_buf) == -1) {
            ui_error(op, "failure to open '%s'", filename);
            free(filename);
            return FALSE;
        }
        
        if (S_ISDIR(stat_buf.st_mode)) {
            remove_directory(op, filename);
        } else {
            if (unlink(filename) != 0) {
                ui_error(op, "Failure removing file %s (%s)",
                         filename, strerror(errno));
            }
        }
        free(filename);
    }

    if (rmdir(victim) != 0) {
        ui_error(op, "Failure removing directory %s (%s)",
                 victim, strerror(errno));
        return FALSE;
    }

    return TRUE;
    
} /* remove_directory() */



/*
 * copy_file() - copy the file specified by srcfile to dstfile, using
 * mmap and memcpy.  The destination file is created with the
 * permissions specified by mode.  Roughly based on code presented by
 * Richard Stevens, in Advanced Programming in the Unix Environment,
 * 12.9.
 */

int copy_file(Options *op, const char *srcfile,
              const char *dstfile, mode_t mode)
{
    int src_fd, dst_fd;
    struct stat stat_buf;
    char *src, *dst;
    
    if ((src_fd = open(srcfile, O_RDONLY)) == -1) {
        ui_error (op, "Unable to open '%s' for copying (%s)",
                  srcfile, strerror (errno));
        goto fail;
    }
    if ((dst_fd = open(dstfile, O_RDWR | O_CREAT | O_TRUNC, mode)) == -1) {
        ui_error (op, "Unable to create '%s' for copying (%s)",
                  dstfile, strerror (errno));
        goto fail;
    }
    if (fstat(src_fd, &stat_buf) == -1) {
        ui_error (op, "Unable to determine size of '%s' (%s)",
                  srcfile, strerror (errno));
        goto fail;
    }
    if (lseek(dst_fd, stat_buf.st_size - 1, SEEK_SET) == -1) {
        ui_error (op, "Unable to set file size for '%s' (%s)",
                  dstfile, strerror (errno));
        goto fail;
    }
    if (write(dst_fd, "", 1) != 1) {
        ui_error (op, "Unable to write file size for '%s' (%s)",
                  dstfile, strerror (errno));
        goto fail;
    }
    if ((src = mmap(0, stat_buf.st_size, PROT_READ,
                    MAP_FILE | MAP_SHARED, src_fd, 0)) == (void *) -1) {
        ui_error (op, "Unable to map source file '%s' for copying (%s)",
                  srcfile, strerror (errno));
        goto fail;
    }
    if ((dst = mmap(0, stat_buf.st_size, PROT_READ | PROT_WRITE,
                    MAP_FILE | MAP_SHARED, dst_fd, 0)) == (void *) -1) {
        ui_error (op, "Unable to map destination file '%s' for copying (%s)",
                  dstfile, strerror (errno));
        goto fail;
    }
    
    memcpy (dst, src, stat_buf.st_size);
    
    if (munmap (src, stat_buf.st_size) == -1) {
        ui_error (op, "Unable to unmap source file '%s' after copying (%s)",
                 srcfile, strerror (errno));
        goto fail;
    }
    if (munmap (dst, stat_buf.st_size) == -1) {
        ui_error (op, "Unable to unmap destination file '%s' after "
                 "copying (%s)", dstfile, strerror (errno));
        goto fail;
    }
    close (src_fd);
    close (dst_fd);
    
    return TRUE;

 fail:
    return FALSE;

} /* copy_file() */
 

/*
 * select_tls_class() - determine which tls class should be installed
 * on the user's machine; if tls_test() fails, just install the
 * classic tls libraries.  If tls_test() passes, install both OpenGL
 * sets, but only the new tls libglx.
 */

void select_tls_class(Options *op, Package *p)
{
    int i;

    if (!tls_test(op)) {
        
        /*
         * tls libraries will not run on this system; just install the
         * classic OpenGL libraries: clear the FILE_TYPE of any
         * FILE_CLASS_NEW_TLS package entries.
         */

        ui_log(op, "Installing classic TLS OpenGL libraries.");
        
        for (i = 0; i < p->num_entries; i++) {
            if (p->entries[i].flags & FILE_CLASS_NEW_TLS) {
                p->entries[i].flags &= ~FILE_TYPE_MASK;
                p->entries[i].dst = NULL;
            }
        }
    } else {

        /*
         * tls libraries will run on this system: install both the
         * classic and new TLS libraries.  However, since the XFree86
         * loader is not equiped to select between tls and non tls
         * libraries like the glibc loader is, clear the FILE_TYPE of
         * any (FILE_TYPE_XFREE86_LIB | FILE_CLASS_CLASSIC_TLS)
         * package entries.
         */

        ui_log(op, "Installing both new and classic TLS OpenGL libraries "
               "(but only new XFree86 TLS libraries.");

        for (i = 0; i < p->num_entries; i++) {
            if ((p->entries[i].flags &
                 (FILE_TYPE_XFREE86_LIB | FILE_CLASS_CLASSIC_TLS)) ==
                (FILE_TYPE_XFREE86_LIB | FILE_CLASS_CLASSIC_TLS)) {
                p->entries[i].flags &= ~FILE_TYPE_MASK;
                p->entries[i].dst = NULL;
            }
        }
    }
} /* select_tls_class() */


/*
 * set_destinations() - given the Options and Package structures,
 * assign the destination field in each Package entry, building from
 * the OpenGL and XFree86 prefixes, the path relative to the prefix,
 * and the filename.  This assumes that the prefixes have already been
 * assigned in the Options struct.
 */

int set_destinations(Options *op, Package *p)
{
    char *prefix, *path, *name;
    int i;
    
    for (i = 0; i < p->num_entries; i++) {

        switch (p->entries[i].flags & FILE_TYPE_MASK) {
            
        case FILE_TYPE_KERNEL_MODULE_CMD:
        case FILE_TYPE_KERNEL_MODULE_SRC:
            
            /* we don't install kernel module sources */
            
            p->entries[i].dst = NULL;
            continue;
            
        case FILE_TYPE_OPENGL_LIB:
        case FILE_TYPE_OPENGL_SYMLINK:
            prefix = op->opengl_prefix;
            path = p->entries[i].path;
            break;
            
        case FILE_TYPE_XFREE86_LIB:
        case FILE_TYPE_XFREE86_SYMLINK:
            prefix = op->xfree86_prefix;
            path = p->entries[i].path;
            break;

            /*
             * XXX should the OpenGL headers and documentation also go
             * under the OpenGL installation prefix?  The Linux OpenGL
             * ABI requires that the header files be installed in
             * /usr/include/GL/.
             */

        case FILE_TYPE_OPENGL_HEADER:
            prefix = op->opengl_prefix;
            path = OPENGL_HEADER_DST_PATH;
            break;
            
        case FILE_TYPE_DOCUMENTATION:
            prefix = op->opengl_prefix;
            path = p->entries[i].path;
            break;
        
        case FILE_TYPE_INSTALLER_BINARY:
            prefix = op->installer_prefix;
            path = INSTALLER_BINARY_DST_PATH;
            break;
            
        case FILE_TYPE_NVIDIA_SETTINGS_BINARY:
            prefix = op->nvidia_settings_prefix;
            path = NVIDIA_SETTINGS_BINARY_DST_PATH;
            break;

        case FILE_TYPE_KERNEL_MODULE:
            
            /*
             * the kernel module dst field has already been
             * initialized in add_kernel_module_to_package()
             */
            
            continue;

        default:
            
            /* 
             * silently ignore anything that doesn't match; libraries
             * of the wrong TLS class may fall in here, for example.
             */
            
            p->entries[i].dst = NULL;
            continue;
        }
        
        name = p->entries[i].name;

        p->entries[i].dst = nvstrcat(prefix, "/", path, "/", name, NULL);
    }
    
    return TRUE;

} /* set_destinations() */



/*
 * get_license_acceptance() - stat the license file to find out its
 * length, open the file, mmap it, and pass it to the ui for
 * acceptance.
 */

int get_license_acceptance(Options *op)
{
    struct stat buf;
    char *text;
    int fd;

    /* trivial accept if the user accepted on the command line */

    if (op->accept_license) {
        ui_log(op, "License accepted by command line option.");
        return TRUE;
    }

    if ((fd = open(LICENSE_FILE, 0x0)) == -1) goto failed;
   
    if (fstat(fd, &buf) != 0) goto failed;
    
    if ((text = (char *) mmap(NULL, buf.st_size, PROT_READ,
                              MAP_FILE|MAP_SHARED,
                              fd, 0x0)) == (char *) -1) goto failed;
    
    if (!ui_display_license(op, text)) {
        ui_log(op, "License not accepted.  Aborting installation.");
        munmap(text, buf.st_size);
        close(fd);
        return FALSE;
    }
    
    ui_log(op, "License accepted.");
    
    munmap(text, buf.st_size);
    close(fd);
    
    return TRUE;

 failed:
    
    ui_error(op, "Unable to open License file '%s' (%s)",
             LICENSE_FILE, strerror(errno));
    return FALSE;

} /* get_license_acceptance() */



/*
 * get_prefixes() - if in expert mode, ask the user for the OpenGL and
 * XFree86 installation prefix.  The default prefixes are already set
 * in parse_commandline().
 */

int get_prefixes (Options *op)
{
    char *ret;
    
    if (op->expert) {
        ret = ui_get_input(op, op->xfree86_prefix,
                           "XFree86 installation prefix (only under "
                           "rare circumstances should this be changed "
                           "from the default)");
        if (ret && ret[0]) {
            op->xfree86_prefix = ret; 
            if (!confirm_path(op, op->xfree86_prefix)) return FALSE;
        }
    }

    remove_trailing_slashes(op->xfree86_prefix);
    ui_expert(op, "X installation prefix is: '%s'", op->xfree86_prefix);
        
    if (op->expert) {
        ret = ui_get_input(op, op->opengl_prefix,
                           "OpenGL installation prefix (only under "
                           "rare circumstances should this be changed "
                           "from the default)");
        if (ret && ret[0]) {
            op->opengl_prefix = ret;
            if (!confirm_path(op, op->opengl_prefix)) return FALSE;
        }
    }

    remove_trailing_slashes(op->opengl_prefix);
    ui_expert(op, "OpenGL installation prefix is: '%s'", op->opengl_prefix);

    if (op->expert) {
        ret = ui_get_input(op, op->installer_prefix,
                           "Installer installation prefix");
        if (ret && ret[0]) {
            op->installer_prefix = ret;
            if (!confirm_path(op, op->installer_prefix)) return FALSE;
        }
    }
    
    remove_trailing_slashes(op->installer_prefix);
    ui_expert(op, "Installer installation prefix is: '%s'",
              op->installer_prefix);
    
    return TRUE;
    
} /* get_prefixes() */



/*
 * add_kernel_module_to_package() - append the kernel module
 * (contained in p->kernel_module_build_directory) to the package list
 * for installation.
 */

int add_kernel_module_to_package(Options *op, Package *p)
{
    int n, len;

    n = p->num_entries;
    
    p->entries =
        (PackageEntry *) nvrealloc(p->entries, (n + 1) * sizeof(PackageEntry));

    len = strlen(p->kernel_module_build_directory) +
        strlen(p->kernel_module_filename) + 2;
    p->entries[n].file = (char *) nvalloc(len);
    snprintf(p->entries[n].file, len, "%s/%s",
             p->kernel_module_build_directory, p->kernel_module_filename);
    
    p->entries[n].path = NULL;
    p->entries[n].target = NULL;
    p->entries[n].flags = FILE_TYPE_KERNEL_MODULE;
    p->entries[n].mode = 0644;
    
    p->entries[n].name = strrchr(p->entries[n].file, '/');
    if (p->entries[n].name) p->entries[n].name++;
    if (!p->entries[n].name) p->entries[n].name = p->entries[n].file;
    
    len = strlen(op->kernel_module_installation_path) +
        strlen(p->kernel_module_filename) + 2;
    p->entries[n].dst = (char *) nvalloc(len);
    snprintf (p->entries[n].dst, len, "%s/%s",
              op->kernel_module_installation_path, p->kernel_module_filename);
    
    p->num_entries++;

    return TRUE;

} /* add_kernel_module_to_package() */



/*
 * remove_trailing_slashes() - begin at the end of the given string,
 * and overwrite slashes with NULL as long as we find slashes.
 */

void remove_trailing_slashes(char *s)
{
    int len = strlen(s);

    while (s[len-1] == '/') s[--len] = '\0';
    
} /* remove_trailing_slashes() */



/*
 * mode_string_to_mode() - convert the string s 
 */

int mode_string_to_mode(Options *op, char *s, mode_t *mode)
{
    char *endptr;
    long int ret;
    
    ret = strtol(s, &endptr, 8);

    if ((ret == LONG_MIN) || (ret == LONG_MAX) || (*endptr != '\0')) {
        ui_error(op, "Error parsing permission string '%s' (%s)",
                 s, strerror (errno));
        return FALSE;
    }

    *mode = (mode_t) ret;

    return TRUE;
    
} /* mode_string_to_mode() */



/*
 * mode_to_permission_string() - given a mode bitmask, allocate and
 * write a permission string.
 */

char *mode_to_permission_string(mode_t mode)
{
    char *s = (char *) nvalloc(10);
    memset (s, '-', 9);
    
    if (mode & (1 << 8)) s[0] = 'r';
    if (mode & (1 << 7)) s[1] = 'w';
    if (mode & (1 << 6)) s[2] = 'x';
    
    if (mode & (1 << 5)) s[3] = 'r';
    if (mode & (1 << 4)) s[4] = 'w';
    if (mode & (1 << 3)) s[5] = 'x';
    
    if (mode & (1 << 2)) s[6] = 'r';
    if (mode & (1 << 1)) s[7] = 'w';
    if (mode & (1 << 0)) s[8] = 'x';
    
    s[9] = '\0';
    return s;

} /* mode_to_permission_string() */



/*
 * directory_exists() - 
 */

int directory_exists(Options *op, const char *dir)
{
    struct stat stat_buf;

    if ((stat (dir, &stat_buf) == -1) || (!S_ISDIR(stat_buf.st_mode))) {
        return FALSE;
    } else {
        return TRUE;
    }
} /* directory_exists() */



/*
 * confirm_path() - check that the path exists; if not, ask the user
 * if it's OK to create it and then do mkdir().
 *
 * XXX for a while, I had thought that it would be better for this
 * function to only ask the user if it was OK to create the directory;
 * there are just too many reasons why mkdir might fail, though, so
 * it's better to do mkdir() so that we can fail at this point in the
 * installation.
 */

int confirm_path(Options *op, const char *path)
{
    /* return TRUE if the path already exists and is a directory */

    if (directory_exists(op, path)) return TRUE;
    
    if (ui_yes_no(op, TRUE, "The directory '%s' does not exist; "
                  "create?", path)) {
        if (mkdir_recursive(op, path, 0755)) {
            return TRUE;
        } else {
            return FALSE;
        }
    }
    
    ui_message(op, "Not creating directory '%s'; aborting installation.",
               path);
    
    return FALSE;

} /* confirm_path() */



/* 
 * mkdir_recursive() - create the path specified, also creating parent
 * directories as needed; this is equivalent to `mkdir -p`
 */

int mkdir_recursive(Options *op, const char *path, const mode_t mode)
{
    char *c, *tmp, ch;
    
    if (!path || !path[0]) return FALSE;
        
    tmp = nvstrdup(path);
    remove_trailing_slashes(tmp);
        
    c = tmp;
    do {
        c++;
        if ((*c == '/') || (*c == '\0')) {
            ch = *c;
            *c = '\0';
            if (!directory_exists(op, tmp)) {
                if (mkdir(tmp, mode) != 0) {
                    ui_error(op, "Failure creating directory '%s': (%s)",
                             tmp, strerror(errno));
                    free(tmp);
                    return FALSE;
                }
            }
            *c = ch;
        }
    } while (*c);

    free(tmp);
    return TRUE;

} /* mkdir_recursive() */



/*
 * get_symlink_target() - get the target of the symbolic link
 * 'filename'.  On success, a newly malloced string containing the
 * target is returned.  On error, an error message is printed and NULL
 * is returned.
 */

char *get_symlink_target(Options *op, const char *filename)
{
    struct stat stat_buf;
    int ret, len = 0;
    char *buf = NULL;

    if (lstat(filename, &stat_buf) == -1) {
        ui_error(op, "Unable to get file properties for '%s' (%s).",
                 filename, strerror(errno));
        return NULL;
    }
    
    if (!S_ISLNK(stat_buf.st_mode)) {
        ui_error(op, "File '%s' is not a symbolic link.", filename);
        return NULL;
    }
    
    /*
     * grow the buffer to be passed into readlink(2), until the buffer
     * is big enough to hold the whole target.
     */

    do {
        len += NV_LINE_LEN;
        if (buf) free(buf);
        buf = nvalloc(len);
        ret = readlink(filename, buf, len - 1);
        if (ret == -1) {
            ui_error(op, "Failure while reading target of symbolic "
                     "link %s (%s).", filename, strerror(errno));
            free(buf);
            return NULL;
        }
    } while (ret >= (len - 1));

    buf[ret] = '\0';
    
    return buf;

} /* get_symlink_target() */



/*
 * install_file() - install srcfile as dstfile; this is done by
 * extracting the directory portion of dstfile, and then calling
 * copy_file().
 */ 

int install_file(Options *op, const char *srcfile,
                 const char *dstfile, mode_t mode)
{   
    int retval; 
    char *dirc, *dname;

    dirc = nvstrdup(dstfile);
    dname = dirname(dirc);
    
    if (!mkdir_recursive(op, dname, 0755)) {
        free(dirc);
        return FALSE;
    }

    retval = copy_file(op, srcfile, dstfile, mode);
    free(dirc);

    return retval;

} /* install_file() */



size_t get_file_size(Options *op, const char *filename)
{
    struct stat stat_buf;
    
    if (stat(filename, &stat_buf) == -1) {
        ui_error(op, "Unable to determine file size of '%s' (%s).",
                 filename, strerror(errno));
        return 0;
    }

    return stat_buf.st_size;

} /* get_file_size() */



size_t fget_file_size(Options *op, const int fd)
{
    struct stat stat_buf;
    
    if (fstat(fd, &stat_buf) == -1) {
        ui_error(op, "Unable to determine file size of file "
                 "descriptor %d (%s).", fd, strerror(errno));
        return 0;
    }

    return stat_buf.st_size;

} /* fget_file_size() */



char *get_tmpdir(Options *op)
{
    char *tmpdirs[] = { NULL, "/tmp", ".", NULL };
    int i;

    tmpdirs[0] = getenv("TMPDIR");
    tmpdirs[3] = getenv("HOME");
    
    for (i = 0; i < 4; i++) {
        if (tmpdirs[i] && directory_exists(op, tmpdirs[i])) {
            return (tmpdirs[i]);
        }
    }
    
    return NULL;

} /* get_tmpdir() */



/*
 * nvrename() - replacement for rename(2), because rename(2) can't
 * cross filesystem boundaries.  Get the src file attributes, copy the
 * src file to the dst file, stamp the dst file with the src file's
 * timestamp, and delete the src file.  Returns FALSE on error, TRUE
 * on success.
 */

int nvrename(Options *op, const char *src, const char *dst)
{
    struct stat stat_buf;
    struct utimbuf utime_buf;

    if (stat(src, &stat_buf) == -1) {
        ui_error(op, "Unable to determine file attributes of file "
                 "%s (%s).", src, strerror(errno));
        return FALSE;
    }
        
    if (!copy_file(op, src, dst, stat_buf.st_mode)) return FALSE;

    utime_buf.actime = stat_buf.st_atime; /* access time */
    utime_buf.modtime = stat_buf.st_mtime; /* modification time */

    if (utime(dst, &utime_buf) == -1) {
        ui_warn(op, "Unable to transfer timestamp from '%s' to '%s' (%s).",
                   src, dst, strerror(errno));
    }
    
    if (unlink(src) == -1) {
        ui_error(op, "Unable to delete '%s' (%s).", src, strerror(errno));
        return FALSE;
    }

    return TRUE;
    
} /* nvrename() */



/*
 * check_for_existing_rpms() - check if any of the previous NVIDIA
 * rpms are installed on the system.  If we find any, ask the user if
 * we may remove them.
 */

int check_for_existing_rpms(Options *op)
{
    /* list of rpms to remove; should be in dependency order */

    const char *rpms[2] = { "NVIDIA_GLX", "NVIDIA_kernel" };

    char *data, *cmd;
    int i, ret;

    for (i = 0; i < 2; i++) {
        
        cmd = nvstrcat("rpm --query ", rpms[i], NULL);
        ret = run_command(op, cmd, NULL, FALSE, 0);
        nvfree(cmd);

        if (ret == 0) {
            if (!ui_yes_no(op, TRUE, "An %s rpm appears to already be "
                           "installed on your system.  As part of installing "
                           "the new driver, this %s rpm will be uninstalled.  "
                           "Are you sure you want to continue? ('no' will "
                           "abort installation)", rpms[i], rpms[i])) {
                ui_log(op, "Installation aborted.");
                return FALSE;
            }
            
            cmd = nvstrcat("rpm --erase --nodeps ", rpms[i], NULL);
            ret = run_command(op, cmd, &data, op->expert, 0);
            nvfree(cmd);
            
            if (ret == 0) {
                ui_log(op, "Removed %s.", rpms[i]);
            } else {
                ui_warn(op, "Unable to erase %s rpm: %s", rpms[i], data);
            }
            
            nvfree(data);
        }
    }

    return TRUE;
    
} /* check_for_existing_rpms() */



/*
 * copy_directory_contents() - copy the contents of directory src to
 * directory dst.  This only copies files; subdirectories are ignored.
 */

int copy_directory_contents(Options *op, const char *src, const char *dst)
{
    DIR *dir;
    struct dirent *ent;
    char *srcfile, *dstfile;
    struct stat stat_buf;

    if ((dir = opendir(src)) == NULL) {
        ui_error(op, "Unable to open directory '%s' (%s).",
                 src, strerror(errno));
        return FALSE;
    }

    while ((ent = readdir(dir)) != NULL) {
        
        if (((strcmp(ent->d_name, ".")) == 0) ||
            ((strcmp(ent->d_name, "..")) == 0)) continue;
        
        srcfile = nvstrcat(src, "/", ent->d_name, NULL);

        /* only copy regular files */

        if ((stat(srcfile, &stat_buf) == -1) || !(S_ISREG(stat_buf.st_mode))) {
            nvfree(srcfile);
            continue;
        }
        
        dstfile = nvstrcat(dst, "/", ent->d_name, NULL);
        
        if (!copy_file(op, srcfile, dstfile, stat_buf.st_mode)) return FALSE;

        nvfree(srcfile);
        nvfree(dstfile);
    }

    if (closedir(dir) != 0) {
        ui_error(op, "Failure while closing directory '%s' (%s).",
                 src, strerror(errno));
        
        return FALSE;
    }
    
    return TRUE;
    
} /* copy_directory_contents() */



/*
 * pack_precompiled_kernel_interface() - 
 */

int pack_precompiled_kernel_interface(Options *op, Package *p)
{
    char *cmd, time_str[256], *proc_version_string;
    char major[16], minor[16], patch[16];
    char *result, *descr;
    time_t t;
    struct utsname buf;
    int ret;

    ui_log(op, "Packaging precompiled kernel interface.");

    /* make sure the precompiled_kernel_interface_directory exists */

    mkdir_recursive(op, p->precompiled_kernel_interface_directory, 0755);
    
    /* use the time in the output string... should be fairly unique */

    t = time(NULL);
    snprintf(time_str, 256, "%lu", t);
    
    /* read the proc version string */

    proc_version_string = read_proc_version(op);

    /* get the version strings */

    snprintf(major, 16, "%d", p->major);
    snprintf(minor, 16, "%d", p->minor);
    snprintf(patch, 16, "%d", p->patch);
    
    /* use the uname string as the description */

    uname(&buf);
    descr = nvstrcat(buf.sysname, " ",
                     buf.release, " ",
                     buf.version, " ",
                     buf.machine, NULL);
    
    /* build the mkprecompiled command */

    cmd = nvstrcat("./usr/bin/mkprecompiled --interface=",
                   p->kernel_module_build_directory, "/",
                   PRECOMPILED_KERNEL_INTERFACE_FILENAME,
                   " --output=", p->precompiled_kernel_interface_directory,
                   "/", PRECOMPILED_KERNEL_INTERFACE_FILENAME,
                   "-", p->version_string, ".", time_str,
                   " --description=\"", descr, "\"",
                   " --proc-version=\"", proc_version_string, "\"",
                   " --major=", major,
                   " --minor=", minor,
                   " --patch=", patch, NULL);

    /* execute the command */
    
    ret = run_command(op, cmd, &result, FALSE, 0);
    
    nvfree(cmd);
    nvfree(proc_version_string);
    nvfree(descr);
    
    /* remove the old kernel interface file */

    cmd = nvstrcat(p->kernel_module_build_directory, "/",
                   PRECOMPILED_KERNEL_INTERFACE_FILENAME, NULL);
    
    unlink(cmd); /* XXX what to do if this fails? */

    nvfree(cmd);
    
    if (ret != 0) {
        ui_error(op, "Unable to package precompiled kernel interface: %s",
                 result);
    }

    nvfree(result);

    if (ret == 0) return TRUE;
    else return FALSE;
    
} /* pack_kernel_interface() */
