/*
 * nvidia-installer: A tool for installing NVIDIA software packages on
 * Unix and Linux systems.
 *
 * Copyright (C) 2003 NVIDIA Corporation
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the:
 *
 *      Free Software Foundation, Inc.
 *      59 Temple Place - Suite 330
 *      Boston, MA 02111-1307, USA
 */

#include <string.h>
#include <unistd.h>
#include <errno.h>
#include <sys/utsname.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <ctype.h>
#include <stdlib.h>
#include <dirent.h>
#include <fcntl.h>
#include <sys/mman.h>

#include "nvidia-installer.h"
#include "kernel.h"
#include "user-interface.h"
#include "files.h"
#include "misc.h"
#include "precompiled.h"
#include "snarf.h"
#include "crc.h"

/* local prototypes */

static char *default_kernel_module_installation_path(Options *op);
static char *default_kernel_include_path(Options *op);
static int check_for_loaded_kernel_module(Options *op, const char *);
static int rmmod_kernel_module(Options *op, const char *);
static PrecompiledInfo *download_updated_kernel_interface(Options*, Package*,
                                                          const char*);

/*
 * Message text that is used by several error messages.
 */

static const char install_your_kernel_headers[] = 
"Please make sure you have installed the kernel header files "
"for your kernel; on Red Hat Linux systems, for example, be "
"sure you have the 'kernel-source' rpm installed.  If you know the "
"correct kernel header files are installed, you may specify the "
"kernel include path with the '--kernel-include-path' "
"commandline option.";

 


/*
 * determine_kernel_module_installation_path() - get the installation
 * path for the kernel module.  The order is:
 *
 * - if op->kernel_module_installation_path is non-NULL, then it must
 *   have been initialized by the commandline parser, and therefore we
 *   should obey that (so just return).
 *
 * - get the default installation path
 *
 * - if in expert mode, ask the user, and use what they gave, if
 *   non-NULL
 */

int determine_kernel_module_installation_path(Options *op)
{
    char *result;
    int count = 0;
    
    if (op->kernel_module_installation_path) return TRUE;
    
    op->kernel_module_installation_path =
        default_kernel_module_installation_path(op);
    
    if (!op->kernel_module_installation_path) return FALSE;

    if (op->expert) {

    ask_for_kernel_install_path:

        result = ui_get_input(op, op->kernel_module_installation_path,
                              "Kernel module installation path");
        if (result && result[0]) {
            free(op->kernel_module_installation_path);
            op->kernel_module_installation_path = result;
            if (!confirm_path(op, op->kernel_module_installation_path)) {
                return FALSE;
            }
        } else {
            if (result) free(result);
            
            if (++count < NUM_TIMES_QUESTIONS_ASKED) {
                ui_warn(op, "Invalid kernel module installation path.");
                goto ask_for_kernel_install_path;
            } else {
                ui_error(op, "Unable to determine kernel module "
                         "installation path.");

                return FALSE;
            }
        }
    }

    if (!mkdir_recursive(op, op->kernel_module_installation_path, 0755))
        return FALSE;

    ui_expert(op, "Kernel module installation path: %s",
              op->kernel_module_installation_path);

    return TRUE;
    
} /* determine_kernel_module_installation_path() */



/*
 * determine_kernel_include_path() - find the include path to the
 * kernel header files.  This is called from install_from_cwd() if we
 * need to compile the kernel interface files.  Assigns
 * op->kernel_include_path and returns TRUE if successful.  Returns
 * FALSE if no kernel header files were found.
 */

int determine_kernel_include_path(Options *op)
{
    char *result;
    int count = 0;
    
    /* determine the kernel include path */
    
    op->kernel_include_path = default_kernel_include_path(op);
    
    if (op->expert) {
        
    ask_for_kernel_include_path:
        
        result = ui_get_input(op, op->kernel_include_path,
                              "Kernel include path");
        if (result && result[0]) {
            if (!directory_exists(op, result)) {
                ui_warn(op, "Kernel include path '%s' does not exist.",
                        result);
                free(result);
                
                if (++count < NUM_TIMES_QUESTIONS_ASKED) {
                    goto ask_for_kernel_include_path;
                } else {
                    op->kernel_include_path = NULL;
                }
            } else {
                op->kernel_include_path = result;
            }
        } else {
            ui_warn(op, "Invalid kernel include path.");
            if (result) free(result);
            
            if (++count < NUM_TIMES_QUESTIONS_ASKED) {
                goto ask_for_kernel_include_path;
            } else {
                op->kernel_include_path = NULL;
            }
        }
    }

    /* if we STILL don't have a kernel include path, give up */
    
    if (!op->kernel_include_path) {
        ui_error(op, "Unable to find the kernel header files for the "
                 "currently running kernel.  %s", install_your_kernel_headers);
        
        /*
         * I suppose we could ask them here for the kernel include
         * path, but we've already given them multiple methods of
         * specifying their kernel headers
         */
        
        return FALSE;
    }
    
    /* check that the kernel include path exists */

    if (!directory_exists(op, op->kernel_include_path)) {
        ui_error (op, "The kernel include path '%s' does not exist.  %s",
                  op->kernel_include_path, install_your_kernel_headers);
        op->kernel_include_path = NULL;
        return FALSE;
    }

    /* check that <path>/linux/kernel.h exists */

    result = nvstrcat(op->kernel_include_path, "/linux/kernel.h", NULL);
    if (access(result, F_OK) == -1) {
        ui_error(op, "The kernel header file '%s' does not exist.  "
                 "The most likely reason for this is that the kernel include "
                 "path '%s' is incorrect.  %s", result,
                 op->kernel_include_path, install_your_kernel_headers);
        free(result);
        return FALSE;
    }
    free(result);

    /* check that <path>/linux/modversions.h exists */

    result = nvstrcat(op->kernel_include_path, "/linux/modversions.h", NULL);
    if (access(result, F_OK) == -1) {
        ui_error(op, "The kernel header file '%s' does not exist.  "
                 "The most likely reason for this is that the kernel header "
                 "files in '%s' have not been configured.",
                 result, op->kernel_include_path);
        free(result);
        return FALSE;
    }
    free(result);
    
    /* OK, we seem to have a path to configured kernel headers */
    
    ui_log(op, "Kernel include path: '%s'\n", op->kernel_include_path);
    
    return TRUE;

} /* determine_kernel_include_path() */



/*
 * link_kernel_module() - link the prebuilt kernel interface against
 * the binary-only core of the kernel module.  This results in a
 * complete kernel module, ready for installation.
 *
 *
 * ld -r -o nvidia.o nv-linux.o nv-kernel.o
 */

int link_kernel_module(Options *op, Package *p)
{
    char *cmd, *result;
    int len, ret;
    
    len = strlen(p->kernel_module_build_directory) +
        strlen(op->utils[LD]) +
        strlen(LD_OPTIONS) +
        strlen(p->kernel_module_filename) +
        strlen(PRECOMPILED_KERNEL_INTERFACE_FILENAME) + 32;

    cmd = (char *) nvalloc(len);

    snprintf(cmd, len, "cd %s; %s %s -o %s %s nv-kernel.o",
             p->kernel_module_build_directory,
             op->utils[LD],
             LD_OPTIONS,
             p->kernel_module_filename,
             PRECOMPILED_KERNEL_INTERFACE_FILENAME);
    
    ret = run_command(op, cmd, &result, TRUE, 0);

    if (ret != 0) {
        ui_error(op, "Unable to link kernel module.");
        return FALSE;
    }

    ui_log(op, "Kernel module linked successfully.");

    return TRUE;
   
} /* link_kernel_module() */


/*
 * build_kernel_module() - determine the kernel include directory,
 * copy the kernel module source files into a temporary directory, and
 * compile nvidia.o.
 *
 * XXX depends on things like make, gcc, ld, existing.  Should we
 * check that the user has these before doing this?
 */

int build_kernel_module(Options *op, Package *p)
{
    char *result, *cmd, *tmp;
    int len, ret;
    
    ui_log(op, "Cleaning kernel module build directory.");
    
    len = strlen(p->kernel_module_build_directory) + 32;
    cmd = nvalloc(len);
    
    snprintf(cmd, len, "cd %s; make clean", p->kernel_module_build_directory);

    ret = run_command(op, cmd, &result, TRUE, 0);
    free(result);
    free(cmd);
    
    ui_status_begin(op, "Building kernel module:", "Building");
    
    len = strlen(op->kernel_include_path) +
        strlen(p->kernel_module_build_directory) +
        strlen(p->kernel_module_filename) + 32;
    cmd = nvalloc(len);

    snprintf(cmd, len, "cd %s; make %s SYSINCLUDE=%s",
             p->kernel_module_build_directory,
             p->kernel_module_filename, op->kernel_include_path);
    
    ret = run_command(op, cmd, &result, TRUE, 25);

    free(cmd);

    if (ret != 0) {
        ui_status_end(op, "Error.");
        ui_error(op, "Unable to build the NVIDIA kernel module.");
        /* XXX need more descriptive error message */
        return FALSE;
    }
    
    /* check that the file actually exists */

    tmp = nvstrcat(p->kernel_module_build_directory, "/",
                   p->kernel_module_filename, NULL);
    if (access(tmp, F_OK) == -1) {
        free(tmp);
        ui_status_end(op, "Error.");
        ui_error(op, "The NVIDIA kernel module was not created.");
        return FALSE;
    }
    free(tmp);
    
    ui_status_end(op, "done.");

    ui_log(op, "Kernel module compilation complete.");

    return TRUE;
    
} /* build_kernel_module() */


/*
 * test_kernel_module() - attempt to insmod the kernel module and then
 * rmmod it.  Return TRUE if the insmod succeeded, or FALSE otherwise.
 *
 * Pass the special silence_nvidia_output option to the kernel module
 * to prevent any console output while testing.
 */

int test_kernel_module(Options *op, Package *p)
{
    char *cmd = NULL, *data;
    int ret;

    /* 
     * If we're building/installing for a different kernel, then we
     * can't test the module now.
     */

    if (op->kernel_name) return TRUE;

    cmd = nvstrcat(op->utils[INSMOD], " ",
                   p->kernel_module_build_directory, "/",
                   p->kernel_module_filename, " silence_nvidia_output=1",
                   NULL);
    
    /* only output the result of the test if in expert mode */

    ret = run_command(op, cmd, &data, op->expert, 0);

    if (ret != 0) {
        ui_error(op, "Unable to load the kernel module '%s'.  This is "
                 "most likely because the kernel module was built using "
                 "the wrong kernel header files.  %s",
                 p->kernel_module_filename, install_your_kernel_headers);

        /*
         * if in expert mode, run_command() would have caused this to
         * be written to the log file; so if not in expert mode, print
         * the output now.
         */

        if (!op->expert) ui_log(op, "Kernel module load error: %s", data);
        ret = FALSE;
    } else {
        free(cmd);
        cmd = nvstrcat(op->utils[RMMOD], " ", p->kernel_module_name, NULL);
        run_command(op, cmd, NULL, FALSE, 0); /* what if we fail to rmmod? */
        ret = TRUE;
    }
    
    if (cmd) free(cmd);
    if (data) free(data);
    
    return ret;
    
} /* test_kernel_module() */



/*
 * load_kernel_module() - modprobe the kernel module
 */

int load_kernel_module(Options *op, Package *p)
{
    char *cmd, *data;
    int len, ret;

    len = strlen(op->utils[MODPROBE]) + strlen(p->kernel_module_name) + 2;

    cmd = (char *) nvalloc(len);
    
    snprintf(cmd, len, "%s %s", op->utils[MODPROBE], p->kernel_module_name);
    
    ret = run_command(op, cmd, &data, FALSE, 0);

    if (ret != 0) {
        if (op->expert) {
            ui_error(op, "Unable to load the kernel module: '%s'", data);
        } else {
            ui_error(op, "Unable to load the kernel module.");
        }
        ret = FALSE;
    } else {
        ret = TRUE;
    }

    if (cmd) free(cmd);
    if (data) free(data);
    
    return ret;

} /* load_kernel_module() */






/*
 * check_kernel_module_version() - check that the driver version
 * indicated in the /proc filesystem is the same as the driver version
 * specified in the package description.
 */

int check_kernel_module_version(Options *op, Package *p)
{
    int major, minor, patch;
    int proc_major, proc_minor, proc_patch;
    FILE *fp = 0;
    char *buf;
    int eof;

    fp = fopen(NVIDIA_VERSION_PROC_FILE, "r");
    buf = fget_next_line(fp, &eof);
    
    if (!nvid_version(buf, &proc_major, &proc_minor, &proc_patch)) {
        free(buf);
        return FALSE;
    }

    if (!nvid_version(p->version_string, &major, &minor, &patch)) {
        return FALSE;
    }

    if ((proc_major != major) ||
        (proc_minor != minor) || 
        (proc_patch != patch)) {
        return FALSE;
    }

    return TRUE;

} /* check_kernel_module_version() */



/*
 * check_for_unloaded_kernel_module() - test if any of the "bad"
 * kernel modules are loaded; if they are, then try to unload it.  If
 * we can't unload it, then report an error and return FALSE;
 */

int check_for_unloaded_kernel_module(Options *op, Package *p)
{
    int n = 0;
    int loaded = FALSE;
    unsigned int bits = 0;
    
    while (p->bad_modules[n]) {
        if (check_for_loaded_kernel_module(op, p->bad_modules[n])) {
            loaded = TRUE;
            bits |= (1 << n);
        }
        n++;
    }

    if (!loaded) return TRUE;
    
    /* one or more kernel modules is loaded... try to unload them */

    n = 0;
    while (p->bad_modules[n]) {
        if (!(bits & (1 << n))) {
            n++;
            continue;
        }
        
        rmmod_kernel_module(op, p->bad_modules[n]);

        /* check again */
        
        if (check_for_loaded_kernel_module(op, p->bad_modules[n])) {
            ui_error(op,  "An NVIDIA kernel module '%s' appears to already "
                     "be loaded in your kernel.  This may be because it is "
                     "in use (for example, by the X server).  Please be "
                     "sure you have exited X before attempting to upgrade "
                     "your driver.  If you have exited X but still receive "
                     "this message, then an error has occured that has "
                     "confused the usage count of the kernel module; the "
                     "simplest remedy is to reboot your computer.",
                     p->bad_modules[n]);
    
            return FALSE;
        }
        n++;
    }

    return TRUE;

} /* check_for_unloaded_kernel_module() */





/*
 * find_precompiled_kernel_interface() - do assorted black magic to
 * determine if the given package contains a precompiled kernel interface
 * for the kernel on this system.
 *
 * XXX it would be nice to extend this so that a kernel module could
 * be installed for a kernel other than the currently running one.
 */

int find_precompiled_kernel_interface(Options *op, Package *p)
{
    DIR *dir;
    struct dirent *ent;
    char *filename, *d, *proc_version_string, *output_filename;
    PrecompiledInfo *info = NULL;

    /* allow the user to completely skip this search */
    
    if (op->no_precompiled_interface) {
        ui_log(op, "Not probing for precompiled kernel interfaces.");
        return FALSE;
    }
    
    /* retrieve the proc version string for the running kernel */

    proc_version_string = read_proc_version(op);
    
    d = p->precompiled_kernel_interface_directory;
    
    /* build the output filename */

    output_filename = nvstrcat(p->kernel_module_build_directory, "/",
                               PRECOMPILED_KERNEL_INTERFACE_FILENAME, NULL);

    if (!proc_version_string || !d) goto failed;
    
    if ((dir = opendir(d)) == NULL) {

        /* 
         * couldn't open the directory; maybe it doesn't exit? try
         * creating it
         */

        if (!mkdir_recursive(op, d, 0755)) goto failed;

        /* now try opening it again */

        if ((dir = opendir(d)) == NULL) goto failed;
    }

    /*
     * loop over all contents of the directory, looking for a
     * precompiled kernel interface that matches the running kernel
     */
    
    while ((ent = readdir(dir)) != NULL) {
        
        if (((strcmp(ent->d_name, ".")) == 0) ||
            ((strcmp(ent->d_name, "..")) == 0)) continue;
        
        filename = nvstrcat(d, "/", ent->d_name, NULL);
        
        info = precompiled_unpack(op, filename, output_filename,
                                  proc_version_string);

        if (info) break;
        
        free(filename);
        filename = NULL;
    }

    if (closedir(dir) != 0) {
        ui_error(op, "Failure while closing directory '%s' (%s).",
                 p->precompiled_kernel_interface_directory,
                 strerror(errno));
    }

    /*
     * If we didn't find a matching precompiled kernel interface, ask
     * if we should try to download one.
     */

    if (!info) {
        if (ui_yes_no(op, TRUE, "No precompiled kernel interface was "
                      "found to match "
                      "your kernel; would you like the installer to attempt "
                      "to download a kernel interface for your kernel from "
                      "the NVIDIA ftp site (%s)?", op->ftp_site)) {
            
            info = download_updated_kernel_interface(op, p,
                                                     proc_version_string);
            if (!info) {
                ui_message(op, "No matching precompiled kernel interface was "
                           "found on the NVIDIA ftp site; this means that the "
                           "installer will need to compile a kernel interface "
                           "for your kernel.");
                return FALSE;
            }
        }
    }

    /* If we found one, ask expert users if they really want to use it */

    if (info && op->expert) {
        if (!ui_yes_no(op, TRUE, "A precompiled kernel interface for the "
                       "kernel '%s' has been found.  Would you like to "
                       "use this? (answering 'no' will require the "
                       "installer to compile the interface)",
                       info->description)) {
            /* XXX free info */
            info = NULL;
        }
    }

    if (info) {
        /* XXX free info */
        return TRUE;
    }

 failed:

    ui_message(op, "No precompiled kernel interface was found to match "
               "your kernel; this means that the installer will need to "
               "compile a new kernel interface.");
        
    return FALSE;
    
} /* find_precompiled_kernel_interface() */



/*
 ***************************************************************************
 * local static routines
 ***************************************************************************
 */



/*
 * default_kernel_module_installation_path() - do the equivalent of:
 *
 * KERNDIR = /lib/modules/$(shell uname -r)
 *
 * ifeq ($(shell if test -d $(KERNDIR)/kernel; then echo yes; fi),yes)
 *   INSTALLDIR = $(KERNDIR)/kernel/drivers/video
 * else
 *   INSTALLDIR = $(KERNDIR)/video
 * endif
 */

static char *default_kernel_module_installation_path(Options *op)
{
    struct utsname uname_buf;
    char *str, *tmp;
    
    if (op->kernel_name) {
        tmp = op->kernel_name;
    } else {
        if (uname(&uname_buf) == -1) {
            ui_error (op, "Unable to determine kernel version (%s)",
                      strerror (errno));
            return NULL;
        }
        tmp = uname_buf.release;
    }

    str = nvstrcat("/lib/modules/", tmp, "/kernel", NULL);
    
    if (directory_exists(op, str)) {
        free(str);
        str = nvstrcat("/lib/modules/", tmp, "/kernel/drivers/video", NULL);
        return str;
    }

    free(str);
    
    str = nvstrcat("/lib/modules/", tmp, "/video", NULL);

    return str;

} /* default_kernel_module_installation_path() */



/*
 * default_kernel_include_path() - determine the default kernel
 * include path, if possible.  Return NULL if no default kernel path
 * is found.
 *
 * Here is the logic:
 *
 * if --kernel-include-dir was set, use that
 * 
 * else if SYSINCLUDE is set, use that
 *
 * else if /lib/modules/`uname -r`/builds/include exists use that
 *
 * else if /usr/src/linux/include exists use that
 *
 * else return NULL
 *
 * One thing to note is that for the first two methods
 * (--kernel-include-dir and $SYSINCLUDE) we don't check for directory
 * existence before returning.  This is intentional: if the user set
 * one of these, then they're trying to set a particular path.  If
 * that directory doesn't exist, then better to abort installation with
 * an appropriate error message in determine_kernel_include_path().
 * Whereas, for the later two (/lib/modules/`uname -r`/builds/include
 * and /usr/src/linux/include), these are not explicitly requested by
 * the user, so it makes sense to only use them if they exist.
 */ 

static char *default_kernel_include_path(Options *op)
{
    struct utsname uname_buf;
    char *str, *tmp;
    
    str = tmp = NULL;
    
    /* check --kernel-include-dir */

    if (op->kernel_include_path) {
        ui_log(op, "Using the kernel include path '%s' as specified by the "
               "'--kernel-include-dir' commandline option.",
               op->kernel_include_path);
        return op->kernel_include_path;
    }

    /* check SYSINCLUDE */
    
    str = getenv("SYSINCLUDE");
    if (str) {
        ui_log(op, "Using the kernel include path '%s', as specified by the "
               "SYSINCLUDE environment variable.", str);
        return str;
    }
    
    /* check /lib/modules/`uname -r`/builds/include */
    
    if (op->kernel_name) {
        tmp = op->kernel_name;
    } else {
        if (uname(&uname_buf) == -1) {
            ui_warn(op, "Unable to determine kernel version (%s).",
                    strerror(errno));
            tmp = NULL;
        } else {
            tmp = uname_buf.release;
        }
    }
    
    if (tmp) {
        str = nvstrcat("/lib/modules/", tmp, "/build/include", NULL);
    
        if (!directory_exists(op, str)) {
            free(str);
            str = NULL;
        } else {
            return str;
        }
    }

    /* finally, try /usr/src/linux/include */

    if (directory_exists(op, "/usr/src/linux/include")) {
        return "/usr/src/linux/include";
    }
    
    return NULL;
    
} /* default_kernel_include_path() */


/*
 * check_for_loaded_kernel_module() - check if the specified kernel
 * module is currently loaded using `lsmod`.  Returns TRUE if the
 * kernel module is loaded; FALSE if it is not.
 *
 * Be sure to check that the character following the kernel module
 * name is a space (to avoid getting false positivies when the given
 * kernel module name is contained within another kernel module name.
 */

static int check_for_loaded_kernel_module(Options *op, const char *module_name)
{
    char *ptr, *result = NULL;
    int ret;

    ret = run_command(op, op->utils[LSMOD], &result, FALSE, 0);
    
    if ((ret == 0) && (result) && (result[0] != '\0')) {
        ptr = strstr(result, module_name);
        if (ptr) {
            ptr += strlen(module_name);
            if(!isspace(*ptr)) ret = 1;
        } else {
            ret = 1;
        }
    }
    
    if (result) free(result);
    
    return ret ? FALSE : TRUE;
    
} /* check_for_loaded_kernel_module() */


/*
 * rmmod_kernel_module() - run `rmmod nvidia`
 */

static int rmmod_kernel_module(Options *op, const char *module_name)
{
    int len, ret;
    char *cmd;
    
    len = strlen(op->utils[RMMOD]) + strlen(module_name) + 2;
    
    cmd = (char *) nvalloc(len);
    
    snprintf(cmd, len, "%s %s", op->utils[RMMOD], module_name);
    
    ret = run_command(op, cmd, NULL, FALSE, 0);
    
    free(cmd);
    
    return ret ? FALSE : TRUE;
    
} /* rmmod_kernel_module() */



/*
 * get_updated_kernel_interfaces() - 
 */

static PrecompiledInfo *
download_updated_kernel_interface(Options *op, Package *p,
                                  const char *proc_version_string)
{
    int fd = -1;
    int dst_fd = -1;
    char *url = NULL;
    char *tmpfile = NULL;
    char *dstfile = NULL;
    char *buf = NULL;
    char *output_filename = NULL;
    char *str = (void *) -1;
    char *ptr, *s;
    struct stat stat_buf;
    PrecompiledInfo *info = NULL;
    uint32 crc;
    
    /* initialize the tmpfile and url strings */
    
    tmpfile = nvstrcat(op->tmpdir, "/nv-updates-XXXXXX", NULL);
    url = nvstrcat(op->ftp_site, "/XFree86/", INSTALLER_OS, "-",
                   INSTALLER_ARCH, "/", p->version_string,
                   "/updates/updates.txt", NULL);
    
    /*
     * create a temporary file in which to write the list of available
     * updates
     */

    if ((fd = mkstemp(tmpfile)) == -1) {
        ui_error(op, "Unable to create temporary file (%s)", strerror(errno));
        goto done;
    }

    /* download the updates list */

    if (!snarf(op, url, fd, SNARF_FLAGS_DOWNLOAD_SILENT)) goto done;
    
    /* get the length of the file */

    if (fstat(fd, &stat_buf) == -1) goto done;
    
    /* map the file into memory for easier reading */
    
    str = mmap(0, stat_buf.st_size, PROT_READ, MAP_FILE | MAP_SHARED, fd, 0);
    if (str == (void *) -1) goto done;
    
    /*
     * loop over each line of the updates file: each line should be of
     * the format: "[filename]:::[proc version string]"
     */

    ptr = str;

    while (TRUE) {
        buf = get_next_line(ptr, &ptr);
        if ((!buf) || (buf[0] == '\0')) goto done;

        s = strstr(buf, ":::");
        if (!s) {
            ui_error(op, "Invalid updates.txt list.");
            goto done;
        }
        
        s += 3; /* skip past the ":::" separator */
        
        if (strcmp(proc_version_string, s) == 0) {
            
            /* proc versions strings match */
            
            /*
             * terminate the string at the start of the ":::"
             * separator so that buf is the filename
             */
            
            s -= 3;
            *s = '\0';

            /* build the new url and dstfile strings */

            nvfree(url);
            url = nvstrcat(op->ftp_site, "/XFree86/",
                           INSTALLER_OS, "-", INSTALLER_ARCH, "/",
                           p->version_string, "/updates/", buf, NULL);
                
            dstfile = nvstrcat(p->precompiled_kernel_interface_directory,
                               "/", buf, NULL);
            
            /* create dstfile */
            
            dst_fd = creat(dstfile, S_IRUSR | S_IWUSR);
            if (dst_fd == -1) {
                ui_error(op, "Unable to create file '%s' (%s).",
                         dstfile, strerror(errno));
                goto done;
            }
            
            /* download the file */
            
            if (!snarf(op, url, dst_fd, SNARF_FLAGS_STATUS_BAR)) goto done;
            
            close(dst_fd);
            dst_fd = -1;
            
            /* XXX once we have gpg setup, should check the file here */

            /* build the output filename string */
            
            output_filename = nvstrcat(p->kernel_module_build_directory, "/",
                                       PRECOMPILED_KERNEL_INTERFACE_FILENAME,
                                       NULL);
            
            /* unpack the downloaded file */
            
            info = precompiled_unpack(op, dstfile, output_filename,
                                      proc_version_string);
            
            /* compare checksums */
            
            crc = compute_crc(op, output_filename);
            
            if (info && (info->crc != crc)) {
                ui_error(op, "The embedded checksum of the downloaded file "
                         "'%s' (%d) does not match the computed checksum ",
                         "(%d); not using.", buf, info->crc, crc);
                unlink(dstfile);
                /* XXX free info */
                info = NULL;
            }
            
            goto done;
        }

        nvfree(buf);
    }
    

 done:

    if (dstfile) nvfree(dstfile);
    if (buf) nvfree(buf);
    if (str != (void *) -1) munmap(str, stat_buf.st_size);
    if (dst_fd > 0) close(dst_fd);
    if (fd > 0) close(fd);

    unlink(tmpfile);
    if (tmpfile) nvfree(tmpfile);
    if (url) nvfree(url);

    return info;

} /* get_updated_kernel_interfaces() */
