/* _NVRM_COPYRIGHT_BEGIN_
 *
 * Copyright 2001 by NVIDIA Corporation.  All rights reserved.  All
 * information contained herein is proprietary and confidential to NVIDIA
 * Corporation.  Any use, reproduction, or disclosure without the written
 * permission of NVIDIA Corporation is prohibited.
 *
 * _NVRM_COPYRIGHT_END_
 */


#ifndef _NV_LINUX_H_
#define _NV_LINUX_H_

#include "nv.h"

#include <linux/config.h>

#if defined (CONFIG_MODVERSIONS) && !defined (MODVERSIONS)
#  define MODVERSIONS
#endif

#if defined (MODVERSIONS)
#include <linux/modversions.h>
#endif

#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/version.h>

#if LINUX_VERSION_CODE < KERNEL_VERSION(2, 2, 12)
#  error This driver does not support 2.2.11 or earlier kernels!
#elif LINUX_VERSION_CODE < KERNEL_VERSION(2, 3, 0)
#  define KERNEL_2_2
#  warning NVIDIA is considering dropping support for linux-2.2 
#  warning kernels. While end users are free to maintain their 
#  warning own patches, or stick with current drivers, our new 
#  warning drivers will not work "out of the box." If you are 
#  warning concerned about lack of support for 2.2 kernels, 
#  warning please let us know at linux-bugs@nvidia.com; we would
#  warning like to know how many users would be seriously 
#  warning impacted by this decision.
#elif LINUX_VERSION_CODE < KERNEL_VERSION(2, 4, 0)
#  error This driver does not support 2.3.x development kernels!
#elif LINUX_VERSION_CODE < KERNEL_VERSION(2, 5, 0)
#  define KERNEL_2_4
#elif LINUX_VERSION_CODE < KERNEL_VERSION(2, 6, 0)
#  error This driver does not support 2.5.x development kernels!
#  define KERNEL_2_5
#else
#  error This driver does not support 2.6.x or newer kernels!
#endif

#if defined (__ia64)
#  if LINUX_VERSION_CODE < KERNEL_VERSION(2, 4, 13)
#    error This driver does not support 2.4.12 or earlier kernels!
#  endif
#endif

#if defined (CONFIG_SMP) && !defined (__SMP__)
#define __SMP__
#endif

#include <linux/types.h>            /* pic_t, size_t, __u32, etc        */
#include <linux/errno.h>            /* error codes                      */
#include <linux/list.h>             /* circular linked list             */
#include <linux/stddef.h>           /* NULL, offsetof                   */
#include <linux/wait.h>             /* wait queues                      */
#include <linux/tqueue.h>           /* struct tq_struct                 */

#include <linux/slab.h>             /* kmalloc, kfree, etc              */
#include <linux/vmalloc.h>          /* vmalloc, vfree, etc              */

#include <linux/poll.h>             /* poll_wait                        */
#include <linux/delay.h>            /* mdelay, udelay                   */

#include <linux/pci.h>              /* pci_find_class, etc              */
#include <linux/wrapper.h>          /* mem_map_reserve                  */
#include <linux/interrupt.h>        /* mark_bh, init_bh, remove_bh      */
#include <linux/timer.h>

#include <asm/system.h>             /* cli, sli, save_flags             */
#include <asm/io.h>                 /* ioremap, virt_to_phys            */
#include <asm/uaccess.h>            /* access_ok                        */
#include <asm/page.h>               /* PAGE_OFFSET                      */
#include <asm/pgtable.h>            /* pte bit definitions              */

#if !defined (KERNEL_2_2)
#include <linux/spinlock.h>
#include <asm/semaphore.h>
#include <linux/highmem.h>
#else
#include <asm/spinlock.h>
#include <asm/semaphore.h>
#endif

#ifdef CONFIG_PROC_FS
#include <linux/proc_fs.h>
#endif

#ifdef CONFIG_DEVFS_FS
#include <linux/devfs_fs_kernel.h>
#endif

#ifdef CONFIG_KMOD
#include <linux/kmod.h>
#endif

#ifdef CONFIG_PM
#include <linux/pm.h>
#endif

#ifdef CONFIG_MTRR
#include <asm/mtrr.h>
#endif

#ifdef CONFIG_KDB
#include <linux/kdb.h>
#include <asm/kdb.h>
#endif

#if defined (CONFIG_AGP) || defined (CONFIG_AGP_MODULE)
#ifndef NOAGPGART
#  define AGPGART
#  include <linux/agp_backend.h>
#  include <linux/agpgart.h>
#endif
#endif

#if !defined (list_for_each)
#define list_for_each(pos, head) \
        for (pos = (head)->next; pos != (head); pos = (pos)->next)
#endif

#if LINUX_VERSION_CODE < KERNEL_VERSION(2, 2, 18)
#define vmalloc_32 vmalloc
#endif

#if !defined (KERNEL_2_2)
#  define LINUX_VMA_OFFS(vma)           (((vma)->vm_pgoff) << PAGE_SHIFT)
#  define GET_MODULE_SYMBOL(mod,sym)	(const void *) inter_module_get(sym)
#  define PUT_MODULE_SYMBOL(sym)        inter_module_put((char *) sym)
#  define GET_MAP_NR(phys_page)         virt_to_page(__va(phys_page))
#  define MEM_MAP_READ_COUNT(map_nr)    (atomic_read(&(map_nr)->count))
#  define MEM_MAP_INC_COUNT(map_nr)     (atomic_inc(&(map_nr)->count))
#  define MEM_MAP_DEC_COUNT(map_nr)     (atomic_dec(&(map_nr)->count))
#  define GET_EVENT_QUEUE(nv)           ((struct __wait_queue_head *) ((nv)->event_queue))
#  define VMA_PRIVATE(vma)              ((vma)->vm_private_data)
#else
#  define in_irq()                      (local_irq_count[smp_processor_id()])
#  define LINUX_VMA_OFFS(vma)           ((vma)->vm_offset)
#  define GET_MODULE_SYMBOL(mod, sym)   (void*) get_module_symbol((mod), (sym))
#  define PUT_MODULE_SYMBOL(sym)
#  define GET_MAP_NR(phys_page)         MAP_NR(__va(phys_page))
#  define MEM_MAP_READ_COUNT(map_nr)    (atomic_read(&mem_map[map_nr].count))
#  define MEM_MAP_INC_COUNT(map_nr)     (atomic_inc(&mem_map[map_nr].count))
#  define MEM_MAP_DEC_COUNT(map_nr)     (atomic_dec(&mem_map[map_nr].count))
#  define GET_EVENT_QUEUE(nv)           ((struct wait_queue **) &((nv)->event_queue))
#  define VMA_PRIVATE(vma)              ((void*)((vma)->vm_pte))
#endif

#if defined(REMAP_PAGE_RANGE_5)
#define NV_REMAP_PAGE_RANGE(a, b...)    remap_page_range(vma, a, ## b)
#elif defined(REMAP_PAGE_RANGE_4)
#define NV_REMAP_PAGE_RANGE(a, b...)    remap_page_range(a, ## b)
#else
#error "Couldn't determine number of arguments expected by remap_page_range!"
#endif

#if defined(pte_offset_atomic)
#define NV_PTE_OFFSET(addres, pg_mid_dir, pte) \
    { \
        pte_t *pte__ = pte_offset_atomic(pg_mid_dir, address); \
        pte = *pte__; \
        pte_kunmap(pte__); \
    }
#elif defined(pte_offset)
#define NV_PTE_OFFSET(addres, pg_mid_dir, pte) \
    pte = *pte_offset(pg_mid_dir, address)
#else
#define NV_PTE_OFFSET(addres, pg_mid_dir, pte) \
    { \
        pte_t *pte__ = pte_offset_map(pg_mid_dir, address); \
        pte = *pte__; \
        pte_unmap(pte__); \
    }
#endif

#define NV_PAGE_ALIGN(addr)             ( ((addr) + PAGE_SIZE - 1) / PAGE_SIZE)
#define NV_MASK_OFFSET(addr)            ( (addr) & (PAGE_SIZE - 1) )

#ifndef MAXMEM  /* temporary define for 2.2 kernels */
#define MAXMEM  (-PAGE_OFFSET - (64 * 1024 * 1024))
#endif

#ifndef NV01_ROOT
#define NV01_ROOT 0x00000000
#endif 

#if defined(NVCPU_IA64)
#define NV_GFP_HW (GFP_KERNEL | __GFP_DMA)
#define CACHE_FLUSH()
#else
#define NV_GFP_HW (GFP_KERNEL)
#define CACHE_FLUSH()  asm volatile("wbinvd":::"memory")
#endif

#if defined(NVCPU_X86) || defined(NVCPU_X86_64)
/* this isn't defined in some older 2.2 kernel header files */
#define NV_CPU_INTERRUPT_FLAGS_BIT (1<<9)
#elif defined(NVCPU_IA64)
/* For whatever reason this is not defined an any header file I could
 * find.  From Intel IA64 Architecture Software Developers Manual Volume 2: 
 * IA64 System Architecture page 3-7 we have:
 */
#define NV_CPU_INTERRUPT_FLAGS_BIT (1<<14)
#else
#error define NV_CPU_INTERRUPT_FLAGS_BIT
#endif

/*
 * AMD Athlon processors expose a subtle bug in the Linux
 * kernel, that may lead to AGP memory corruption. Recent
 * kernel versions had a workaround for this problem, but
 * 2.4.20 is the first kernel to address it properly. The
 * page_attr API provides the means to solve the problem. 
 */
#if defined(NVCPU_X86) && (LINUX_VERSION_CODE >= KERNEL_VERSION(2,4,20))
static inline void NV_SET_AMD_PAGE_ATTRIB(void **page_list, int count)
    {
        int i;
        for (i = 0; i < count; i++)
        {
            struct page *page = virt_to_page(__va(page_list[i]));
            change_page_attr(page, 1, PAGE_KERNEL_NOCACHE);
        }
    }
static inline void NV_CLEAR_AMD_PAGE_ATTRIB(void **page_list, int count)
    {
        int i;
        for (i = 0; i < count; i++)
        {
            struct page *page = virt_to_page(__va(page_list[i]));
            change_page_attr(page, 1, PAGE_KERNEL);
        }
    }
#else
#define NV_SET_AMD_PAGE_ATTRIB(page_list, count)
#define NV_CLEAR_AMD_PAGE_ATTRIB(page_list, count)
#endif

static inline int NV_IRQL_IS_RAISED()
    {
        unsigned long int eflags;
        __save_flags(eflags);
        return !(eflags & NV_CPU_INTERRUPT_FLAGS_BIT);
    }
 
static inline int calc_order(size)
    {
        int order = 0;
        while ( ((1 << order) * PAGE_SIZE) < (size))
        {
            order++;
        }
        return order;
    }

#if !defined (pgprot_noncached)
static inline pgprot_t pgprot_noncached(pgprot_t old_prot)
    {
        pgprot_t new_prot = old_prot;
        if (boot_cpu_data.x86 > 3)
            new_prot = __pgprot(pgprot_val(old_prot) | _PAGE_PCD);
        return new_prot;
    }
#endif

/*
 * An allocated bit of memory using NV_MEMORY_ALLOCATION_OFFSET
 *   looks like this in the driver
 */

typedef struct nv_alloc_s {
    struct nv_alloc_s *next;    
    struct vm_area_struct *vma;
    unsigned int   usage_count;
    unsigned int   flags;
    unsigned int   agp_offset;
    unsigned int   num_pages;
    void         **page_table;          /* list of physical pages allocated */
    void          *key_mapping;         /* mapping used as a key for finding this nv_alloc_t */
                                        /*   may be the same as page_table                   */
    unsigned int   class;
    void          *priv_data;
} nv_alloc_t;

#define NV_ALLOC_TYPE_PCI      (1<<0)
#define NV_ALLOC_TYPE_AGP      (1<<1)
#define NV_ALLOC_TYPE_CONTIG   (1<<2)
#define NV_ALLOC_TYPE_KERNEL   (1<<3)
#define NV_ALLOC_TYPE_VMALLOC  (1<<4)


/* linux-specific version of old nv_state_t */
/* this is a general os-specific state structure. the first element *must* be
   the general state structure, for the generic unix-based code */
typedef struct {
    nv_state_t nv_state;

    nv_alloc_t *alloc_queue;

    // bottom half interrupt handler info; per device
    /* keep track of any pending bottom-halves */
    struct tq_struct *bh;
    atomic_t bh_count;

    U032 vblank_notifier;
    U032 waiting_for_vblank;

    /* queue for for NV's OS events */
    void *event_queue;

    /* get a timer callback every second */
    struct timer_list rc_timer;

    /* per-device locking mechanism for access to core rm */
    spinlock_t rm_lock;

    /* lock for linux-specific data, not used by core rm */
    /* XXX eventually change to sleeping locks? */
    spinlock_t ldata_lock;

    /* lock for linux-specific alloc queue */
    /* XXX eventually change to sleeping locks? */
    spinlock_t at_lock;
} nv_linux_state_t;


/*
 * poll(2) support for event notifiers
 *
 * A notifer can be turned into an "OS" event.
 * On UNIX, this means you can block using poll() while waiting for any
 * of your OS events to fire.
 *
 * When poll() on the NV file descriptor notices that you have one or more
 * events, it returns POLLPRI.  At that time it clears the count of fired
 * notifiers.
 *
 * Upon return from poll() just poll your notifiers.
 */

/*
 * Per-file private data structure.
 * We keep stuff here that is separate on a per-file basis.
 */

typedef struct {
    U032 any_fired_notifiers;
    void *nv_ptr;
} nv_file_private_t;

/* for the card devices */
#define NV_GET_NVL_FROM_FILEP(filep) \
    (nv_linux_state_t*) (((nv_file_private_t *)((filep)->private_data))->nv_ptr)

/* for the control device */
#define NV_GET_NV_FROM_FILEP(filep) \
    (nv_state_t*) (((nv_file_private_t *)((filep)->private_data))->nv_ptr)

#define NV_HIDE_IN_FILEP(filep, nvp) \
    (((nv_file_private_t *)((filep)->private_data))->nv_ptr = (void *) (nvp))
    

#define NV_GET_NVL_FROM_NV_STATE(nv) \
    ((nv_linux_state_t *) nv->os_state)

#define NV_STATE_PTR(nvl)   (&((nvl)->nv_state))

/* all of the below kernel calls could sleep.
 * if we sleep with the nv lock, we could hang an incoming interrupt
 * and lock the system. So, drop the lock, perform the call, and
 * regrab the lock.
 */

#define NV_VMALLOC(ptr, size) \
    { \
        (void *) (ptr) = vmalloc_32(size); \
    }

#define NV_VFREE(ptr) \
    { \
        vfree((void *) (ptr)); \
    }

/* only use this because GFP_KERNEL may sleep..
 * GFP_ATOMIC is ok, it won't sleep
 */
#define NV_KMALLOC(ptr, size) \
    { \
        (void *) (ptr) = kmalloc(size, GFP_KERNEL); \
    }

#define NV_KFREE(ptr) \
    { \
        kfree((void *) (ptr)); \
    }

#define NV_ATOMIC_INC(data)  atomic_inc(&(data))
#define NV_ATOMIC_DEC(data)  atomic_dec(&(data))
#define NV_ATOMIC_READ(data) atomic_read(&(data))

/*
 * Basic support for kgdb assertions.
 */
#if defined(CONFIG_X86_REMOTE_DEBUG)
#include <linux/gdb.h>

#define NV_ASSERT(message, condition) KGDB_ASSERT(message, condition)
#else
#define NV_ASSERT(message, condition)
#endif


#endif  /* _NV_LINUX_H_ */
