/* _NVRM_COPYRIGHT_BEGIN_
 *
 * Copyright 1999-2001 by NVIDIA Corporation.  All rights reserved.  All
 * information contained herein is proprietary and confidential to NVIDIA
 * Corporation.  Any use, reproduction, or disclosure without the written
 * permission of NVIDIA Corporation is prohibited.
 *
 * _NVRM_COPYRIGHT_END_
 */


#ifndef _NV_H_
#define _NV_H_

#include <nvtypes.h>

/* Nvidia's reserved major device number
 * Set this to 0 to request dynamic major number allocation. */
#define NV_MAJOR_DEVICE_NUMBER 195

/* most cards in a single system */
#define NV_MAX_DEVICES 8

#define NV_IOCTL_MAGIC      'F'

typedef struct nv_ioctl_card_info
{
    NvU32    flags;               /* see below                   */
    NvU32    bus;		  /* bus number (PCI, AGP, etc)  */
    NvU32    slot;                /* card slot                   */
    NvU32    vendor_id;           /* PCI vendor id               */
    NvU32    device_id;
    NvU32    interrupt_line;
    NvU32    reg_address;         /* register aperture           */
    NvU32    reg_size;
    NvU32    fb_address;          /* framebuffer aperture        */
    NvU32    fb_size;
} nv_ioctl_card_info_t;

#define NV_IOCTL_CARD_INFO_FLAG_PRESENT       0x0001
#define NV_IOCTL_CARD_INFO_FLAG_NEED_MSYNC    0x0002

#define NV_IOCTL_CARD_INFO     \
    _IOWR(NV_IOCTL_MAGIC, 2, NvU8[NV_MAX_DEVICES*sizeof(nv_ioctl_card_info_t)])


typedef struct nv_ioctl_primary_card
{
    NvU32 bus;
    NvU32 slot;
} nv_ioctl_primary_card_t;

/* notify resman which card is primary */
#define NV_IOCTL_PRIMARY_CARD  _IOWR(NV_IOCTL_MAGIC, 4, nv_ioctl_primary_card_t)

typedef struct nv_ioctl_sim_env
{
    NvU32 sim_env;
} nv_ioctl_sim_env_t;

#define NV_IOCTL_SIM_ENV  _IOWR(NV_IOCTL_MAGIC, 5, nv_ioctl_sim_env_t)

/* rm api check
 * used to verify client/rm interaction both ways.
 * structure overloads card_info structure passed into resman, so resman can 
 * verify the client is using the correct version of the api.
 * client then does api_version ioctl to verify the rm is the correct version.
 * we do both to avoid cases where one but not the other supports api checking.
 * esp w/ recent memory/mmap changes to resman, we've seen some cases where 
 * mix-matching rm/clients can case serious damage.
 */
typedef struct nv_ioctl_rm_api_version
{
    NvU32 magic;
    NvU32 version;
    NvU32 major;
    NvU32 minor;
    NvU32 patch;
} nv_ioctl_rm_api_version_t;

#define NV_RM_API_VERSION_MAGIC_REQ  0x9797fade
#define NV_RM_API_VERSION_MAGIC_REP  0xbead2929
#define NV_RM_API_VERSION 1

#define NV_IOCTL_RM_API_VERSION _IOWR(NV_IOCTL_MAGIC, 6, nv_ioctl_rm_api_version_t)

/* debug tool to force module count to 0 */
#define NV_IOCTL_MODULE_RESET  _IO(NV_IOCTL_MAGIC, 99)


#ifdef __KERNEL__

/*
 * ptr arithmetic convenience
 */

typedef union
{
    volatile V008 Reg008[1];
    volatile V016 Reg016[1];
    volatile V032 Reg032[1];
} nv_hwreg_t, * nv_phwreg_t;

/*
 * data structure to keep track of open clients for abnormal termination
 * cleanup.  See nv.c for discussion of cleanup issues
 * NOTE: this can't be kept in 'nv' since the nv current during root allocate
 *       is the control 'nv'.
 */

#define NV_MAX_CLIENTS 128         /* *system-wide* limit, not per device */
typedef struct {
    NvU32  client;
    int    pid;
    void  *file;
} nv_client_t;
extern nv_client_t nv_clients[NV_MAX_CLIENTS];

typedef struct {
    NvU32 address;
    NvU32 size;
    NvU32 *map;
    nv_phwreg_t map_u;
} nv_aperture_t;


typedef struct {
    nv_aperture_t regs;
    nv_aperture_t fb;
} nv_bar_t;

typedef struct {
    char *node;
    char *name;
    unsigned int *data;
} nv_parm_t;


/*
 * per device state
 */

typedef struct
{
    void  *pdev;		    /* you know who */
    void  *priv;                    /* private data */
    void  *os_state;                /* os-specific version of this file */

    int    rmInitialized;
    int    flags;
    int    device_number;
    int    usage_count;

    /* mapped addresses */
    NvU32  *rom;

    /* PCI config info */
    int bus;
    int slot;
    int vendor_id;
    int device_id;

    /* physical characteristics */
    nv_bar_t bar;
    nv_aperture_t *regs;
    nv_aperture_t *fb;
    nv_aperture_t agp;

    NvU32  interrupt_line;

    U032 agp_config;
    U032 agp_buffers;
    U032 agp_teardown;

    /* save/restore eflags register */
    U032 saved_eflags;

    U032 sim_env;

    U032 rc_timer_enabled;

} nv_state_t;

/*
 * flags
 */

#define NV_FLAG_OPEN            0x0001
#define NV_FLAG_POSTED          0x0002
#define NV_FLAG_WAITING         0x0004
#define NV_FLAG_CONTROL         0x0010
#define NV_FLAG_HOTKEY_OCCURRED 0x0100

#define NV_AGP_ENABLED(nv)      ((nv)->agp_config)
#define NV_AGP_DISABLED(nv)     (!(nv)->agp_config)
#define NV_NVAGP_ENABLED(nv)    ((nv)->agp_config & NVOS_AGP_CONFIG_NVAGP)
#define NV_OSAGP_ENABLED(nv)    ((nv)->agp_config & NVOS_AGP_CONFIG_OSAGP)

/*
** where we hide our nv_state_t * in a pdev
** subject to change...
*/
#define NV_HIDE_NV_IN_PDEV(pdev) \
                     ((nv_state_t *) (pdev)->pOsHwInfo)

#define NV_WAITING_FOR_VBLANK(nv) \
                     (nv->flags & NV_FLAG_WAITING)

/* mmap(2) offsets */

#define IS_REG_OFFSET(nv, offset, length)                                      \
             (((offset) >= (nv)->regs->address) &&                             \
             (((offset) + (length)) <= (nv)->regs->address + (nv)->regs->size))

#define IS_FB_OFFSET(nv, offset, length)                                       \
             (((offset) >= (nv)->fb->address) &&                               \
             (((offset) + (length)) <= (nv)->fb->address + (nv)->fb->size))

#define IS_AGP_OFFSET(nv, offset, length)                                      \
             (((offset) >= (nv)->agp.address) &&                               \
             (((offset) + (length)) <= (nv)->agp.address + (nv)->agp.size))

/* duplicated from nvos.h for external builds */
#ifndef NVOS_AGP_CONFIG_DISABLE_AGP
#  define NVOS_AGP_CONFIG_DISABLE_AGP (0x00000000)
#endif
#ifndef NVOS_AGP_CONFIG_NVAGP
#  define NVOS_AGP_CONFIG_NVAGP       (0x00000001)
#endif
#ifndef NVOS_AGP_CONFIG_OSAGP
#  define NVOS_AGP_CONFIG_OSAGP       (0x00000002)
#endif
#ifndef NVOS_AGP_CONFIG_ANYAGP
#  define NVOS_AGP_CONFIG_ANYAGP      (0x00000003)
#endif


/* device name length; must be atleast 8 */

#define NV_DEVICE_NAME_LENGTH 40


/*
 * driver internal interfaces
 */

/* need a fake device number for control device; just to flag it for msgs */
#define NV_CONTROL_DEVICE_NUMBER     100

#ifndef NVWATCH

/*
 * ---------------------------------------------------------------------------
 *
 * Function prototypes for UNIX specific OS interface.
 *
 * ---------------------------------------------------------------------------
 */

VOID*  nv_find_kernel_mapping       (nv_state_t *, unsigned long); 
VOID*  nv_find_agp_kernel_mapping   (nv_state_t *, unsigned long); 
ULONG  nv_get_phys_address          (ULONG);

VOID   nv_lock_rm                   (nv_state_t *);
VOID   nv_unlock_rm                 (nv_state_t *);
VOID   nv_lock_bh                   (nv_state_t *);
VOID   nv_unlock_bh                 (nv_state_t *);

VOID   nv_post_event                (VOID *);
VOID   nv_post_vblank               (nv_state_t *);
VOID   nv_set_hotkey_occurred_flag  (VOID);
S032   nv_int10h_call               (nv_state_t *, U032 *, U032 *, U032 *, U032 *, VOID *);

S032   nv_alloc_pages               (nv_state_t *, VOID **, U032, U032, U032, U032, U032, U032, VOID **);
S032   nv_free_pages                (nv_state_t *, VOID **, U032, U032, VOID **);

S032   nv_agp_init                  (nv_state_t *, VOID **, VOID **, VOID *, U032);
S032   nv_agp_teardown              (nv_state_t *);
S032   nv_agp_translate_address     (nv_state_t *, VOID *, U032, U032 *);

S032   nv_start_rc_timer            (nv_state_t *);
S032   nv_stop_rc_timer             (nv_state_t *);


/*
 * ---------------------------------------------------------------------------
 *
 * Function prototypes for Resource Manager interface.
 *
 * ---------------------------------------------------------------------------
 */

BOOL       rm_init_rm               (VOID);
BOOL       rm_shutdown_rm           (VOID);
BOOL       rm_init_adapter          (nv_state_t *);
BOOL       rm_disable_adapter       (nv_state_t *);
BOOL       rm_shutdown_adapter      (nv_state_t *);
BOOL       rm_ioctl                 (nv_state_t *, VOID *, U032, VOID *);
BOOL       rm_isr                   (U032, U032 *);
VOID       rm_isr_bh                (VOID *);
RM_STATUS  rm_power_management      (nv_state_t *, U032, U032);
U032       rm_get_vbios_version     (nv_state_t *, U032 *, U032 *, U032 *, U032 *, U032 *);
VOID       rm_free_unused_clients   (nv_state_t *, U032, VOID *);

VOID       rm_update_agp_config     (nv_state_t *);
RM_STATUS  rm_init_agp              (nv_state_t *);
RM_STATUS  rm_shutdown_agp          (nv_state_t *);

RM_STATUS  rm_alloc_agp_pages       (nv_state_t *, VOID **, U032, U032, VOID **, U032 *);
RM_STATUS  rm_map_agp_pages         (nv_state_t *, VOID **, U032, VOID *);
RM_STATUS  rm_free_agp_pages        (nv_state_t *, VOID **, VOID *);

RM_STATUS  rm_alloc_agp_bitmap      (nv_state_t *, U032, U032 *);
RM_STATUS  rm_free_agp_bitmap       (nv_state_t *, U032, U032);
RM_STATUS  rm_set_agp_bitmap        (nv_state_t *, VOID *);
RM_STATUS  rm_clear_agp_bitmap      (nv_state_t *, VOID **);

RM_STATUS  rm_load_registry         (nv_parm_t *);
RM_STATUS  rm_read_registry_dword   (nv_state_t *, U008 *, U008 *, U032 *);
RM_STATUS  rm_write_registry_dword  (nv_state_t *, U008 *, U008 *, U032);
RM_STATUS  rm_read_registry_binary  (nv_state_t *, U008 *, U008 *, U008 *, U032 *);
RM_STATUS  rm_write_registry_binary (nv_state_t *, U008 *, U008 *, U008 *, U032);

RM_STATUS  rm_run_rc_callback       (nv_state_t *);
RM_STATUS  rm_get_device_name       (U032, U032, U008*);

NvUI64     nv_rdtsc                 (VOID);

#endif /* NVWATCH */

#endif /* __KERNEL__ */

#endif
