/* _NVRM_COPYRIGHT_BEGIN_
 *
 * Copyright 1999-2001 by NVIDIA Corporation.  All rights reserved.  All
 * information contained herein is proprietary and confidential to NVIDIA
 * Corporation.  Any use, reproduction, or disclosure without the written
 * permission of NVIDIA Corporation is prohibited.
 *
 * _NVRM_COPYRIGHT_END_
 */



/******************* Operating System Interface Routines *******************\
*                                                                           *
* Module: interface.c                                                       *
*   This is the OS interface module.  All operating system transactions     *
*   pass through these routines.  No other operating system specific code   *
*   or data should exist in the source.                                     *
*                                                                           *
\***************************************************************************/

#include <linux/config.h>

#if defined(CONFIG_SMP) && !defined(__SMP__)
#define __SMP__
#endif

// pull in the versioned kernel api defines if needed
#ifdef CONFIG_MODVERSIONS
#define MODVERSIONS
#include <linux/modversions.h>
#endif

#define __NO_VERSION__          // tell module.h to not define __module_kernel_version
                                // it was already defined for nv.c

#include <linux/module.h>

#include <nv-linux.h>                 // NV device driver interface

#include <linux/stddef.h>
#include <linux/kernel.h>       // printk()
#include <linux/errno.h>        // error codes
#include <linux/types.h>        // size_t
#include <linux/interrupt.h>    // in_interrupt()
#include <linux/delay.h>        // udelay()
#include <linux/pci.h>          // pci_*() functions
#include <linux/kmod.h>         // request_module
#include <asm/io.h>             // ioremap() and iounmap()
#ifdef CONFIG_MTRR
#include <asm/mtrr.h>           // mtrr_add()
#endif
#ifdef CONFIG_KDB
#include <asm/kdb.h>
#endif
#include <asm/page.h>
#include <linux/wrapper.h>      // for mem_map_reserve() 
#include <asm/uaccess.h>        // user access stuff like copy_from_user
#include <asm/smp.h>
#if !defined(IA64)
#include <asm/ldt.h>
#include <asm/desc.h>
#endif

#define RM_HEADERS_MINIMAL      // we just need top half of nvrm.h, not the entire
                                // tree of include files
#include <nvrm.h>

#include <os-interface.h>

// registry controls

#define NV_MODULE_NAME       "NVdriver"
#define NV_SYM_PREFIX        "NVreg_"
#define NV_SYM_PREFIX_LENGTH 6
#define NV_MAX_SYM_NAME      128


#ifdef DEBUG

// memory allocation tracking data structs and globals
typedef struct _MEM_ALLOC_LOG_ENTRY
{
    VOID *address;
    U032 handle;
    U032 size;
    struct _MEM_ALLOC_LOG_ENTRY *next;
    struct _MEM_ALLOC_LOG_ENTRY *last;

} MEM_ALLOC_LOG_ENTRY;
MEM_ALLOC_LOG_ENTRY *memAllocLog = NULL;
MEM_ALLOC_LOG_ENTRY *memAllocLogTail = NULL;
U032 memAllocEntries = 0;
U032 memAllocTotal = 0;
VOID osLogMemAlloc(VOID *address, U032 size);
VOID osUnlogMemAlloc(VOID *address);

// instance memory allocation tracking data structs and globals
typedef struct _INST_MEM_ALLOC_LOG_ENTRY
{
    U032 handle;
    U032 instance;
    U032 size;
    U032 alignment;
    struct _INST_MEM_ALLOC_LOG_ENTRY *next;
    struct _INST_MEM_ALLOC_LOG_ENTRY *last;

} INST_MEM_ALLOC_LOG_ENTRY;
INST_MEM_ALLOC_LOG_ENTRY *instMemAllocLog = NULL;
INST_MEM_ALLOC_LOG_ENTRY *instMemAllocLogTail = NULL;
U032 instMemAllocEntries = 0;
U032 instMemAllocTotal = 0;

#endif // DEBUG

// return TRUE if the caller is the super-user
BOOL osIsAdministrator(
    PHWINFO pDev
)
{
    return suser();
}

//
// Some quick and dirty library functions.
// This is an OS function because some operating systems supply their
// own version of this function that they require you to use instead
// of the C library function.  And some OS code developers may decide to
// use the actual C library function instead of this code.  In this case,
// just replace the code within osStringCopy with a call to the C library
// function strcpy.
//
char *osStringCopy(
    char *dst,
    const char *src
)
{
    return strcpy(dst, src);
}

RM_STATUS osStringNCopyFromUser(
    char *dst,
    const char *src,
    int n
)
{
#if defined(IA64)
    return copy_from_user(dst, src, n) ? RM_ERR_BAD_ADDRESS : RM_OK;
#else
    return strncpy_from_user(dst, src, n) ? RM_ERR_BAD_ADDRESS : RM_OK;
#endif
}

S032 osStringCompare(
    const char *s1,
    const char *s2
)
{
    return strcmp(s1, s2);
}

U032 osStringLength(const char * str)
{
    return strlen(str);
}

void * osMemCopy(void *dst, const void* src, U032 length)
{
    return memcpy(dst, src, length);
}

RM_STATUS osMemCopyFromUser(void *dst, const void* src, U032 length)
{
    return copy_from_user(dst, src, length) ? RM_ERR_BAD_ADDRESS : RM_OK;
}

RM_STATUS osMemCopyToUser(void *dst, const void* src, U032 length)
{
    return copy_to_user(dst, src, length) ? RM_ERR_BAD_ADDRESS : RM_OK;
}

void * osMemSet(void * dst, int c, U032 length)
{
    return memset(dst, c, length);
}

int osMemCmp(const void *buf0, const void* buf1, U032 length)
{
    return memcmp(buf0, buf1, length);
}


VOID osDbgBreakPoint(void);


//---------------------------------------------------------------------------
//
//  Operating System Memory functions.
//
//  On UNIX, there are some special considerations here.
//
//  There are 2 interesting things about the resource manager memory allocation:
//
//      1. It allocates *lots* of big memory, eg: a single allocation of 164k
//
//      2. Implicitly believes it is ok to allocate memory from an interrupt handler
//
//  On linux, the first of these requires that we use vmalloc() which pulls
//  together non-contiguous pages.
//  But the 2nd requires that we use kmalloc(0).
//
//  So we have to be smart and do the right thing at run time.
//  Big requests use vmalloc unless called from isr in which case they fail.
//  Otherwise use kmalloc.
//  Save the type info in the buffer.
//
//---------------------------------------------------------------------------

/* 
 * kmalloc() can allocate "memory objects" of different sizes, with
 * a minimum of 32 bytes on i386 and 64 bytes on ia64. The available
 * allocation sizes (as defined in mm/slab.c) are powers of two, the
 * largest is 128kB on both i386 and ia64 at this point. Attempts to
 * allocate more memory with kmalloc() will fail.
 */

#define OS_MEM_SIZE_BIG     131072

#define OS_MEM_SIZE_ALIGN   sizeof(void*)

#ifdef DEBUG
#define MEM_DEBUG_ENTRIES 3
#else
#define MEM_DEBUG_ENTRIES 1
#endif

#define MEM_OVERHEAD  (OS_MEM_SIZE_ALIGN * MEM_DEBUG_ENTRIES)

#define OS_V_TYPE 0
#define OS_K_TYPE 1

#define PACK_INFO(p, sz, typ)    ((sz) | (typ))
#define UNPACK_INFO(p, sz, typ) \
    do {                                      \
       U032 pval = p[0];                      \
       (sz) =  pval & ~(OS_MEM_SIZE_ALIGN-1); \
       (typ) = pval & (OS_MEM_SIZE_ALIGN-1);  \
    } while (0)

RM_STATUS osAllocMem(
    VOID **pAddress,
    U032   Size
)
{
    U032     *ptr = NULL;
    int       type = OS_K_TYPE;

    // reserve space for our stuff
    Size += MEM_OVERHEAD;

    // round up to make sure sentinels don't generate misaligned data
    if (Size & (OS_MEM_SIZE_ALIGN - 1))
        Size = NV_ALIGN_UP(Size, OS_MEM_SIZE_ALIGN);
    
    //
    // if we're in an interrupt or bottom half on this CPU  (in_interrupt() == TRUE)
    //   or if we're holding the NV lock, must use kmalloc(ATOMIC)
    //

    if (in_interrupt())
    {
        if (Size <= OS_MEM_SIZE_BIG)
        {
            type = OS_K_TYPE;
            ptr = kmalloc(Size, GFP_ATOMIC);  /* atomic is ok */
        }
    }
    else 
    {
        //  Not in an interrupt, we can pick "best"
        // "big" allocations use vmalloc(); "small" ones use kmalloc()

        if (Size > OS_MEM_SIZE_BIG)
        {
            type = OS_V_TYPE;
            NV_VMALLOC(ptr, Size);
        }
        else
        {
            type = OS_K_TYPE;
            NV_KMALLOC(ptr, Size);
        }
    }

    if (ptr == NULL)
    {
        *pAddress = (void *) 0;
        return RM_ERR_NO_FREE_MEM;
    }
    ptr[0] = PACK_INFO(ptr, Size, type);

#ifndef DEBUG
    // align the pointer to prevent access faults on the IA64
    ptr += MEM_DEBUG_ENTRIES;
    ptr = NV_ALIGN_PTR_UP(ptr, OS_MEM_SIZE_ALIGN);
#else
    // memory allocation tracking
    osLogMemAlloc(ptr, Size);
    ptr[1] = NV_MARKER1;
    ptr[(Size/sizeof(U032)) - 1] = NV_MARKER2;
    ptr += MEM_DEBUG_ENTRIES - 1;
    osMemSet(ptr, 0x69, Size - MEM_OVERHEAD);
#endif

    *pAddress = ptr;
    return RM_OK;
}


RM_STATUS osFreeMem(
    VOID *pAddress
)
{
    U032      Size;
    int       type;
    U032     *ptr;

    ptr = pAddress;

#ifndef DEBUG
    ptr  -= MEM_DEBUG_ENTRIES;
    ptr = NV_ALIGN_PTR_DOWN(ptr, OS_MEM_SIZE_ALIGN);
    UNPACK_INFO(ptr, Size, type);
#else
    ptr -= MEM_DEBUG_ENTRIES - 1;
    UNPACK_INFO(ptr, Size, type);

    if ((ptr[1] != NV_MARKER1) || (ptr[(Size/sizeof(U032))-1] != NV_MARKER2))
    {
        if (ptr[1] != NV_MARKER1)
            nv_printf("NV: Invalid address to osFreeMem: 0x%p\n", ptr);
        else
            nv_printf("NV: Memory overrun in structure to osFreeMem: 0x%p\n",
                       ptr);
        osDbgBreakPoint();
        return RM_ERR_MEM_NOT_FREED;
    }

    ptr[1] = 'DEAD';
    ptr[(Size/sizeof(U032))-1] = 'DEAD';
    Size -= MEM_OVERHEAD;
    osUnlogMemAlloc(pAddress);
#endif // DEBUG

    if (type == OS_V_TYPE)
    {
        NV_VFREE(ptr);
    } else {
        NV_KFREE(ptr);
    }

    return RM_OK;
}

#ifdef DEBUG

// add a memory allocation log entry
VOID osLogMemAlloc(VOID *address, U032 size)
{
    static U032 memAllocAttempts = 0;
    static U032 memAllocId = 0;
    MEM_ALLOC_LOG_ENTRY *newElement;

    NV_KMALLOC(newElement, sizeof(MEM_ALLOC_LOG_ENTRY));
    if (newElement)
    {
        // stats
        memAllocEntries++;
        memAllocTotal += size;

        // add the new element to the head of the list
        newElement->handle = memAllocId++;
        newElement->address = address;
        newElement->size = size;
        newElement->next = memAllocLog;
        newElement->last = NULL;
        memAllocLog = newElement;

        // set the tail
        if (memAllocLogTail == NULL)
        {
            memAllocLogTail = newElement;
        }

        // fixup the back link of the next element
        if (newElement->next)
        {
            newElement->next->last = newElement;
        }
    }

    // more stats
    memAllocAttempts++;
}

// remove a memory allocation log entry
VOID osUnlogMemAlloc(VOID *address)
{
    static U032 memFreeAttempts = 0;
    MEM_ALLOC_LOG_ENTRY *element;

    for (element = memAllocLog; element; element = element->next)
    {
        if (element->address == address)
        {
            // stats
            memAllocEntries--;
            memAllocTotal -= element->size;

            // unlink the element from the list
            if (element->last)
            {
                element->last->next = element->next;
            }
            else
            {
                memAllocLog = element->next;
            }
            if (element->next)
            {
                element->next->last = element->last;
            }

            // adjust the tail
            if (element == memAllocLogTail)
            {
                memAllocLogTail = element->last;
            }

            // free the element
            NV_KFREE(element);
            break;
        }
    }

    // more stats
    memFreeAttempts++;
}

// add a memory allocation log entry
VOID osLogInstMemAlloc(U032 instance, U032 size, U032 alignment)
{
    static U032 instMemAllocAttempts = 0;
    static U032 instMemAllocId = 0;
    INST_MEM_ALLOC_LOG_ENTRY *newElement;

    NV_KMALLOC(newElement, sizeof(INST_MEM_ALLOC_LOG_ENTRY));
    if (newElement)
    {
        // stats
        instMemAllocEntries++;
        instMemAllocTotal += size;

        // add the new element to the head of the list
        newElement->handle = instMemAllocId++;
        newElement->instance = instance;
        newElement->size = size;
        newElement->alignment = alignment;
        newElement->next = instMemAllocLog;
        newElement->last = NULL;
        instMemAllocLog = newElement;

        // set the tail
        if (instMemAllocLogTail == NULL)
        {
            instMemAllocLogTail = newElement;
        }

        // fixup the back link of the next element
        if (newElement->next)
        {
            newElement->next->last = newElement;
        }
    }

    // more stats
    instMemAllocAttempts++;
}

// remove a memory allocation log entry
VOID osUnlogInstMemAlloc(U032 instance, U032 size)
{
    static U032 instMemFreeAttempts = 0;
    INST_MEM_ALLOC_LOG_ENTRY *element;

    for (element = instMemAllocLog; element; element = element->next)
    {
        if (element->instance == instance)
        {
            // stats
            instMemAllocEntries--;
            instMemAllocTotal -= size;

            // unlink the element from the list
            if (element->last)
            {
                element->last->next = element->next;
            }
            else
            {
                instMemAllocLog = element->next;
            }
            if (element->next)
            {
                element->next->last = element->last;
            }

            // adjust the tail
            if (element == instMemAllocLogTail)
            {
                instMemAllocLogTail = element->last;
            }

            // free the element
            NV_KFREE(element);
            break;
        }
    }

    // more stats
    instMemFreeAttempts++;
}

#endif // DEBUG

/*****************************************************************************
*
*   Name: osGetCurrentTime
*
*****************************************************************************/

RM_STATUS osGetCurrentTime(
    U032 *seconds,
    U032 *useconds
)
{
    struct timeval tm;

    do_gettimeofday(&tm);

    *seconds = tm.tv_sec;
    *useconds = tm.tv_usec;

    return RM_OK;
}

//---------------------------------------------------------------------------
//
//  Misc services.
//
//---------------------------------------------------------------------------

#define NV_MAX_ISR_UDELAY           20000
#define NV_MAX_ISR_MDELAY           (NV_MAX_ISR_UDELAY / 1000)
#define NV_MSECS_PER_JIFFIE         (1000 / HZ)
#define NV_MSECS_TO_JIFFIES(msec)   ((msec) * HZ / 1000)

/*
 * It is generally a bad idea to use udelay() to wait for more than
 * a few milliseconds. Since the caller is most likely not aware of
 * this, we use mdelay() for any full millisecond to be safe.
 */

RM_STATUS osDelayUs(U032 MicroSeconds)
{
    unsigned long mdelay_safe_msec;
    unsigned long usec;

    if (in_irq() && MicroSeconds > NV_MAX_ISR_UDELAY)
        return RM_ERROR;
    
    mdelay_safe_msec = MicroSeconds / 1000;
    if (mdelay_safe_msec)
        mdelay(mdelay_safe_msec);

    usec = MicroSeconds % 1000;
    if (usec)
        udelay(usec);

    return RM_OK;
}

/* 
 * On Linux, a jiffie represents the time passed in between two timer
 * interrupts. The number of jiffies per second (HZ) varies across the
 * supported platforms. On i386, where HZ is 100, a timer interrupt is
 * generated every 10ms; the resolution is a lot better on ia64, where
 * HZ is 1024. NV_MSECS_TO_JIFFIES should be accurate independent of
 * the actual value of HZ; any partial jiffies will be 'floor'ed, the
 * remainder will be accounted for with mdelay().
 */

RM_STATUS osDelay(U032 MilliSeconds)
{
    unsigned long jiffies;
    unsigned long mdelay_safe_msec;
    
    if (in_irq() && MilliSeconds > NV_MAX_ISR_MDELAY)
        return RM_ERROR;

    if (in_interrupt()) 
    {
        mdelay(MilliSeconds);
        return RM_OK;
    }

    /* do we have a full jiffie to wait? */
    jiffies = NV_MSECS_TO_JIFFIES(MilliSeconds);

    if (jiffies) 
    {
        /* give up the cpu */
        current->state = TASK_INTERRUPTIBLE;
        schedule_timeout(jiffies);
    }

    /* catch the remainder, if any */
    if (NV_MSECS_PER_JIFFIE) 
    {
        mdelay_safe_msec = MilliSeconds % NV_MSECS_PER_JIFFIE;
        if (mdelay_safe_msec)
            mdelay(mdelay_safe_msec);
    }

    return RM_OK;
}

/* return CPU frequency in MHz */
U032 osGetCpuFrequency()
{
    u64 tsc[2];
    u32 tsc_d;
    u32 cpu_mhz;

    tsc[0] = nv_rdtsc();
    mdelay(250);
    tsc[1] = nv_rdtsc();

    /* we timed 1/4th a second */
    tsc_d = (tsc[1] - tsc[0]) * 4;

    /* convert from Hz to MHz */
    cpu_mhz = tsc_d / 1000000;

    return cpu_mhz;
}

RM_STATUS osGetCurrentProcess(
    U032 *pPid
)
{
    *pPid = current->pid;
    return RM_OK;
}

RM_STATUS osKillProcess(
    U032 pid,
    U032 sig
)
{
    return kill_proc(pid, sig, 1) ? RM_ERR_OPERATING_SYSTEM : RM_OK;
}

/*******************************************************************************/
/*                                                                             */
/* Debug and logging utilities follow                                          */
/*                                                                             */
/*******************************************************************************/

#undef DEBUG
#define DEBUG 1         // for rest of this file, DEBUG is always on

/*
 *  print or log error and freakout messages 
 *
 * These routines provide general purpose error reporting.
 * nv_error reports an error to stderr and allows use of
 * printf style formatting.  A newline is appended to all messages.
 * 
 * error_flag can be specified as any of the following:
 *
 *      NV_ERROR_ERRNO       -- include errno text in output
 *      NV_ERROR_STATUS      -- RM status is in error_flag
 *      NV_ERROR_PANIC       -- halts local system after output
 *      NV_ERROR_MESSAGE     -- just do the printing; no hint of error
 *
 *
 * It can also include an errno value which can be OR'd
 * with the above flags.  In this case, the errno within error_flag
 * is used instead of the global errno.
 * 
 * 
 * EXAMPLE
 *  #include <nv.h>
 *  nv_error(0, "stray interrupt %d", intr);
 *
 * EXAMPLE
 *        if ((fd = open(pathname, O_RDNLY)) < 0)
 *        {
 *            nv_error(NV_ERROR_ERRNO, "open of '%s' failed", pathname);
 *            goto failed;
 *        }
 *
 */

//---------------------------------------------------------------------------
//
//  Debugging support.
//
//---------------------------------------------------------------------------

//
// this is what actually outputs the data.
//
inline void out_string(const char *str)
{
    printk("%s", str);
}    

const char *nv_rm_status_text(
    int local_errno
)
{
    static char buff[32];
    static struct {
        unsigned int key;
        char *text;
    } *ep, errs[] = {
    { RM_OK,                            "RM_OK" },
    { RM_ERROR,                         "RM_ERROR" },
    { NV_TYPE_CONFLICT,                 "NV_TYPE_CONFLICT" },
    { NV_OUT_OF_RANGE,                  "NV_OUT_OF_RANGE" },
    { NV_NO_CONNECTION,                 "NV_NO_CONNECTION" },
    { NV_NO_SUCH_OBJECT,                "NV_NO_SUCH_OBJECT" },
    { NV_NAME_IN_USE,                   "NV_NAME_IN_USE" },
    { NV_OUT_OF_RESOURCES,              "NV_OUT_OF_RESOURCES" },
    { NV_TRANSLATION_VIOLATION,         "NV_TRANSLATION_VIOLATION" },
    { NV_PROTECTION_VIOLATION,          "NV_PROTECTION_VIOLATION" },
    { NV_BUFFER_BUSY,                   "NV_BUFFER_BUSY" },
    { NV_ILLEGAL_ACCESS,                "NV_ILLEGAL_ACCESS" },
    { NV_BAD_COLORMAP_FORMAT,           "NV_BAD_COLORMAP_FORMAT" },
    { NV_BAD_COLOR_FORMAT,              "NV_BAD_COLOR_FORMAT" },
    { NV_BAD_MONOCHROME_FORMAT,         "NV_BAD_MONOCHROME_FORMAT" },
    { NV_BAD_PATTERN_SHAPE,             "NV_BAD_PATTERN_SHAPE" },
    { NV_BAD_SUBDIVIDE,                 "NV_BAD_SUBDIVIDE" },
    { NV_NO_CURRENT_POINT,              "NV_NO_CURRENT_POINT" },
    { NV_NO_DMA_TRANSLATION,            "NV_NO_DMA_TRANSLATION" },
    { NV_INCOMPLETE_METHOD,             "NV_INCOMPLETE_METHOD" },
    { NV_RESERVED_ADDRESS,              "NV_RESERVED_ADDRESS" },
    { NV_UNIMPLEMENTED_PATCH,           "NV_UNIMPLEMENTED_PATCH" },
    { NV_OS_NAME_ERROR,                 "NV_OS_NAME_ERROR" },
    { NV_INCOMPLETE_PATCH,              "NV_INCOMPLETE_PATCH" },
    { NV_BUFFERGAP_ERROR,               "NV_BUFFERGAP_ERROR" },
    { NV_FIFO_OVERFLOW,                 "NV_FIFO_OVERFLOW" },
    { RM_WARN_NULL_OBJECT,              "RM_WARN_NULL_OBJECT" },
    { RM_WARN_GRAPHICS_DISABLED,        "RM_WARN_GRAPHICS_DISABLED" },
    { RM_ERR_NO_FREE_MEM,               "RM_ERR_NO_FREE_MEM" },
    { RM_ERR_MEM_NOT_FREED,             "RM_ERR_MEM_NOT_FREED" },
    { RM_ERR_PAGE_TABLE_NOT_AVAIL,      "RM_ERR_PAGE_TABLE_NOT_AVAIL" },
    { RM_ERR_NO_FREE_FIFOS,             "RM_ERR_NO_FREE_FIFOS" },
    { RM_ERR_CANT_CREATE_CLASS_OBJS,    "RM_ERR_CANT_CREATE_CLASS_OBJS" },
    { RM_ERR_BAD_OBJECT,                "RM_ERR_BAD_OBJECT" },
    { RM_ERR_INSERT_DUPLICATE_NAME,     "RM_ERR_INSERT_DUPLICATE_NAME" },
    { RM_ERR_OBJECT_NOT_FOUND,          "RM_ERR_OBJECT_NOT_FOUND" },
    { RM_ERR_CREATE_BAD_CLASS,          "RM_ERR_CREATE_BAD_CLASS" },
    { RM_ERR_DELETE_BAD_CLASS,          "RM_ERR_DELETE_BAD_CLASS" },
    { RM_ERR_FIFO_RUNOUT_OVERFLOW,      "RM_ERR_FIFO_RUNOUT_OVERFLOW" },
    { RM_ERR_FIFO_BAD_ACCESS,           "RM_ERR_FIFO_BAD_ACCESS" },
    { RM_ERR_FIFO_OVERFLOW,             "RM_ERR_FIFO_OVERFLOW" },
    { RM_ERR_METHOD_ORDER,              "RM_ERR_METHOD_ORDER" },
    { RM_ERR_METHOD_COUNT,              "RM_ERR_METHOD_COUNT" },
    { RM_ERR_ILLEGAL_OBJECT,            "RM_ERR_ILLEGAL_OBJECT" },
    { RM_ERR_DMA_IN_USE,                "RM_ERR_DMA_IN_USE" },
    { RM_ERR_BAD_DMA_SPECIFIER,         "RM_ERR_BAD_DMA_SPECIFIER" },
    { RM_ERR_INVALID_XLATE,             "RM_ERR_INVALID_XLATE" },
    { RM_ERR_INVALID_START_LENGTH,      "RM_ERR_INVALID_START_LENGTH" },
    { RM_ERR_DMA_MEM_NOT_LOCKED,        "RM_ERR_DMA_MEM_NOT_LOCKED" },
    { RM_ERR_DMA_MEM_NOT_UNLOCKED,      "RM_ERR_DMA_MEM_NOT_UNLOCKED" },
    { RM_ERR_NOTIFY_IN_USE,             "RM_ERR_NOTIFY_IN_USE" },
    { RM_ERR_ILLEGAL_ADDRESS,           "RM_ERR_ILLEGAL_ADDRESS" },
    { RM_ERR_BAD_ADDRESS,               "RM_ERR_BAD_ADDRESS" },
    { RM_ERR_INVALID_COLOR_FORMAT,      "RM_ERR_INVALID_COLOR_FORMAT" },
    { RM_ERR_INVALID_MONO_FORMAT,       "RM_ERR_INVALID_MONO_FORMAT" },
    { RM_ERR_OBJECT_TYPE_MISMATCH,      "RM_ERR_OBJECT_TYPE_MISMATCH" },
    { RM_ERR_INCOMPLETE_PATCH,          "RM_ERR_INCOMPLETE_PATCH" },
    { RM_ERR_INVALID_PATCH,             "RM_ERR_INVALID_PATCH" },
    { RM_ERR_PATCH_TOO_COMPLEX,         "RM_ERR_PATCH_TOO_COMPLEX" },
    { RM_ERR_MAX_PATCH_FANOUT,          "RM_ERR_MAX_PATCH_FANOUT" },
    { RM_ERR_DEVICE_DISCONNECTED,       "RM_ERR_DEVICE_DISCONNECTED" },
#if 0
    /* whence NV_BUFFER_GAP? */
    { RM_ERR_BUFFER_GAP,                "RM_ERR_BUFFER_GAP" },
#endif
    { RM_ERR_INVALID_OS_NAME,           "RM_ERR_INVALID_OS_NAME" },
    { RM_ERR_ILLEGAL_ACTION,            "RM_ERR_ILLEGAL_ACTION" },
    { RM_ERR_ILLEGAL_DIRECTION,         "RM_ERR_ILLEGAL_DIRECTION" },
    { RM_ERR_OUT_OF_TIMER_CALLBACKS,    "RM_ERR_OUT_OF_TIMER_CALLBACKS" },
    { RM_ERR_OPERATING_SYSTEM,          "RM_ERR_OPERATING_SYSTEM" },
    { RM_ERR_BAD_OBJECT_PARENT,         "RM_ERR_BAD_OBJECT_PARENT" },
    { RM_ERR_BAD_OBJECT_HANDLE,         "RM_ERR_BAD_OBJECT_HANDLE" },
    { RM_ERR_OBJECT_IN_USE,             "RM_ERR_OBJECT_IN_USE" },
    { RM_ERR_OBJECT_HAS_CHILDERN,       "RM_ERR_OBJECT_HAS_CHILDERN" },
    { RM_ERR_BAD_CLASS,                 "RM_ERR_BAD_CLASS" },
    { RM_ERR_INSUFFICIENT_RESOURCES,    "RM_ERR_INSUFFICIENT_RESOURCES" },
    { RM_ERR_BAD_FLAGS,                 "RM_ERR_BAD_FLAGS" },
    { RM_ERR_BAD_BASE,                  "RM_ERR_BAD_BASE" },
    { RM_ERR_BAD_LIMIT,                 "RM_ERR_BAD_LIMIT" },
    { RM_ERR_PROTECTION_FAULT,          "RM_ERR_PROTECTION_FAULT" },
    { RM_ERR_MULTIPLE_MEMORY_TYPES,     "RM_ERR_MULTIPLE_MEMORY_TYPES" },
    { RM_ERR_BAD_OBJECT_ERROR,          "RM_ERR_BAD_OBJECT_ERROR" },
    { RM_ERR_BAD_OBJECT_BUFFER,         "RM_ERR_BAD_OBJECT_BUFFER" },
    { RM_ERR_BAD_OFFSET,                "RM_ERR_BAD_OFFSET" },
    { 0, 0 },
    };

    for (ep = errs; ep->text; ep++)
        if ((ep->key & ~NV_ERROR_MASK) == local_errno)
            return ep->text;

    sprintf(buff, "unknown error 0x%x", local_errno);
    return buff;
}

// The current debug display level (default to maximum debug level)
int cur_debuglevel = 0xaaaaaaaa;

// override initial debug level from registry
VOID osDbgInit(void)
{
    U032 new_debuglevel;
    if (RM_OK == osReadRegistryDword((PHWINFO) 0,
                                     "",
                                     "resman_debuglevel",
                                     &new_debuglevel))
    {
        if (new_debuglevel != ~0)
            cur_debuglevel = new_debuglevel;
    }
}

/* we used to have a global lock that we had to drop so incoming interrupts
 * wouldn't hang the machine waiting for the lock. In the switch to finer
 * grained locking, I've had to remove that code. We don't know which card has
 * a lock or which threw the breakpoint. I should probably scan the list of
 * nv_state_t's and drop any held locks before throwing this breakpoint.
 */
VOID osDbgBreakPoint(void)
{
    out_string("Break\n");

#if defined(CONFIG_X86_REMOTE_DEBUG)
     __asm__ __volatile__ ("int $3");
#elif defined(CONFIG_KDB)
    KDB_ENTER();
#else
    out_string("Skipping INT 3, we don't like kernel panics\n");
#endif
}



#define MAX_ERROR_STRING 256
    
/*
 * Report an error using a variable argument pointer.
 * This is the utility routine for nv_error, nv_abort,
 * and nv_panic.
 * error_flag is as above; printf_format is a normal
 * printf(3) format string, with its concommitant arguments.
 * 
 * Returns the number of characters written.
 */

int nv_verror(
    int   error_flag,
    const char *printf_format,
    va_list arglist
)
{
    int  local_errno;
    char nv_error_string[MAX_ERROR_STRING];
    char *p = nv_error_string;
    
    local_errno = error_flag & ~NV_ERROR_MASK;

    p = nv_error_string;

    /*
     * Put "ERROR" on all messages, unless requested not to.
     */
    
    if ( ! (error_flag & NV_ERROR_MESSAGE))
        p += sprintf(p, "%s: ",
                     (error_flag & NV_ERROR_WARNING) ? "WARNING" : "ERROR");

    /*
     * Do the actual print of user data and args
     */
    
    p += vsprintf(p, printf_format, arglist);
    
    /*
     * Add in "errno" or NV status info if requested
     */
    
    if ((error_flag & NV_ERROR_ERRNO) && (local_errno != 0))
        p += sprintf(p, " (errno=%d)\n", local_errno);
    else if (error_flag & NV_ERROR_STATUS)
        p += sprintf(p, " (%s)\n", nv_rm_status_text(local_errno));
    else
    p += sprintf(p, "\n");

    out_string(nv_error_string);

    /*
     * Bug out if requested
     */

    if (error_flag & NV_ERROR_PANIC)
    {
        nv_error(0, "fatal error");
        osDbgBreakPoint();
    }

    return p - nv_error_string;
}


/*
 * Report an error using nv_verror.
 * 
 * Returns the number of characters written.
 */

int nv_error(
    int   error_flag,
    const char *printf_format,
    ...
  )
{
    va_list arglist;
    int chars_written;

    va_start(arglist, printf_format);
    chars_written = nv_verror(error_flag, printf_format, arglist);
    va_end(arglist);
    
    return chars_written;
}

/*
 * nv_panic is shorthand for nv_error(NV_ERROR_PANIC, ...)
 */

void nv_panic(
    const char *printf_format,
    ...
  )
{
    va_list arglist;

    va_start(arglist, printf_format);
    /* | 1 sets local_errno & makes sure 'exit' gets non-zero value    */
    (void) nv_verror(NV_ERROR_PANIC | 1, printf_format, arglist);
    /* NORETURN */
    for (;;) ;
}

/*
 * nv_warning is shorthand for nv_error(NV_ERROR_WARNING, ...)
 */

void nv_warning(
    const char *printf_format,
    ...
  )
{
    va_list arglist;

    va_start(arglist, printf_format);
    (void) nv_verror(NV_ERROR_WARNING, printf_format, arglist);
    va_end(arglist);
}


/*
 * nv_message is shorthand for nv_error(NV_ERROR_MESSAGE, ...)
 * So it doesn't print the word ERROR on every message.
 */

void nv_message(
    const char *printf_format,
    ...
  )
{
    va_list arglist;

    va_start(arglist, printf_format);
    (void) nv_verror(NV_ERROR_MESSAGE, printf_format, arglist);
    va_end(arglist);
}


/*
 * nv_printf() prints to the "error" log for the driver.
 * Just like printf, adds no newlines or names
 */

int nv_printf(
    const char *printf_format,
    ...
)
{
    char nv_error_string[MAX_ERROR_STRING];
    char *p = nv_error_string;
    int chars_written;
    va_list arglist;

    va_start(arglist, printf_format);
    chars_written = vsprintf(p, printf_format, arglist);
    va_end(arglist);

    out_string(p);

    return chars_written;
}

/*
 * Like nv_printf(), but pays attention to debuglevel
 * 
 * Returns the number of characters written.
 */

int nv_debug(
    int   debuglevel,
    const char *printf_format,
    ...
  )
{
    char nv_error_string[MAX_ERROR_STRING];
    char *p = nv_error_string;
    va_list arglist;
    int chars_written = 0;

    if (debuglevel >= cur_debuglevel)
    {
        va_start(arglist, printf_format);
        chars_written = vsprintf(p, printf_format, arglist);
        va_end(arglist);
        out_string(p);
    }

    return chars_written;
}

// AGP additions
// This AGP support is intended for pre-existing installations that
// haven't patched their kernel with Jeff's module, and don't have
// 2.3.XX or 2.4.XX. Over time, we'll add support for Jeff's module

BOOL
osPciDevicePresent(unsigned short vendor, unsigned short device)
{
    if (pci_find_device(vendor, device, NULL))
        return 1;
    return 0;
}

VOID *
osPciInitHandle(U008 bus, U008 slot, U008 function, U016 *vendor, U016 *device)
{
    struct pci_dev *dev;
    dev = pci_find_slot(bus, PCI_DEVFN(slot, function));
    if (!dev)
        return NULL;
    if (vendor) *vendor = dev->vendor;
    if (device) *device = dev->device;
    return (VOID *) dev;
}

U008
osPciReadByte(VOID *handle, U008 offset)
{
    U008 value;
    pci_read_config_byte( (struct pci_dev *) handle, offset, &value);
    return value;
}

U016
osPciReadWord(VOID *handle, U008 offset)
{
    U016 value;
    pci_read_config_word( (struct pci_dev *) handle, offset, &value);
    return value;
}

U032
osPciReadDword(VOID *handle, U008 offset) 
{
    U032 value;
    pci_read_config_dword( (struct pci_dev *) handle, offset, (u32 *) &value);
    return value;
}

VOID
osPciWriteByte(VOID *handle, U008 offset, U008 value)
{
    pci_write_config_byte( (struct pci_dev *) handle, offset, value);
}

VOID
osPciWriteWord(VOID *handle, U008 offset, U016 value)
{
    pci_write_config_word( (struct pci_dev *) handle, offset, value);
}

VOID
osPciWriteDword(VOID *handle, U008 offset, U032 value) {
    pci_write_config_dword( (struct pci_dev *) handle, offset, value);
}

RM_STATUS 
osMapIOPortSpace( 
    PHWINFO pDev,
    U032 Address, 
    U032 Length,
    U032 *uVirtualAddress
) 
{
    *uVirtualAddress = Address;
    return( RM_OK );
}

void
osIoWriteByte(PHWINFO pdev, unsigned long address, unsigned char value)
{
    outb(value, address);
}

void
osIoWriteWord(PHWINFO pdev, unsigned long address, unsigned short value)
{
    outw(value, address);
}

void
osIoWriteDword(PHWINFO pdev, unsigned long address, unsigned int value)
{
    outl(value, address);
}

unsigned char
osIoReadByte(PHWINFO pdev, unsigned long address)
{
    return inb(address);
}

unsigned short
osIoReadWord(PHWINFO pdev, unsigned long address)
{
    return inw(address);
}

unsigned int
osIoReadDword(PHWINFO pdev, unsigned long address)
{
    return inl(address);
}

unsigned int 
osCli(unsigned int flags)
{
    save_flags(flags);
    cli();
    return flags;
}

unsigned int 
osSti(unsigned int flags)
{
    restore_flags(flags);
    return flags;
}


typedef struct wb_mem_t {
    unsigned int start;
    unsigned int size;
    struct wb_mem_t *next;
} wb_mem;

static wb_mem *wb_list = NULL;

int save_wb(unsigned int start, unsigned int size)
{
#ifdef CONFIG_MTRR
    struct wb_mem_t *ptr;

    NV_KMALLOC(ptr, sizeof(struct wb_mem_t));
    if (ptr == NULL)
        return 0;

    ptr->start = start;
    ptr->size  = size;
    ptr->next  = wb_list;
    wb_list    = ptr;

    return 1;
#endif
    return 0;
}

int del_wb(unsigned int start, unsigned int size)
{
#ifdef CONFIG_MTRR
    struct wb_mem_t *ptr;
    struct wb_mem_t *prev;

    prev = ptr = wb_list;

    while (ptr != NULL) {
        if (ptr->start == start && ptr->size == size) {
            if (ptr == wb_list)
                wb_list = ptr->next;
            else
                prev->next = ptr->next;

            NV_KFREE(ptr);
            return 1;
        }
        prev = ptr;
        ptr = prev->next;
    }
#endif
    return 0;
}

/*
 * Don't assume MTRR-specific, as we could add support for PATs to 
 * achieve the same results on a PIII or higher
 */
RM_STATUS
osSetMemRange(U032 start, U032 size, U032 mode)
{
#ifdef CONFIG_MTRR
    int err;

    /* for now, we only set memory to write-combined */
    if (mode != NV_MEMORY_WRITECOMBINED)
        return RM_ERROR;

    err = mtrr_add(start, size, MTRR_TYPE_WRCOMB, 0x0);
    if (err < 0)
    {
        printk("nvidia: cannot write-combine 0x%08x, %dM\n",
            start, size / (1024 * 1024));
        return RM_ERROR;
    }

    save_wb(start, size);

    return RM_OK;
#endif
    return RM_ERROR;
}

RM_STATUS
osUnsetMemRange(U032 start, U032 size)
{
#ifdef CONFIG_MTRR
    if (del_wb(start, size))
    {
        mtrr_del(-1, start, size);
    }
    return RM_OK;
#endif
    return RM_ERROR;
}


/*
 * This needs to remap the AGP Aperture from IO space to kernel space,
 * or at least return a pointer to a kernel space mapping of the Aperture
 * should this also check for Write-Combining??
 */
void *
osMapKernelSpace(unsigned int start, unsigned int size_bytes, U032 mode)
{
    void *vaddr;

    if (in_interrupt())
    {
        nv_printf("trying to map 0x%x to kernel space in interrupt!\n", start);
        osDbgBreakPoint();
        return NULL;
    }

    {
#if defined (KERNEL_2_2)
        unsigned long map_nr = MAP_NR(__va(start));
        if (map_nr < max_mapnr) {
            vaddr = __va(start);
        } else 
#endif
        {
            if (mode == NV_MEMORY_DEFAULT) {
                 vaddr = ioremap(start, size_bytes);
            } else {
                 vaddr = ioremap_nocache(start, size_bytes);
            }
        }
    }

#ifdef DEBUG
    if (mode == NV_MEMORY_WRITECOMBINED) {
        printk("nvidia: request to write-combine remapped memory!");
        osDbgBreakPoint();
    }
    else if (mode != NV_MEMORY_UNCACHED) {
        printk("nvidia: request to cache remapped memory!");
        osDbgBreakPoint();
    }
#endif 

    return vaddr;
}

void
osUnmapKernelSpace(void *addr, unsigned int size_bytes)
{
#if defined (KERNEL_2_2)
    if (MAP_NR(addr) < max_mapnr) {
        // if we didn't want the memory cached, this isn't necessary
        // but we shouldn't be in a timing critical piece of code.
        asm volatile("wbinvd":::"memory");
    } else 
#endif
    {
        iounmap(addr);
    }
}

/* these two functions are for NT, and are only pass-through functions
 * under Linux.
 */
RM_STATUS
osAllocPool(void **addr, unsigned int size_bytes)
{
    return RM_OK;
}

RM_STATUS
osFreePool(void *addr)
{
   return RM_OK;
}

/* #### this whole section is currently only used to allocate the AGP GATT */
/* XXX hard-coded globals exist here for now, clean up as soon as possible */

inline int
count_num_pages(int order)
{
    int i, num_pages = 1;

    for (i = 0; i < order; i++)
    {
        num_pages *= 2;
    }
    return num_pages;
}

static void *gatt_table = NULL;
static int gatt_pages = 0;
static int gatt_order= 0;

void *
osAllocContigPages(unsigned long size)
{
    int order = 0;
    void *paddr, *tmp_paddr;
    unsigned long num_pages = size / PAGE_SIZE;

    if (gatt_table != NULL)
        nv_printf("RM Warning: multiple Contiguous allocs!\n");

    calc_order(size, order);
    num_pages = count_num_pages(order);

    tmp_paddr = paddr = (void *) __get_free_pages(GFP_KERNEL, order);
    if (paddr == 0x0)
        return paddr;

    gatt_table = paddr;
    gatt_pages = num_pages;
    gatt_order = order;

    while (num_pages--)
    {
	mem_map_reserve(GET_MAP_NR(__pa(tmp_paddr)));
        tmp_paddr = (void *)((unsigned long)tmp_paddr + PAGE_SIZE);
    }

    return paddr;
}

void
osFreeContigPages(void *addr)
{
    void *tmp_paddr = addr;

    if (gatt_table == NULL)
    {
        nv_printf("RM Warning: couldn't find contiguous alloc!\n");
        return; 
    }

    while (gatt_pages--)
    {
	mem_map_unreserve(GET_MAP_NR(__pa(tmp_paddr)));
        tmp_paddr = (void *)((unsigned long)tmp_paddr + PAGE_SIZE);
    }
    free_pages((unsigned long) addr, gatt_order);

    gatt_order = 0;
    gatt_pages = 0;
    gatt_table = NULL;
}

/* ### end of section */

void *
osMapUserSpace(void *kaddr, void **priv, unsigned int size_bytes, U032 mode)
{
    int err;
    void *paddr = NULL, *uaddr = NULL;

    if ( (size_t) kaddr > VMALLOC_START) {
        nv_ioctl_memory_vtop_t parms;
        parms.buffer = kaddr;
        parms.byte_length = 1;
        parms.physical_addresses = (unsigned int *) &paddr;
        nv_vtop(&parms, 0);
    } else {
        paddr = (void *) virt_to_phys(kaddr);
    }

    uaddr = *priv;

    /* finally, let's do it! */
    err = remap_page_range( (size_t) uaddr, (size_t) paddr, size_bytes, 
                            PAGE_SHARED);

    if (err != 0)
    {
        return (void *) NULL;
    }

    return uaddr;
}

void osUnmapUserSpace(void *uaddr, void *priv)
{
    // I don't think I need to do anything here...
}

void *osMapIOSpace(unsigned int start, unsigned int size_bytes,
                   void **priv, U032 user, U032 mode)
{
    int err;
    void *uaddr = NULL;

    if (!user)
        return NULL;

    uaddr = *priv;

    /* finally, let's do it! */
    err = remap_page_range( (size_t) uaddr, (size_t) start, size_bytes, 
                            PAGE_SHARED);

    if (err != 0)
    {
        return (void *) NULL;
    }

    return uaddr;
}

void osUnmapIOSpace(void *addr, unsigned int size_bytes, void *priv, U032 user)
{
}


U032
osGetKernPhysAddr(void *vaddr)
{
    U032 ret = 0UL;

    if ( (size_t) vaddr > VMALLOC_START) {
        nv_ioctl_memory_vtop_t parms;
        parms.buffer = vaddr;
        parms.byte_length = 1;
        parms.physical_addresses = (unsigned int *) &ret;
        nv_vtop(&parms, 0);
    } else {
        ret = virt_to_phys(vaddr);
    }

    return ret;
}

/*
 * "registry" read/write routines
 *
 * Currently, this just looks up variables defined in resman.
 * These could be set during insmod time.
 *
 * Example:
 *    registry entry "foofoo" becomes a symbol lookup for
 *    a symbol defined by NVdriver named: NVreg_foofoo.
 *
 * Symbols w/ embedded chars that are illegal in C symbols are munged slightly.
 *    Illegal chars are deleted, except for ' ' which is replaced by '_'
 */

#if !defined (KERNEL_2_2)
static spinlock_t unload_lock = SPIN_LOCK_UNLOCKED;
#endif

void *osRegistryLookup(
    PHWINFO pDev,
    char   *regDevNode,
    char   *regParmStr
)
{
    char symbol_name[NV_MAX_SYM_NAME+1];
#if !defined (KERNEL_2_2)
    unsigned long symbol_value = 0;
#else
    void *symbol_value = 0;
#endif
    char *symp;
    char *parmp;
    char ch;

    if ((strlen(regParmStr) + NV_SYM_PREFIX_LENGTH) > NV_MAX_SYM_NAME)
        goto done;

    strcpy(symbol_name, NV_SYM_PREFIX);
    
    // copy the symbol name over, fixing any bad chars in it.
    symp = symbol_name + NV_SYM_PREFIX_LENGTH;
    for (parmp = regParmStr; (ch = *parmp); parmp++)
        if ((ch >= 'a' && (ch <= 'z')) ||
            (ch >= 'A' && (ch <= 'Z')) ||
            (ch >= '0' && (ch <= '9')) ||
            (ch == '_'))
        {
            *symp++ = ch;
        }
        else if (ch == ' ')
            *symp++ = '_';

    *symp = '\0';

    NV_GET_REG_SYMBOL(symbol_value, NV_MODULE_NAME, symbol_name);
     
 done:
    return (void *) symbol_value;
}

RM_STATUS osReadRegistryBoolean(
    PHWINFO pDev,
    char *regDevNode,
    char *regParmStr
)
{
    return 1;
}

RM_STATUS osReadRegistryBinary(
    PHWINFO pDev,
    char   *regDevNode,
    char   *regParmStr,
    U008   *Data,
    U032   *cbLen
)
{
    return RM_ERROR;
}

RM_STATUS osWriteRegistryBinary(
    PHWINFO pDev,
    char   *regDevNode,
    char   *regParmStr,
    U008   *Data,
    U032    cbLen
)
{
    return RM_ERROR;
}

// XXX doesn't currently save registry value between reboots, but will
// so long as we're running (which is all we need for some things..)
RM_STATUS osWriteRegistryDword(
    PHWINFO pDev,
    char   *regDevNode,
    char   *regParmStr,
    U032    Data
)
{
    U032 *ptr;

    ptr = osRegistryLookup(pDev, regDevNode, regParmStr);
    if (ptr)
    {
        *ptr = Data;
        return RM_OK;
    }

    return RM_ERROR;
}


RM_STATUS osReadRegistryDword(
    PHWINFO pDev,
    char   *regDevNode,
    char   *regParmStr,
    U032   *Data
)
{
    U032 *ptr;

    ptr = osRegistryLookup(pDev, regDevNode, regParmStr);
    if (ptr)
    {
        *Data = *ptr;

        /* XXX Ugly, ugly hack to get the correct value for the
         * Mobile registry value when the user didn't specify
         * it.  Note that this is only a temporary fix until we can
         * get this information from the video bios.
         */
        
        if ((strcmp(regParmStr, "Mobile") == 0) && (*Data == 0xFFFFFFFF))
            *Data = RmGetMobileValue(pDev);
        
        return RM_OK;
    }

    return RM_ERROR;
}

/* AGPGART support - mostly by Christian Zander [phoenix@minion.de] */

#ifdef AGPGART

typedef struct {
    agp_memory *ptr;
    int num_pages;
    int offset;
#if defined(IA64)
    struct vm_area_struct *vma;
    void  *handle;
#endif
} agp_priv_data;

typedef struct {
    agp_memory *memory;
    void *aperture;
#ifdef CONFIG_MTRR
    int  mtrr;
#endif
    int  ready;
} agp_gart;

typedef struct {
    int           (*backend_acquire)(void);
    void          (*backend_release)(void);
    void          (*copy_info)(agp_kern_info *);
    agp_memory *  (*allocate_memory)(size_t, unsigned int);
    void          (*free_memory)(agp_memory *);
    int           (*bind_memory)(agp_memory *, off_t);
    int           (*unbind_memory)(agp_memory *);
    void          (*enable)(unsigned int);
} agp_operations_struct;

agp_operations_struct agp_ops;
agp_kern_info         agpinfo;
agp_gart              gart;
#if !defined (KERNEL_2_2)
const drm_agp_t       *drm_agp_p;
#endif

#if defined (KERNEL_2_2)
	#define GET_AGPGART_SYMBOL(sym, sym_string)						\
		sym = (void*) GET_MODULE_SYMBOL(0, sym_string);					\
		if (sym == NULL)								\
		{										\
			printk("NVRM: AGPGART: unable to retrieve symbol %s\n", sym_string);	\
			return 1;								\
		}

	#define AGP_BACKEND_ACQUIRE_SYM   __MODULE_STRING(agp_backend_acquire)
	#define AGP_BACKEND_RELEASE_SYM   __MODULE_STRING(agp_backend_release)
	#define AGP_COPY_INFO_SYM         __MODULE_STRING(agp_copy_info)
	#define AGP_ALLOCATE_MEMORY_SYM   __MODULE_STRING(agp_allocate_memory)
	#define AGP_FREE_MEMORY_SYM       __MODULE_STRING(agp_free_memory)
	#define AGP_BIND_MEMORY_SYM       __MODULE_STRING(agp_bind_memory)
	#define AGP_UNBIND_MEMORY_SYM     __MODULE_STRING(agp_unbind_memory)
	#define AGP_ENABLE_SYM            __MODULE_STRING(agp_enable)
#endif

BOOL KernInitAGP(VOID **ap_phys_base, VOID **ap_mapped_base, U032 *apsize)
{
    U032  agp_rate;
    U032  agp_sba;
    U032  agp_fw;
    char* chipset;

    memset( (void *) &gart, 0, sizeof(agp_gart));

#if !defined (KERNEL_2_2)
    if (!(drm_agp_p = inter_module_get_request("drm_agp", "agpgart"))) {
        printk(KERN_ERR "NVRM: AGPGART: unable to retrieve symbol table\n");
        return 1;
    }
    
    agp_ops.backend_acquire = drm_agp_p->acquire;
    agp_ops.backend_release = drm_agp_p->release;
    agp_ops.copy_info       = drm_agp_p->copy_info;
    agp_ops.allocate_memory = drm_agp_p->allocate_memory;
    agp_ops.free_memory     = drm_agp_p->free_memory;
    agp_ops.bind_memory     = drm_agp_p->bind_memory;
    agp_ops.unbind_memory   = drm_agp_p->unbind_memory;
    agp_ops.enable          = drm_agp_p->enable;

#else
	#if defined(CONFIG_KMOD)
		if ( request_module("agpgart") ) {
			printk(KERN_INFO "NVRM: AGPGART: not loading agpgart.o\n");
			return 1;
		}
	#endif

    GET_AGPGART_SYMBOL(agp_ops.backend_acquire, AGP_BACKEND_ACQUIRE_SYM);
    GET_AGPGART_SYMBOL(agp_ops.backend_release, AGP_BACKEND_RELEASE_SYM);
    GET_AGPGART_SYMBOL(agp_ops.copy_info,       AGP_COPY_INFO_SYM);
    GET_AGPGART_SYMBOL(agp_ops.allocate_memory, AGP_ALLOCATE_MEMORY_SYM);
    GET_AGPGART_SYMBOL(agp_ops.free_memory,     AGP_FREE_MEMORY_SYM);
    GET_AGPGART_SYMBOL(agp_ops.bind_memory,     AGP_BIND_MEMORY_SYM);
    GET_AGPGART_SYMBOL(agp_ops.unbind_memory,   AGP_UNBIND_MEMORY_SYM);
    GET_AGPGART_SYMBOL(agp_ops.enable,          AGP_ENABLE_SYM);
#endif

    /* NOTE: from here down, return an error code of '-1'
     * that indicates that agpgart is loaded, but we failed to use it
     * in some way. This is so we don't try to use nvagp and lock up
     * the memory controller.
     */

    if ( (*(agp_ops.backend_acquire))() ) {
        printk(KERN_ERR "NVRM: AGPGART: backend in use\n");
        return -1;
    }

    if (osReadRegistryDword(NULL, "NVreg", "ReqAGPRate", &agp_rate) == RM_ERROR)
        agp_rate = 0x00000007;
    agp_rate &= 0x00000007;

    if (osReadRegistryDword(NULL, "NVreg", "EnableAGPSBA", &agp_sba) == RM_ERROR)
        agp_sba = 1;
    agp_sba &= 0x00000001;

    if (osReadRegistryDword(NULL, "NVreg", "EnableAGPFW", &agp_fw) == RM_ERROR)
        agp_fw = 1;
    agp_fw &= 0x00000001;
    
    (*(agp_ops.copy_info))(&agpinfo);

    switch ( agpinfo.chipset ) {
        case INTEL_GENERIC:    chipset = "Intel";            break;
        case INTEL_LX:         chipset = "Intel 440LX";      break;
        case INTEL_BX:         chipset = "Intel 440BX";      break;
        case INTEL_GX:         chipset = "Intel 440GX";      break;
        case INTEL_I810:       chipset = "Intel i810";       break;
        case INTEL_I840:       chipset = "Intel i840";       break;
#if !defined (KERNEL_2_2)
        case INTEL_I815:       chipset = "Intel i815";       break;
#if !defined(__rh_config_h__)
        case INTEL_I850:       chipset = "Intel i850";       break;
#endif
#endif
#if defined(IA64)
        case INTEL_460GX:      chipset = "Intel 460GX";      break;
#endif
        case VIA_GENERIC:      chipset = "VIA";              break;
        case VIA_VP3:          chipset = "VIA VP3";          break;
        case VIA_MVP3:         chipset = "VIA MVP3";         break;
        case VIA_MVP4:         chipset = "VIA MVP4";         break;
#if !defined (KERNEL_2_2)
        case VIA_APOLLO_KX133: chipset = "VIA Apollo KX133"; break;
        case VIA_APOLLO_KT133: chipset = "VIA Apollo KT133"; break;
#endif
        case VIA_APOLLO_PRO:   chipset = "VIA Apollo Pro";   break;
        case SIS_GENERIC:      chipset = "SiS";              break;
        case AMD_GENERIC:      chipset = "AMD";              break;
        case AMD_IRONGATE:     chipset = "AMD Irongate";     break;
        case ALI_M1541:        chipset = "ALi M1541";        break;
        case ALI_GENERIC:      chipset = "ALi";              break;
        case NOT_SUPPORTED:    chipset = "unsupported";      break;
        default:               chipset = "unknown";
    }

    printk(KERN_INFO "NVRM: AGPGART: %s chipset\n", chipset);

#ifdef CONFIG_MTRR
    {
        if ((gart.mtrr = mtrr_add(agpinfo.aper_base, agpinfo.aper_size * 0x100000, MTRR_TYPE_WRCOMB, 0x0)) < 0)
            printk(KERN_ERR "NVRM: AGPGART: unable to set MTRR write-combining\n");
    }
#endif

    /*
     * Just map a single page for drivers to osGetAddressInfo() will work.
     * This will allow a client to create a DMA context for the entire
     * aperture without mapping the entire aperture.
     */

    if (!(gart.aperture = (void*) ioremap(agpinfo.aper_base, NVHW_PAGE_SIZE))) {
        printk(KERN_ERR "NVRM: AGPGART: unable to remap aperture\n");
        return -1;
    }

    printk(KERN_INFO "NVRM: AGPGART: aperture: %ldM @ 0x%08lx\n", (unsigned long)agpinfo.aper_size, (unsigned long)agpinfo.aper_base);
    printk(KERN_INFO "NVRM: AGPGART: aperture mapped from 0x%08lx to 0x%08lx\n", agpinfo.aper_base, (unsigned long) gart.aperture);

    if (!agp_sba) agpinfo.mode &= ~0x00000200;
    if (!agp_fw)  agpinfo.mode &= ~0x00000010;

    if (!(agp_rate & 0x00000004)) agpinfo.mode &= ~0x00000004;
    if (!(agp_rate & 0x00000002)) agpinfo.mode &= ~0x00000002;
    
    (*(agp_ops.enable))(agpinfo.mode);
    
    if (agpinfo.mode & 0x00000200)
        printk(KERN_INFO "NVRM: AGPGART: backend supports sba\n");
    if (agpinfo.mode & 0x00000010)
        printk(KERN_INFO "NVRM: AGPGART: backend supports fw\n");
    if (agpinfo.mode & 0x00000004)
        printk(KERN_INFO "NVRM: AGPGART: mode 4x\n");
    else if (agpinfo.mode & 0x00000002)
        printk(KERN_INFO "NVRM: AGPGART: mode 2x\n");
    else if (agpinfo.mode & 0x00000001)
        printk(KERN_INFO "NVRM: AGPGART: mode 1x\n");

    *ap_phys_base   = (void*) agpinfo.aper_base;
    *ap_mapped_base = (void*) gart.aperture;
    *apsize         = (agpinfo.aper_size * 0x100000) - 1;

    gart.ready = 1;

    return 0;
}

BOOL KernTeardownAGP()
{
    /* sanity check to make sure we should actually be here. */
    if (!gart.ready)
    {
        printk(KERN_INFO "NVRM: AGPGART: not ready so not releasing backend\n");
        return 0;
    }

    if (gart.aperture != NULL) {
#ifdef CONFIG_MTRR
        if (gart.mtrr > 0)
            mtrr_del(gart.mtrr, 0, 0);
#endif
        iounmap(gart.aperture);
    }

    (*(agp_ops.backend_release))();

#if !defined (KERNEL_2_2)
    inter_module_put("drm_agp");
#endif

    printk(KERN_INFO "NVRM: AGPGART: backend released\n");
    return 0;
}

#define DEBUG_AGPGART_ALLOCATIONS 0

#if DEBUG_AGPGART_ALLOCATIONS
static void
print_agp_table()
{
    agp_memory *ptr;

    printk("AGP Table:\n"); 
    for (ptr = gart.memory; ptr; ptr = ptr->next)
    {
        printk("  0x%04x - 0x%04x (%d): 0x%p\n", 
                   (unsigned int) ptr->pg_start, 
                   (unsigned int) ptr->pg_start + ptr->page_count,
                   ptr->page_count,
                   ptr->memory);
    } 
}
#endif

#if DEBUG_AGPGART_ALLOCATIONS
#define PRINT_AGPGART_TABLE() print_agp_table()
#else
#define PRINT_AGPGART_TABLE()
#endif

RM_STATUS KernAllocAGPPages(VOID **pAddress, U032 PageCount,
                            VOID **pPriv_data)
{
    agp_memory *ptr, *i;
    off_t offset;
    int err;
    agp_priv_data *data;
    RM_STATUS status;

    if (!gart.ready) {
        *pAddress = (void*) 0;
        printk(KERN_ERR "NVRM: AGPGART: backend not ready\n");
        return RM_ERROR;
    }
    
    ptr = (*agp_ops.allocate_memory)(PageCount, AGP_NORMAL_MEMORY);
    if (ptr == NULL) {
        *pAddress = (void*) 0;
        printk(KERN_ERR "NVRM: AGPGART: no pages available\n");
        return RM_ERR_NO_FREE_MEM;
    }
    
    offset = 0;

    if (gart.memory == NULL) {
        ptr->next = NULL;
        ptr->prev = NULL;
        gart.memory = ptr;
    } else {
        for (i = gart.memory; i != NULL; i = i->next) {
            if (i->next == NULL && offset == 0) {
                ptr->next = NULL;
                ptr->prev = i;
                i->next = ptr;
                offset = i->pg_start + i->page_count;
                break;
            }
            if (i->pg_start + i->page_count + PageCount < i->next->pg_start) {
                ptr->next = i->next;
                ptr->prev = i;
                i->next->prev = ptr;
                i->next = ptr;
                offset = i->pg_start + i->page_count;
                break;
            }
        }
    }

    err = (*(agp_ops.bind_memory))(ptr, offset);
    if (err) {
        printk(KERN_ERR "NVRM: AGPGART: unable to bind %lu pages\n",
               (unsigned long) PageCount);
        goto fail;
    }

    PRINT_AGPGART_TABLE();
  
    /* return the agp aperture address */ 
    *pAddress = (void *) (agpinfo.aper_base + (offset << PAGE_SHIFT));

    status = osAllocMem((VOID **)&data, sizeof(agp_priv_data));
    if (status != RM_OK) {
        printk(KERN_ERR "NVRM: memory allocation failed\n");
        (*(agp_ops.unbind_memory))(ptr);
        goto fail;
    }

    data->ptr = ptr;
    data->num_pages = PageCount;
    data->offset = offset;

    *pPriv_data = data;
    
    printk(KERN_INFO "NVRM: AGPGART: allocated %lu pages\n",
           (unsigned long) PageCount);
    return RM_OK;

fail:
    if (!(ptr->prev == NULL)) {
        if (!(ptr->next == NULL)) {
            ptr->prev->next = ptr->next;
            ptr->next->prev = ptr->prev;
        } else ptr->prev->next = NULL;
    } else gart.memory = ptr->next;

    (*(agp_ops.free_memory))(ptr);
    *pAddress = (void*) 0;

    return RM_ERROR;
}

#if !defined(IA64) || (LINUX_VERSION_CODE < KERNEL_VERSION(2, 4, 9))

RM_STATUS 
KernMapAGPPages(void *pvma, void *priv_data)
{
    struct vm_area_struct *vma = (struct vm_area_struct *) pvma;
    agp_priv_data *agp_data = (agp_priv_data *) priv_data;
    unsigned long agp_addr;
    int err;

    if ((agp_data == NULL) || (vma == NULL))
        return RM_ERROR;

    agp_addr = agpinfo.aper_base + (agp_data->offset << PAGE_SHIFT);

    err = remap_page_range(vma->vm_start, (size_t) agp_addr, 
                           agp_data->num_pages << PAGE_SHIFT,
#if defined(IA64)
                           vma->vm_page_prot);
#else
                           PAGE_SHARED);
#endif
        
    if (err) {
        printk(KERN_ERR "NVRM: AGPGART: unable to remap %lu pages\n",
               (unsigned long)agp_data->num_pages);
        (*(agp_ops.unbind_memory))(agp_data->ptr);
        goto fail;
    }
    
#if defined(IA64)
    vma->vm_pgoff = agp_addr >> PAGE_SHIFT;
    agp_data->vma = vma;
    agp_register_map(vma);
#endif

    return RM_OK;

fail:
    /* XXX what else needs to happen here? */
    return RM_ERROR;
}

#else

RM_STATUS 
KernMapAGPNopage(void *address, void *pvma, void *priv_data, void **ppage)
{
    struct vm_area_struct *vma = (struct vm_area_struct *) pvma;
    agp_priv_data *agp_data = (agp_priv_data *) priv_data;
    agp_memory *agp_memory_ptr = agp_data->ptr;
    struct page *page_ptr;
    char *cp;
    unsigned long phys_addr;
    int idx, i;

    idx = ((unsigned long)address - vma->vm_start) >> PAGE_SHIFT;
    phys_addr = (agp_memory_ptr->memory[idx] & 0xffffff) << 12;
    page_ptr = virt_to_page(__va(phys_addr));
    get_page(page_ptr);
    *ppage = page_ptr;
    
    /*
     * Reliability is dramatically improved when invalidating the cache lines
     * prior to using write combined mappings.  The "fc" instruction is
     * specified to affect at least 32 bytes, thus the 32 byte increment below.
     */ 
    cp = (char *)__va(phys_addr);
    for (i = 0; i < PAGE_SIZE/32; i++) {
        ia64_fc (cp);
        cp += 32;
    }
    ia64_sync_i(); /* paranoia */

    return RM_OK;
}
#endif

RM_STATUS KernFreeAGPPages(VOID **pAddress, VOID *priv_data)
{
    agp_memory *ptr;
    agp_priv_data *data = priv_data;
    size_t pages;
    
#if defined(IA64) && (LINUX_VERSION_CODE < KERNEL_VERSION(2, 4, 9))
    agp_unregister_map(data->vma);
#endif
    ptr = data->ptr;

    osFreeMem(data);

    if (!(ptr == NULL)) {
        pages = ptr->page_count;

        if (!(ptr->prev == NULL)) {
            if (!(ptr->next == NULL)) {
                ptr->prev->next = ptr->next;
                ptr->next->prev = ptr->prev;
            } else ptr->prev->next = NULL;
        } else {
            gart.memory = ptr->next;
            if (gart.memory)
                gart.memory->prev = NULL;
        }
    
        (*(agp_ops.unbind_memory))(ptr);
        (*(agp_ops.free_memory))(ptr);

        printk(KERN_INFO "NVRM: AGPGART: freed %ld pages\n", (unsigned long)pages);
        PRINT_AGPGART_TABLE();
        return RM_OK;
    }
    return RM_ERROR;
}

#else

BOOL KernInitAGP(VOID **ap_phys_base, VOID **ap_mapped_base, U032 *apsize)
{
    return 1;
};

BOOL KernTeardownAGP()
{
    return 1;
};

RM_STATUS KernAllocAGPPages(VOID **pAddress, U032 PageCount,
                            VOID **pPriv_data)
{
    *pAddress = (void*) 0;
    return RM_ERROR;
};

RM_STATUS KernMapAGPPages(VOID *pvma, VOID *priv_data)
{
    return RM_ERROR;
}

RM_STATUS KernFreeAGPPages(VOID **pAddress, VOID  *priv_data)
{
    return RM_OK;
};

#endif

// SGI SwapGroup/SwapBarrier extension
// This requires special hardware plus an external driver
// XXX - add code to allocate struct as needed, for each barrier id

// #define TESTING_SWAP 1

#if TESTING_SWAP
// testing code, showing how I tested things

#define ADD_BARRIER_FUNC    "sgitest_add_barrier"
#define REMOVE_BARRIER_FUNC "sgitest_remove_barrier"
#define SWAP_READY_FUNC     "sgitest_swap_ready"

// functions added to serial driver in kernel, from shrijeet
extern int sgi_add_barrier(int id);
extern int sgi_remove_barrier(int id);
extern int sgi_swap_ready(int id, void (*callback)(void *), void *data);

int
sgitest_add_barrier(int id)
{
    return sgi_add_barrier(id);
}

int 
sgitest_remove_barrier(int id)
{
    return sgi_remove_barrier(id);
}

int 
sgitest_swap_ready(int id, void (*callback)(void *), void *data)
{
    return sgi_swap_ready(id, callback, data);
}
#else

#define ADD_BARRIER_FUNC    "sgi_add_barrier"
#define REMOVE_BARRIER_FUNC "sgi_remove_barrier"
#define SWAP_READY_FUNC     "sgi_swap_ready"

#endif


struct barrier_funcs {
    int (*add_barrier)(int id);
    int (*remove_barrier)(int id);
    int (*swap_ready)(int id, void (*callback)(void *), void *data);
};

static struct barrier_funcs sgi_funcs = { NULL, NULL, NULL };

RM_STATUS
osInitSwapBarrier(int id)
{
    // if we haven't been called before, initialize
    if (sgi_funcs.add_barrier == NULL)
    {
#if defined(TESTING_SWAP)
#if !defined (KERNEL_2_2)
        inter_module_register(ADD_BARRIER_FUNC,    THIS_MODULE, sgitest_add_barrier);
        inter_module_register(REMOVE_BARRIER_FUNC, THIS_MODULE, sgitest_remove_barrier);
        inter_module_register(SWAP_READY_FUNC,     THIS_MODULE, sgitest_swap_ready);
#endif
#endif
        sgi_funcs.add_barrier = GET_MODULE_SYMBOL(0, ADD_BARRIER_FUNC);
        sgi_funcs.remove_barrier = GET_MODULE_SYMBOL(0, REMOVE_BARRIER_FUNC);
        sgi_funcs.swap_ready = GET_MODULE_SYMBOL(0, SWAP_READY_FUNC);
    }

    // failed to initialize
    if (sgi_funcs.add_barrier == NULL)
    {
#if defined(TESTING_SWAP)
        printk("FAILED TO INITIALIZE SWAP!!\n");
#endif
        return RM_ERROR;
    }

    return sgi_funcs.add_barrier(id);
}

RM_STATUS
osRemoveSwapBarrier(int id)
{
    RM_STATUS rmStatus;

    if (sgi_funcs.remove_barrier == NULL)
        return RM_ERROR;

    rmStatus = sgi_funcs.remove_barrier(id);

    if (sgi_funcs.add_barrier)
    {
        PUT_MODULE_SYMBOL(ADD_BARRIER_FUNC);
        sgi_funcs.add_barrier = NULL;
    }

    if (sgi_funcs.remove_barrier)
    {
        PUT_MODULE_SYMBOL(REMOVE_BARRIER_FUNC);
        sgi_funcs.remove_barrier = NULL;
    }

    if (sgi_funcs.swap_ready)
    {
        PUT_MODULE_SYMBOL(SWAP_READY_FUNC);
        sgi_funcs.swap_ready = NULL;
    }

    return rmStatus;
}

RM_STATUS
osSwapBarrier(int id, VOID (*callback)(VOID *), VOID *data)
{
    (*callback)((VOID *)data);
    return RM_OK;
}



