/* _NVRM_COPYRIGHT_BEGIN_
 *
 * Copyright 1993-2001 by NVIDIA Corporation.  All rights reserved.  All
 * information contained herein is proprietary and confidential to NVIDIA
 * Corporation.  Any use, reproduction, or disclosure without the written
 * permission of NVIDIA Corporation is prohibited.
 *
 * _NVRM_COPYRIGHT_END_
 */

#ifndef _NVRM_H_
#define _NVRM_H_

/**************** Resource Manager Defines and Structures ******************\
*                                                                           *
* Module: NVRM.H                                                            *
*       Resource Manager defines and structures used throughout the code.   *
*                                                                           *
*****************************************************************************
*                                                                           *
* History:                                                                  *
*                                                                           *
\***************************************************************************/

//-------------------------------------------------------------------------//
//                       Code Path Defines                                 //
//-------------------------------------------------------------------------//
//#define USE_NVRM_TRACE 1

//-------------------------------------------------------------------------//
//                                                                         //
//                Version Number of the Resource Manager                   //
//                                                                         //
//  This number is designed to track the _build_ version number.  This is  //
//  important because it allows us to know which changelist the            //
//  NVRM_VERSION corresponds to.  At the moment the only usage of this     //
//  number is in stamping the "driver" (Resman and its HAL) so that        //
//  writers of dynamically loadable HAL modules have the abilty to track   //
//  down version differences via changelists.                              //
//                                                                         //
//  The format of this 32 bit number is 0xcoMMMmmm, where                  //
//      - c is the cpu # as defined below                                  //
//      - o is the os # as defined below                                   //
//      - MMM is the stream major #                                        //
//      - mmm is the stream minor #                                        //
//                                                                         //
//   NOTE: prior to this scheme:                                           //
//   <= rel10 == 0x00030200                                                //
//   which maps to x86 on an unknown CPU,OS.                               //
//   a few windows 20.x's (and probably mac r2.2's) went out with:         //
//   0x0001000{1,2} which again                                            //
//   maps to unknown CPU,OS.                                               //
//-------------------------------------------------------------------------//

#define NVRM_VERSION_CPU_UNK   ((NvU32)0x0)
#define NVRM_VERSION_CPU_X86   ((NvU32)0x1)
#define NVRM_VERSION_CPU_PPC   ((NvU32)0x2)
#define NVRM_VERSION_CPU_IA64  ((NvU32)0x3)
#define NVRM_VERSION_CPU_MIPS  ((NvU32)0x4)
#define NVRM_VERSION_CPU_ALL   ((NvU32)0xf)

#define NVRM_VERSION_OS_UNK    ((NvU32)0x0)
#define NVRM_VERSION_OS_WIN9X  ((NvU32)0x1)
#define NVRM_VERSION_OS_WIN2K  ((NvU32)0x2)
#define NVRM_VERSION_OS_MACOS9 ((NvU32)0x3)
#define NVRM_VERSION_OS_MACOSX ((NvU32)0x4)
#define NVRM_VERSION_OS_WINXP  ((NvU32)0x5)
#define NVRM_VERSION_OS_LINUX  ((NvU32)0x6)
#define NVRM_VERSION_OS_ALL    ((NvU32)0xf)

#define NVRM_VERSION_MAJ_ALL   ((NvU32)0xfff)
#define NVRM_VERSION_MIN_ALL   ((NvU32)0xfff)

#define NVRM_VERSION    (( ((NvU32)NVRM_VERSION_CPU) << 28) | ( ((NvU32)NVRM_VERSION_OS) << 24) | (((NvU32)NVRM_VERSION_MAJ) << 12) | ((NvU32)NVRM_VERSION_MIN))

#define NVRM_VERSION_CPU_MASK ((NvU32)0xf0000000)
#define NVRM_VERSION_OS_MASK  ((NvU32)0x0f000000)
#define NVRM_VERSION_MAJ_MASK ((NvU32)0x00fff000)
#define NVRM_VERSION_MIN_MASK ((NvU32)0x00000fff)

#define NVRM_VERSION_GET_CPU(Z) (((Z) & NVRM_VERSION_CPU_MASK)>>28)
#define NVRM_VERSION_GET_OS(Z)  (((Z) & NVRM_VERSION_OS_MASK)>>24)
#define NVRM_VERSION_GET_MAJ(Z) (((Z) & NVRM_VERSION_MAJ_MASK)>>12)
#define NVRM_VERSION_GET_MIN(Z)  ((Z) & NVRM_VERSION_MIN_MASK)

#define NVRM_VERSION_SET_CPU(Z) ( NVRM_VERSION_CPU_MASK &  ((Z)<<28) )
#define NVRM_VERSION_SET_OS(Z)  ( NVRM_VERSION_OS_MASK  &  ((Z)<<24) )
#define NVRM_VERSION_SET_MAJ(Z) ( NVRM_VERSION_MAJ_MASK &  ((Z)<<12) )
#define NVRM_VERSION_SET_MIN(Z) ( NVRM_VERSION_MIN_MASK &  (Z))


//
// Contort the #defines of the world to glean NVRM_VERSION_{CPU,OS,MAJOR,MINOR}
//

#ifdef IA64
#   define NVRM_VERSION_CPU NVRM_VERSION_CPU_IA64
#endif

#ifdef _WIN32
#   ifdef _WIN32_WINNT
#      if (_WIN32_WINNT >= 0x0500) && (_WIN32_WINNT < 0x0600)
#         define NVRM_VERSION_OS NVRM_VERSION_OS_WIN2K
#      else
#         define NVRM_VERSION_OS NVRM_VERSION_OS_WINXP
#      endif
#   else
#      define NVRM_VERSION_OS NVRM_VERSION_OS_WIN9X
#   endif
#   ifndef NVRM_VERSION_CPU
#      define NVRM_VERSION_CPU NVRM_VERSION_CPU_X86
#   endif
#   ifndef NV_DRIVER_VERSION
#      include "nvVer.h"
#   endif
#   define NVRM_VERSION_MAJ ((NV_DRIVER_VERSION)/100)
#   define NVRM_VERSION_MIN ((NV_DRIVER_VERSION)%100)
#endif


#ifdef WIN32
#   ifndef NVRM_VERSION_OS
#      define NVRM_VERSION_CPU NVRM_VERSION_CPU_X86
#      define NVRM_VERSION_OS  NVRM_VERSION_OS_WIN9X
#      ifndef NV_DRIVER_VERSION
#         include "nvVer.h"
#      endif
#      define NVRM_VERSION_MAJ ((NV_DRIVER_VERSION)/100)
#      define NVRM_VERSION_MIN ((NV_DRIVER_VERSION)%100)
#   endif
#endif


#ifndef NVRM_VERSION_CPU 
#   define NVRM_VERSION_CPU NVRM_VERSION_CPU_UNK
#endif

#ifndef NVRM_VERSION_OS 
#   define NVRM_VERSION_OS NVRM_VERSION_OS_UNK
#endif

#ifndef NVRM_VERSION_MAJ
#   define NVRM_VERSION_MAJ NVRM_VERSION_MAJ_ALL
#endif

#ifndef NVRM_VERSION_MIN
#   define NVRM_VERSION_MIN NVRM_VERSION_MIN_ALL
#endif

//---------------------------------------------------------------------------
//  define some inline assembly instructions

#ifdef __WATCOMC__

void my_sti();
#pragma aux my_sti = \
    "pushf" \
    "sti";

void my_cli();
#pragma aux my_cli =  \
    "popf";

#else   // if not Watcom, its the Microsoft compiler

#define my_sti \
    _asm    pushf   \
    _asm    sti
#define my_cli \
    __asm   popf

#endif  // _WATCOMC_

//---------------------------------------------------------------------------
//
//  NVidia data types.
//
//---------------------------------------------------------------------------

#include <nvtypes.h>

//
// Fixed point math types and macros.
//
typedef signed long     S016d016;
typedef unsigned long   U016d016;
typedef signed long     S008d024;
typedef unsigned long   U008d024;
typedef signed long     S012d020;
typedef unsigned long   U012d020;

#define FIX16_POINT             16
#define FIX16_SCALE             ((U032)1<<FIX16_POINT)
#define FIX16_INT_MASK          ((U032)~0<<FIX16_POINT)
#define FIX16_FRAC_MASK         (~FIX16_INT_MASK)
#define FIX16_INT(f)            ((f)>>FIX16_POINT)
#define FIX16_FRAC(f)           ((f)&FIX16_FRAC_MASK)
#define INT_FIX16(f)            ((f)<<FIX16_POINT)
#define FIX16(i)                (S016d016)((i)*FIX16_SCALE))
#define FIX16_MUL(f1,f2)        ((FIX16_INT(f1)*FIX16_INT(f2)<<FIX16_POINT)     \
                                +(FIX16_INT(f1)*FIX16_FRAC(f2))                 \
                                +(FIX16_FRAC(f1)*FIX16_INT(f2))                 \
                                +(FIX16_FRAC(f1)*FIX16_FRAC(f2)>>FIX16_POINT))
#define FIX16_DIV(f1,f2)        (((f1)<<(FIX16_POINT/2))/((f2)>>(FIX16_POINT/2)))
#define FIX16_DIVI(f1,f2)       ((f1)/((f2)>>(FIX16_POINT)))
#define FIX16_IDIV(f1,f2)       (((f1)<<(FIX16_POINT))/(f2))
#define FIX24_POINT             24
#define FIX24_SCALE             ((U032)1<<FIX24_POINT)
#define FIX24_INT_MASK          ((U032)~0<<FIX24_POINT)
#define FIX24_FRAC_MASK         (~FIX24_INT_MASK)
#define FIX24_TO_INT(f)         ((f)>>FIX24_POINT)
#define FIX24(i)                ((S008d024)((i)*FIX24_SCALE))
#define FIX24_MUL(f1,f2)        (((f1)>>(FIX24_POINT/2))*((f2)>>(FIX24_POINT/2)))
#define FIX24_DIV(f1,f2)        (((f1)<<(FIX24_POINT/2))/((f2)>>(FIX24_POINT/2)))
#define FIX24_DIVI(f1,f2)       ((f1)/((f2)>>(FIX24_POINT)))
#define FIX24_IDIV(f1,f2)       (((f1)<<(FIX24_POINT))/(f2))

//---------------------------------------------------------------------------
//
//  Common types.
//
//---------------------------------------------------------------------------
#ifndef VOID
#define VOID    void
#endif
#ifndef BOOL
#define BOOL    S032
#endif
#ifndef TRUE
#define TRUE    1L
#endif
#ifndef FALSE
#define FALSE   0L
#endif
#ifndef NULL
#define NULL    0L
#endif
    
//---------------------------------------------------------------------------
//
//  Error codes.
//
//---------------------------------------------------------------------------

//
// External codes.
//
#define NV_ERROR_1                      0x0100
#define NV_TYPE_CONFLICT                (0x01|NV_ERROR_1)
#define NV_OUT_OF_RANGE                 (0x02|NV_ERROR_1)
#define NV_NO_CONNECTION                (0x03|NV_ERROR_1)
#define NV_NO_SUCH_OBJECT               (0x04|NV_ERROR_1)
#define NV_NAME_IN_USE                  (0x05|NV_ERROR_1)
#define NV_OUT_OF_RESOURCES             (0x06|NV_ERROR_1)
#define NV_TRANSLATION_VIOLATION        (0x07|NV_ERROR_1)
#define NV_PROTECTION_VIOLATION         (0x08|NV_ERROR_1)
#define NV_BUFFER_BUSY                  (0x09|NV_ERROR_1)
#define NV_ILLEGAL_ACCESS               (0x0A|NV_ERROR_1)
#define NV_BAD_COLORMAP_FORMAT          (0x0B|NV_ERROR_1)
#define NV_BAD_COLOR_FORMAT             (0x0C|NV_ERROR_1)
#define NV_BAD_MONOCHROME_FORMAT        (0x0D|NV_ERROR_1)
#define NV_BAD_PATTERN_SHAPE            (0x0E|NV_ERROR_1)
#define NV_BAD_SUBDIVIDE                (0x0F|NV_ERROR_1)
#define NV_NO_CURRENT_POINT             (0x10|NV_ERROR_1)
#define NV_NO_DMA_TRANSLATION           (0x12|NV_ERROR_1)
#define NV_INCOMPLETE_METHOD            (0x13|NV_ERROR_1)
#define NV_RESERVED_ADDRESS             (0x14|NV_ERROR_1)
#define NV_UNIMPLEMENTED_PATCH          (0x15|NV_ERROR_1)
#define NV_OS_NAME_ERROR                (0x16|NV_ERROR_1)
#define NV_INCOMPLETE_PATCH             (0x17|NV_ERROR_1)
#define NV_BUFFERGAP_ERROR              (0x18|NV_ERROR_1)
#define NV_FIFO_OVERFLOW                (0x19|NV_ERROR_1)

//
// Internal codes.
//
typedef U032 RM_STATUS;
#define NUM_RM_ERRORS                   22
#define RM_OK                           0x00000000
#define RM_ERROR                        0xFFFFFFFF
#define RM_WARN_NULL_OBJECT             (0x10000000|RM_OK)
#define RM_WARN_GRAPHICS_DISABLED       (0x30000000|RM_OK)
#define RM_ERR_NO_FREE_MEM              (0x01000000|NV_OUT_OF_RESOURCES)
#define RM_ERR_MEM_NOT_FREED            (0x01010000|NV_OUT_OF_RESOURCES)
#define RM_ERR_PAGE_TABLE_NOT_AVAIL     (0x01020000|NV_OUT_OF_RESOURCES)
#define RM_ERR_NO_FREE_FIFOS            (0x02000000|NV_OUT_OF_RESOURCES)
#define RM_ERR_CANT_CREATE_CLASS_OBJS   (0x02010000|NV_OUT_OF_RESOURCES)
#define RM_ERR_BAD_OBJECT               (0x03000000|NV_NO_SUCH_OBJECT)
#define RM_ERR_INSERT_DUPLICATE_NAME    (0x03010000|NV_NAME_IN_USE)
#define RM_ERR_OBJECT_NOT_FOUND         (0x03020000|NV_NO_SUCH_OBJECT)
#define RM_ERR_OTHER_DEVICE_FOUND       (0x03040000|NV_NO_SUCH_OBJECT)
#define RM_ERR_CREATE_BAD_CLASS         (0x04000000|NV_NO_SUCH_OBJECT)
#define RM_ERR_DELETE_BAD_CLASS         (0x04010000|NV_NO_SUCH_OBJECT)
#define RM_ERR_FIFO_RUNOUT_OVERFLOW     (0x05000000|NV_FIFO_OVERFLOW)
#define RM_ERR_FIFO_BAD_ACCESS          (0x05010000|NV_RESERVED_ADDRESS)
#define RM_ERR_FIFO_OVERFLOW            (0x05020000|NV_FIFO_OVERFLOW)
#define RM_ERR_METHOD_ORDER             (0x06000000|NV_INCOMPLETE_METHOD)
#define RM_ERR_METHOD_COUNT             (0x06010000|NV_INCOMPLETE_METHOD)
#define RM_ERR_ILLEGAL_OBJECT           (0x06020000|NV_TYPE_CONFLICT)
#define RM_ERR_DMA_IN_USE               (0x07000000|NV_BUFFER_BUSY)
#define RM_ERR_BAD_DMA_SPECIFIER        (0x07010000|NV_TRANSLATION_VIOLATION)
#define RM_ERR_INVALID_XLATE            (0x07020000|NV_TRANSLATION_VIOLATION)
#define RM_ERR_INVALID_START_LENGTH     (0x07030000|NV_PROTECTION_VIOLATION)
#define RM_ERR_DMA_MEM_NOT_LOCKED       (0x07040000|NV_OUT_OF_RESOURCES)
#define RM_ERR_DMA_MEM_NOT_UNLOCKED     (0x07050000|NV_OUT_OF_RESOURCES)
#define RM_ERR_NOTIFY_IN_USE            (0x07060000|NV_BUFFER_BUSY)
#define RM_ERR_ILLEGAL_ADDRESS          (0x08000000|NV_ILLEGAL_ACCESS)
#define RM_ERR_BAD_ADDRESS              (0x08010000|NV_PROTECTION_VIOLATION)
#define RM_ERR_INVALID_COLOR_FORMAT     (0x09000000|NV_BAD_COLOR_FORMAT)
#define RM_ERR_INVALID_MONO_FORMAT      (0x09010000|NV_BAD_MONOCHROME_FORMAT)
#define RM_ERR_OBJECT_TYPE_MISMATCH     (0x0A000000|NV_TYPE_CONFLICT)
#define RM_ERR_INCOMPLETE_PATCH         (0x0C000000|NV_INCOMPLETE_PATCH)
#define RM_ERR_INVALID_PATCH            (0x0C010000|NV_UNIMPLEMENTED_PATCH)
#define RM_ERR_PATCH_TOO_COMPLEX        (0x0C020000|NV_UNIMPLEMENTED_PATCH)
#define RM_ERR_MAX_PATCH_FANOUT         (0x0C030000|NV_UNIMPLEMENTED_PATCH)
#define RM_ERR_DEVICE_DISCONNECTED      (0x0C040000|NV_NO_CONNECTION)
#define RM_ERR_BUFFER_GAP               (0x0D000000|NV_BUFFER_GAP)
#define RM_ERR_INVALID_OS_NAME          (0x0E000000|NV_OS_NAME_ERROR)
#define RM_ERR_ILLEGAL_ACTION           (0x0E010000|NV_OS_NAME_ERROR)

//
// The DMA direction specified for the DMA object is not allowed in the
// specified address space.
//
#define RM_ERR_ILLEGAL_DIRECTION        (0x0E020000|NV_OS_NAME_ERROR)
#define RM_ERR_OUT_OF_TIMER_CALLBACKS   (0x0F000000|NV_OUT_OF_RESOURCES)

//
// Errors returned from the arch calls.
//
#define NV_ERROR_2                      0x0200
#define RM_ERR_OPERATING_SYSTEM         (0x00000001|NV_ERROR_2)
#define RM_ERR_BAD_OBJECT_PARENT        (0x00000002|NV_ERROR_2)
#define RM_ERR_BAD_OBJECT_HANDLE        (0x00000003|NV_ERROR_2)
#define RM_ERR_OBJECT_IN_USE            (0x00000004|NV_ERROR_2)
#define RM_ERR_OBJECT_HAS_CHILDERN      (0x00000005|NV_ERROR_2)
#define RM_ERR_BAD_CLASS                (0x00000006|NV_ERROR_2)
#define RM_ERR_INSUFFICIENT_RESOURCES   (0x00000007|NV_ERROR_2)
#define RM_ERR_BAD_FLAGS                (0x00000008|NV_ERROR_2)
#define RM_ERR_BAD_BASE                 (0x00000009|NV_ERROR_2)
#define RM_ERR_BAD_LIMIT                (0x0000000A|NV_ERROR_2)
#define RM_ERR_PROTECTION_FAULT         (0x0000000B|NV_ERROR_2)
#define RM_ERR_MULTIPLE_MEMORY_TYPES    (0x0000000C|NV_ERROR_2)
#define RM_ERR_BAD_OBJECT_ERROR         (0x0000000D|NV_ERROR_2)
#define RM_ERR_BAD_OBJECT_BUFFER        (0x0000000E|NV_ERROR_2)
#define RM_ERR_BAD_OFFSET               (0x0000000F|NV_ERROR_2)
#define RM_ERR_BAD_CLIENT               (0x00000010|NV_ERROR_2)

//
// Errors returned from HAL calls.
//
#define NV_ERROR_3                      (0x0400)
#define RM_ERR_VERSION_MISMATCH         (0x00000001|NV_ERROR_3)
#define RM_ERR_BAD_ARGUMENT             (0x00000002|NV_ERROR_3)
#define RM_ERR_INVALID_STATE            (0x00000003|NV_ERROR_3)
#define RM_ERR_STATE_IN_USE             (0x00000004|NV_ERROR_3)

//
// RM_ASSERT macro
//
#ifdef DEBUG
#define RM_ASSERT(cond)                                                     \
    if ((cond) == 0x0)                                                      \
    {                                                                       \
        DBG_PRINT_STRING(DEBUGLEVEL_ERRORS, "NVRM: assertion FAILED!\n");   \
        DBG_PRINT_STRING(DEBUGLEVEL_ERRORS, __FILE__);                      \
        DBG_PRINT_STRING(DEBUGLEVEL_ERRORS, ": line ");                     \
        DBG_PRINT_VALUE(DEBUGLEVEL_ERRORS, (int)__LINE__);                  \
        DBG_PRINT_STRING(DEBUGLEVEL_ERRORS, "\n");                          \
        DBG_BREAKPOINT();                                                   \
    }
#else
#define RM_ASSERT(cond)
#endif

//
// Power of 2 alignment.
//    (Will give unexpected results if 'a' is not a power of 2.)
//
#define NV_ALIGN_DOWN(v, gran)      ((v) & ~((gran) - 1))
#define NV_ALIGN_UP(v, gran)        (((v) + ((gran) - 1)) & ~((gran)-1))

#define NV_ALIGN_PTR_DOWN(p, gran)  ((void *) NV_ALIGN_DOWN(((NV_UINTPTR_T)p), (gran)))
#define NV_ALIGN_PTR_UP(p, gran)    ((void *) NV_ALIGN_UP(((NV_UINTPTR_T)p), (gran)))

#define NV_PAGE_ALIGN_DOWN(value)   NV_ALIGN_DOWN((value), RM_PAGE_SIZE)
#define NV_PAGE_ALIGN_UP(value)     NV_ALIGN_UP((value), RM_PAGE_SIZE)

//
// Useful macros
//
#define NV_MIN(a, b) (((a) < (b)) ? (a) : (b))
#define NV_MAX(a, b) (((a) > (b)) ? (a) : (b))
#define NV_CEIL(a,b) (((a)+(b)-1)/(b))

//
// NV Reference Manual register access definitions.
//
#define BIT(b)                  (1<<(b))
#define DEVICE_BASE(d)          (0?d)
#define DEVICE_EXTENT(d)        (1?d)
#define DRF_SHIFT(drf)          ((0?drf) % 32)
#define DRF_SIZE(drf)           (DEVICE_EXTENT(drf)-DEVICE_BASE(drf)+1)
#define DRF_MASK(drf)           (0xFFFFFFFF>>(31-((1?drf) % 32)+((0?drf) % 32)))
#define DRF_DEF(d,r,f,c)        ((NV ## d ## r ## f ## c)<<DRF_SHIFT(NV ## d ## r ## f))
#define DRF_NUM(d,r,f,n)        (((n)&DRF_MASK(NV ## d ## r ## f))<<DRF_SHIFT(NV ## d ## r ## f))
#define DRF_VAL(d,r,f,v)        (((v)>>DRF_SHIFT(NV ## d ## r ## f))&DRF_MASK(NV ## d ## r ## f))
#define REG_WR_DRF_NUM(d,r,f,n) REG_WR32(NV ## d ## r, DRF_NUM(d,r,f,n))
#define REG_WR_DRF_DEF(d,r,f,c) REG_WR32(NV ## d ## r, DRF_DEF(d,r,f,c))
#define FLD_WR_DRF_NUM(d,r,f,n) REG_WR32(NV##d##r,(REG_RD32(NV##d##r)&~(DRF_MASK(NV##d##r##f)<<DRF_SHIFT(NV##d##r##f)))|DRF_NUM(d,r,f,n))
#define FLD_WR_DRF_DEF(d,r,f,c) REG_WR32(NV##d##r,(REG_RD32(NV##d##r)&~(DRF_MASK(NV##d##r##f)<<DRF_SHIFT(NV##d##r##f)))|DRF_DEF(d,r,f,c))
#define REG_RD_DRF(d,r,f)       (((REG_RD32(NV ## d ## r))>>DRF_SHIFT(NV ## d ## r ## f))&DRF_MASK(NV ## d ## r ## f))

//
// NV Reference Manual instance memory structure access definitions.
//
#define INST_WR32(i,o,d)        REG_WR32((pDev->Pram.HalInfo.PraminOffset+((i)<<4)+(o)),(d))
#define INST_RD32(i,o)          REG_RD32((pDev->Pram.HalInfo.PraminOffset+((i)<<4)+(o)))
#define SF_OFFSET(sf)           (((0?sf)/32)<<2)
#define SF_SHIFT(sf)            ((0?sf)&31)
#undef  SF_MASK
#define SF_MASK(sf)             (0xFFFFFFFF>>(31-(1?sf)+(0?sf)))
#define SF_DEF(s,f,c)           ((NV ## s ## f ## c)<<SF_SHIFT(NV ## s ## f))
#define SF_NUM(s,f,n)           (((n)&SF_MASK(NV ## s ## f))<<SF_SHIFT(NV ## s ## f))
#define SF_VAL(s,f,v)           (((v)>>SF_SHIFT(NV ## s ## f))&SF_MASK(NV ## s ## f))
#define RAM_WR_ISF_NUM(i,s,f,n) INST_WR32(i,SF_OFFSET(NV ## s ## f),SF_NUM(s,f,n))
#define RAM_WR_ISF_DEF(i,s,f,c) INST_WR32(i,SF_OFFSET(NV ## s ## f),SF_DEF(s,f,c))
#define FLD_WR_ISF_NUM(i,s,f,n) INST_WR32(i,SF_OFFSET(NV##s##f),(INST_RD32(i,SF_OFFSET(NV##s##f))&~(SF_MASK(NV##s##f)<<SF_SHIFT(NV##s##f)))|SF_NUM(s,f,n))
#define FLD_WR_ISF_DEF(i,s,f,c) INST_WR32(i,SF_OFFSET(NV##s##f),(INST_RD32(i,SF_OFFSET(NV##s##f))&~(SF_MASK(NV##s##f)<<SF_SHIFT(NV##s##f)))|SF_DEF(s,f,c))
#define RAM_RD_ISF(i,s,f)       (((INST_RD32(i,SF_OFFSET(NV ## s ## f)))>>SF_SHIFT(NV ## s ## f))&SF_MASK(NV ## s ## f))

//
// EEPROM access macros.
//
#define EEPROM_INDEX_RD08(ii, dd)   \
    {REG_WR32(NV_PEEPROM_CONTROL,DRF_NUM(_PEEPROM,_CONTROL,_ADDRESS,(ii))|DRF_DEF(_PEEPROM,_CONTROL,_COMMAND,_READ));\
    osDelay(1);\
    (dd)=REG_RD32(NV_PEEPROM_CONTROL)&0xFF;}

//
// DevInfo update/finish flags.
//
#define UPDATE_HWINFO_DAC_CURSOR_POS           BIT(0)
#define FINISH_HWINFO_DAC_CURSOR_POS           BIT(0)
#define UPDATE_HWINFO_DAC_CURSOR_IMAGE         BIT(1)
#define FINISH_HWINFO_DAC_CURSOR_IMAGE         BIT(1)
#define UPDATE_HWINFO_DAC_CURSOR_COLOR         BIT(2)
#define FINISH_HWINFO_DAC_CURSOR_COLOR         BIT(2)
#define UPDATE_HWINFO_DAC_COLORMAP             BIT(3)
#define FINISH_HWINFO_DAC_COLORMAP             BIT(3)
#define UPDATE_HWINFO_DAC_CURSOR_ENABLE        BIT(4)
#define FINISH_HWINFO_DAC_CURSOR_ENABLE        BIT(4)
#define UPDATE_HWINFO_DAC_CURSOR_DISABLE       BIT(5)
#define FINISH_HWINFO_DAC_CURSOR_DISABLE       BIT(5)
#define UPDATE_HWINFO_DAC_CURSOR_TYPE          BIT(6)
#define FINISH_HWINFO_DAC_CURSOR_TYPE          BIT(6)
#define UPDATE_HWINFO_DAC_CURSOR_IMAGE_DELAYED BIT(7)
#define FINISH_HWINFO_DAC_CURSOR_IMAGE_DELAYED BIT(7)
#define UPDATE_HWINFO_DAC_DPML                 BIT(8)
#define FINISH_HWINFO_DAC_DPML                 BIT(8)
#define UPDATE_HWINFO_DAC_APLL                 BIT(9)
#define FINISH_HWINFO_DAC_APLL                 BIT(9)
#define UPDATE_HWINFO_DAC_VPLL                 BIT(10)
#define FINISH_HWINFO_DAC_VPLL                 BIT(10)
#define UPDATE_HWINFO_DAC_MPLL                 BIT(11)
#define FINISH_HWINFO_DAC_MPLL                 BIT(11)
#define UPDATE_HWINFO_DAC_CURSOR_IMAGE_NEW     BIT(12)
#define FINISH_HWINFO_DAC_CURSOR_IMAGE_NEW     BIT(12)
#define UPDATE_HWINFO_DAC_NVPLL                BIT(13)
#define FINISH_HWINFO_DAC_NVPLL                BIT(13)
#define UPDATE_HWINFO_DAC_IMAGE_OFFSET_FORMAT  BIT(14)
#define FINISH_HWINFO_DAC_IMAGE_OFFSET_FORMAT  BIT(14)
#define UPDATE_HWINFO_BUFFER_FLIP              BIT(0)
#define FINISH_HWINFO_BUFFER_FLIP              BIT(0)
#define UPDATE_HWINFO_BUFFER_PARAMS            BIT(8)
#define FINISH_HWINFO_BUFFER_PARAMS            BIT(8)
#define UPDATE_HWINFO_DISPLAY_PARAMS           BIT(9)
#define FINISH_HWINFO_DISPLAY_PARAMS           BIT(9)
#define UPDATE_HWINFO_REFRESH_PARAMS           BIT(10)
#define FINISH_HWINFO_REFRESH_PARAMS           BIT(10)
#define UPDATE_HWINFO_VIDEO_ENABLE             BIT(0)
#define FINISH_HWINFO_VIDEO_ENABLE             BIT(0)
#define UPDATE_HWINFO_VIDEO_START              BIT(1)
#define FINISH_HWINFO_VIDEO_START              BIT(1)
#define UPDATE_HWINFO_VIDEO_SIZE               BIT(2)
#define FINISH_HWINFO_VIDEO_SIZE               BIT(2)
#define UPDATE_HWINFO_VIDEO_SCALE              BIT(3)
#define FINISH_HWINFO_VIDEO_SCALE              BIT(3)
#define UPDATE_HWINFO_VIDEO_COLORKEY           BIT(4)
#define FINISH_HWINFO_VIDEO_COLORKEY           BIT(4)

//
// Notification otherInfo defines.
//
#define NOTIFY_INFO16_BUFFER_CHAIN_GAP  1
#define NOTIFY_INFO16_BUFFER_YANK       2
#define NOTIFY_INFO16_BUFFER_LOOP       3

//---------------------------------------------------------------------------
//
// 32 bit debug marker values.
//
//---------------------------------------------------------------------------

// Solaris machines are byte reversed from Intel machines
#ifdef SOLARIS
#define NV_MARKER1 (U032)(('N' << 24) | ('V' << 16) | ('R' << 8) | 'M')
#define NV_MARKER2 (U032)(('M' << 24) | ('R' << 16) | ('V' << 8) | 'N')
#else
#define NV_MARKER1 (U032)(('M' << 24) | ('R' << 16) | ('V' << 8) | 'N')
#define NV_MARKER2 (U032)(('N' << 24) | ('V' << 16) | ('R' << 8) | 'M')
#endif

//---------------------------------------------------------------------------
//
// DevInfo state values.
//
//---------------------------------------------------------------------------

//
// BUS type.
//
#define BUS_PCI                         1
#define BUS_VL                          2
#define BUS_NIRV                        3
#define BUS_AGP                         4

//
// AGP defines
//
#define AGP_APERTURE_SIZE               0x2000000       // 32 MB

//
// Define the various NV chip revision ID's
//
#define NV4_REV_A_00                    0x00
#define NV4_REV_B_00                    0x10

//
// Framebuffer values.
//
#define BUFFER_DRAM                     0
#define BUFFER_SGRAM                    1
#define BUFFER_SDRAM                    2
#define BUFFER_DDRAM                    3
#define BUFFER_CRUSH_UMA                4

#define BUFFER_DRAM_EDO                 0
#define BUFFER_DRAM_FPM                 1

#define BUFFER_HSYNC_NEGATIVE           1
#define BUFFER_HSYNC_POSITIVE           0
#define BUFFER_VSYNC_NEGATIVE           1
#define BUFFER_VSYNC_POSITIVE           0
#define BUFFER_CSYNC_DISABLED           0
#define BUFFER_CSYNC_ENABLED            1

//
// Offscreen Instance Memory size.
//
#define NV_PRAM_MIN_SIZE_INSTANCE_MEM  0x1000   // 4KB

//
// DAC values.
//
#define DAC_CURSOR_DISABLED             0
#define DAC_CURSOR_THREE_COLOR          1
#define DAC_CURSOR_TWO_COLOR_XOR        2
#define DAC_CURSOR_TWO_COLOR            3
#define DAC_CURSOR_FULL_COLOR_XOR       4
#define DAC_CURSOR_FULL_COLOR           5
#define DAC_CURSOR_TYPE_NEW             6

#define DAC_CURSOR_CACHE_ENTRY          0x80000000
#define DAC_CURSOR_CACHE_DISPLAY        0x40000000

#define MAX_CACHED_CURSORS              5

//
// Monitor types
//
#define NV_MONITOR_VGA                  0

//
// Display Types
//
#define MONITOR                         0
#define TV                              1
#define DUALSURFACE                     2
#define MONITOR_FLAT_PANEL              3

//
// Queue depth for mobile hotkey events. Should be comfortably more than needed.
//
#define NV_HOTKEY_EVENT_QUEUE_SIZE      16

// 
// Customer coding (bitmasks used to enable/disable functionality)
//
#define CUSTOMER_GENERIC                0x00000000
#define CUSTOMER_CANOPUS                0x00000001
#define CUSTOMER_CANOPUS_TOSHIBA        0x00000002
#define CUSTOMER_CREATIVE               0x00000004
#define CUSTOMER_ASUS                   0x00000008       

//
// Mobile operations enumerants for HWINFO.Power.MobileOperation.
//
#define MOBILE_OPERATION_NONE       0
#define MOBILE_OPERATION_DELL       1
#define MOBILE_OPERATION_TOSHIBA    2
#define MOBILE_OPERATION_QUANTA     3
#define MOBILE_OPERATION_COMPAL     4
#define MOBILE_OPERATION_GATEWAY    5

//
// Desktop panel policy for how to handle desktop panels during enable/disable dac.
//
#define PANEL_POWER_POLICY_DEFAULT      0x00000000  // Default (original) operation (don't run bios panel scripts)
#define PANEL_POWER_POLICY_SCRIPTS      0x00000001  // Run bios panel scripts.


//---------------------------------------------------------------------------
//
//  Max number of FIFO channels used.
//
//---------------------------------------------------------------------------

//
// currently only initClientInfo() uses MAX_FIFOS instead of NUM_FIFOS, since
// NUM_FIFOS may be different per device (e.g. nv4 vs. nv10). Also, the client
// database alloc occurs before an nvInfo has been allocated.
//
#define MAX_FIFOS                   32

//---------------------------------------------------------------------------
//
//  Data structures.
//
//---------------------------------------------------------------------------

//
// Device register access array.
//
typedef union _def_HwReg
{
    volatile V008 Reg008[1];
    volatile V016 Reg016[1];
    volatile V032 Reg032[1];
} HWREG, * PHWREG;

//
// A hardware fifo hash table entry.
//
typedef struct _def_HashTableEntry
{
    V032 ht_ObjectHandle;
    V032 ht_Context;
} HASH_TABLE_ENTRY;
   
//
// Device state and configuration information.
//
typedef struct _def_hw_info *PHWINFO;

#ifndef RM_HEADERS_MINIMAL

//
// setting RM_HEADERS_MINIMAL causes rest of file to be excluded so that
//    we don't pull in all the other include files.   Used by
//    unix version for os-interface files (so we can minimize
//    shipped include files)
//

#include "all_incs.h"

#include <nvhal.h>
#include <nvromdat.h>
#include <nvRmStereo.h>
#include <mvision.h>

#include <nvdload.h>

//
// Dynamically loadable code module 
//
typedef struct _def_dld_module
{
    NV_ImageHeader          *header;
    SectionInfoEntry        *secTable;
    RelocationInfoEntry     *relocTable;
    SymbolTableEntry        *symTable;
    NvU8                    *dataBuffer;
    NvU8                    *codeBuffer;
    NvU8                    *bssBuffer;
    NV_ImageFooter          *footer;
    NvU32                   refCnt;
    void                    *osHook;
} DLD_MODULE, *PDLD_MODULE;

//
// A software hash table entry
//
typedef struct _def_sw_hashTableEntry
{
    POBJECT Object;
    U032    ChID;
} SW_HASH_TABLE_ENTRY, *PSW_HASH_TABLE_ENTRY;

// A repository for different types of screen positioning data, whether
// CRT or TV. This is the basic structure read/written to/from the 
// registry on starup and upon 'commit' to modes and positions.

// Definitions for making an array of positioning units. Both values
// are rather arbitrary; We're assuming 1-4 CRT's and 1-4 TV's.
// CRT DISPLAYS indexes are 0-3 and TV indexes are 4-7.

#define MAX_DISPLAYS        8
#define DISPLAY_TV_OFFSET   4

typedef struct 
{
    //
    // These flags control registry reads; If 'Valid' is true then we won't
    // read the registry -- we'll use the HWINFO value. If NotInRegistry is
    // true then we won't attempt a futile registry re-read -- we'll use
    // defaults.
    //
    struct
    {
        U032 NotInRegistry     : 1;
        U032 ValidFromRegistry : 1;
        U032 Reserved          : 30;
    }Flags;

    U032 numBytes;

    // The position unit
    union 
    {
        DESKTOP_POSITION_ADJUSTMENT DesktopPositionAdjustment;    
        NV_CFGEX_SET_DESKTOP_POSITION_MONITOR_PARAMS    CrtParams;
        NV_CFGEX_DESKTOP_POSITION_TV_MULTI_PARAMS       TVParams;
    } Position;
}POSITIONING_UNIT;


//
// System info structure
//
typedef struct _def_rm_info
{
    struct _def_processor_info
    {
        U032 Size;
        char ID[32];
        U032 Type;
        U032 Clock;
        U032 L1DataCacheSize;          // L1 data (or unified) cache size in KB.
        U032 L2DataCacheSize;          // L2 data (or unified) cache size in KB.
        U032 DataCacheLineSize;        // Bytes per line in the L1 data cache.
        U032 HostPageSize;             // Native page size of host os (4k, 64k, etc).
        char Name[52];                 // Embedded processor name.  
                                       //   Only filled in if CPU has embedded name.      
        PROCESSORHALINFO HalInfo;
    } Processor;

    struct _def_agp_info
    {
        struct _def_agp_nb_addr {
            U008   bus;
            U008   device;
            U008   func;
            U008   valid;
        } NBAddr;                   // bus, device, func of the NB
        U032 AGPCtlrPCI;            // AGP controller is PCI addressable
        U032 AGPCapPtr;             // offset of the AGP capptr in the NB
        NvPhysAddr AGPPhysStart;

        struct                      // TRUE bits indicate that registry overrides exist.
        {
            U032   fwStrap          : 1;
            U032   rate             : 1;    // Note: AGP Rate variable is in pDev->Chip
            U032   sba              : 1;
            U032   fwAdis           : 1;
            U032   fastWrite        : 1;
            U032   agpVrefDisabled  : 1;
            U032   agpIntelAmtt     : 1;
            U032   agpIntelLptt     : 1;
            U032   reserved         : 24;
        } AGPOverrides;

        U032 AGPIntelAmtt;
        U032 AGPIntelLptt;
        U032 AGPsba;
        U032 AGPfw_adis;
        U032 AGPfwstrap;
        U032 AGPfw;
        VOID_PTR AGPLinearStart;
        U032 AGPHeapFree;
        VOID_PTR AGPGartBase;
        VOID_PTR AGPGartContigAlloc;
        VOID_PTR AGPGartBitmap;
        U032 AGPLimit;
        U032 AGPFWEnable;
        U032 AGPFullSpeedFW;
        U032 AGPChipset;            // AGP chipset enum (NT4 AGP support)
        VOID_PTR AGPChipsetMapping; // mapping to AGP registers (NT4 AGP support)
        VOID_PTR pOsAgpInfo;
        U032 AGPRegisterSave[12];   // Chipset Register Save Storage.
    } AGP;

    struct _def_debugger_info
    {
        U032 Handle;                // handle of currently registered DE client
        POBJECT object;             // object data structure
        U032 deStatus;              // debugger status
        U032 deCaps;                // debugger capabilites
    } Debugger;

    struct _def_dload_info
    {
        U032 modulesSz;             // size of the 'images' pointer array
        PDLD_MODULE *modules;       // pointers to NV_Image structs
    } Dload;

} RMINFO, *PRMINFO;

typedef struct _def_hw_info
{
    struct _def_chip_info
    {
        U032 Size;
        char ID[32];
        U032 IntrEn0;
        U032 Bus;
        U032 BiosRevision;
        U032 BiosOEMRevision;
        U032 BiosRMDword;
#define RMDWORD_VPE_INFO_AVAILABLE 0x00010000
#define RMDWORD_VPE_PRESENT        0x00020000
#define RMDWORD_DRIVER_REVISION    0xFFFF
        U032 ReqAGPRate;
#ifdef RM_STATS
        U032 ServiceCount;
        U032 ServiceTimeLo;
        U032 ServiceTimeHi;
        U032 ExceptionTimeLo;
        U032 ExceptionTimeHi;
        U032 osStartInterrupt;
#endif
        MCHALINFO HalInfo;
    } Chip;
    struct _def_mapping_info
    {
        U032 Size;
        char ID[32];
        NvPhysAddr PhysAddr;
        NvPhysAddr PhysFbAddr;
        U032 IntLine;
        U032 IntPin;
#define BUS_DEVICE_BUS_SHIFT    8     
#define BUS_DEVICE_BUS_MASK     ((1 << BUS_DEVICE_BUS_SHIFT) - 1)
        U032 BusDevice;                     // location on the PCI bus
        U032 UpdateFlags;
        U032 FinishFlags;
        NvPhysAddr nvPhys;                  //***
        NvPhysAddr fbPhys;                  //***
        U032 nvIRQ;                         //***
        // NOTE: fbLength is the size of the memory window.
        // IT DOES NOT REFLECT THE AMOUNT OF MEMORY ACTUALLY ON THE BOARD.
        // Use Framebuffer.RamSize for that.
        U032 fbLength;
        U032 nvBusDeviceFunc;
        U032 ChipID;                        //***
        U032 hpicIRQ;                       //***  WIN9X ONLY
        U032 hDev;                          //***  WIN9X ONLY
        U032 doIOToFlushCache;              // A non zero in this field indicates that an IO needs to be done to 
                                            // ensure all the caches are flushed. This is currently set if we
                                            // detect an ALI1541 chipset or a RegistryEntry exists. This info
                                            // is returned in rmConfigGet.
    } Mapping;
    struct _def_pram_info
    {
        U032 Size;
        char ID[32];
        U032 FreeInstSize;
        U032 FreeInstBase;
        U032 FreeInstMax;
        U032 PrivBase;                      // throw this out!
        U032 UpdateFlags;
        U032 FinishFlags;
        PRAMHALINFO HalInfo;
    } Pram;        
    struct _def_fifo_info
    {
        U032 Size;
        char ID[32];
#ifdef RM_STATS
        U032 ServiceCount;
#endif
        U032 UpdateFlags;
        U032 FinishFlags;
        FIFOHALINFO HalInfo;
    } Fifo;
    struct _def_master_info
    {
        U032 Size;
        char ID[32];
        U032 UpdateFlags;
        U032 FinishFlags;
        U032 Enable;
        U032 PciNv19;   // Not Master, but there is no Config
    } Master;
    struct _def_dma_info
    {
        U032 Size;
        char ID[32];
#ifdef LEGACY_ARCH    
        U032 DmaInMemEmptyInstance;
        U032 DmaToMemEmptyInstance;
        U032 DmaFromMemEmptyInstance;
#endif // LEGACY_ARCH
        DMAHALINFO HalInfo;
    } Dma;
    struct _def_framebuffer_info
    {
        U032 Size;
        char ID[32];
        // I think these can go too
        U032 DpmLevel;
        U032 Resolution;
        U032 RefreshRateType;
        U032 HorizFrontPorch;
        U032 HorizSyncWidth;
        U032 HorizBackPorch;
        U032 VertFrontPorch;
        U032 VertSyncWidth;
        U032 VertBackPorch;
        U032 HSyncPolarity;
        U032 VSyncPolarity;
        U032 CSync;
        U032 Count;
#ifdef RM_STATS
        U032 ServiceCount;
#endif
        U032 Current;
        U032 FlipUsageCount;
        U032 FlipTo;
        U032 FlipFrom;
        U032 UpdateFlags;
        U032 FinishFlags;
        U032 FilterEnable;
        U032 Underscan_x;
        U032 Underscan_y;
        U032 Scale_x;               
        U032 Scale_y;
        VOID_PTR HeapHandle;
        U032 CursorAdjust;
        U032 NoAdjustedPitch;
        U032 PrimarySurfaceAlloced;
        U032 ConfigPageHeight;  // for LEGACY_ARCH

        //
        // Registry overrides; These indicators say whether the registry
        // has overridden a value. They are only read on power-up. The
        // indicator says whether to use the override.
        //
        struct
        {
            U032   FilterEnable  : 1;
            U032   MemoryOverride : 1;
            U032   RTLOverride : 1;
            U032   reserved : 29;
        } FBOverrides;

        // Some Actual override values
        U032      MemoryOverride;
        U032      RTLOverride;

        FBHALINFO HalInfo;
    } Framebuffer;
    struct _def_graphics_info
    {
        U032 Size;
        char ID[32];
        U032 Debug0;
        U032 Debug1;
        U032 Debug2;
        U032 Debug3;
        U032 Debug4;
        U032 CurrentChID;
        U032 Enabled;
#ifdef RM_STATS
        U032 ServiceCount;
#endif
        U032 UpdateFlags;
        U032 FinishFlags;
        U032 Capabilities;
        VOID_PTR ClassLocks;        // ptr to lock table
        GRHALINFO HalInfo;
        //
        // Registry overrides; These indicators say whether the registry
        // has overridden a value. They are only read on power-up. The
        // indicator says whether to use the override.
        //
        struct
        {
            U032   SwathControl  : 1;
            U032   reserved : 31;
        } GROverrides;

        // Some Actual override values
        U032 SwathControl;

    } Graphics;
    struct _def_video_info
    {
        U032 Size;
        char ID[32];
        U032 OverlayAllowed;
        U032 CurrentChID;
#ifdef RM_STATS
        U032 ServiceCount;
#endif
        U032 dwHasInited;   
        U032 class63VBlankCount[2];
        U032 FlipOnHsync;
        U032 ClassLocks;
        VIDEOHALINFO HalInfo;
    } Video;
    struct _def_mediaport_info
    {
        U032 Size;
        char ID[32];
#ifdef RM_STATS
        U032 ServiceCount;
#endif
        U032 ImageBufferNumber;
        VOID_PTR ImageCallback;
        VOID_PTR CurrentDecoder;
        VOID_PTR CurrentDecompressor;
        VOID_PTR CurrentParallelBusObj;
        U032 ClassLocks;
        MPHALINFO HalInfo;
    } MediaPort;
    struct _def_mpeg_info
    {
        U032 Size;
        char ID[32];
        U032 MpegAllowed;
        VOID_PTR CurrentChannelMpeg;
#ifdef RM_STATS
        U032 ServiceCount;
#endif
        MPEGHALINFO HalInfo;
    } Mpeg;
    struct _def_dac_info
    {
        U032 Size;
        char ID[32];
        U032 DpmLevel;
        U032 HsyncOff;                      // Ok to set hsync off, vsync on (patent issue)
#ifdef RM_STATS
        U032 ServiceCount;
#endif
        U032 DisplayChanged;
        U032 CurrentDisplay;
        U032 DisplayPitch[MAX_CRTCS];
        U032 EncoderType;
        U032 EncoderID;
        U032 TVCursorMin;
        U032 TVCursorMax;
        U032 TVStandard;                    // NTSC or PAL
        U032 TVoutput;                      // SVideo/Composite/Autoselect
        U032 AuxWinEnabled;                 // auxwin needs special treatment in the modeset (specifically for NV11 auxwin)
        U032 SavedReg09H;                   // Saved brightness register
        U032 TV871_80;                      // 871 user adjustment registers
        U032 TV871_82;
        U032 TV871_92;
        U032 TV871_98;
        U032 TV871_9A;
        U032 TV871_C8;
        U032 TV871_CA;
        U032 TV871_CC;
        U032 TV871_7C;                      // HBURST_BEGIN (needed by macrovision code)
        U032 TV871_7E;                      // HBURST_END (needed by macrovision code)
        U032 TVClockInversionDisabled;      // enable the encoder clock inversion? (from registry)
        U032 TVOverscanDisabled;            // TV overscan or non-overscan? (from registry)
        RM_MACROVISION_CONTROL TVLastMacrovision;
        U032 TVMasterMode;                  // force master-mode protocol? (from registry)
        U032 CH7009_03h;                    // default positioning value for reg 3
        U032 CH7009_05h;                    // default positioning value for reg 5
        U032 CH7009_06h;                    // default positioning value for reg 6
        U032 CH7009_07h;                    // current BLR/brightness setting (w/o Macrovision compensation)
        S032 CH7009_07hAdjust;              // BLR/brightness adjustment (based on Macrovision)
        U032 CH700X_08h;                    // default positioning value for reg 8
        U032 CH700X_0Ah;                    // default positioning value for reg A
        U032 CH700X_0Bh;                    // default positioning value for reg B
        U032 PH710X_70h;                    // Default hpos values for Philips encoders                     
        U032 PH710X_71h;                    
        U032 PH710X_72h;
        U032 PH710X_VDT;
        NV_CFGEX_TV_USER_CONTROLS_PARAMS TVControls; // TV slider controls
        U032 CursorColor1;
        U032 CursorColor2;
        U032 CursorColor3;
        U016 CursorColorImage[32][32];
        U032 PaletteOrGamma[MAX_CRTCS][256]; // Palatte for 8 bpp, otherwise Gamma 
        U032 UpdateFlags;
        U032 CursorImageInstance;
        U016 CursorImageCache[32][32][MAX_CACHED_CURSORS]; // 32x32 words
        U032 CursorImageCacheInstance[MAX_CACHED_CURSORS];
        U032 CursorAddress[MAX_CACHED_CURSORS];
        U032 CursorMaskCRC[MAX_CACHED_CURSORS];
        U008 SavedCursor[(32/8)*32 + 32*32];
        U032 CursorCached;
        U032 CursorCacheCount;
        U032 CursorCacheEntry;
        U008 flickertoggle;                 //***
        U008 CursorBitDepth; 
        U008 I2cIniFlag;
        U008 dummy2;                        //Pads for the previous byte ***
        U032 filterCountdown;               //***
        U032 CursorCacheEnable;
        U032 CRTPortID;
        U032 CRT2PortID;
        U032 TVOutPortID;
        U032 DFPPortID;                     // I2C Port to use for digital flat panel as primary
        U032 DFP2PortID;                    // I2C Port to use for DFP 2 of 2
        U032 I2CPrimaryWriteIndex;          // CRTC Index of primary I2C write port
        U032 I2CPrimaryStatusIndex;         // CRTC Index of primary I2C status port
        U032 I2CSecondaryWriteIndex;        // CRTC Index of secondary I2C write port    
        U032 I2CSecondaryStatusIndex;       // CRTC Index of secondary I2C status port    
        U032 I2CTertiaryWriteIndex;         // CRTC Index of tertiary I2C write port    
        U032 I2CTertiaryStatusIndex;        // CRTC Index of tertiary I2C status port    
        U032 HorizontalVisible;
        U032 HorizontalBlankStart;
        U032 HorizontalRetraceStart;
        U032 HorizontalRetraceEnd;
        U032 HorizontalBlankEnd;
        U032 HorizontalTotal;
        U032 VerticalVisible;
        U032 VerticalBlankStart;
        U032 VerticalRetraceStart;
        U032 VerticalRetraceEnd;
        U032 VerticalBlankEnd;
        U032 VerticalTotal;
        U032 PixelDepth;
        U032 HorizontalSyncPolarity;
        U032 VerticalSyncPolarity;
        U032 DoubleScannedMode;
        U032 fpHActive;
        U032 fpHDispEnd;
        U032 fpHValidEnd;
        U032 fpHSyncStart;
        U032 fpHSyncEnd;
        U032 fpHValidStart;
        U032 fpHTotal;
        U032 fpVActive;
        U032 fpVDispEnd;
        U032 fpVValidEnd;
        U032 fpVSyncStart;
        U032 fpVSyncEnd;
        U032 fpVValidStart;
        U032 fpVTotal;
        U032 fpFlags;
        U032 fpRelativeBrightness;  //current setting of the FP brightness
        U032 fpPWMRegisterContents; //used by PM to shut off backlight and properly restore backlight
        U032 fpExternalInverterCurrentMode;
        U032 PClkVClkRatio;         // for legacy
        U032 ExtDacMaxM;
        U032 FminVco;
        U032 FmaxVco;
        DACCRTCINFO CrtcInfo[MAX_CRTCS];
        DACHALINFO HalInfo;
        DACDCBDEVDESC DCBDesc[DCB_MAX_NUM_DEVDESCS];
        DACDCBI2CRECORD DCBI2c[DCB_MAX_NUM_I2C_RECORDS];
        U032 VBiosDCB[DCB_MAX_NUM_DEVDESCS][2];
        U032 VBiosICB[DCB_MAX_NUM_DEVDESCS];
#define DAC_CRT_OFFSET_IN_MAP       0       // bits 0..7 map to CRT0..CRT7
#define DAC_TV_OFFSET_IN_MAP        8       // bits 8..15 map to TV0..TV7
#define DAC_DD_OFFSET_IN_MAP        16      // bits 16..23 map to DD0..DD7
#define DAC_CRT_MASK_IN_MAP         0x000000ff
#define DAC_TV_MASK_IN_MAP          0x0000ff00
#define DAC_DD_MASK_IN_MAP          0x00ff0000
        U032 DevicesBitMap;
        U032 BootDevicesBitMap;
        U032 PrimaryDeviceBitMap;
        U032 LogicalDeviceMap[MAX_CRTCS];   //Index by logical device to look up mapping to physical head
        struct _def_display_props
        {
            BOOL OverrideSBIOSDevs;
            U032 RMMaintainDevs;            //Ignore VBIOS input and do it ourselves.
            U032 RMMaintainScaling;         //Ignore VBIOS input and do it ourselves.
            U032 LidBehaviorBitMask;
            U032 MaskSecondaryDevice;       //which devices shouldnt be enabled as secondary by default
#define NV_MOBILE_LID_HIDES_DEV         1   //If set, lid closed causes us to consider LCD not present
#define NV_MOBILE_LID_HOTKEYS_NOSPAN    2   //If set, lid closed with LCD enabled causes spoofed hotkey (standard and clone mode)
#define NV_MOBILE_LID_HOTKEYS_SPAN      4   //If set, lid closed with LCD enabled causes spoofed hotkey (spanning mode)
            BOOL Spanning;          //1 = currently in spanning/extended/... mode
            BOOL VideoPlayback;       //1 = DVD playback in progress
            U032 LastSwitchEvent;
#define SWITCH_FROM_COLD_START  0   //Hibernation or cold boot
#define SWITCH_FROM_FS_DOS      1   //Returned from full-screen DOS
#define SWITCH_FROM_WINDOWS     2   //From standby or hotkey or control panel induced device switch event
            U032 nonACPIHotkeyPrimDevScheme;    // non-ACPI only, with ACPI this same method is implemented in display driver
            U032 RMMaintainTVStandard;          // Ignore VBIOS input and do it ourselves: see NV_REG_RM_MAINTAIN_TV_STANDARD.
            U032 UseDPMS;
#define NO_DPMS_REQUIRED        0   //As God intended, SBIOS leaves all display control to us (keep this #define == 0)
#define DPMS_NV11_STYLE         1   //Toshiba approach for NV11 - DPMS calls needed to turn on/off devices
#define DPMS_NV17_STYLE         2   //Toshiba approach for NV17 - as NV11, but we can read registers to determine
                                    // true HW state and SBIOS has an additional int10h (4F14/8C) that can lock
                                    // panel in on or off state
            U032 DevSwitchSuppressMask; //Conditions for which to suppress display switching.
                                        // Bits defines in nvcm.h with NV_CFG_DISPLAY_CHANGE_INHIBITS.
            U032 SBIOSHandlesHotkeyInhibit; // Non-zero if the SBIOS can be trusted to inhibit hotkeys correctly.
        } DeviceProperties;
        U032 DevicesNoCheck;
#define NO_DEVICES_FORCED_PRESENT   0
        U032 DisplayChangeInProgress;
        U032 PreviousLidState;
        U032 CurrentDevices;
#define CUR_DEVS_NOT_INITED             0xFFFFFFFF
        U032 BIOSPanelProgramming;
        U032 PanelOffTimestampLO;
        U032 PanelOffTimestampHI;
#define DAC_NOT_ENABLED               0x0
#define DAC_PANEL_POWER_ENABLED       0x1
#define DAC_PANEL_BACKLIGHT_ENABLED   0x2
#define DAC_PANEL_ENABLED             (DAC_PANEL_POWER_ENABLED | DAC_PANEL_BACKLIGHT_ENABLED)
#define DAC_CRT_ENABLED               0x4
#define DAC_TV_ENABLED                0x8
#define DAC_ALL_DEVICES_ENABLED       (DAC_PANEL_ENABLED | DAC_CRT_ENABLED | DAC_TV_ENABLED)

#define DAC_NV11_NONDDCCRT_FP_DISABLE 0x04000000  // When the flat panel is booted on Head A on Desktop NV11 cards,
                                                  // to detect a Non-DDC monitor on DAC A will require us to unslave
                                                  // the timing generator from the flat panel.  This causes yellow
                                                  // flashing.  So, as a customer option, if we read the registry key
                                                  // of DisableAnalogMonitorDetectCollisions to 1, we'll return FALSE
                                                  // NEGATIVES from the dacMonitorConnectStatus() if a flat panel
                                                  // is already setup on Head A.  This fixes bug 46385.
#define DAC_NV17_TV_REMOVED           0x08000000  // On per board basis, we need to read the strap bit to see if the 
                                                  // NV17 board has TV connected to DAC B else it's a VGA connector.
#define DAC_MONITOR_POWER_STATE_OFF   0x10000000  // Shows that the current monitor power management state is off.
#define DAC_PRE_EDID_INITIALIZED      0x20000000  // Need to initialize panel strap and Mobile EDID before using it.
#define DAC_PANEL_CLK_RESET           0x40000000  // states that we've SW reset the device, and the flat panel needs the reset script
                                                  // if we power on the flat panel in this state, we'll get blooming.
#define DAC_LOW_PCLK_ENABLED          0x80000000  // states that we've lowered the PCLK in dacfp.c
        U032 DevicesEnabled;

        //tmds values that are useful at wakeup
        U008 TmdsPll0[MAX_CRTCS];
        U008 TmdsPll1[MAX_CRTCS];
        U008 TmdsPll2[MAX_CRTCS];
        U008 TmdsIdly[MAX_CRTCS];
        U008 TmdsMode[MAX_CRTCS];
        U008 TmdsLvds[MAX_CRTCS];
        U008 TmdsTrig0[MAX_CRTCS];
        U008 TmdsTrig1[MAX_CRTCS];
        U008 TmdsTrig2[MAX_CRTCS];
        U008 TmdsRotck[MAX_CRTCS];

    struct _def_manufacturer_props
        {
          U032 Version;                         // Version of the block
#define MANUFACTURER_NOT_CAPABLE 0xFFFFFFFF     // BIOS not capable of manufacturer scripts (Must be >= rev 14)
#define MANUFACTURER_NOT_TESTED  0x00000000     // BIOS has not been tested if it supports manufacturer scripts
          U032 Size;                            // Size of each entry ( to allow fields to be added without breaking the old drivers )
          U032 Flags;                           // Script processing flags
#define MANUFACTURER_POWER_OFF_REQUIRED 0x1     //   bit 0   : 1=power off required for reprogramming
#define MANUFACTURER_RESET_REQUIRED     0x2     //   bit 1   : 1=reset required after PCLK change
                                                //   bits 2-7 : reserved
          U032 LVDSInitScript;      // Panel link initialization script index
          U032 LVDSResetScript;     // Panel link reprogram script index 
                                    //   (only needed if Flags[1]=1
          U032 BackLightOnScript;   // Back light ON  single link B table
          U032 BackLightOffScript;  // Back light OFF single link A table
          U032 PanelOnScript;       // Panel and backlight ON single link A table
          U032 PanelOffScript;      // Panel and backlight OFF single link A table
          U032 PanelOffOnDelay;     // Minimum delay between panel off and panel on in 
                                    //   milliseconds (NOT microseconds like the scripts)
        } ManufacturerProperties;
        U008 MobileEDID[128];
        U032 PanelStrap;
        U032 BroadcastSupport;
        U032 NonBroadcastModeStoredCr3B[2];
        U032 PreserveBroadcastState;    // This flag is used in dacClearBroadcastState().
                                        // If set to 1, it means keep broadcast mode as is
                                        // If set to 0, it means we should force it off when !pDev->VGA.Enabled.
        U032 DesktopPanelPowerPolicy;   // Used when running on "special" desktop systems that need special handling for the DFP.

        U032 colorSaturationBoost[MAX_CRTCS];   // Color saturation boost control panel value

        // "Valid From Registry" bits. Just like the 'override' bits in other engines
        // these bits control whether certain values have already been read from the
        // registry, except that in certain circumstances they must be re-read.
        // If the associated values are changed and written to the registry then
        // these "Valid From Registry" bits should be cleared so that the registry
        // read function will re-read them, unless the HWINFO value is also updated.

        struct
        {
            U032 CRTCNotInRegistry     : 1;
            U032 CRTCHtotal            : 1;
            U032 Reserved              : 30;
        } ValidFromRegistry[MAX_CRTCS];

        U032 CRTCHtotal[MAX_CRTCS];

        // CRT and TV positioning units. These also use the 'valid from registry'
        // strategy.

        POSITIONING_UNIT    DisplayPosition[MAX_DISPLAYS][MAX_RESOLUTIONS];

        // Padding to ensure that HWINFO is still a multiple of 4 bytes.
        U008 positionPad[((sizeof(POSITIONING_UNIT) * MAX_DISPLAYS * MAX_RESOLUTIONS) % 4)+4];

        //these are not saved off on a per-head basis, because they don't vary from head to head.
        U032 TVContrast;
        U032 TVBrightness;
        U032 TVFlickerFilter;
        U032 TVSaturation;

        //these are not saved off on a per-head basis, because they don't vary from head to head (yet!)
        CURRENT_TV_POSITION_ADJUSTMENT  TVCurrentPosition;

#define TVO_CONTRAST      BIT(0)
#define TVO_BRIGHTNESS    BIT(1)
#define TVO_FLICKERFILTER BIT(2)
#define TVO_SATURATION    BIT(3)

        U032 TVRegistryValid;
        U032 TVInitFromRegistry;
        U032 dynEngCtrlEnabled;
        U032 unusedEngines;
        U032 modesetBlankDelay;
        U032 enableDvoPm;
        U032 DCBOverrideRegistry;
    } Dac;

#ifdef LEGACY_ARCH    
    struct _def_dgp_chip_info
    {
        U032 Size;
        char ID[32];
        U032 Chip;
        U032 Enable3D;
        U032 Scale;
        U032 HorizDelay;
        U032 HorizOffset;
        U032 HorizScale;
        U032 VertOffset;
        U032 VertScale;
        U032 UpdateFlags;
        U032 FinishFlags;
    } Dgp;
#endif // LEGACY_ARCH
    struct _def_timer_info
    {
        U032 Size;
        char ID[32];
        U032 Denominator;
        U032 Numerator;
#ifdef RM_STATS
        U032 ServiceCount;
#endif
        U032 UpdateFlags;
        U032 FinishFlags;
        U032 tmrSave[2];          //***
    } Timer;
    struct _def_vga_info
    {
        U032 Size;
        char ID[32];
        U032 Enabled;
        U032 DOSBox;
        U032 PrimaryVga;       // The actual VGA Device in Win '9X
#ifdef NV_VGA
        U032 GlobalEnable;
        U032 RomVersion;
        U032 RomFixupCode;
        U032 UpdateFlags;
        U032 FinishFlags;
#endif // NV_VGA
    } Vga;
    struct _def_classes_info
    {
        U032 Size;
        char ID[32];
        U032 UpdateFlags;
        U032 FinishFlags;
    } Classes;
#ifdef RM_STATS
    struct _def_stat_info
    {
        U032 Size;
        char ID[32];
        U032 FifoExceptionCount;
        U032 FramebufferExceptionCount;
        U032 GraphicsExceptionCount;
        U032 TimerExceptionCount;
        U032 TotalExceptionCount;
        U032 IntTimeLo;
        U032 IntTimeHi;
        U032 ExecTimeLo;
        U032 ExecTimeHi;
        U032 MemAlloced;
        U032 MemLocked;
        U032 UpdateLo;
        U032 UpdateHi;
    } Statistics;
#endif
    struct _def_power_info
    {
        U032 State;
        U032 DisplayState[MAX_CRTCS];
        U032 strapInfo;
        U032 coeff_select;
        U032 nvpll;
        U032 mpll;
        U032 vpll;
        U032 PbusDebug1;
        U032 PbusDebug3;
        U032 MobileOperation;
        U032 PostNVCLK;
        U032 PostMCLK;

//
// This should really go in nvReg.h
//
#define NV_REG_POWER_MIZER_ENABLE  "PowerMizerEnable"
#define NV_REG_POWER_MIZER_LEVEL   "PowerMizerLevel"
#define NV_REG_POWER_MIZER_DEFAULT "PowerMizerDefault"
#define POWER_MIZER_DEFAULT  1                    // Maximum performance PM slider setting.
#define POWER_MIZER_AC       POWER_MIZER_DEFAULT  // Use default level when the laptop is plugged in.
#define POWER_MIZER_MIN      0
#define POWER_MIZER_MAX      4

        U032 PmNVCLK[POWER_MIZER_MAX + 1];
        U032 PmMCLK[POWER_MIZER_MAX + 1];
        U032 ForceSwapInterval[POWER_MIZER_MAX + 1];
        U032 PowerMizerEnable;    // 0 - PowerMizer disabled.  1 - PowerMizer Enabled.
        U032 PowerMizerDefault;   // Default value for "Restore Defaults" on property page.
        U032 PowerMizerSelected;  // Value set on  property page slider and registry.
        U032 PowerMizerDynamic;   // Value the driver is currently using.  Affected by AC.  
        U032 ACPowerState;        // 0 - Running from battery.  1 - Plugged into AC.
        U032 doSpeedStep3DDisable;     // workaround for NV11 on SpeedStep laptops
        U032 IntelSpeedStepState;      // saved GEN_PMCON_1 (offset 0xA0) in the ICH3
        U032 DisableSpeedStepCount;    // disable SpeedStep counter (0 = SpeedStep enabled)
        U032 PciIoMemSpace;
        U032 GPIOExt;
        U008 cr51hA;
        U008 dummy1;    //Pads for the previous byte
        U008 cr51hB;
        U008 dummy2;    //Pads for the previous byte

#define DAC_POWER_PENDING             0x1
#define DAC_PM_SUPPORTED              0x80000000
        U032 Flags;
        U032 osD0;
        U032 osD1;
        U032 osD3;
        U032 osD4;

#define CR3B_EXTERNAL_DEVICES         0xf0          // Bits for external devices; 4=LCD, 5=CRT, 6=TV, 7=PANEL or other.
        U008 cr3BHibernateState[MAX_CRTCS];         // Pre-hibernation state of CR3B registers. NOTE: Only if !Broadcast Mode
        U008 cr3BHSReserved[(4-(MAX_CRTCS%4))%4];   ///RYAN@COMPILE, reserved to ensure sizeof(HWINFO) is a multiple of 4.

        U032 activeEngines;
        // Chip-dependent power data moved to MC section.
    } Power;
    struct _def_registry_info
    {
        U032 Size;
        char ID[32];
        char DBstrDevNodeDisplay[128];
        char DBstrDevNodeDisplayNumber[128];
    } Registry;

    struct _def_hotkey_info
    {
        BOOL enabled;
        U032 queueHead;
        U032 queueTail;
        struct _def_hotkey_queue
        {
            U032 event;
            U032 status;
        } Queue[NV_HOTKEY_EVENT_QUEUE_SIZE];
    } HotKeys;

    U016                                  filler1;
    U008                                  filler2;

    // NOTE: This filler above fixed a really strange problem. We don't know why it
    // fixed the problem. Without this filler, phantom drive letters would appear!
    // See bug # 19981209-132200. We also don't know how a misalignment here causes
    // the registry to go bad.

    U008                DeviceDisabled;
    PSW_HASH_TABLE_ENTRY DBhashTable;                 //***
    U008                *DBfbInstBitmap;              //***
    U032                DBfbKDASharedMem;
    VOID                *DBfbIconBeginEventHandle;
    VOID                *DBfbIconEndEventHandle;
    PHWREG              DBnvAddr;                     //***
    PHWREG              DBromAddr;                    //***
    PHWREG              DBbiosAddr;                   //***
    PHWREG              DBfbAddr;                     //***
    PHWREG              DBagpAddr;                    //***
    PHWREG              DBimAddr;                     //***                               
    PFIFO               DBfifoTable;                  //***
    PDMAOBJECT          DBDmaList;                    //***
    PDMAOBJECT          DBVidDmaObject;               //***
    PVBLANKNOTIFY       DBgrVBlankList[2];            //***
    PTIMERCALLBACK      DBtmrCallbackList;            //***
    PTIMERCALLBACK      DBtmrCallbackFreeList;        //***
    PTIMERCALLBACK      DBtmrCallbackTable;           //***
    PVBLANKNOTIFY       DBclass63VBlankList[2];
    POVERLAYSERVICEPROC pOverlayServiceHandler;
    POBJECT             DBpHardwareObject;
    U032                (*DBdisplayModeHook)(U032);
    SURFACEATTRIBUTES   DBOverlaySurfaceAttribs[20];
    SURFACEATTRIBUTES   DBImageSurfaceAttribs[20];
    U032                *fb_save;
    PVIDEOCOLORMAPOBJECT colormapObjectToNotify;

    ENGINEDB            engineDB;                   // HAL-derived engine info
    CLASSDB             classDB;                    // HAL-derived class info

    HALHWINFO           halHwInfo;
    VOID_PTR            pHalEngineMem;
    PHALFUNCS           pHalFuncs;
    PHALFUNCS           pHalCompatFuncs;

    U008               *EDIDBuffer;
    U032                I2CAcquired;

    U032                DBdevInstance;
    U032                biosOffset;                 // has bios been relocated within rom?

    PRMINFO             pRmInfo;                    // pointer to our system information (RMINFO NvRmInfo)
    VOID_PTR            pOsHwInfo;                  // pointer to our OS dependent information

    // Can't move this to WINNTHWINFO currently because of a hack in state.c
#ifdef NTRM
    VOID*               DBmpDeviceExtension;        // miniport device extension
#endif

    //Stereo stuff
    LPNV_CFG_STEREO_PARAMS      pStereoParams;  
    U008*               LastCursorAddr;             // For keeping track of the last interlaced cursor. (For interlaced stereo only). 
#define STEREO_ADJACENT_EYES    1                   // Left and Right eye buffers have to be adjacent.
    U032                StereoFlags;

    // DosVedanya2: Quadro protection code
    U032                QuadroProtectionFlags;      // If any bit is active, a Quadro-Hack was detected and the damage routing matching the bit
                                                    // should do some work.
    U032                QuadroProtectionCounter;    // Counter used to track how often we ran into it. This is also used 
                                                    // as performance degradation value.
} HWINFO;

// DosVedanya2: Quadro protection code
// flags used in HWINFO.QuadroProtectionFlags
#define RM_QUADROPROTECTION_DISABLE_IRQS            BIT(0) // if set will not enable IRQs again
#define RM_QUADROPROTECTION_DONT_RESET_VBLANK_IRQS  BIT(1) // if will not reset vblank handled immediately
#define RM_QUADROPROTECTION_DISABLE_GAMMA_UPDATE    BIT(2) // if set will not allow to update gamma table


//
// This macro will produce a divide by 0 error at compile time if the sizeof HWINFO
// struct is not a multiple of 4. However, it does not help if some sub struct of
// HWINFO is not aligned properly.
//
#define ASSERT_NVINFO_SIZE if (1/((sizeof(HWINFO) % 4) ? 0 : 1))

#define MAX_INSTANCE        8
#define TEST_DEVICE_NUM     0

extern  U032    ClientInstance;
extern  PHWINFO NvDBPtr_Table[MAX_INSTANCE];
extern  RMINFO  NvRmInfo;

//
// nv10 needs 2 dwords, but nv4 only needs one, so we'll rename one of the
// fields to be the nv4 define
//
#define VideoScale          VideoScaleX
#define Buff0VideoScale     Buff0VideoScaleX
#define Buff1VideoScale     Buff1VideoScaleX
#define nvAddr              pDev->DBnvAddr 
#define romAddr             pDev->DBromAddr
#define biosAddr            pDev->DBbiosAddr
#define fbAddr              pDev->DBfbAddr 
#define CurDacAdr(Head)     GETCRTCHALINFO(pDev,Head,RegOffset)
#define NUM_FIFOS           pDev->Fifo.HalInfo.Count
#define devInstance         pDev->DBdevInstance

//
// Things that were moved from nvarch.h
//
#define NV10_CHIP_ARCHITECTURE          0x10

//
// HW description
// TO DO: change this to DEVICE_EXTENT of instance memory so that we will ALWAYS map enuff memory
//
#define NV_DEVICE_SIZE                   0x740000       // 7M of regs + 256K of instance mem (first 64K is BIOS)
#define INSTANCE_MEMORY_BASE             (pDev->Mapping.PhysAddr + pDev->Pram.HalInfo.PraminOffset)
#define INSTANCE_MEMORY_PHYSICAL_BASE(d) ((d)->Mapping.PhysAddr + pDev->Pram.HalInfo.PraminOffset)
#define INSTANCE_MEMORY_LOGICAL_BASE(d)  ((U008*)(d)->DBnvAddr + pDev->Pram.HalInfo.PraminOffset)
#define INSTANCE_MEMORY_SIZE             0x40000        // 256K
#define ROM_SIZE                         0x8000
#define CHANNEL_LENGTH                   0x10000        // 64K
#define INT_A                            0x01
#define FIFO_USER_BASE(d)                (d->Fifo.HalInfo.UserBase)

#endif // RM_HEADERS_MINIMAL
#endif // _NVRM_H_
