/* _NVRM_COPYRIGHT_BEGIN_
 *
 * Copyright 1999-2001 by NVIDIA Corporation.  All rights reserved.  All
 * information contained herein is proprietary and confidential to NVIDIA
 * Corporation.  Any use, reproduction, or disclosure without the written
 * permission of NVIDIA Corporation is prohibited.
 *
 * _NVRM_COPYRIGHT_END_
 */


#ifndef _NV_H_
#define _NV_H_

#include <nvtypes.h>

/* Nvidia's reserved major device number
 * Set this to 0 to request dynamic major number allocation. */
#define NV_MAJOR_DEVICE_NUMBER 195

/* most cards in a single system */
#define NV_MAX_DEVICES 4

typedef struct nv_ioctl_memory_vtop
{
    void *buffer;
    int   byte_length;
    unsigned int *physical_addresses;
} nv_ioctl_memory_vtop_t;

#define NV_IOCTL_MAGIC      'F'
#define NV_IOCTL_MEMORY_VTOP    _IOWR(NV_IOCTL_MAGIC, 1, sizeof(void *))

typedef struct nv_ioctl_card_info
{
    int    flags;               /* see below                   */
    int    bus;			/* bus number (PCI, AGP, etc)  */
    int    slot;                /* card slot                   */
    int    vendor_id;           /* PCI vendor id               */
    int    device_id;
    int    interrupt_line;
    unsigned int reg_address;   /* register aperture           */
    unsigned int reg_size;
    unsigned int fb_address;    /* framebuffer aperture        */
    unsigned int fb_size;
} nv_ioctl_card_info_t;

#define NV_IOCTL_CARD_INFO_FLAG_PRESENT       0x0001

#define NV_IOCTL_CARD_INFO      _IOWR(NV_IOCTL_MAGIC, 2, sizeof(void *))


/* allow interrupts to be turned off/on.
 * hopefully just for debug or working around buggy Xservers :-)
 * Specify 'arg' as a 0 (disable) or non-zero (enable)
 */
#define NV_IOCTL_INTERRUPTS     _IOWR(NV_IOCTL_MAGIC, 3, sizeof(void *))

typedef struct nv_ioctl_primary_card
{
    int index;			// primary card index
} nv_ioctl_primary_card_t;

/* notify resman which card is primary */
#define NV_IOCTL_PRIMARY_CARD  _IOWR(NV_IOCTL_MAGIC, 4, sizeof(void *))

/* debug tool to force module count to 0 */
#define NV_IOCTL_MODULE_RESET  _IO(NV_IOCTL_MAGIC, 99)

/* in some cases, the os may have a different page size, but the hardware (fb, regs, etc)
 * still address and "think" in 4k pages. make sure we can mask and twiddle with these
 * addresses when PAGE_SIZE isn't what we want.
 */
#define NVHW_PAGE_SIZE                4096

#define	NVHW_PAGES_PER_OS_PAGES       (PAGE_SIZE/NVHW_PAGE_SIZE)
#define	NVHW_PAGES_TO_OS_PAGES(count) (((count) + NVHW_PAGES_PER_OS_PAGES - 1) \
                                       / NVHW_PAGES_PER_OS_PAGES)


/* mmap(2) offsets */

#define NV_MMAP_REG_OFFSET              (0)
#define NV_MMAP_FB_OFFSET                (256 * 1024 * 1024)
/* Magic mmap() offset for the memory allocator */
#define NV_MMAP_ALLOCATION_OFFSET (1 * 1024 * 1024 * 1024)
/* AGP indicator, intended to be or'd with magic allocator offset */
#define NV_MMAP_AGP_OFFSET ((unsigned long)(2)*(1024 * 1024 * 1024))

/* duplicated from nvos.h for external builds */
#ifndef NVOS_AGP_CONFIG_DISABLE_AGP
#  define NVOS_AGP_CONFIG_DISABLE_AGP (0x00000000)
#endif
#ifndef NVOS_AGP_CONFIG_NVAGP
#  define NVOS_AGP_CONFIG_NVAGP       (0x00000001)
#endif
#ifndef NVOS_AGP_CONFIG_OSAGP
#  define NVOS_AGP_CONFIG_OSAGP       (0x00000002)
#endif
#ifndef NVOS_AGP_CONFIG_ANYAGP
#  define NVOS_AGP_CONFIG_ANYAGP      (0x00000003)
#endif

#ifdef __KERNEL__

/*
 * ptr arithmetic convenience
 */

typedef union
{
    volatile V008 Reg008[1];
    volatile V016 Reg016[1];
    volatile V032 Reg032[1];
} nv_hwreg_t, * nv_phwreg_t;

/*
 * data structure to keep track of open clients for abnormal termination
 * cleanup.  See nv.c for discussion of cleanup issues
 * NOTE: this can't be kept in 'nv' since the nv current during root allocate
 *       is the control 'nv'.
 */

#define NV_MAX_CLIENTS 128         /* *system-wide* limit, not per device */
typedef struct {
    NvU32  client;
    int    pid;
    void  *file;
} nv_client_t;
extern nv_client_t nv_clients[NV_MAX_CLIENTS];


#define SAVED_BIOS_IMAGE_SIZE 0x10000
#define NUM_FB_PAGES_CACHED 10

typedef struct {
    NvU32 address;
    NvU32 size;
    NvU32 *map;
    nv_phwreg_t map_u;
} nv_aperture_t;

/*
 * per device state
 */

typedef struct
{
    void  *pdev;		    /* you know who */
    void  *os_state;                /* os-specific version of this file */

    int    flags;
    int    device_number;
    int    usage_count;

    /* mapped addresses */
    NvU32  *rom;

    /* PCI config info */
    int bus;
    int slot;
    int vendor_id;
    int device_id;

    /* physical characteristics */
    nv_aperture_t regs;
    nv_aperture_t fb;

    NvU32  interrupt_line;

    U032 agp_config;
    U032 agp_buffers;
    U032 agp_teardown;

    /* keep track of any pending bottom-halves */
    int bh_count;

    /* copy of the video bios in system memory */
    /* used by general resman code to query bios-set values */
    void *vbios;

    nv_aperture_t fb_cache[NUM_FB_PAGES_CACHED];

} nv_state_t;

/*
 * flags
 */

#define NV_FLAG_OPEN            0x0001
#define NV_FLAG_POSTED          0x0002
#define NV_FLAG_WAITING         0x0004
#define NV_FLAG_CONTROL         0x0010
#define NV_FLAG_HOTKEY_OCCURRED 0x0100

#define NV_AGP_ENABLED(nv)      ((nv)->agp_config)
#define NV_AGP_DISABLED(nv)     (!(nv)->agp_config)
#define NV_NVAGP_ENABLED(nv)    ((nv)->agp_config & NVOS_AGP_CONFIG_NVAGP)
#define NV_OSAGP_ENABLED(nv)    ((nv)->agp_config & NVOS_AGP_CONFIG_OSAGP)

/*
** where we hide our nv_state_t * in a pdev
** subject to change...
*/
#define NV_HIDE_NV_IN_PDEV(pdev) \
                     ((nv_state_t *) (pdev)->DBmpDeviceExtension)

#define NV_WAITING_FOR_VBLANK(nv) \
                     (nv->flags & NV_FLAG_WAITING)

/*
 * driver internal interfaces
 */

/* need a fake device number for control device; just to flag it for msgs */
#define NV_CONTROL_DEVICE_NUMBER     100

/* unix-system-specific functions exported to rest of resman */
int   nv_vtop(nv_ioctl_memory_vtop_t *params, int in_kernel);
void *nv_find_kernel_mapping(nv_state_t *, void *address);
unsigned long nv_get_phys_address(unsigned long address);

/* unix-specific locking mechanism for access to core resman */
void nv_lock_rm(nv_state_t *);
void nv_unlock_rm(nv_state_t *);

/* unix-specific locking mechanism to protect bottom half data */
void nv_lock_bh(nv_state_t *);
void nv_unlock_bh(nv_state_t *);

void nv_post_event(void *filep);
void nv_post_vblank(nv_state_t *nv);


void nv_set_hotkey_occurred_flag(void);
int  nv_int10h_call(nv_state_t *, U032 *, U032 *, U032 *, U032 *, void *);

int
nv_alloc_pages( nv_state_t *, void **, unsigned int, unsigned int, unsigned int, void **);
int nv_free_pages( nv_state_t *, void **, unsigned int, unsigned int, void **);

/* functions in general unix module that may be called by os-specific implementations */
void  nv_unix_free_all_unused_clients(nv_state_t *nv, int pid, void *file);

/* move some linux-specific agp code into the included source files for
 * easier external changes
 * if agp isn't implemented for your os, these should all return '-1'
 */
int nv_agp_init(nv_state_t *, void **, void **, void *, U032);
int nv_agp_teardown(nv_state_t *);
int nv_agp_translate_address(nv_state_t *, void *, U032, U032 *);
int nv_agp_retrieve_mapping(nv_state_t *nv, void *, unsigned long *);


#endif /* __KERNEL__ */

/*
 * resman interfaces the driver has to know about
 */

int  RmInitRm(void);
int  RmShutdownRm(void);
int  RmIsr(U032 deviceReference, U032 *need_to_run_bottom_half);
void RmIsrBottomHalf(void *pdev_void);
int  RmPowerManagement(void *pdev_void, U032 head, U032 command);
U032 RmGetMobileValue (void *pdev_v);
U032 RmGetVideoBiosVersion (void *pdev_v);


#ifdef __KERNEL__

int  RmInitAdapter(nv_state_t *nv);
int  RmDisableAdapter(nv_state_t *nv);
int  RmShutdownAdapter(nv_state_t *nv);
int  nvExtEscape(void *file, nv_state_t *nv, int cmd, void *pvParms);

#endif

#endif
