/* _NVRM_COPYRIGHT_BEGIN_
 *
 * Copyright 2001 by NVIDIA Corporation.  All rights reserved.  All
 * information contained herein is proprietary and confidential to NVIDIA
 * Corporation.  Any use, reproduction, or disclosure without the written
 * permission of NVIDIA Corporation is prohibited.
 *
 * _NVRM_COPYRIGHT_END_
 */


#ifndef _NV_LINUX_H_
#define _NV_LINUX_H_

#include "nv.h"

#ifdef __KERNEL__

#include <linux/config.h>
#if defined(CONFIG_SMP) && !defined(__SMP__)
#define __SMP__
#endif

#include <linux/version.h>
#include <linux/wait.h>
#include <linux/fs.h>
#include <linux/poll.h>
#include <linux/slab.h>      /* kmalloc().. */
#include <linux/vmalloc.h>   /* vmalloc().. */


#if LINUX_VERSION_CODE < KERNEL_VERSION(2, 2, 12)
#  error This driver does not support 2.2.11 or earlier kernels!
#elif LINUX_VERSION_CODE < KERNEL_VERSION(2, 3, 0)
#  define KERNEL_2_2
#elif LINUX_VERSION_CODE < KERNEL_VERSION(2, 4, 0)
#  error This driver does not support 2.3.x development kernels!
#elif LINUX_VERSION_CODE < KERNEL_VERSION(2, 5, 0)
#  define KERNEL_2_4
#elif LINUX_VERSION_CODE < KERNEL_VERSION(2, 6, 0)
#  error This driver does not support 2.5.x development kernels!
#  define KERNEL_2_5
#else
#  error This driver does not support 2.6.x or newer kernels!
#endif

#if defined (__ia64)
#  if LINUX_VERSION_CODE < KERNEL_VERSION(2, 4, 13)
#    error This driver does not support 2.4.12 or earlier kernels!
#  endif
#endif

#include <linux/autoconf.h>

#ifndef NOAGPGART
#  ifdef CONFIG_AGP
#    define AGPGART
#  endif
#  ifdef CONFIG_AGP_MODULE
#    define AGPGART
#  endif
#endif

#ifdef AGPGART
#include <linux/types.h>
#include <asm/page.h>
#include <asm/semaphore.h>
#include <linux/agp_backend.h>  // agpgart backend
#include <linux/agpgart.h>      // agpgart
#endif

#if !defined (KERNEL_2_2)
#  define GET_MODULE_SYMBOL(mod,sym)	(const void *) inter_module_get(sym)
#  define PUT_MODULE_SYMBOL(sym)        inter_module_put((char *) sym)
#  define GET_MAP_NR(phys_page)         virt_to_page(__va(phys_page))
#  define MEM_MAP_READ_COUNT(map_nr)    (atomic_read(&(map_nr)->count))
#  define MEM_MAP_INC_COUNT(map_nr)     (atomic_inc(&(map_nr)->count))
#  define MEM_MAP_DEC_COUNT(map_nr)     (atomic_dec(&(map_nr)->count))
#  define GET_EVENT_QUEUE(nv)           ((struct __wait_queue_head *) ((nv)->event_queue))
#else
#  define GET_MODULE_SYMBOL(mod, sym)   (void*) get_module_symbol((mod), (sym))
#  define PUT_MODULE_SYMBOL(sym)
#  define GET_MAP_NR(phys_page)         MAP_NR(__va(phys_page))
#  define MEM_MAP_READ_COUNT(map_nr)    (atomic_read(&mem_map[map_nr].count))
#  define MEM_MAP_INC_COUNT(map_nr)     (atomic_inc(&mem_map[map_nr].count))
#  define MEM_MAP_DEC_COUNT(map_nr)     (atomic_dec(&mem_map[map_nr].count))
#  define GET_EVENT_QUEUE(nv)           ((struct wait_queue **) &((nv)->event_queue))
#endif

#ifndef THIS_MODULE
#define THIS_MODULE  (&__this_module)
#endif

#if defined(DEBUG)
#define NV_DEBUG_LOCKS 1
#endif

#define calc_order(size, order)                         \
    {                                                   \
        order = 0;                                      \
        while ( ((1 << order) * PAGE_SIZE) < (size))    \
        {                                               \
            order++;                                    \
        }                                               \
    }

#if !defined (KERNEL_2_2)
#include <linux/spinlock.h>
extern spinlock_t unload_lock;
#define NV_GET_REG_SYMBOL(sym_value, mod, sym_name) \
    { \
        struct module *mp = THIS_MODULE; \
        struct module_symbol *sym; \
        int i; \
        spin_lock(&unload_lock); \
        if (MOD_CAN_QUERY(mp) && (mp->nsyms > 0)) { \
            for (i = mp->nsyms, sym = mp->syms; i > 0; --i, ++sym) { \
                if (strcmp(sym->name, sym_name) == 0) { \
                    sym_value = sym->value; \
                    break; \
                } \
            } \
        } \
        spin_unlock(&unload_lock); \
    }
#else
#define NV_GET_REG_SYMBOL(value, mod, symbol) \
    value = GET_MODULE_SYMBOL(mod, symbol)
#endif

/*
 * An allocated bit of memory using NV_MEMORY_ALLOCATION_OFFSET
 *   looks like this in the driver
 */

typedef struct nv_alloc_s {
    struct nv_alloc_s *next;    
    struct vm_area_struct *vma;
    unsigned int   process_id;
    unsigned int   thread_gid;
    unsigned int   num_pages;
    void          *direct_kmapping;
    void         **page_table;          /* list of physical pages allocated */
    void          *agp_mapping;         /* ptr to linear agp mapping of pages */
    /* for nvagp */
    unsigned int   class;
    void          *priv_data;
} nv_alloc_t;


/* linux-specific version of old nv_state_t */
/* this is a general os-specific state structure. the first element *must* be
   the general state structure, for the generic unix-based code */
typedef struct {
    nv_state_t nv_state;

    nv_alloc_t *alloc_queue;

    // bottom half interrupt handler info; per device
    struct tq_struct *bh;

    U032 vblank_notifier;
    U032 waiting_for_vblank;

    /* queue for for NV's OS events */
    void *event_queue;

    /* per-device locking mechanism for access to core rm */
    spinlock_t rm_lock;
    unsigned long rm_pflags;

    /* lock for linux-specific data, not used by core rm */
    /* XXX eventually change to sleeping locks? */
    spinlock_t ldata_lock;
    unsigned long ldata_pflags;

    /* lock for linux-specific alloc queue */
    /* XXX eventually change to sleeping locks? */
    spinlock_t at_lock;
    unsigned long at_pflags;

    /* lock for bottom half, not used by core rm */
    spinlock_t bh_lock;
    unsigned long bh_pflags;

#if defined(NV_DEBUG_LOCKS)
    /* only used in debug builds to track who has the locks */
    unsigned long rm_locked;
    unsigned long rm_waiting;
    unsigned long ldata_locked;
    unsigned long ldata_waiting;
    unsigned long at_locked;
    unsigned long at_waiting;
    unsigned long bh_locked;
#endif

} nv_linux_state_t;


/*
 * poll(2) support for event notifiers
 *
 * A notifer can be turned into an "OS" event.
 * On UNIX, this means you can block using poll() while waiting for any
 * of your OS events to fire.
 *
 * When poll() on the NV file descriptor notices that you have one or more
 * events, it returns POLLPRI.  At that time it clears the count of fired
 * notifiers.
 *
 * Upon return from poll() just poll your notifiers.
 */

/*
 * Per-file private data structure.
 * We keep stuff here that is separate on a per-file basis.
 */

typedef struct {
    U032 any_fired_notifiers;
    void *nv_ptr;
} nv_file_private_t;

/* for the card devices */
#define NV_GET_NVL_FROM_FILEP(filep) \
    (nv_linux_state_t*) (((nv_file_private_t *)((filep)->private_data))->nv_ptr)

/* for the control device */
#define NV_GET_NV_FROM_FILEP(filep) \
    (nv_state_t*) (((nv_file_private_t *)((filep)->private_data))->nv_ptr)

#define NV_HIDE_IN_FILEP(filep, nvp) \
    (((nv_file_private_t *)((filep)->private_data))->nv_ptr = (void *) (nvp))
    

#define NV_GET_NVL_FROM_NV_STATE(nv) \
    ((nv_linux_state_t *) nv->os_state)

#define NV_STATE_PTR(nvl)   (&((nvl)->nv_state))

/* all of the below kernel calls could sleep.
 * if we sleep with the nv lock, we could hang an incoming interrupt
 * and lock the system. So, drop the lock, perform the call, and
 * regrab the lock.
 */

#define NV_VMALLOC(ptr, size) \
    { \
        (void *) ptr = vmalloc(size); \
    }

#define NV_VFREE(ptr) \
    { \
        vfree((void *) ptr); \
    }

/* only use this because GFP_KERNEL may sleep..
 * GFP_ATOMIC is ok, it won't sleep
 */
#define NV_KMALLOC(ptr, size) \
    { \
        (void *) ptr = kmalloc(size, GFP_KERNEL); \
    }

#define NV_KFREE(ptr) \
    { \
        kfree((void *) ptr); \
    }



#endif  /* __KERNEL__ */
#endif  /* _NV_LINUX_H_ */
