/*
 * ggiplay.c by Andreas Beck becka@ggi-project.org in 1999
 *
 *   This software is placed in the public domain and can be used freely
 *   for any purpose. It comes without any kind of warranty, either
 *   expressed or implied, including, but not limited to the implied
 *   warranties of merchantability or fitness for a particular purpose.
 *   Use it at your own risk. the author is not responsible for any damage
 *   or consequences raised by use or inability to use this program.
 *
 */
                 

#include <stdio.h>
#include <math.h>
#include <unistd.h>
#include <sys/time.h>

#include <ggi/ggi.h>
#include <ggi/errors.h>
#include "quicktime.h"

/* Get access to a directbuffer.
 */

unsigned char *GetDBwritebuf(ggi_visual_t *vis)
{
	int maxdb,x;
	const ggi_directbuffer *buf;

	maxdb=ggiDBGetNumBuffers(vis);
	for(x=0;x<maxdb;x++) {

		if (NULL==(buf=ggiDBGetBuffer(vis,x))) break;

		if (buf->type&GGI_DB_SIMPLE_PLB) {
			return buf->write;
		}

	}

	return NULL; /* Ouch - we shouldn't reach that ! */
}

/* Swap RGB->BGR (which most graphics cards use) "in-situ".
 */
void swaprgb(unsigned char *ptr,int length)
{
	unsigned char buf;
	while(length--) {
		buf=*ptr;
		*ptr=*(ptr+2);
		*(ptr+2)=buf;
		ptr+=3;
	}
}

/* Play a track of a quicktime movie in a preopened visual. 
 * Visual will be resized as  needed by the movie. Returns 
 * error codes, if something goes wrong.
 */
int ggiPlayQuicktimeTrack(ggi_visual_t vis, ggi_graphtype type, quicktime_t *file, int track)
{
	ggi_visual_t memvis;	/* We use a memvisual as decoding buffer. */
	ggi_mode suggested_mode;

	int width, height;	/* Some bits of information about the track. */
	long vidlen,curframe;
	double framerate;
	
	struct timeval start,stop;

	unsigned char **rowptrs;	/* A few helpers for getting at the QT data */
	unsigned char *hlpptr;
	int x;
	int stepping,autospeed;

	/* Can we play that track at all - or rather: Can the QT lib do so for us ? 
	 */
	if (!quicktime_supported_video(file,track)) {
		printf("Codec not supported by lib.\n");
		return GGI_EARGINVAL;
	}

	/* Gather some informative bits about the track.
	 */
	width=quicktime_video_width(file,track);
	height=quicktime_video_height(file,track);
	vidlen=quicktime_video_length(file,track);
	framerate=quicktime_frame_rate(file,track);
	if (fabs(framerate)>0.01) framerate=1.0/framerate;
	else framerate=1.0/25.0;

	ggiCheckGraphMode(vis,width,height,width,height,type,
		&suggested_mode);
	if (ggiSetMode(vis,&suggested_mode)) {
		return GGI_EARGINVAL;
	}

	/*Now first allocate a memvisual for decoding */
	if (NULL==(memvis=ggiOpen("memory",NULL,NULL)))
		return GGI_ENOMEM;

	ggiSetGraphMode(memvis,width,height,width,height, GT_24BIT);
	/* maybe once:	GT_CONSTRUCT(24, GT_TRUECOLOR | GT_SUB_REVERSE_ENDIAN , 24))); */
	{
		ggi_color black={0,0,0};
		ggi_color white={0xffff,0xffff,0xffff};
		
		ggiSetGCForeground(vis,ggiMapColor(vis,&white));
		ggiSetGCBackground(vis,ggiMapColor(vis,&black));
	}

	if (NULL==(rowptrs=malloc(height*sizeof(unsigned char *)))) {
		ggiClose(memvis);
		return GGI_ENOMEM;
	}
	if (NULL==(hlpptr=GetDBwritebuf(memvis))) {
		ggiClose(memvis);
		return GGI_EFATAL;
	}

	/* Set up the rowpointers - FIXME stride ! */
	for(x=0;x<height;x++) {
		rowptrs[x]=hlpptr+width*3*x;
	}

	stepping=1;
	autospeed=1;
	for(curframe=0;curframe<vidlen&&curframe>=0;curframe+=stepping) {

		/* Start Timing */
		gettimeofday(&start,NULL);

		/* Decode and show movie */
		quicktime_set_video_position(file,curframe,track);
		quicktime_decode_video(file,rowptrs,track);
		swaprgb(hlpptr,width*height);

		/* FIXME - use doublebuffering here */
		ggiCrossBlit(memvis,0,0,width,height,vis,0,0);
		ggiFlush(vis);

		/* Stop timing */
		gettimeofday(&stop,NULL);

		/* calculate remaining time. */
		start.tv_usec-=stop.tv_usec;
		start.tv_usec+=(framerate-floor(framerate))*1e6;
		start.tv_sec-=stop.tv_sec;
		start.tv_sec+=floor(framerate);
		while(start.tv_usec>=1000000) {
			start.tv_usec-=1000000;start.tv_sec++;
		}
		while(start.tv_usec<0) {
			start.tv_usec+=1000000;start.tv_sec--;
		}
		if (start.tv_sec<0) { 
			/* Ooops - we are slow */ 
			start.tv_sec=start.tv_usec=0; 
			ggiPutc(vis,0,0,'R');
			ggiFlushRegion(vis,0,0,8,8);
			if (autospeed) {
				int oldstep=stepping;
				if (oldstep) {
					stepping+= stepping>0 ? 1 : -1;
					framerate=framerate*(stepping/oldstep);
				}
			}
		}
		/* Fixme - read/act on some events here. */
		while(ggiEventPoll(vis,emKeyboard|emPointer,&start)>0) {
			x=ggiEventsQueued(vis,emKeyboard|emPointer);
			while(x--) {
				gii_event event;
				ggiEventRead(vis,&event,emKeyboard|emPointer);
				switch(event.any.type)  {
					case evKeyPress:
					case evKeyRepeat:
						switch(event.key.label) {
							case 'P': /* Play/Pause */
							case 'p':
								stepping=!stepping;
								break;
							case 'r': /* Rewind */
							case 'R':
							case GIIK_Up:
								curframe=0;
								break;
							case 'e': /* End */
							case 'E':
							case GIIK_Down:
								curframe=vidlen-1;
								break;
							case 'Q':
							case 'q':
							case GIIUC_Escape:
								curframe=-1;
								stepping=0;
								break;
							case 'f': /* forward */
							case 'F':
							case GIIK_Right:
								if (stepping) stepping=1;
								else curframe++;
								break;
							case 'b': /* backward */
							case 'B':
							case GIIK_Left:
								if (stepping) stepping=-1;
								else curframe--;
								break;
							case '+':
								if (framerate>=0.001) framerate-=0.001;
								else framerate=0.0;
								break;
							case '-':
								framerate+=0.001;
								break;
						}
						break;
				}
			}
		}
	}
	free(rowptrs);

	ggiClose(memvis);

	return 0;
}

int main(int argc, char *argv[])
{
	quicktime_t *file;
	int numtracks,curtrack;
	ggi_visual_t vis;
	
	if(argc < 2) {
		fprintf(stderr,"Need a movie.\n");
		exit(1);
	}

	if (!quicktime_check_sig(argv[1])) {
		fprintf(stderr,"This file doesn't look like a QT movie.\n");
		exit(1);
	}

	if(!(file = quicktime_open(argv[1], 1, 0)))
	{
		fprintf(stderr,"Open failed\n");
		exit(1);
	}

	numtracks=quicktime_video_tracks(file);

	ggiInit();

	if (NULL==(vis=ggiOpen(NULL))) {
		fprintf(stderr,"Cannot open default LibGGI target.\n");
	} else {
		ggiSetFlags(vis,GGIFLAG_ASYNC);
		for(curtrack=0;curtrack<numtracks;curtrack++) {
			ggiPlayQuicktimeTrack(vis,GT_AUTO,file,curtrack);
		}
		quicktime_close(file);
		ggiClose(vis);
	}

	ggiExit();
	return 0;
}
