/* this file written from scratch by Nathan Strong (gblues@gstis.net)
 * All code contained is under the GPL. See COPYING for details.
 *
 * Some modifications (c) 1999 Andreas Beck [becka@ggi-project.org]
 *
 * cmdline.c -- this file contains any/all functions related to parsing
 * the command-line options.
 */

#include "peggy.h"

extern struct OPTIONS Options;

void CheckCommandLine(int *argc, char **argv);
void DoOpt(char *command);
void SetBool(char *command, char *baseline, ggi_bool *option);
void SetResolution(char *command);
void SetTime(char *command);
void DoRes(char *command);

/* CheckCommandLine() -- searches through the command line for options,
 * processes the options, and re-organizes argv[] such that it contains
 * only files.
 */
void CheckCommandLine(int *argc, char **argv) {
	int i, numfiles = 0, numopts = 0;
	char *Files[*argc];
	char *Cmds[*argc];
	for (i = 1; i < *argc; i++) {
		if(argv[i][0] != '-')
			Files[numfiles++] = argv[i];
		else
			Cmds[numopts++] = argv[i];
	}

	for (i = 0; i < numopts; i++)
		DoOpt(strdup(Cmds[i]));

	for (i = 0; i < *argc; i++) {
		if (i < numfiles) 
			argv[i] = Files[i];
		else {
			*argc = i;
			break;
		}
	}
}

/* DoOpt() -- validates the option sent to it. If the option is invald, it
 * exits the program. Note that this function relies on the fact that each
 * valid command's first character is unique.
 */
void DoOpt(char *command) {
	char firstletter;
	static int res = 0, t = 0, h = 0, rnd = 0, ver = 0;

	while(*command == '-') command++;
	firstletter = *command;

	if ( (firstletter == 'g') && !(res) ) { 
		SetResolution(command); res++; 
	} else if ( (firstletter == 't') && !(t) ) { 
		SetTime(command); t++; 
	} else if ( (firstletter == 'h') && !(h) ) {
		 SetBool(command, "help", &Options.help); h++; 
	} else if ( (firstletter == 'r') && !(rnd) ) { 
		SetBool(command, "random", &Options.rnd); rnd++; 
	} else if ( (firstletter == 'v') && !(ver) ) { 
		SetBool(command, "version", &Options.version); ver++; 
	}
}

/* SetBool() -- toggles boolean options in the OPTIONS structure.
 * since the procedure is the same for all current boolean variables,
 * I merged them into this function.
 */
void SetBool(char *command, char *baseline, ggi_bool *option) {
	int len = strlen(command), i;
	if (len > strlen(baseline)) {
		printf("Invalid option: %s\n", command);
		exit(1);
	}
	for (i=0; i < len; i++) {
		if (command[i] != baseline[i]) {
			printf("Invalid Option: %s\n", command);
			exit(1);
		}
	}
	*option = 1;
}

/* DoRes() -- extracts the resolution from the --geometry parameter.
 * Note that this function will reject any --geometry resolutions >
 * 9999x9999. If for some insane reason you need a >10000 resolution,
 * this is the function to change. Also note that this function does
 * not validate that the given resolution is a legal one, this is done
 * in InitGraphics() (look in peggy.c)
 */
void DoRes(char *command) {
	char _x[5] = "", _y[5] = "";
	int j, _xindex = 0, _yindex = 0, xory = 0;	

	for(j = 0; j < strlen(command); j++) {
		if ( ( (_yindex < 5) && (xory) ) || 
		     ( (_xindex < 5) && !(xory) ) ) {
			if (isdigit(command[j]))
				if (xory) _y[_yindex++] = command[j];
				else _x[_xindex++] = command[j];
			else if ((command[j] == 'x') && !(xory))
				xory = 1;
			else {
				printf("Invalid resolution: %s\n", command);
				exit(1);
			}
		} else {
			printf("Invalid resolution: %s\n", command);
			exit(1);
		}
	}
	if((!xory) || !(_y[0])) {
		printf("Invalid resolution: %s\n", command);
		exit(1);
	}
	Options.xsize = atoi(_x);
	Options.ysize = atoi(_y);
}

/* SetResolution() -- this checks to make sure the --geometry command
 * is actually specified (i.e. so you can specify --geo=fooxbar instead
 * of --geometry=fooxbar). Once it verifies that yes, indeed, --geometry
 * was specified, it chops everything past the '=' off and sends it to
 * DoRes().
 */
void SetResolution(char *command) {
	char dummy[9] = "";
	char *baseline = "geometry";
	int i = 0, j;
	int len = strlen(command);

	while ( (i < len) && (command[i] != '=') )
		dummy[i] = command[i++];
	for (j=0; j < strlen(baseline); j++)
		if ((dummy[j] != baseline[j]) && (dummy[j])) {
			printf("Invalid option: %s\n", command);
			exit(1);
		}
	DoRes(command+i+1);
}

/* SetTime() -- this function verifies the --time parameter, and sets up
 * the delay accordingly. Far less complicated than SetResolution()
 * because everything after the '=' has to be a number (no bloody 'x' in
 * the middle).
 */
void SetTime(char *command) {
	char dummy[5] = "";
	char tstring[11] = "";
	char *baseline = "time";
	int i = 0, j, k = 0;
	int len = strlen(command);
	while ( (i < len) && (command[i] != '=') )
		dummy[i] = command[i++];
	if (i > 5) {
		printf("Invalid option: %s\n", command);
		exit(1);
	}
	for (j=0; j < strlen(baseline); j++)
		if ((dummy[j] != baseline[j]) && (dummy[j])) {
			printf("Invalid option: %s\n", command);
			exit(1);
		}
	for (j=++i; j < len; j++) {
		if(isdigit(command[j]))
			tstring[k++] = command[j];
	else	{
		printf("Invalid option: %s\n", command);
		exit(1);
		}
	}
	Options.timeout = atoi(tstring);
}

