/* Tab=4, Linewrap=off */

#include <stdio.h>
#include <unistd.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <dirent.h>
#include <ggi/ggi.h>

#include "gbm.h"
#include "ggv.h"

#include "logo.h"

static char select_help[][80]={
	" "," ",
	"up arrow ( also p ) \t\t move file selector to up",
	"down arrow ( also n ) \t move file selector down",
	"left arrow ( also l ) \t\t move file selector left",
	"right arrow ( also r ) \t move file selector right",
	"b \t\t\t\t move file selector to beginning",
	"e \t\t\t\t move file selector to end",
	" ",
	"enter \t\t\t display current file",
	" \t\t\t\t or change directory",
	" ",
	"h \t\t\t\t help page",
	"f \t\t\t\t toggle fullscreen select mode",
	"u \t\t\t\t update thumbnails",
	"c \t\t\t\t create collage",
	" ",
	"q \t\t\t\t quit ggv",
	""
};

/* Private */
static void version(void);
static void usage(void);
static int  main_loop(void);
static int  set_default_mode(void);
static int  set_default_palette(void);
static int  show_screen(void);
static int  read_directory(void);
static void free_directory(void);
static int  show_directory(int startfrom, int unshow);
static int  show_bar(int current, int selected, int startfrom);
static void show_select_help(void);
static void prettyfile(char *buf, struct filelist *gifptr);
static int  centertxt(int x,int fsiz,char *str);
static struct filelist *ll_move(int count);
static void draw_logo(int a, int b);

int main(int argc, char **argv)
{
	int x, f, total=0;
	int ret;
	first_file=NULL;

	/* Check command line args */
	while( -1 != (x=getopt(argc,argv,"vh")) )
		{
		switch(x)
			{
			case 'v':
				version();
				return EXIT_SUCCESS;
				break;
			case 'h':
				usage();
				return EXIT_SUCCESS;
				break;
			case '?':
				return EXIT_FAILURE;
				break;
			default:
				fprintf(stdout,"Unknown error. Exiting.\n");
				return EXIT_FAILURE;
				break;
			}
		}
	for(x=optind;x<argc;x++)
		total++;

	/* Read in the config file */
	if( GGV_ERROR == (ret=read_config()) )
		return EXIT_FAILURE;

	/* Set things up */
	ggiInit();
	vis=ggiOpen(NULL);
	set_default_mode();
	set_default_palette();
		
	/* If files on command line */
	if(total>0)
		{
		for(f=optind;f<argc;f++)
			view_file(argv[f]);
		}
	/* Else normal startup */
	else
		main_loop();

	ggiFlush(vis);
	ggiClose(vis);
	ggiExit();

return EXIT_SUCCESS;
}

static void version(void)
{
	fprintf(stdout,"ggv : Version %s\n",GGV_VERSION);
	fprintf(stdout,"Copyright (c) 1997-1998 Willie Daniel <gith@quicklink.net> \n\n");
}

static void usage(void)
{
	version();
	fprintf(stdout,"Usage: ggv -vh [image1...imageN]\n\n");
}

static int main_loop(void)
{
	int key;
	int current;
	int oldcurrent;
	int startfrom;
	int oldstart;
	int quit;
	struct filelist *current_file;

	startfrom=oldstart=current=oldcurrent=1;
	quit=0;

	show_screen();
	read_directory();
	show_directory(startfrom,0);
	show_bar(current,1,startfrom);

	while(!quit)
		{
		oldcurrent=current;
		key=ggiGetc(vis);
		switch(GII_KTYP(key))
			{
			case GII_KT_LATIN1:
				switch(GII_KVAL(key))
					{
					case 'f': /* toggle fullscreen select */
						cfg.select_fullscreen=!cfg.select_fullscreen;
						show_screen();
						show_directory(startfrom,0);
						show_bar(current,1,startfrom);
						quit=0;
						break;
					case 'h': /* show help screen */
						show_select_help();
						show_screen();
						show_directory(startfrom,0);
						show_bar(current,1,startfrom);
						quit=0;
						break;
					case 'u': /* update thumbnails */
						quit=0;
						break;
					case 'c': /* collage */
						collage();
						show_screen();
						read_directory();
						show_directory(startfrom,0);
						show_bar(current,1,startfrom);
						quit=0;
						break;
					case 'q': /* quit */
						quit=1;
						break;
					case 'n': /* next file */
						current++;
						if(current>totalpics)
							current=totalpics;
						quit=0;
						break;
					case 'p': /* previous file */
						current--;
						if(current<1)
							current=1;
						quit=0;
						break;
					case 'l': /* move left */
						current-=YSIZ;
						if(current<1)
							current=1;
						quit=0;
						break;
					case 'r': /* move right */
						current+=YSIZ;
						if(current>totalpics)
							current=totalpics;
						quit=0;
						break;
					case 'b': /* goto beginning */
						current=1;
						quit=0;
						break;
					case 'e': /* goto end */
						current=totalpics;
						quit=0;
						break;
					case GIIUC_Return:
						goto is_enter;
					}
				break;
			default:
				switch(key)
					{
					case GIIK_Enter:
						is_enter:
						current_file=ll_move(current);
						if(current_file->isdir)
							{
							chdir(current_file->name);
							startfrom=oldstart=current=oldcurrent=1;
							read_directory();
							show_screen();
							show_directory(startfrom,0);
							show_bar(current,1,startfrom);
							}
						else
							{
							view_file(current_file->name);
							set_default_palette();
							show_screen();
							show_directory(startfrom,0);
							show_bar(current,1,startfrom);
							}
						quit=0;
						break;
					case GIIK_Down: /* next file - same as "n" */
						current++;
						if(current>totalpics)
							current=totalpics;
						quit=0;
						break;
					case GIIK_Up: /* previous file - same as "p" */
						current--;
						if(current<1)
							current=1;
						quit=0;
						break;
					case GIIK_Left: /* move left - same as "l" */
						current-=YSIZ;
						if(current<1)
							current=1;
						quit=0;
						break;
					case GIIK_Right: /* move right - same as "r" */
						current+=YSIZ;
						if(current>totalpics)
							current=totalpics;
						quit=0;
						break;
					case GIIK_PageUp:
						current-=YSIZ*3;
						if(current<1)
							current=1;
						quit=0;
						break;
					case GIIK_PageDown:
						current+=YSIZ*3;
						if(current>totalpics)
							current=totalpics;
						quit=0;
						break;
					}
				break;
			}
		oldstart=startfrom;
		while(current<startfrom)
			startfrom-=YSIZ;
		while(fwinxpos(current-startfrom+1)+BARWIDTH>RMARGIN)
			startfrom+=YSIZ;
		if(startfrom<1)
			startfrom=1;
		if(startfrom!=oldstart)
			{
			show_bar(oldcurrent,0,oldstart);
			show_directory(oldstart,1);
			show_directory(startfrom,0);
			show_bar(current,1,startfrom);
			}
		else
			if(oldcurrent!=-1 && (current!=oldcurrent))
				{
				show_bar(oldcurrent,0,startfrom);
				show_bar(current,1,startfrom);
				}
		ggiFlush(vis);
		}

return GGV_NOERROR;
}

static int set_default_mode(void)
{
	int err;

	err=ggiSetGraphMode(vis,cfg.default_width,cfg.default_height,
                            cfg.default_width,cfg.default_height,
                            cfg.default_bpp);
	if(err)
		ggiPanic("Can't set default screen mode.\n");

return GGV_NOERROR;
}

static int set_default_palette(void)
{
    ggiSetPalette(vis,0,256,default_pal);

    idx_light=ggiMapColor(vis,&default_pal[216]);
    idx_medium=ggiMapColor(vis,&default_pal[217]);
    idx_dark=ggiMapColor(vis,&default_pal[218]);
    idx_black=ggiMapColor(vis,&default_pal[219]);
    idx_marked=ggiMapColor(vis,&default_pal[220]);

return GGV_NOERROR;
}

static int show_screen(void)
{
	ggiFlush(vis);
	ggiSetGCBackground(vis,idx_medium);
	ggiSetGCForeground(vis,idx_medium);
	ggiFillscreen(vis);

	barheight=GDFSIZ*6+6+70;
	yofs=(cfg.select_fullscreen?10:110);

	if(cfg.select_fullscreen)
		{
		draw3dbox(0, 0, cfg.default_width - 1, cfg.default_height - 1, 2, 1, idx_light, idx_dark);
		draw3dbox(10, 10, cfg.default_width - 11, cfg.default_height - 11, 1, 0, idx_light, idx_dark);
		}
	else
		{
		draw3dbox(0, 0, cfg.default_width - 1, 99, 2, 1, idx_light, idx_dark);
		draw3dbox(10, 10, cfg.default_width - 11, 89, 1, 0, idx_light, idx_dark);
		draw3dbox(0, 100, cfg.default_width - 1, cfg.default_height - 1, 2, 1, idx_light, idx_dark);
		draw3dbox(10, 110, cfg.default_width - 11, cfg.default_height - 11, 1, 0, idx_light, idx_dark);
		ggiSetGCForeground(vis,idx_black);
		draw_logo( (cfg.default_width-logow) / 2, 10);
		ggiSetGCForeground(vis,idx_medium);
		}

	ggiFlush(vis);

return GGV_NOERROR;
}

static int read_directory(void)
{
	DIR *dp;
	struct dirent *ep;
	struct stat statbuf;
	struct filelist *new_file, *current_file, *previous_file=NULL;
	int isdir;
	int l;

	totalpics=0;

	if(NULL != first_file)
		free_directory();

	dp = opendir (".");
	if(NULL == dp)
		return GGV_ERROR;
	while((ep=readdir(dp)))
		{
		if(ep->d_name[0]=='.' && ep->d_name[1]!='.')
			continue;
		if((stat(ep->d_name,&statbuf))==-1)
			statbuf.st_mode=0;
		isdir=S_ISDIR(statbuf.st_mode);

/*XXX This is nasty. Get rid of it. */

		if(((l=strlen(ep->d_name))>4) || isdir)
			if((!strcasecmp(ep->d_name+l-4,".gif")) ||
(!strcasecmp(ep->d_name+l-4,".bmp")) ||
(!strcasecmp(ep->d_name+l-4,".vga")) ||
(!strcasecmp(ep->d_name+l-4,".bga")) ||
(!strcasecmp(ep->d_name+l-4,".rle")) ||
(!strcasecmp(ep->d_name+l-4,".dib")) ||
(!strcasecmp(ep->d_name+l-4,".rl4")) ||
(!strcasecmp(ep->d_name+l-4,".rl8")) ||
(!strcasecmp(ep->d_name+l-4,".pcx")) ||
(!strcasecmp(ep->d_name+l-4,".pcc")) ||
(!strcasecmp(ep->d_name+l-4,".pgm")) ||
(!strcasecmp(ep->d_name+l-4,".ppm")) ||
(!strcasecmp(ep->d_name+l-4,".tga")) ||
(!strcasecmp(ep->d_name+l-4,".vst")) ||
(!strcasecmp(ep->d_name+l-4,".afi")) ||
(!strcasecmp(ep->d_name+l-4,".tif")) ||
(!strcasecmp(ep->d_name+l-5,".tiff")) ||
(!strcasecmp(ep->d_name+l-4,".xbm")) ||
(!strcasecmp(ep->d_name+l-4,".iff")) ||
(!strcasecmp(ep->d_name+l-4,".lbm")) ||
(!strcasecmp(ep->d_name+l-4,".jpg")) ||
(!strcasecmp(ep->d_name+l-4,".jpeg")) ||
			isdir)
			{
			totalpics++;
			new_file=calloc(1,sizeof(struct filelist));
			if(NULL == new_file)
				return GGV_ERROR;
			strcpy(new_file->name,ep->d_name);
			new_file->isdir=isdir;
			new_file->xvw=0;
			new_file->xvh=0;
			new_file->marked=0;
			new_file->next=NULL;
			/* Now sort the list */
			if(NULL == first_file)
				first_file = new_file;
			else
				{
				current_file = first_file;
				while(current_file != NULL && \
					strcmp(new_file->name,current_file->name) >0)
					{
					previous_file=current_file;
					current_file=current_file->next;
					}
				if(current_file == first_file)
					first_file=new_file;
				else previous_file->next=new_file;
					new_file->next=current_file;
				}
			}
		}

	closedir(dp);

return GGV_NOERROR;
}

static void free_directory(void)
{
	struct filelist *current_file, *next_file;

	current_file=first_file;
	while(NULL != current_file)
		{
		next_file=current_file->next;
		free(current_file);
		current_file=next_file;
		}
	first_file = NULL;
}

static int show_directory(int startfrom, int unshow)
{
	char cdir[1024];
	char ctmp[1024];
	struct filelist *current_file, *next_file;
	int xpos,ypos,xt,w,h,y;
	int loopcount;
	unsigned char *image;

	getcwd(cdir,1023);
	sprintf(ctmp,"Directory of %s",cdir);

	set_max_text_width(560);
	if(unshow)
		undrawtext3d(40,yofs+25,4,ctmp);
	else
		drawtext3d(40,yofs+25,4,ctmp,1, idx_light,idx_dark,idx_black);

	set_max_text_width(BAR_RESTRICT_WIDTH);

	current_file=ll_move(startfrom);
	loopcount=startfrom;
	while(NULL != current_file)
		{
		next_file=current_file->next;

		xpos=fwinxpos(loopcount-startfrom+1);
		if(xpos+BARWIDTH>RMARGIN) break;
		ypos=fwinypos(loopcount-startfrom+1);

		prettyfile(ctmp,current_file);
		xt=centertxt(xpos,GDFSIZ,ctmp);
		ggiSetGCForeground(vis,unshow?idx_medium:(current_file->marked?idx_marked:idx_black));
		vgadrawtext(xt,ypos+3,GDFSIZ,ctmp);
		ggiSetGCForeground(vis,unshow?idx_medium:idx_black);

		if(unshow)
			{
			image=malloc(96);
			if(image!=NULL)
				{
				memset(image,idx_medium,96);
				for(y=-2;y<62;y++)
				ggiPutHLine(vis,xpos+(BARWIDTH-80)/2-2,ypos+y+GDFSIZ*6+9,96,image);
				free(image);
				}
			}
		else
			{
			if(current_file->isdir)
				{
				/* 'folder' icon */
				ggiSetGCForeground(vis,unshow?idx_medium:idx_black);
				xt=xpos+(BARWIDTH-80)/2;
				ypos+=GDFSIZ*6+9;
				/* main bit */
				ggiDrawLine(vis,xt+10,ypos+50,xt+70,ypos+50);
				ggiDrawLine(vis,xt+70,ypos+50,xt+70,ypos+20);
				ggiDrawLine(vis,xt+70,ypos+20,xt+65,ypos+15);
				ggiDrawLine(vis,xt+65,ypos+15,xt+15,ypos+15);
				ggiDrawLine(vis,xt+15,ypos+15,xt+10,ypos+20);
				ggiDrawLine(vis,xt+10,ypos+20,xt+10,ypos+50);
				/* top bit */
				ggiDrawLine(vis,xt+15,ypos+15,xt+20,ypos+10);
				ggiDrawLine(vis,xt+20,ypos+10,xt+35,ypos+10);
				ggiDrawLine(vis,xt+35,ypos+10,xt+40,ypos+15);
				ypos-=GDFSIZ*6+9;
				current_file->xvw=w=80; current_file->xvh=h=60;
				}
			else
				{
				/* default icon */
				ggiSetGCForeground(vis,unshow?idx_medium:idx_black);
				xt=xpos+(BARWIDTH-80)/2;
				ypos+=GDFSIZ*6+9;
				/* main bit */
				ggiDrawLine(vis,xt+20,ypos+50,xt+60,ypos+50);
				ggiDrawLine(vis,xt+60,ypos+50,xt+60,ypos+20);
				ggiDrawLine(vis,xt+60,ypos+20,xt+50,ypos+10);
				ggiDrawLine(vis,xt+50,ypos+10,xt+20,ypos+10);
				ggiDrawLine(vis,xt+20,ypos+10,xt+20,ypos+50);
				/* 'folded' bit */
				ggiDrawLine(vis,xt+50,ypos+10,xt+50,ypos+20);
				ggiDrawLine(vis,xt+50,ypos+20,xt+60,ypos+20);
				ypos-=GDFSIZ*6+9;
				current_file->xvw=w=80; current_file->xvh=h=60;
				}
			if(current_file->xvw!=0)
				{
				draw3dbox(xpos+(BARWIDTH-w)/2-1,ypos+GDFSIZ*6+38-h/2,
	                            xpos+(BARWIDTH-w)/2+w,ypos+GDFSIZ*6+39-h/2+h,1,0,
	                            idx_light,idx_dark);
				}
			}
		loopcount++;
		current_file=next_file;
		}

	set_max_text_width(NO_CLIP_FONT);

	ggiFlush(vis);

return GGV_NOERROR;
}

static int show_bar(int current, int selected, int startfrom)
{
	struct filelist *current_file;
	int xpos,ypos,xt;
	char ctmp[1024];

	current_file=ll_move(current);
	strcpy(ctmp,current_file->name);

	xpos=fwinxpos(current-startfrom+1);
	if((xpos<1)||(xpos+BARWIDTH>RMARGIN))
		return GGV_ERROR;
	ypos=fwinypos(current-startfrom+1);
	prettyfile(ctmp,current_file);
	set_max_text_width(BAR_RESTRICT_WIDTH);
	xt=centertxt(xpos,GDFSIZ,ctmp);

	if(selected)
		{
		draw3dbox(xpos,ypos,xpos+BARWIDTH-1,ypos+barheight-1,1,1,idx_light,idx_dark);
		drawtext3d(xt,ypos+3,GDFSIZ,ctmp,0, idx_light,idx_dark,
                   current_file->marked?idx_marked:idx_black);
		/* box if cfg.xvpic_index and is an xvpic being used */
		if(current_file->xvw!=0)
			draw3dbox(xpos+(BARWIDTH-current_file->xvw)/2-2,
                      ypos+GDFSIZ*6+39-current_file->xvh/2-2,
                      xpos+(BARWIDTH-current_file->xvw)/2+current_file->xvw+1,
                      ypos+GDFSIZ*6+39-current_file->xvh/2+current_file->xvh+1,
                      1,0, idx_light,idx_dark);
		}
	else
		{
		undraw3dbox(xpos,ypos,xpos+BARWIDTH-1,ypos+barheight-1,1);
		undrawtext3d(xt,ypos+3,GDFSIZ,ctmp);
		ggiSetGCForeground(vis,current_file->marked?idx_marked:idx_black);
		vgadrawtext(xt,ypos+3,GDFSIZ,ctmp);
		/* undraw box if cfg.xvpic_index and is an xvpic being used */
		if(current_file->xvw!=0)
			undraw3dbox(xpos+(BARWIDTH-current_file->xvw)/2-2,
                        ypos+GDFSIZ*6+39-current_file->xvh/2-2,
                        xpos+(BARWIDTH-current_file->xvw)/2+current_file->xvw+1,
                        ypos+GDFSIZ*6+39-current_file->xvh/2+current_file->xvh+1,
                        1);
		}

	set_max_text_width(NO_CLIP_FONT);

	ggiFlush(vis);

return GGV_NOERROR;
}

static void show_select_help(void)
{
	int x;

	ggiFlush(vis);
	ggiSetGCBackground(vis,idx_medium);
	ggiSetGCForeground(vis,idx_medium);
	ggiFillscreen(vis);

	draw3dbox(0, 0, cfg.default_width - 1, cfg.default_height - 1, 2, 1, idx_light, idx_dark);
	draw3dbox(10, 10, cfg.default_width - 11, cfg.default_height - 11, 1, 0, idx_light, idx_dark);

	drawtext3d( (cfg.default_width/3),15,5,"GGV Help Page",0,idx_light,idx_black,idx_black);

	x=0;
	while(strlen(select_help[x])>0)
		{
		x++;
		drawtext3d(20,30+x*20,3,select_help[x],0,idx_light,idx_black,idx_black);
		}

	ggiFlush(vis);

	ggiGetc(vis);
}

static void prettyfile(char *buf, struct filelist *gifptr)
{
	if(gifptr->isdir)
		{
		buf[0]='(';
		strcpy(buf+1,gifptr->name);
		strcat(buf,")");
		}
	else
		strcpy(buf,gifptr->name);
}

static int centertxt(int x,int fsiz,char *str)
{
	int a;
	a=vgatextsize(fsiz,str);

return(x+(BARWIDTH-a)/2);
}

static struct filelist *ll_move(int count)
{
    struct filelist *current_file, *next_file;
	int x;

    current_file=first_file;
	for(x=1;x!=count;x++)
        {
        next_file=current_file->next;
        current_file=next_file;
        }

return current_file;
}

void draw_logo(int a, int b)
{
	int x,y,bw,c=0;
	byte *ptr;

	ptr=ggvlogo;
	bw=logow>>3;
	if((logow%8)>0) bw+=1;
	for(y=0;y<logoh;y++)
		{
		ptr=ggvlogo+y*bw;
		for(x=0;x<logow;x++)
			{
			if((x%8)==0)
				c=*ptr;
			if((c&128)==0)
				ggiDrawPixel(vis,a+x,b+y);
			c<<=1;
			c&=0xff;
			if((x%8)==7)
				ptr++;
			}
		}

	ggiFlush(vis);
}
