/* Tab=4, Linewrap=off */

#include <stdio.h>
#include <unistd.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <dirent.h>
#include <ctype.h>
#include <ggi/ggi.h>

#include "gbm.h"
#include "ggv.h"

#include "colors_x.h"
#include "colors_n.h"

/* Public */
int read_config(void);

/* Private */
static char *strlwr(char *string);
static char *str_trimsp(char *str);

#define GGV_LINE_LEN_MAX    128                  /* actual max line length  */
#define GGV_BUFFERSIZE      GGV_LINE_LEN_MAX +2  /* ... including \n and \0 */

static struct cfg_strings {
	char *name;
	char *data;
} cfg_data[] = {
	{ "default_width", NULL },
	{ "default_height", NULL },
	{ "default_bpp", NULL },
	{ "select_fullscreen", NULL },
	{ "color_light", NULL },
	{ "color_medium", NULL },
	{ "color_dark", NULL },
	{ "color_black", NULL },
	{ "color_marked", NULL },
	{ NULL, NULL }
};

static char *strlwr(char *string)
{
    char *s;

    if (string)
        {
        for (s = string; *s; ++s)
            *s = tolower(*s);
        }

return string;
}

static char *str_trimsp(char *str)
{
    char *ibuf, *obuf;
    if(str)
        {
        for(ibuf = obuf = str; *ibuf;)
            {
            while (*ibuf && (isspace (*ibuf)))
                ibuf++;
            while (*ibuf && (!isspace (*ibuf)))
                *(obuf++) = *(ibuf++);
            }
        *obuf = '\0';
        }
return str;
}

static int read_cfg_file( char *filename, struct cfg_strings *cfg_info )
{
    char buffer[ GGV_BUFFERSIZE ];

    char *work_ptr;
    char *cfg_name;
    char *cfg_data;

    struct cfg_strings * cfg;
    FILE *cfg_file;

    cfg_file = fopen( filename, "r" );
    if( NULL == cfg_file )
        {
        fprintf(stderr,"Error: Failed to open %s for reading.\n", filename);
        return 1;
        }

    while( NULL != fgets( buffer, GGV_BUFFERSIZE, cfg_file ))
        {
        /* clip off optional comment tail indicated by a semi-colon */
        if( NULL != (work_ptr = strchr( buffer, '#' )))
            *work_ptr = '\0';
        else
            work_ptr = buffer + strlen( buffer );

        /* clip off trailing and leading white space */
        work_ptr--;
        while( isspace( *work_ptr ) && work_ptr >= buffer )
            *work_ptr-- = '\0';
        work_ptr = buffer;
        while( isspace( *work_ptr ))
            work_ptr++;
        if( 0 == strlen( work_ptr ))
            continue;

        cfg_name = strtok( work_ptr, " =" );
        if( NULL != cfg_name )
            {
            /* Condition the name (lower case required),
            ** and strip leading white and a 'late' = from data part.
            */
            strlwr( cfg_name );
            cfg_data = strtok( NULL, "" );
            while( isspace( *cfg_data ))
                cfg_data++;
            if( '=' == *cfg_data )
                cfg_data++;
            while( isspace( *cfg_data ))
                cfg_data++;

			str_trimsp(cfg_data);

            /* look for matching 'name' */
            cfg = cfg_info ;
            while( NULL != cfg->name && 0 != strcmp( cfg->name, cfg_name ))
                cfg++;

            /* duplicate the data if the name is found. */
            if( NULL != cfg->name )
                {
                cfg->data = strdup( cfg_data ); /* strdup is not ANSI    */
                                            /* memory leaks if Cfg->data */
                                            /* is malloc'ed already      */
                if( NULL == cfg->data )
                {
                    fclose( cfg_file );
                    return 1;
                }
            }   /* undetected error on failure should not be a problem  */
        }       /* but test and handle it anyway ...                    */
    }
    fclose( cfg_file );
    return 0;
}

int read_config(void)
{
	char cfgfile[1024];
	struct stat statbuf;
    int i;
	int x;
	int foundit=0;
	int tempbpp;

	strcpy(cfgfile,getenv("HOME"));
	strcat(cfgfile,"/.ggvrc");
	if( !stat(cfgfile,&statbuf) )
		read_cfg_file(cfgfile, cfg_data );
	else if( !stat("/etc/ggv.conf",&statbuf) )
		read_cfg_file("/etc/ggv.conf", cfg_data );
	else if( !stat("/usr/local/etc/ggv.conf",&statbuf) )
		read_cfg_file("/usr/local/etc/ggv.conf", cfg_data );
    else if( !stat("/usr/local/lib/ggv.conf",&statbuf) )
        read_cfg_file("/usr/local/lib/ggv.conf", cfg_data );
	else
		{
		fprintf(stderr,"Error: Cannot find configuration file.\n");
		return GGV_ERROR;
		}

	/* Quick check for something not set or missing */
    for( x = 0; x < 9 ; x++ )
		{
		if(!strlen(cfg_data[x].data))
			{
			fprintf(stderr,"Error: %s= information not set.\n",cfg_data[x].name);
			return GGV_ERROR;
			}
		}

	/* Set default_width */
	for( x = 0; x < 9 ; x++ )
		{
		if(!strcmp(cfg_data[x].name,"default_width") )
			{
			sscanf(cfg_data[x].data,"%d",&cfg.default_width);
			if(cfg.default_width<640)
				{
				fprintf(stderr,"Error: %s=%s : Width must be at least 640\n",cfg_data[x].name,cfg_data[x].data);
				return GGV_ERROR;
				}
			}
		}

	/* Set default_height */
	for( x = 0; x < 9 ; x++ )
		{
		if(!strcmp(cfg_data[x].name,"default_height") )
			{
			sscanf(cfg_data[x].data,"%d",&cfg.default_height);
			if(cfg.default_height<480)
				{
				fprintf(stderr,"Error: %s=%s : Height must be at least 480\n",cfg_data[x].name,cfg_data[x].data);
				return GGV_ERROR;
				}
			}
		}

	/* Set default_bpp */
	for( x = 0; x < 9 ; x++ )
		{
		if(!strcmp(cfg_data[x].name,"default_bpp") )
			{
			sscanf(cfg_data[x].data,"%d",&tempbpp);
			switch(tempbpp)
				{
				case 8:
					cfg.default_bpp=GT_8BIT;
					break;
				case 15:
					cfg.default_bpp=GT_15BIT;
					break;
				case 16:
					cfg.default_bpp=GT_16BIT;
					break;
				case 24:
					cfg.default_bpp=GT_24BIT;
					break;
				default:
					fprintf(stderr,"Error: %s=%s : invalid value.\n",cfg_data[x].name,cfg_data[x].data);
					return GGV_ERROR;
					break;
				}
			}
		}

	/* Set select_fullscreen */
	for( x = 0; x < 9 ; x++ )
		{
		if(!strcmp(cfg_data[x].name,"select_fullscreen") )
			{
			if( !strcmp(cfg_data[x].data,"on") )
				cfg.select_fullscreen=1;
			else if( !strcmp(cfg_data[x].data,"off") )
				cfg.select_fullscreen=0;
			else
				{
				fprintf(stderr,"Error: %s= : Must be either Off or On\n",cfg_data[x].name);
				return GGV_ERROR;
				}
			}
		}


	/* Copy color names */
	for( x = 0; x < 9 ; x++ )
		{
		if(!strcmp(cfg_data[x].name,"color_light") )
			strcpy(cfg.color_light,cfg_data[x].data);
		}
	for( x = 0; x < 9 ; x++ )
		{
		if(!strcmp(cfg_data[x].name,"color_medium") )
			strcpy(cfg.color_medium,cfg_data[x].data);
		}
	for( x = 0; x < 9 ; x++ )
		{
		if(!strcmp(cfg_data[x].name,"color_dark") )
			strcpy(cfg.color_dark,cfg_data[x].data);
		}
	for( x = 0; x < 9 ; x++ )
		{
		if(!strcmp(cfg_data[x].name,"color_black") )
			strcpy(cfg.color_black,cfg_data[x].data);
		}
	for( x = 0; x < 9 ; x++ )
		{
		if(!strcmp(cfg_data[x].name,"color_marked") )
			strcpy(cfg.color_marked,cfg_data[x].data);
		}

	/* Find and set the configurable colors */
	for(i = 0; i < X_COLORS; i++)
		{
		if(!strcmp(cfg.color_light,x_colors[i].name))
			{
			foundit=1;
			default_pal[216].r=x_colors[i].red;
			default_pal[216].g=x_colors[i].green;
			default_pal[216].b=x_colors[i].blue;
			break;
			}
		}
	if(!foundit)
		{
		fprintf(stderr,"Error: Unknown color: %s\n",cfg.color_light);
		return GGV_ERROR;
		}
	foundit=0;

	for(i = 0; i < X_COLORS; i++)
		{
		if(!strcmp(cfg.color_medium,x_colors[i].name))
			{
			foundit=1;
			default_pal[217].r=x_colors[i].red;
			default_pal[217].g=x_colors[i].green;
			default_pal[217].b=x_colors[i].blue;
			break;
			}
		}
	if(!foundit)
		{
		fprintf(stderr,"Error: Unknown color: %s\n",cfg.color_medium);
		return GGV_ERROR;
		}
	foundit=0;

	for(i = 0; i < X_COLORS; i++)
		{
		if(!strcmp(cfg.color_dark,x_colors[i].name))
			{
			foundit=1;
			default_pal[218].r=x_colors[i].red;
			default_pal[218].g=x_colors[i].green;
			default_pal[218].b=x_colors[i].blue;
			break;
			}
		}
	if(!foundit)
		{
		fprintf(stderr,"Error: Unknown color: %s\n",cfg.color_dark);
		return GGV_ERROR;
		}
	foundit=0;

	for(i = 0; i < X_COLORS; i++)
		{
		if(!strcmp(cfg.color_black,x_colors[i].name))
			{
			foundit=1;
			default_pal[219].r=x_colors[i].red;
			default_pal[219].g=x_colors[i].green;
			default_pal[219].b=x_colors[i].blue;
			break;
			}
		}
	if(!foundit)
		{
		fprintf(stderr,"Error: Unknown color: %s\n",cfg.color_black);
		return GGV_ERROR;
		}
	foundit=0;

	for(i = 0; i < X_COLORS; i++)
		{
		if(!strcmp(cfg.color_marked,x_colors[i].name))
			{
			foundit=1;
			default_pal[220].r=x_colors[i].red;
			default_pal[220].g=x_colors[i].green;
			default_pal[220].b=x_colors[i].blue;
			break;
			}
		}
	if(!foundit)
		{
		fprintf(stderr,"Error: Unknown color: %s\n",cfg.color_marked);
		return GGV_ERROR;
		}
	foundit=0;

	/* Set the default colors */
	for(i=0;i<N_COLORS;i++)
		{
		default_pal[i].r=( n_colors[i].red + 1) * 256;
		default_pal[i].g=( n_colors[i].green + 1) * 256;
		default_pal[i].b=( n_colors[i].blue + 1) * 256;
		}

return GGV_NOERROR;
}
