/*****************************************************************************
*
* gbm.h - General Bitmap Module
*
*****************************************************************************/

#ifndef GBM_H
#define	GBM_H

#define GBM_VERSION "1.0.10"

#ifndef BOOLEAN_DEFINED
#define	BOOLEAN_DEFINED
typedef	int GBM_BOOLEAN;
#define GBM_FALSE 0
#define	GBM_TRUE  !GBM_FALSE
#endif

#ifndef BASICTYPES_DEFINED
#define	BASICTYPES_DEFINED

#if defined(linux) || defined(__linux) || defined(__linux__)
	typedef signed char         gbm_sint8;
	typedef unsigned char       gbm_uint8;
	typedef signed short        gbm_sint16;
	typedef unsigned short      gbm_uint16;
	typedef signed int          gbm_sint32;
	typedef unsigned int        gbm_uint32;
#else
	#error  Please define system dependent data types.
#endif

typedef gbm_uint8  byte;
typedef gbm_uint16 word;
typedef gbm_uint32 dword;

#endif /* BASICTYPES_DEFINED */


/* Errors */
typedef int GBM_ERR;
#define	GBM_ERR_OK          ((GBM_ERR) 0)
#define	GBM_ERR_MEM         ((GBM_ERR) 1)
#define	GBM_ERR_NOT_SUPP    ((GBM_ERR) 2)
#define	GBM_ERR_BAD_OPTION  ((GBM_ERR) 3)
#define	GBM_ERR_NOT_FOUND   ((GBM_ERR) 4)
#define	GBM_ERR_BAD_MAGIC   ((GBM_ERR) 5)
#define	GBM_ERR_BAD_SIZE    ((GBM_ERR) 6)
#define	GBM_ERR_READ        ((GBM_ERR) 7)
#define	GBM_ERR_WRITE       ((GBM_ERR) 8)
#define	GBM_ERR_BAD_ARG     ((GBM_ERR) 9)


/* Flags */
#define	GBM_FT_R1       0x0001
#define	GBM_FT_R4       0x0002
#define	GBM_FT_R8       0x0004
#define	GBM_FT_R24      0x0008
#define	GBM_FT_W1       0x0010
#define	GBM_FT_W4       0x0020
#define	GBM_FT_W8       0x0040
#define	GBM_FT_W24      0x0080

typedef struct {
    char *short_name;       /* Eg: "Targa"                       */
    char *long_name;        /* Eg: "Truevision Targa / Vista"    */
    char *extensions;       /* Eg: "TGA VST"                     */
    int flags;              /* What functionality exists         */
} GBMFT;

typedef struct {
	byte r;
	byte g;
	byte b;
} GBMRGB;

#define	PRIV_SIZE 2000

typedef struct {
    int w, h, bpp;          /* Bitmap dimensions                 */
    byte priv[PRIV_SIZE];   /* Private internal buffer           */
} GBM;

/*****************************************************************************
*
* gbm.c
*
*****************************************************************************/

extern GBM_ERR  gbm_init(void);
extern GBM_ERR  gbm_deinit(void);

extern GBM_ERR  gbm_io_setup(int  (*open  )(const char *fn, int mode),
                             int  (*create)(const char *fn, int mode),
                             void (*close )(int fd),
                             long (*lseek )(int fd, long pos, int whence),
                             int  (*read  )(int fd, void *buf, int len),
                             int  (*write )(int fd, const void *buf, int len)
                             );

extern int   gbm_io_open  (const char *fn, int mode);
extern int   gbm_io_create(const char *fn, int mode);
extern void  gbm_io_close (int fd);
extern long  gbm_io_lseek (int fd, long pos, int whence);
extern int   gbm_io_read  (int fd, void *buf, int len);
extern int   gbm_io_write (int fd, const void *buf, int len);

extern GBM_ERR  gbm_query_n_filetypes(int *n_ft);
extern GBM_ERR  gbm_query_filetype(int ft, GBMFT *gbmft);
extern GBM_ERR  gbm_guess_filetype(const char *fn, int *ft);

extern GBM_ERR gbm_read_header(const char *fn,
                               int fd, int ft,
                               GBM *gbm,
                               const char *opt
                               );

extern GBM_ERR gbm_read_palette(int fd, int ft, GBM *gbm, GBMRGB *gbmrgb);
extern GBM_ERR gbm_read_data(int fd, int ft, GBM *gbm, byte *data);
extern GBM_ERR gbm_write(const char *fn,
                         int fd, int ft,
                         const GBM *gbm,
                         const GBMRGB *gbmrgb,
                         const byte *data,
                         const char *opt
                         );
extern const char *gbm_err(GBM_ERR rc);

/*****************************************************************************
*
* gbmmir.c - Mirror Image
*
*****************************************************************************/

extern GBM_BOOLEAN gbm_ref_vert(const GBM *gbm, byte *data);
extern GBM_BOOLEAN gbm_ref_horz(const GBM *gbm, byte *data);
extern void gbm_transpose(const GBM *gbm, const byte *data, byte *data_t);

/*****************************************************************************
*
* gbmrect.c - Subrectangle Transfer
*
*****************************************************************************/

extern void gbm_subrectangle(
	const GBM *gbm,
	int x, int y, int w, int h,
	const byte *data_src, byte *data_dst
	);

extern void gbm_blit(
	const byte *s, int sw, int sx, int sy,
	      byte *d, int dw, int dx, int dy,
	int w, int h,
	int bpp
	);

/*****************************************************************************
*
* gbmscale.c - Scaling
*
*****************************************************************************/

extern GBM_ERR gbm_simple_scale(
	const byte *s, int sw, int sh,
	      byte *d, int dw, int dh,
	int bpp
	);

/*****************************************************************************
*
* gbmerr.c - Error Diffusion
*
*****************************************************************************/

extern void    gbm_errdiff_line_24(byte *src, byte *dest, short *errs, int cx, byte rm, byte gm, byte bm);
extern GBM_BOOLEAN gbm_errdiff_24(const GBM *gbm, byte *data24, byte *data24a, byte rm, byte gm, byte bm);

extern void    gbm_errdiff_pal_6R6G6B(GBMRGB *gbmrgb);
extern void    gbm_errdiff_line_6R6G6B(byte *src, byte *dest, short *errs, int cx);
extern GBM_BOOLEAN gbm_errdiff_6R6G6B(const GBM *gbm, const byte *data24, byte *data8);

extern void    gbm_errdiff_pal_7R8G4B(GBMRGB *gbmrgb);
extern void    gbm_errdiff_line_7R8G4B(byte *src, byte *dest, short *errs, int cx);
extern GBM_BOOLEAN gbm_errdiff_7R8G4B(const GBM *gbm, const byte *data24, byte *data8);

extern void    gbm_errdiff_pal_VGA(GBMRGB *gbmrgb);
extern void    gbm_errdiff_line_VGA(byte *src, byte *dest, short *errs, int cx);
extern GBM_BOOLEAN gbm_errdiff_VGA(const GBM *gbm, const byte *data24, byte *data4);

extern void    gbm_errdiff_pal_8(GBMRGB *gbmrgb);
extern void    gbm_errdiff_line_8(byte *src, byte *dest, short *errs, int cx);
extern GBM_BOOLEAN gbm_errdiff_8(const GBM *gbm, const byte *data24, byte *data4);

extern void    gbm_errdiff_pal_4G(GBMRGB *gbmrgb);
extern void    gbm_errdiff_line_4G(byte *src, byte *dest, short *errs, int cx);
extern GBM_BOOLEAN gbm_errdiff_4G(GBM *gbm, byte *data24, byte *data4);

extern void    gbm_errdiff_pal_BW(GBMRGB *gbmrgb);
extern void    gbm_errdiff_line_BW(byte *src, byte *dest, short *errs, int cx);
extern GBM_BOOLEAN gbm_errdiff_BW(GBM *gbm, byte *data24, byte *data1);

/*****************************************************************************
*
* gbmtrunc.c - Error Diffusion
*
*****************************************************************************/

extern void gbm_trunc_line_24(const byte *src, byte *dest, int cx, byte rm, byte gm, byte bm);
extern void gbm_trunc_24(const GBM *gbm, const byte *data24, byte *data24a, byte rm, byte gm, byte bm);

extern void gbm_trunc_pal_6R6G6B(GBMRGB *gbmrgb);
extern void gbm_trunc_line_6R6G6B(const byte *src, byte *dest, int cx);
extern void gbm_trunc_6R6G6B(const GBM *gbm, const byte *data24, byte *data8);

extern void gbm_trunc_pal_7R8G4B(GBMRGB *gbmrgb);
extern void gbm_trunc_line_7R8G4B(const byte *src, byte *dest, int cx);
extern void gbm_trunc_7R8G4B(const GBM *gbm, const byte *data24, byte *data8);

extern void gbm_trunc_pal_VGA(GBMRGB *gbmrgb);
extern void gbm_trunc_line_VGA(const byte *src, byte *dest, int cx);
extern void gbm_trunc_VGA(const GBM *gbm, const byte *data24, byte *data4);

extern void gbm_trunc_pal_8(GBMRGB *gbmrgb);
extern void gbm_trunc_line_8(const byte *src, byte *dest, int cx);
extern void gbm_trunc_8(const GBM *gbm, const byte *data24, byte *data4);

extern void gbm_trunc_pal_4G(GBMRGB *gbmrgb);
extern void gbm_trunc_line_4G(const byte *src, byte *dest, int cx);
extern void gbm_trunc_4G(const GBM *gbm, const byte *data24, byte *data4);

extern void gbm_trunc_pal_BW(GBMRGB *gbmrgb);
extern void gbm_trunc_line_BW(const byte *src, byte *dest, int cx);
extern void gbm_trunc_BW(const GBM *gbm, const byte *data24, byte *data1);

/*****************************************************************************
*
* gbmht.c - Halftoner
*
*****************************************************************************/

extern void gbm_ht_24_2x2(const GBM *gbm, const byte *data24, byte *data24a, byte rm, byte gm, byte bm);

extern void gbm_ht_pal_6R6G6B(GBMRGB *gbmrgb);
extern void gbm_ht_6R6G6B_2x2(const GBM *gbm, const byte *data24, byte *data8);

extern void gbm_ht_pal_7R8G4B(GBMRGB *gbmrgb);
extern void gbm_ht_7R8G4B_2x2(const GBM *gbm, const byte *data24, byte *data8);

extern void gbm_ht_pal_VGA(GBMRGB *gbmrgb);
extern void gbm_ht_VGA_2x2(const GBM *gbm, const byte *src24, byte *dest4);
extern void gbm_ht_VGA_3x3(const GBM *gbm, const byte *src24, byte *dest4);

extern void gbm_ht_pal_8(GBMRGB *gbmrgb);
extern void gbm_ht_8_2x2(const GBM *gbm, const byte *src24, byte *dest4);
extern void gbm_ht_8_3x3(const GBM *gbm, const byte *src24, byte *dest4);

/*****************************************************************************
*
* gbmhist.c - Histogram/Frequency-of-use
*
*****************************************************************************/

typedef struct {
    byte b, g, r;
    dword freq;
    byte nearest;
} FREQ;

typedef struct {
    int n_cols;
    byte rm, gm, bm;
    FREQ f[2049];   /* N_COLS */
    word ht[5191];  /* N_HASH */
} GBMHIST;

extern GBMHIST *gbm_create_hist(byte rm, byte gm, byte bm);

extern void gbm_delete_hist(GBMHIST *hist);

extern GBM_BOOLEAN gbm_add_to_hist(
    GBMHIST *hist,
    const GBM *gbm,
    const byte *data24
    );

extern void gbm_pal_hist(
	GBMHIST *hist,
	GBMRGB gbmrgb[],
	int n_cols_wanted
	);

extern void gbm_map_hist(
	GBMHIST *hist,
	const GBM *gbm, const byte *data24, byte *data8
	);

extern GBM_BOOLEAN gbm_hist(
	const GBM *gbm, const byte *data24,
	GBMRGB gbmrgb[],
	byte *data8,
	int n_cols_wanted,
	byte rm, byte gm, byte bm
	);

/*****************************************************************************
*
* gbmmcut.c - Median Cut
*
*****************************************************************************/

typedef struct {
    dword freq;
    byte r0,r1,g0,g1,b0,b1;
    byte dividable;
} CELL;

typedef struct {
    dword freqs[0x20][0x20][0x20]; /* 128Kb */
    dword total;
    int n_cells;
    CELL cells[0x100];
} GBMMCUT;

extern GBMMCUT *gbm_create_mcut(void);

extern void gbm_delete_mcut(GBMMCUT *mcut);

extern void gbm_add_to_mcut(
	GBMMCUT *mcut,	
	const GBM *gbm, const byte *data24
	);

extern void gbm_pal_mcut(
	GBMMCUT *mcut,
	GBMRGB gbmrgb[],
	int n_cols_wanted
	);

extern void gbm_map_mcut(
	GBMMCUT *mcut,
	const GBM *gbm, const byte *data24, byte *data8
	);

extern GBM_BOOLEAN gbm_mcut(
	const GBM *gbm, const byte *data24,
	GBMRGB gbmrgb[],
	byte *data8,
	int n_cols_wanted
	);

#endif /* GBM_H */
