/*
 * interface to the v4l driver
 *
 *   (c) 1997-99 Gerd Knorr <kraxel@goldbach.in-berlin.de>
 *
 */
#include "config.h"

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <math.h>
#include <errno.h>
#include <fcntl.h>
#include <string.h>
#include <signal.h>
#include <endian.h>
#include <sys/types.h>
#include <sys/time.h>
#include <sys/ioctl.h>
#include <sys/stat.h>
#include <sys/mman.h>

//#include <X11/Intrinsic.h>

#include "grab.h"
#include "colorspace.h"

#include <asm/types.h>		/* XXX glibc */
#include <linux/videodev.h>

#define SYNC_TIMEOUT 1

/* ---------------------------------------------------------------------- */

/* open+close */
static int   grab_open(char *filename);
static int   grab_close();

/* overlay */
static int   grab_setupfb(int sw, int sh, int format, void *base, int width);
static int   grab_overlay(int x, int y, int width, int height, int format,
			  struct OVERLAY_CLIP *oc, int count);

/* capture */
static int   grab_setparams(int format, int *width, int *height,
			    int *linelength);
static void* grab_capture(int single);
static void  grab_cleanup();
#if 0
static void* grab_scr(void *dest, int width, int height, int single);
static void* grab_one(int width, int height);
#endif

/* control */
static int   grab_tune(unsigned long freq);
static int   grab_tuned();
static int   grab_input(int input, int norm);
static int   grab_picture(int color, int bright, int hue, int contrast);
static int   grab_audio(int mute, int volume, int *mode);

/* internal helpers */
static int   grab_wait(struct video_mmap *gb);

/* ---------------------------------------------------------------------- */

static char *device_cap[] = {
    "capture", "tuner", "teletext", "overlay", "chromakey", "clipping",
    "frameram", "scales", "monochrome", NULL
};

static char *device_pal[] = {
    "-", "grey", "hi240", "rgb16", "rgb24", "rgb32", "rgb15",
    "yuv422", "yuyv", "uyvy", "yuv420", "yuv411", "raw",
    "yuv422p", "yuv411p"
};
#define PALETTE(x) ((x < sizeof(device_pal)/sizeof(char*)) ? device_pal[x] : "UNKNOWN")

/*                            PAL  NTSC SECAM */
static int    maxwidth[]  = { 768, 640, 768 };
static int    maxheight[] = { 576, 480, 576 };

static struct STRTAB norms[] = {
    {  0, "PAL" },
    {  1, "NTSC" },
    {  2, "SECAM" },
    {  3, "AUTO" },
    { -1, NULL }
};
static struct STRTAB norms_bttv[] = {
    {  0, "PAL" },
    {  1, "NTSC" },
    {  2, "SECAM" },
    {  3, "PAL-NC" },
    {  4, "PAL-M" },
    {  5, "PAL-N" },
    {  6, "NTSC-JP" },
    { -1, NULL }
};
static struct STRTAB *inputs;

static int    fd = -1;

/* generic informations */
static struct video_capability  capability;
static struct video_channel     *channels;
static struct video_audio       *audios;
static struct video_tuner       *tuner;
static struct video_picture     pict;
static int                      cur_input;
static int                      cur_norm;

/* overlay */
static struct video_window      ov_win;
static struct video_clip        ov_clips[32];
static struct video_buffer      ov_fbuf;

/* screen grab */
static struct video_mmap        gb_even;
static struct video_mmap        gb_odd;
static int                      even,pixmap_bytes;
static int                      gb_grab,gb_sync;
static struct video_mbuf        gb_buffers = { 2*0x151000, 0, {0,0x151000 }};
static int gb_pal[] = {
    0, 0, 0, 0, 0,
    0, 0, 0, 0, 0,
    0, 0, 0, 0, 0
};

static unsigned short format2palette[] = {
    0,				/* unused */
    VIDEO_PALETTE_HI240,	/* RGB8   */
    VIDEO_PALETTE_GREY,		/* GRAY8  */
#if __BYTE_ORDER == __BIG_ENDIAN
    0,
    0,
    VIDEO_PALETTE_RGB555,	/* RGB15_BE  */
    VIDEO_PALETTE_RGB565,	/* RGB16_BE  */
    0,
    0,
    VIDEO_PALETTE_RGB24,	/* RGB24     */
    VIDEO_PALETTE_RGB32,	/* RGB32     */
#else
    VIDEO_PALETTE_RGB555,	/* RGB15_LE  */
    VIDEO_PALETTE_RGB565,	/* RGB16_LE  */
    0,
    0,
    VIDEO_PALETTE_RGB24,	/* BGR24     */
    VIDEO_PALETTE_RGB32,	/* BGR32     */
    0,
    0,
#endif
};

static char *map = NULL;

/* state */
static int                      overlay, swidth, sheight;

/* pass 0/1 by reference */
static int                      one = 1, zero = 0;

struct GRABBER grab_v4l = {
    "v4l",
    0,
    norms,NULL,

    grab_open,
    grab_close,

    grab_setupfb,
    NULL /* grab_overlay */,

    grab_setparams,
    grab_capture,
    grab_cleanup,

    grab_tune,
    grab_tuned,
    grab_input,
    grab_picture,
    grab_audio
};

/* ---------------------------------------------------------------------- */

static void
sigalarm(int signal)
{
    fprintf(stderr,"v4l: oops: got sigalarm\n");
}

static void
siginit(void)
{
    struct sigaction act,old;
    
    memset(&act,0,sizeof(act));
    act.sa_handler  = sigalarm;
    sigemptyset(&act.sa_mask);
    sigaction(SIGALRM,&act,&old);
}

/* main() appends any options needed for v4l-conf */
extern char v4l_conf[];
static void
run_v4l_conf(void)
{    
/*    if (!x11_is_remote) {*/
	switch (system(v4l_conf)) {
	case -1: /* can't run */
	    fprintf(stderr,"could'nt start v4l-conf\n");
	    break;
	case 0: /* ok */
	    break;
	default: /* non-zero return */
	    fprintf(stderr,"v4l-conf had some trouble, "
		    "trying to continue anyway\n");
	    break;
	}
/*    }*/
}

/* ---------------------------------------------------------------------- */

static int
grab_open(char *filename)
{
    int i;
    int mode = -1;

    if (-1 != fd)
	goto err;

    run_v4l_conf();
    if (-1 == (fd = open(filename ? filename : "/dev/video",O_RDWR))) {
	fprintf(stderr,"open %s: %s\n",
		filename ? filename : "/dev/video",strerror(errno));
	goto err;
    }

    if (-1 == ioctl(fd,VIDIOCGCAP,&capability))
	goto err;

    if (debug)
	fprintf(stderr, "v4l: open\n");

    fcntl(fd,F_SETFD,FD_CLOEXEC);
    siginit();
    if (debug)
	fprintf(stderr,"v4l: device is %s\n",capability.name);    
    sprintf(grab_v4l.name = malloc(strlen(capability.name)+8),
	    "v4l: %s",capability.name);
    if (debug) {
	fprintf(stderr,"capabilities: ");
	for (i = 0; device_cap[i] != NULL; i++)
	    if (capability.type & (1 << i))
		fprintf(stderr," %s",device_cap[i]);
	fprintf(stderr,"\n");
    }

    /* input sources */
    if (debug)
	fprintf(stderr,"  channels: %d\n",capability.channels);
    channels = malloc(sizeof(struct video_channel)*capability.channels);
    memset(channels,0,sizeof(struct video_channel)*capability.channels);
    inputs = malloc(sizeof(struct STRTAB)*(capability.channels+1));
    memset(inputs,0,sizeof(struct STRTAB)*(capability.channels+1));
    for (i = 0; i < capability.channels; i++) {
	channels[i].channel = i;
	if (-1 == ioctl(fd,VIDIOCGCHAN,&channels[i]))
	    perror("ioctl VIDIOCGCHAN"), exit(0);
	inputs[i].nr  = i;
	inputs[i].str = channels[i].name;
	if (debug)
	    fprintf(stderr,"    %s: %d %s%s %s%s\n",
		    channels[i].name,
		    channels[i].tuners,
		    (channels[i].flags & VIDEO_VC_TUNER)   ? "tuner "  : "",
		    (channels[i].flags & VIDEO_VC_AUDIO)   ? "audio "  : "",
		    (channels[i].type & VIDEO_TYPE_TV)     ? "tv "     : "",
		    (channels[i].type & VIDEO_TYPE_CAMERA) ? "camera " : "");
    }
    inputs[i].nr  = -1;
    inputs[i].str = NULL;
    grab_v4l.inputs =inputs;

    /* ioctl probe, switch to input 0 */
    if (-1 == ioctl(fd,VIDIOCSCHAN,&channels[0])) {
	fprintf(stderr,"v4l: you need a newer bttv version (>= 0.5.14)\n");
	goto err;
    }

    /* audios */
    if (debug)
	fprintf(stderr,"  audios  : %d\n",capability.audios);
    if (capability.audios) {
	audios = malloc(sizeof(struct video_audio)*capability.audios);
	memset(audios,0,sizeof(struct video_audio)*capability.audios);
	for (i = 0; i < capability.audios; i++) {
	    audios[i].audio = i;
	    if (-1 == ioctl(fd,VIDIOCGAUDIO,&audios[i]))
		perror("ioctl VIDIOCGCAUDIO") /* , exit(0) */ ;
	    if (debug) {
		fprintf(stderr,"    %d (%s): ",i,audios[i].name);
		if (audios[i].flags & VIDEO_AUDIO_MUTABLE)
		    fprintf(stderr,"muted=%s ",
			    (audios[i].flags&VIDEO_AUDIO_MUTE) ? "yes":"no");
		if (audios[i].flags & VIDEO_AUDIO_VOLUME)
		    fprintf(stderr,"volume=%d ",audios[i].volume);
		if (audios[i].flags & VIDEO_AUDIO_BASS)
		    fprintf(stderr,"bass=%d ",audios[i].bass);
		if (audios[i].flags & VIDEO_AUDIO_TREBLE)
		    fprintf(stderr,"treble=%d ",audios[i].treble);
		fprintf(stderr,"\n");
	    }
	}
	if (audios[0].flags & VIDEO_AUDIO_VOLUME)
	    grab_v4l.flags |= CAN_AUDIO_VOLUME;
    } else
	grab_v4l.grab_audio = NULL;

    if (debug)
	fprintf(stderr,"  size    : %dx%d => %dx%d\n",
		capability.minwidth,capability.minheight,
		capability.maxwidth,capability.maxheight);

    /* tuner (more than one???) */
    if (capability.type & VID_TYPE_TUNER) {
	tuner = malloc(sizeof(struct video_tuner));
	memset(tuner,0,sizeof(struct video_tuner));
	if (-1 == ioctl(fd,VIDIOCGTUNER,tuner))
	    perror("ioctl VIDIOCGTUNER");
	if (debug)
	    fprintf(stderr,"  tuner   : %s %lu-%lu",
		    tuner->name,tuner->rangelow,tuner->rangehigh);
	for (i = 0; norms[i].str != NULL; i++) {
	    if (tuner->flags & (1<<i)) {
		if (debug)
		    fprintf(stderr," %s",norms[i].str);
	    } else
		norms[i].nr = -1;
	}
	if (debug)
	    fprintf(stderr,"\n");
    } else {
	struct video_channel vchan;

	memcpy(&vchan, &channels[0], sizeof(struct video_channel));
	for (i = 0; norms[i].str != NULL; i++) {
	    vchan.norm = i;
	    if (-1 == ioctl(fd,VIDIOCSCHAN,&vchan))
		norms[i].nr = -1;
	    else if (debug)
		fprintf(stderr," %s",norms[i].str);
	}
	if (debug)
	    fprintf(stderr,"\n");
	if (-1 == ioctl(fd,VIDIOCSCHAN,&channels[0])) {
	    fprintf(stderr,"v4l: you need a newer bttv version (>= 0.5.14)\n");
	    goto err;
	}
	grab_v4l.grab_tune  = NULL;
	grab_v4l.grab_tuned = NULL;
    }
#if 1
#define BTTV_VERSION  	        _IOR('v' , BASE_VIDIOCPRIVATE+6, int)
    /* dirty hack time / v4l design flaw -- works with bttv only
     * this adds support for a few less common PAL versions */
    if (-1 != ioctl(fd,BTTV_VERSION,0)) {
	grab_v4l.norms = norms_bttv;
    }
#endif
    
    /* frame buffer */
    if (-1 == ioctl(fd,VIDIOCGFBUF,&ov_fbuf))
	perror("ioctl VIDIOCGFBUF");
    if (debug)
	fprintf(stderr,"  fbuffer : base=0x%p size=%dx%d depth=%d bpl=%d\n",
		ov_fbuf.base, ov_fbuf.width, ov_fbuf.height,
		ov_fbuf.depth, ov_fbuf.bytesperline);

    /* picture parameters */
    if (-1 == ioctl(fd,VIDIOCGPICT,&pict))
	perror("ioctl VIDIOCGPICT");

    if (debug) {
	fprintf(stderr,
		"  picture : brightness=%d hue=%d colour=%d contrast=%d\n",
		pict.brightness, pict.hue, pict.colour, pict.contrast);
	fprintf(stderr,
		"  picture : whiteness=%d depth=%d palette=%s\n",
		pict.whiteness, pict.depth, PALETTE(pict.palette));
    }

    /* map grab buffer */
    if (-1 == ioctl(fd,VIDIOCGMBUF,&gb_buffers)) {
	perror("ioctl VIDIOCGMBUF");
    }
    map = mmap(0,gb_buffers.size,PROT_READ|PROT_WRITE,MAP_SHARED,fd,0);
    if ((char*)-1 == map) {
	perror("mmap");
    }

#if 0
    if (map == (char*)-1) {
	/* XXX no mmap grabbing, use read */
    }
#endif

    /* unmute */
    if (grab_v4l.grab_audio)
	grab_audio(0,-1,&mode);
    return fd;

err:
    if (fd != -1) {
	close(fd);
	fd = -1;
    }
    return -1;
}

static int
grab_close()
{
    int mode = -1;

    if (-1 == fd)
	return 0;

    if (gb_grab > gb_sync)
	grab_wait(even ? &gb_even : &gb_odd);

    if ((char*)-1 != map)
	munmap(map,gb_buffers.size);

    if (debug)
	fprintf(stderr, "v4l: close\n");

    /* mute */
    if (grab_v4l.grab_audio)
	grab_audio(1,-1,&mode);

    close(fd);
    fd = -1;
    return 0;
}

/* ---------------------------------------------------------------------- */

static int
grab_setupfb(int sw, int sh, int format, void *base, int bpl)
{
    int settings_ok = 1;
    swidth  = sw;
    sheight = sh;

    /* double-check settings */
    fprintf(stderr,"v4l: %dx%d, %d bit/pixel, %d byte/scanline\n",
	    ov_fbuf.width,ov_fbuf.height,
	    ov_fbuf.depth,ov_fbuf.bytesperline);
    if ((bpl > 0 && ov_fbuf.bytesperline != bpl) ||
	(ov_fbuf.width  != sw) ||
	(ov_fbuf.height != sh)) {
	fprintf(stderr,"WARNING: v4l and dga disagree about the screen size\n");
	fprintf(stderr,"WARNING: Is v4l-conf installed correctly?\n");
	settings_ok = 0;
    }
    if (format2depth[format] != ((ov_fbuf.depth+7)&0xf8)) {
	fprintf(stderr,"WARNING: v4l and dga disagree about the color depth\n");
	fprintf(stderr,"WARNING: Is v4l-conf installed correctly?\n");
	fprintf(stderr,"%d %d\n",format2depth[format],ov_fbuf.depth);
	settings_ok = 0;
    }
    if (settings_ok) {
	grab_v4l.grab_overlay = grab_overlay;
	return 0;
    } else {
	fprintf(stderr,"WARNING: overlay mode disabled\n");
	return -1;
    }
}

static int
grab_overlay(int x, int y, int width, int height, int format,
	     struct OVERLAY_CLIP *oc, int count)
{
    int i,xadjust=0,yadjust=0;

#if 0
    if (gb_grab > gb_sync) {
	if (-1 == ioctl(fd,VIDIOCSYNC,even ? &zero : &one))
	    perror("ioctl VIDIOCSYNC");
	else
	    gb_sync++;
    }
#endif

    if (width == 0 || height == 0) {
	if (debug)
	    fprintf(stderr,"v4l: overlay off\n");
	ioctl(fd, VIDIOCCAPTURE, &zero);
	overlay = 0;
	return 0;
    }

    ov_win.x          = x;
    ov_win.y          = y;
    ov_win.width      = width;
    ov_win.height     = height;
    ov_win.flags      = 0;

#if 1
    /* check against max. size */
    if (ov_win.width > maxwidth[cur_norm]) {
	ov_win.width = maxwidth[cur_norm];
	ov_win.x += (width - ov_win.width)/2;
    }
    if (ov_win.height > maxheight[cur_norm]) {
	ov_win.height = maxheight[cur_norm];
	ov_win.y +=  (height - ov_win.height)/2;
    }

    /* pass aligned values -- the driver does'nt get it right yet */
    ov_win.width  &= ~3;
    ov_win.height &= ~3;
    ov_win.x      &= ~3;
    if (ov_win.x              < x)        ov_win.x     += 4;
    if (ov_win.x+ov_win.width > x+width)  ov_win.width -= 4;

    /* fixups */
    xadjust = ov_win.x - x;
    yadjust = ov_win.y - y;
#endif

    if (capability.type & VID_TYPE_CLIPPING) {
	ov_win.clips      = ov_clips;
	ov_win.clipcount  = count;
	
	for (i = 0; i < count; i++) {
	    ov_clips[i].x      = oc[i].x1 - xadjust;
	    ov_clips[i].y      = oc[i].y1 - yadjust;
	    ov_clips[i].width  = oc[i].x2-oc[i].x1 /* XXX */;
	    ov_clips[i].height = oc[i].y2-oc[i].y1;
	    if (debug)
		fprintf(stderr,"v4l: clip=%dx%d+%d+%d\n",
			ov_clips[i].width,ov_clips[i].height,
			ov_clips[i].x,ov_clips[i].y);
#if 0
	    if (ov_clips[i].x < 0 || ov_clips[i].y < 0 ||
		ov_clips[i].width < 0 || ov_clips[i].height < 0) {
		fprintf(stderr,"v4l: bug trap - overlay off\n");
		ioctl(fd, VIDIOCCAPTURE, &zero);
		overlay = 0;
		return 0;		
	    }
#endif
	}
    }
    if (capability.type & VID_TYPE_CHROMAKEY) {
	ov_win.chromakey  = 0;    /* XXX */
    }
    if (-1 == ioctl(fd, VIDIOCSWIN, &ov_win))
	perror("ioctl VIDIOCSWIN");

    if (!overlay) {
	pict.palette =
	    (format < sizeof(format2palette)/sizeof(unsigned short))?
	    format2palette[format]: 0;
	if(pict.palette == 0) {
	    fprintf(stderr,"v4l: unsupported overlay video format: %s\n",
		    format_desc[format]);
	    return -1;
	}
	if (-1 == ioctl(fd,VIDIOCSPICT,&pict))
	    perror("ioctl VIDIOCSPICT");
	if (-1 == ioctl(fd, VIDIOCCAPTURE, &one))
	    perror("ioctl VIDIOCCAPTURE");
	overlay = 1;
    }

    if (debug)
	fprintf(stderr,"v4l: overlay win=%dx%d+%d+%d, %d clips\n",
		width,height,x,y, count);

    return 0;
}

static int
grab_queue(struct video_mmap *gb, int probe)
{
    if (debug > 1)
	fprintf(stderr,"g%d",gb->frame);
#if 0
    /* might be useful for debugging driver problems */
    memset(map + gb_buffers.offsets[gb->frame],0,
	   gb_buffers.size/gb_buffers.frames);
#endif
    if (-1 == ioctl(fd,VIDIOCMCAPTURE,gb)) {
	if (errno == EAGAIN)
	    fprintf(stderr,"grabber chip can't sync (no station tuned in?)\n");
	else
	    if (!probe || debug)
		fprintf(stderr,"ioctl VIDIOCMCAPTURE(%d,%s,%dx%d): %s\n",
			gb->frame,PALETTE(gb->format),gb->width,gb->height,
			strerror(errno));
	return -1;
    }
    if (debug > 1)
	fprintf(stderr,"* ");
    gb_grab++;
    return 0;
}

static int
grab_wait(struct video_mmap *gb)
{
    int ret = 0;
    
    alarm(SYNC_TIMEOUT);
    if (debug > 1)
	fprintf(stderr,"s%d",gb->frame);

    if (-1 == ioctl(fd,VIDIOCSYNC,&(gb->frame))) {
	perror("ioctl VIDIOCSYNC");
	ret = -1;
    }
    gb_sync++;
    if (debug > 1)
	fprintf(stderr,"* ");
    alarm(0);
    return ret;
}

static int
grab_probe(int format)
{
    struct video_mmap gb;

    if (0 != gb_pal[format])
	goto done;

    gb.frame  = 0;
    gb.width  = 64;
    gb.height = 48;

    if (debug)
	fprintf(stderr, "v4l: capture probe %s...\t", device_pal[format]);
    gb.format = format;
    if (-1 == grab_queue(&gb,1)) {
	gb_pal[format] = 2;
	goto done;
    }
    if (-1 == grab_wait(&gb)) {
	gb_pal[format] = 2;
	goto done;
    }
    if (debug)
	fprintf(stderr, "ok\n");
    gb_pal[format] = 1;

 done:
    return gb_pal[format] == 1;
}

static int
grab_setparams(int format, int *width, int *height, int *linelength)
{
    /* finish old stuff */
    if (gb_grab > gb_sync)
	grab_wait(even ? &gb_even : &gb_odd);

    /* verify parameters */
    if (*width > maxwidth[cur_norm])
	*width = maxwidth[cur_norm];
    if (*height > maxheight[cur_norm])
	*height = maxheight[cur_norm];
    *linelength = *width * format2depth[format] / 8;

    /* initialize everything */
    gb_even.format = gb_odd.format =
	(format < sizeof(format2palette)/sizeof(unsigned short))?
	format2palette[format]: 0;
    if(gb_even.format == 0 || !grab_probe(gb_even.format)) {
	return -1;
    }
    pixmap_bytes   = format2depth[format] / 8;
    gb_even.frame  = 0;
    gb_odd.frame   = 1;
    gb_even.width  = *width;
    gb_even.height = *height;
    gb_odd.width   = *width;
    gb_odd.height  = *height;
    even = 0;

    return 0;
}

static void* grab_capture(int single)
{
    void *buf;

    if (!single && gb_grab == gb_sync)
	/* streaming capture started */
	if (-1 == grab_queue(even ? &gb_even : &gb_odd,0))
	    return NULL;

    if (single && gb_grab > gb_sync)
	/* clear streaming capture */
	grab_wait(even ? &gb_even : &gb_odd);

    /* queue */
    if (-1 == grab_queue(even ? &gb_odd : &gb_even,0))
	return NULL;
    if (gb_grab > gb_sync+1) {
	/* wait -- streaming */
	grab_wait(even ? &gb_even : &gb_odd);
	buf = map + gb_buffers.offsets[even ? 0 : 1];
    } else {
	/* wait -- single */
	grab_wait(even ? &gb_odd : &gb_even);
	buf = map + gb_buffers.offsets[even ? 1 : 0];
    }
    even = !even;

    return buf;
}

static void
grab_cleanup()
{
    if (gb_grab > gb_sync)
	grab_wait(even ? &gb_even : &gb_odd);
}

#if 0
static void*
grab_scr(void *dest, int width, int height, int single)
{
    void *buf;

    if ((char*)-1 == map)
	return NULL;
    if (!gb_even.format)
	return NULL;
    
    gb_even.width  = width;
    gb_even.height = height;
    gb_odd.width  = width;
    gb_odd.height = height;

    if (single) {
	if (gb_grab > gb_sync)
	    grab_wait(even ? &gb_even : &gb_odd);
    } else {
	if (gb_grab == gb_sync)
	    if (-1 == grab_queue(even ? &gb_even : &gb_odd,0))
		return NULL;
    }

    if (-1 == grab_queue(even ? &gb_odd : &gb_even,0))
	return NULL;

    if (gb_grab > gb_sync+1) {
	grab_wait(even ? &gb_even : &gb_odd);
	buf = map + gb_buffers.offsets[even ? 0 : 1];
    } else {
	grab_wait(even ? &gb_odd : &gb_even);
	buf = map + gb_buffers.offsets[even ? 1 : 0];
    }
    even = !even;

    if (pixmap_bytes == 4) {
	rgb24_to_rgb32(dest, buf, width, height);
    } else {
	memcpy(dest, buf, width*height*pixmap_bytes);
    }
    
    return dest;
}

static void*
grab_one(int width, int height)
{
    struct video_mmap gb;

    if ((char*)-1 == map)
	return NULL;
    
    if (gb_grab > gb_sync)
	grab_wait(even ? &gb_even : &gb_odd);
    
    gb.format = VIDEO_PALETTE_RGB24;
    gb.frame  = 0;
    gb.width  = width;
    gb.height = height;

    memset(map,0,width*height*3);
    if (-1 == grab_queue(&gb,0))
	return NULL;
    grab_wait(&gb);

    swap_rgb24(map,width*height);
    return map;
}
#endif

/* ---------------------------------------------------------------------- */

static int
grab_tune(unsigned long freq)
{
    if (debug)
	fprintf(stderr,"v4l: freq: %.3f\n",(float)freq/16);
    if (-1 == ioctl(fd, VIDIOCSFREQ, &freq))
	perror("ioctl VIDIOCSFREQ");
    return 0;
}

static int
grab_tuned()
{
    usleep(10000);
#if 0
    /* (quick & dirty -- grabbing works with hsync only) */
    return grab_one(64,48) ? 1 : 0;
#else
    if (-1 == ioctl(fd,VIDIOCGTUNER,tuner)) {
	perror("ioctl VIDIOCGTUNER");
	return 0;
    }
    return tuner->signal ? 1 : 0;
#endif
}

static int
grab_input(int input, int norm)
{
    if (-1 != input) {
	if (debug)
	    fprintf(stderr,"v4l: input: %d\n",input);
	cur_input = input;
    }
    if (-1 != norm) {
	if (debug)
	    fprintf(stderr,"v4l: norm : %d\n",norm);
	cur_norm = norm;
    }

    channels[cur_input].norm = cur_norm;
    if (-1 == ioctl(fd, VIDIOCSCHAN, &channels[cur_input]))
	perror("ioctl VIDIOCSCHAN");
    return 0;
}

int
grab_picture(int color, int bright, int hue, int contrast)
{
    if (color != -1)
	pict.colour = color;
    if (contrast != -1)
	pict.contrast = contrast;
    if (bright != -1)
	pict.brightness = bright;
    if (hue != -1)
	pict.hue = hue;

    if (-1 == ioctl(fd,VIDIOCSPICT,&pict))
	perror("ioctl VIDIOCSPICT");

    return 0;
}

int
grab_audio(int mute, int volume, int *mode)
{
    if (-1 == mute && -1 == volume && mode && -1 == *mode) {
	/* only read mode */
	if (-1 == ioctl(fd,VIDIOCGAUDIO,&audios[0]))
	    perror("ioctl VIDIOCGAUDIO");
	*mode = audios[0].mode;
	return 0;
    }

    /* read ... */
    if (-1 == ioctl(fd,VIDIOCGAUDIO,&audios[0]))
	perror("ioctl VIDIOCGAUDIO");

    /* ... modify .. */
    if (mute != -1) {
	if (mute)
	    audios[0].flags |= VIDEO_AUDIO_MUTE;
	else
	    audios[0].flags &= ~VIDEO_AUDIO_MUTE;
    }
    if (volume != -1)
	audios[0].volume = volume;
    audios[0].audio = cur_input;
    audios[0].mode = mode ? *mode : 0;

    /* ... write */
    if (-1 == ioctl(fd,VIDIOCSAUDIO,&audios[0]))
	perror("ioctl VIDIOCSAUDIO");

    return 0;
}
