/* 
    channel for Bt848 frame grabber driver

    Copyright (C) 1996,97 Marcus Metzler (mocm@thp.uni-koeln.de)

    many changes by Gerd Knorr <kraxel@goldbach.in-berlin.de>
        [ hmm, think by now nearly nothing left from the original code ... ]

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <ctype.h>
#include <math.h>

#include <X11/Intrinsic.h>

#include "grab.h"
#include "channel.h"
#include "frequencies.h"
#include "sound.h"

/* ----------------------------------------------------------------------- */
/* misc common stuff, not only channel related                             */ 

struct CHANNEL  defaults    = { "defaults", NULL,
				"5", 0, 0, 0,
				CAPTURE_OVERLAY, 0, 0,
				32768, 32768, 32768, 32768 };
struct CHANNEL  **channels  = NULL;
int             count       = 0;
int             alloc_count = 0;
int             have_mixer  = 0;

int    last_sender = -1, cur_sender = -1, cur_channel = 5, cur_fine = 0;
int    cur_norm = -1, cur_input = -1;

int have_config;
int jpeg_quality = 75;
int fs_width,fs_height,fs_xoff,fs_yoff,pix_width,pix_height;
int cur_color,cur_bright,cur_hue,cur_contrast,cur_capture;

char *mixer  = NULL;

struct LAUCH *lauch = NULL;
int nlauch          = 0;

/* ----------------------------------------------------------------------- */

int lookup_channel(char *channel)
{
    int    i,nr1,nr2;
    char   tag1[5],tag2[5];

    if (isdigit(channel[0])) {
	tag1[0] = 0;
	nr1  = atoi(channel);
    } else {
	sscanf(channel,"%4[A-Za-z]%d",tag1,&nr1);
    }

    for (i = 0; i < chancount; i++) {
	if (isdigit(chanlist[i].name[0])) {
	    tag2[0] = 0;
	    nr2  = atoi(chanlist[i].name);
	} else {
	    sscanf(chanlist[i].name,"%4[A-Za-z]%d",tag2,&nr2);
	}
	if (tag1[0] && tag2[0])
	    if (nr1 == nr2 && 0 == strcmp(tag1,tag2))
		break;
	if (!tag1[0] && !tag2[0])
	    if (nr1 == nr2)
		break;
    }
    if (i == chancount)
	return -1;

    return i;
}

int  get_freq(int i)
{
    if (i < 0 || i >= chancount)
	return -1;
    return chanlist[i].freq*16/1000;
}

int  cf2freq(char *name, int fine)
{
    int i;
    
    if (-1 == (i = lookup_channel(name)))
	return -1;
    return get_freq(i)+fine;
}

/* ----------------------------------------------------------------------- */

static struct STRTAB captab[] = {
    {  CAPTURE_OFF,         "off"         },
    {  CAPTURE_OFF,         "no"          },
    {  CAPTURE_OFF,         "false"       },
    {  CAPTURE_OVERLAY,     "on"          },
    {  CAPTURE_OVERLAY,     "yes"         },
    {  CAPTURE_OVERLAY,     "true"        },
    {  CAPTURE_OVERLAY,     "overlay"     },
    {  CAPTURE_GRABDISPLAY, "grab"        },
    {  CAPTURE_GRABDISPLAY, "grabdisplay" },
    {  -1, NULL,     },
};

/* just malloc memory for a new channel ... */
struct CHANNEL*
add_channel(char *name)
{
    struct CHANNEL *channel;

    if (alloc_count == count) {
	alloc_count += 16;
	if (alloc_count == 16)
	    channels = malloc(sizeof(struct CHANNEL*)*alloc_count);
	else
	    channels = realloc(channels,sizeof(struct CHANNEL*)*alloc_count);
    }
    channel = channels[count++] = malloc(sizeof(struct CHANNEL));
    memcpy(channel,&defaults,sizeof(struct CHANNEL));
    channel->name = strdup(name);
    return channel;
}

/* delete channel */
void
del_channel(int i)
{
    free(channels[i]->name);
    if (channels[i]->key)
	free(channels[i]->key);
    free(channels[i]);
    count--;
    if (i < count)
	memmove(channels+i,channels+i+1,(count-i)*sizeof(struct CHANNEL*));
}

void
calc_frequencies()
{
    int i;

    for (i = 0; i < count; i++) {
	channels[i]->channel = lookup_channel(channels[i]->cname);
	if (-1 == channels[i]->channel)
	    channels[i]->freq = -1;
	else
	    channels[i]->freq = get_freq(channels[i]->channel)
		+ channels[i]->fine;
    }
}

/* ----------------------------------------------------------------------- */

void
read_config()
{
    FILE *fp;
    char filename[100],line[100], tag[32], val[100], *h;
    char label[32],key[16],cmdline[128];
    int i,nr = 0;
    struct CHANNEL *current = &defaults;

    sprintf(filename,"%s/%s",getenv("HOME"),".xawtv");
    fp = fopen(filename,"r");
    if (NULL == fp)
	fp = fopen("/usr/X11R6/lib/X11/xawtvrc","r");
    if (NULL == fp) {
	fprintf(stderr,"can't open config file %s\n",filename);
	return;
    }
    have_config = 1;
    
    while (NULL != fgets(line,99,fp)) {
	nr++;
	if (line[0] == '\n' || line[0] == '#' || line[0] == '%')
	    continue;
	if (1 == sscanf(line,"[%99[^]]]",val)) {
	    for (i = 0; i < count; i++)
		if (0 == strcasecmp(channels[i]->name,val))
		    break;
	    if (i == count)
		current = add_channel(val);
	    else
		current = channels[i];
	    continue;
	}
	if (2 != sscanf(line," %31[^= ] = %99[^\n]",tag,val)) {
	    fprintf(stderr,"%s:%d: parse error\n",filename,nr);
	    continue;
	}
	for (h = val+strlen(val); h > val && *h == ' '; h--)
	    *h = '\0';

	if (0 == strcmp(tag,"key"))
	    current->key = strdup(val);
	
	else if (0 == strcmp(tag,"capture")) {
	    if (-1 != (i = str_to_int(val,captab)))
		current->capture = i;
	    else
		fprintf(stderr,"%s:%d: invalid value for %s: %s\n",
			filename,nr,tag,val);
	} else if (0 == strcmp(tag,"source")) {
	    if (-1 != (i = str_to_int(val,grabber->inputs)))
		current->source = i;
	    else
		fprintf(stderr,"%s:%d: invalid value for %s: %s\n",
			filename,nr,tag,val);
	} else if (0 == strcmp(tag,"norm")) {
	    if (-1 != (i = str_to_int(val,grabber->norms)))
		current->norm = i;
	    else
		fprintf(stderr,"%s:%d: invalid value for %s: %s\n",
			filename,nr,tag,val);
	} else if (0 == strcmp(tag,"channel")) {
	    current->cname   = strdup(val);
#if 0
	    current->channel = lookup_channel(current->cname);
	    if (-1 == current->channel)
		fprintf(stderr,"%s:%d: invalid value for %s: %s\n",
			filename,nr,tag,val);
#endif
	}
	else if (0 == strcmp(tag,"fine"))
	    current->fine = atoi(val);

	else if (0 == strcmp(tag,"color"))
	    current->color = atoi(val);
	else if (0 == strcmp(tag,"bright"))
	    current->bright = atoi(val);
	else if (0 == strcmp(tag,"hue"))
	    current->hue = atoi(val);
	else if (0 == strcmp(tag,"contrast"))
	    current->contrast = atoi(val);
	
	else if (0 == count && 0 == strcmp(tag,"freqtab")) {
	    for (i = 0; chanlists[i].name != NULL; i++)
		if (0 == strcasecmp(val,chanlists[i].name))
		    break;
	    if (chanlists[i].name != NULL) {
		chantab   = i;
		chanlist  = chanlists[chantab].list;
		chancount = chanlists[chantab].count;
	    } else
		fprintf(stderr,"%s:%d: invalid value for %s: %s\n",
			filename,nr,tag,val);

	} else if (0 == count && 0 == strcmp(tag,"fullscreen")) {
	    if (2 != sscanf(val,"%d x %d",&fs_width,&fs_height)) {
		fprintf(stderr,"%s:%d: invalid value for %s: %s\n",
			filename,nr,tag,val);
		fs_width = fs_height = 0;
	    }

	} else if (0 == count && 0 == strcmp(tag,"pixsize")) {
	    if (2 != sscanf(val,"%d x %d",&pix_width,&pix_height)) {
		fprintf(stderr,"%s:%d: invalid value for %s: %s\n",
			filename,nr,tag,val);
		pix_width = 128;
		pix_height = 96;
	    }

	} else if (0 == count && 0 == strcmp(tag,"wm-off-by")) {
	    if (2 != sscanf(val,"%d %d",&fs_xoff,&fs_yoff)) {
		fprintf(stderr,"%s:%d: invalid value for %s: %s\n",
			filename,nr,tag,val);
		fs_xoff = fs_yoff = 0;
	    }

	} else if (0 == count && 0 == strcmp(tag,"jpeg-quality")) {
	    jpeg_quality = atoi(val);

	} else if (0 == count && 0 == strcmp(tag,"lauch")) {
	    if (3 != sscanf(val,"%31[^,], %15[^,], %127[^\n]",
			    label,key,cmdline)) {
		fprintf(stderr,"%s:%d: invalid value for %s: %s\n",
			filename,nr,tag,val);
	    } else {
		lauch = realloc(lauch,sizeof(struct LAUCH)*(nlauch+1));
		lauch[nlauch].name    = strdup(label);
		lauch[nlauch].key     = strdup(key);
		lauch[nlauch].cmdline = strdup(cmdline);
		nlauch++;
	    }

	} else {
	    fprintf(stderr,"%s:%d: unknown tag %s\n",filename,nr,tag);

	}
    }
    fclose(fp);

    /* calculate channel frequencies */
    defaults.channel = lookup_channel(defaults.cname);
    defaults.freq    = get_freq(defaults.channel) + defaults.fine;
    calc_frequencies();
}

/* ----------------------------------------------------------------------- */

void
save_config()
{
    char filename1[100], filename2[100];
    FILE *fp;
    int i;

    sprintf(filename1,"%s/%s",getenv("HOME"),".xawtv");
    sprintf(filename2,"%s/%s",getenv("HOME"),".xawtv~");

    /* delete old backup */
    unlink(filename2);

    /* current becomes backup */
    if (0 == link(filename1,filename2))
	unlink(filename1);

    /* write new one... */
    fp = fopen(filename1,"w");
    if (NULL == fp) {
	fprintf(stderr,"can't open config file %s\n",filename1);
	return;
    }

    /* write defaults */
    fprintf(fp,"norm = %s\n",int_to_str(cur_norm,grabber->norms));
    fprintf(fp,"capture = %s\n",int_to_str(cur_capture,captab));
    fprintf(fp,"source = %s\n",
	    int_to_str(cur_input,grabber->inputs));

    if (cur_color != 32768)
	fprintf(fp,"color = %d\n",cur_color);
    if (cur_bright != 32768)
	fprintf(fp,"bright = %d\n",cur_bright);
    if (cur_hue != 32768)
	fprintf(fp,"hue = %d\n",cur_hue);
    if (cur_contrast != 32768)
	fprintf(fp,"contrast = %d\n",cur_contrast);

    fprintf(fp,"\n");

    if (fs_width && fs_height)
	fprintf(fp,"fullscreen = %d x %d\n",fs_width,fs_height);
    if (fs_xoff || fs_yoff)
	fprintf(fp,"wm-off-by = %+d%+d\n",fs_xoff,fs_yoff);
    fprintf(fp,"freqtab = %s\n",chanlists[chantab].name);
    fprintf(fp,"pixsize = %d x %d\n",pix_width,pix_height);
    fprintf(fp,"jpeg-quality = %d\n",jpeg_quality);
    if (mixer)
	fprintf(fp,"mixer = %s\n",mixer);

    fprintf(fp,"\n");

    for (i = 0; i < nlauch; i++) {
	fprintf(fp,"lauch = %s, %s, %s\n",
		lauch[i].name,lauch[i].key,lauch[i].cmdline);
    }
    
    fprintf(fp,"\n");

    /* write channels */
    for (i = 0; i < count; i++) {

	fprintf(fp,"[%s]\n",channels[i]->name);
	fprintf(fp,"channel = %s\n",chanlist[channels[i]->channel].name);
	if (cur_fine != channels[i]->fine)
	    fprintf(fp,"fine = %+d\n", channels[i]->fine);
	if (cur_norm != channels[i]->norm)
	    fprintf(fp,"norm = %s\n",
		    int_to_str(cur_norm,grabber->norms));
	if (channels[i]->key != NULL)
	    fprintf(fp,"key = %s\n",channels[i]->key);
	if (channels[i]->capture != cur_capture)
	    fprintf(fp,"capture = %s\n",
		    int_to_str(channels[i]->capture,captab));
	if (channels[i]->source != cur_input)
	    fprintf(fp,"source = %s\n",
		    int_to_str(channels[i]->source,grabber->inputs));
	  
	if (cur_color != channels[i]->color)
	    fprintf(fp,"color = %d\n",channels[i]->color);
	if (cur_bright != channels[i]->bright)
	    fprintf(fp,"bright = %d\n",channels[i]->bright);
	if (cur_hue != channels[i]->hue)
	    fprintf(fp,"hue = %d\n",channels[i]->hue);
	if (cur_contrast != channels[i]->contrast)
	    fprintf(fp,"contrast = %d\n",channels[i]->contrast);

	fprintf(fp,"\n");
    }
    fclose(fp);
}

/* ----------------------------------------------------------------------- */

struct STRTAB booltab[] = {
    {  0, "no" },
    {  0, "false" },
    {  0, "off" },
    {  1, "yes" },
    {  1, "true" },
    {  1, "on" },
    { -1, NULL }
};

int
str_to_int(char *str, struct STRTAB *tab)
{
    int i;
    
    if (str[0] >= '0' && str[0] <= '9')
	return atoi(str);
    for (i = 0; tab[i].str != NULL; i++)
	if (0 == strcasecmp(str,tab[i].str))
	    return(tab[i].nr);
    return -1;
}

char*
int_to_str(int n, struct STRTAB *tab)
{
    int i;
    
    for (i = 0; tab[i].str != NULL; i++)
	if (tab[i].nr == n)
	    return tab[i].str;
    return NULL;
}

char**
split_cmdline(char *line, int *count)
{
    static char cmdline[1024];
    static char *argv[32];
    int  argc,i;

    strcpy(cmdline,line);
    for (argc=0, i=0; argc<31;) {
	argv[argc++] = cmdline+i;
	while (cmdline[i] != ' ' &&
	       cmdline[i] != '\t' &&
	       cmdline[i] != '\0')
	    i++;
	if (cmdline[i] == '\0')
	    break;
	cmdline[i++] = '\0';
	while (cmdline[i] == ' ' ||
	       cmdline[i] == '\t')
	    i++;
	if (cmdline[i] == '\0')
	    break;
    }
    argv[argc] = NULL;

    *count = argc;
    return argv;
}
