/* ------------------------------------------------------------------------
 *	stringbox.c  --  part of DownScript
 * ------------------------------------------------------------------------
 *
 *	Copyright (C) 1998-1999  Andrew Apted  <ajapted@netspace.net.au>
 *
 *	This program is free software; you can redistribute it and/or
 *	modify it under the terms of the GNU General Public License as
 *	published by the Free Software Foundation; either version 2, or
 *	(at your option) any later version.
 *
 *	You should have received a copy of the GNU General Public
 *	License along with this program; see the file COPYING.  If
 *	not, write to the Free Software Foundation, Inc., 59 Temple
 *	Place - Suite 330, Boston, MA 02111-1307, USA
 *
 * ------------------------------------------------------------------------
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>

#include "memory.h"
#include "list.h"
#include "chars.h"
#include "fonts.h"

#include "stringbox.h"


#define MAX_STRING_TEXT  1024

#define BREAK_STYLES	(STYLE_COMPRESSED | STYLE_EXPANDED |    \
			 STYLE_SUPERSCRIPT | STYLE_SUBSCRIPT |  \
			 STYLE_COURIER | STYLE_SYMBOL)


string_box *new_string_box(int x, int y, int width,
			   int font_width, int font_height, 
			   int font_style, char *text)
{
	string_box *result;

	char strbuf[MAX_STRING_TEXT];
	

	recode_string(strbuf, MAX_STRING_TEXT, text);

	result = (string_box *) new_node(sizeof(string_box));

	result->x = x;
	result->y = y;
	result->width  = width;
	result->font_width  = font_width;
	result->font_height = font_height;
	result->font_style  = font_style;

	result->text = safe_strdup(strbuf);

	result->dist_to_next = -1;

	return result;
}

void free_string_box(string_box *sb)
{
	safe_free(sb->text);

	free_node((Node *) sb);
}

void set_super_and_subscript(string_box *sb, int base_y)
{
	int threshold = sb->font_height / 4;

	if (sb->y > base_y + threshold) {
		sb->font_style |= STYLE_SUPERSCRIPT;
	}

	if (sb->y < base_y - threshold) {
		sb->font_style |= STYLE_SUBSCRIPT;
	}
}

void set_word_gap(string_box *L, string_box *R, int xfudge)
{
	int dist;
	int len;

	if ((L == NULL) || (R == NULL)) {
		return;
	}

	len = strlen(L->text);

	dist = R->x - (L->x + L->width + CALC_FUDGE(L->font_width, xfudge));

	if (dist <= 0) {
		L->dist_to_next = 0;
	} else {
		L->dist_to_next = 1 + (dist - 1) / L->font_width;
	}

	if ((len > 0) && isspace(L->text[len-1])) {
		L->dist_to_next++;
	}

	if (isspace(R->text[0])) {
		L->dist_to_next++;
	}
}

int word_abuts(string_box *L, string_box *R, int xfudge)
{
	if ((L == NULL) || (R == NULL)) {
		return 0;
	}

	if (L->dist_to_next >= 1) {
		return 0;
	}

	if ((L->font_style & BREAK_STYLES) != 
	    (R->font_style & BREAK_STYLES)) {
		return 0;
	}

	/* size checks */

	if ((L->font_style & (STYLE_LARGE | STYLE_HUGE)) &&
	    (R->font_style & (STYLE_SMALL | STYLE_TINY))) {
		return 0;
	}
	if ((L->font_style & (STYLE_SMALL | STYLE_TINY)) &&
	    (R->font_style & (STYLE_LARGE | STYLE_HUGE))) {
		return 0;
	}

	return 1;
}
