/* ------------------------------------------------------------------------
 *	parabox.c  --  part of DownScript
 * ------------------------------------------------------------------------
 *
 *	Copyright (C) 1998-1999  Andrew Apted  <ajapted@netspace.net.au>
 *
 *	This program is free software; you can redistribute it and/or
 *	modify it under the terms of the GNU General Public License as
 *	published by the Free Software Foundation; either version 2, or
 *	(at your option) any later version.
 *
 *	You should have received a copy of the GNU General Public
 *	License along with this program; see the file COPYING.  If
 *	not, write to the Free Software Foundation, Inc., 59 Temple
 *	Place - Suite 330, Boston, MA 02111-1307, USA
 *
 * ------------------------------------------------------------------------
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "maths.h"
#include "memory.h"
#include "list.h"
#include "chars.h"
#include "stringbox.h"
#include "parabox.h"


para_box *new_para_box(int x, int y, int x2, int y2, int join_x2)
{
	para_box *result;

	result = (para_box *) new_node(sizeof(para_box));

	result->x  = x;
	result->y  = y;
	result->x2 = x2;
	result->y2 = y2;

	result->join_x2 = join_x2;

	return result;
}

void free_para_box(para_box *para)
{
	free_node(&para->n);
}


/* ---------------------------------------------------------------------- */


static int para_intersect(para_box *cur, int x, int y, int x2, int y2)
{
	if (x2 < cur->x)  return 0;
	if (x  > cur->x2) return 0;

	if (y2 < cur->y)  return 0;
	if (y  > cur->y2) return 0;

	return 1;
}

static int para_intersect_total(para_box *cur, int x, int y, int x2, int y2)
{
	if (x2 < cur->x)  return 0;
	if (x  > cur->join_x2) return 0;

	if (y2 < cur->y)  return 0;
	if (y  > cur->y2) return 0;

	return 1;
}

void add_stringbox_to_paralist(List *plist, string_box *sb,
			       int xfudge, int yfudge)
{
	para_box *cur;
	para_box *first = NULL;
	para_box *next;

	int x2 = sb->x + sb->width +
		 CALC_FUDGE(sb->font_width, xfudge);

	int y2 = sb->y + sb->font_height + 
		 CALC_FUDGE(sb->font_height, yfudge);

	int x3 = x2 + sb->font_width;

	int count = 0;


	/* Wander along the list, seeing if the string box intersects
	 * with any of the paraboxes.
	 */

	for (cur = (para_box *) plist->head;
	     cur != NULL; 
	     cur = next) {

		next = (para_box *) cur->n.succ;

		if (! para_intersect_total(cur, sb->x, sb->y, x3, y2)) {
			continue;
		}

		count++;

		if (count == 1) {

			first = cur;
			
			/* expand parabox to include the string box */

			first->x = MIN(first->x, sb->x);
			first->y = MIN(first->y, sb->y);

			first->x2 = MAX(first->x2, x2);
			first->y2 = MAX(first->y2, y2);

			first->join_x2 = MAX(first->join_x2, x3);

		} else if (count > 1) {

			/* The string box spans more than one parabox
			 * (what a paradox !). Thus we need to merge
			 * this second parabox into the first (which has
			 * already been enlarged to hold the string box).
			 */
			 
			first->x = MIN(first->x, cur->x);
			first->y = MIN(first->y, cur->y);

			first->x2 = MAX(first->x2, cur->x2);
			first->y2 = MAX(first->y2, cur->y2);

			first->join_x2 = MAX(first->join_x2, cur->join_x2);

			remove_node(plist, &cur->n);

			free_para_box(cur);
		}
	}

	/* No intersections ?  Then add the single bounding box of this
	 * string.
	 */

	if (count == 0) {
		cur = new_para_box(sb->x, sb->y, x2, y2, x3);
		
		add_to_tail(plist, &cur->n);
	}
}

int test_sbox_inside_parabox(para_box *para, string_box *sb)
{
	return  (para->x <= sb->x) && (sb->x <= para->x2) &&
		(para->y <= sb->y) && (sb->y <= para->y2);
}

static int test_horiz_aggregation(para_box *p1, para_box *p2, 
				  List *L1, List *L2, int x_divide)
{
	/* Rules for horizontal aggregation:
	 * ---------------------------------
	 *
	 * 1. Horizontal separation between p1 and p2.
	 *
	 * 2. No vertical separation between p1 and p2.
	 *
	 * 3. When x_divide is >= 0, then either p1 or p2 must cross it,
	 * or both p1 and p2 must be on the same side of it.
	 *
	 * 4. No other paraboxes lie within (completely or in part) the
	 * bounding box of the combined p1 & p2.
	 */

	para_box *cur;

	int x1, y1;
	int x2, y2;
	
	if ((p1->x2 < p2->x) || (p2->x2 < p1->x)) {
		/* OK */
	} else {
		return 0;
	}

	if ((p1->y2 < p2->y) || (p2->y2 < p1->y)) {
		return 0;
	}

	if (x_divide >= 0) {

		if (((p1->x <= x_divide) && (x_divide <= p1->x2)) ||
		    ((p2->x <= x_divide) && (x_divide <= p2->x2))) {
			/* OK */
		} else if (((p1->x2 < x_divide) && (p2->x2 < x_divide)) ||
			   ((p1->x  > x_divide) && (p2->x  > x_divide))) {
			/* OK */
		} else {	
			return 0;
		}
	}

	x1 = MIN(p1->x, p2->x);
	y1 = MIN(p1->y, p2->y);

	x2 = MAX(p1->x2, p2->x2);
	y2 = MAX(p1->y2, p2->y2);

	for (cur = (para_box *) L1->head;
	     cur != NULL; 
	     cur = (para_box *) cur->n.succ) {

		if ((cur == p1) || (cur == p2)) {
			continue;
		}

		if (para_intersect(cur, x1, y1, x2, y2)) {
			return 0;
		}
	}

	for (cur = (para_box *) L2->head;
	     cur != NULL; 
	     cur = (para_box *) cur->n.succ) {

		if ((cur == p1) || (cur == p2)) {
			continue;
		}

		if (para_intersect(cur, x1, y1, x2, y2)) {
			return 0;
		}
	}

	return 1;
}

void horizontal_aggregation(List *plist, int x_divide)
{
	List /* of parabox */ cur_list;
	
	
	/* First, move all paraboxes onto the `cur_list'.  Each parabox
	 * that cannot be aggregrated with another is moved back to the
	 * resulting list.  This is done until all paraboxes have been
	 * moved or aggregrated.
	 */

	init_list(&cur_list);

	concat_list_after(&cur_list, plist);

	while (! list_empty(&cur_list)) {

		para_box *p1 = (para_box *) cur_list.head;
		para_box *p2;
		para_box *next;

		for (p2 = (para_box *) p1->n.succ;
		     p2 != NULL; 
		     p2 = next) {

			next = (para_box *) p2->n.succ;
		
			if (! test_horiz_aggregation(p1, p2, &cur_list,
					plist, x_divide)) {
				continue;
			}

			/* Aha, we've found a parabox (p2) that can
			 * aggregate with p1.
			 */

			p1->x  = MIN(p1->x,  p2->x);
			p1->y  = MIN(p1->y,  p2->y);

			p1->x2 = MAX(p1->x2, p2->x2);
			p1->y2 = MAX(p1->y2, p2->y2);

			remove_node(&cur_list, &p2->n);

			free_para_box(p2);
		}

		add_to_tail(plist, remove_node(&cur_list, &p1->n));
	}
}
