/* ------------------------------------------------------------------------
 *	page.c  --  part of DownScript
 * ------------------------------------------------------------------------
 *
 *	Copyright (C) 1998-1999  Andrew Apted  <ajapted@netspace.net.au>
 *
 *	This program is free software; you can redistribute it and/or
 *	modify it under the terms of the GNU General Public License as
 *	published by the Free Software Foundation; either version 2, or
 *	(at your option) any later version.
 *
 *	You should have received a copy of the GNU General Public
 *	License along with this program; see the file COPYING.  If
 *	not, write to the Free Software Foundation, Inc., 59 Temple
 *	Place - Suite 330, Boston, MA 02111-1307, USA
 *
 * ------------------------------------------------------------------------
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "maths.h"
#include "memory.h"
#include "list.h"
#include "sort.h"
#include "fonts.h"
#include "stringbox.h"
#include "linebox.h"
#include "graphics.h"
#include "output.h"
#include "page.h"


static void to_pixel_coord(view_info *view, int *paper_x, int *paper_y)
{
	*paper_x = *paper_x - view->xpos;
	*paper_y = view->ypos - *paper_y;

	*paper_x = *paper_x / view->scale;
	*paper_y = *paper_y / view->scale;
}

static void to_paper_coord(view_info *view, int *pixel_x, int *pixel_y)
{
	*pixel_x = *pixel_x * view->scale;
	*pixel_y = *pixel_y * view->scale;

	*pixel_x = *pixel_x + view->xpos;
	*pixel_y = view->ypos - *pixel_y;
}

static void draw_page_outline(view_info *view, int width, int height)
{
	int x1 = 0;
	int y1 = height;

	int x2 = width;
	int y2 = 0;
	
	to_pixel_coord(view, &x1, &y1);
	to_pixel_coord(view, &x2, &y2);

	draw_box(view->gfx, 0,  0,  view->gfx->width-1, y1-1, COLOR_BLUE);
	draw_box(view->gfx, 0,  y1, x1-1, y2-1, COLOR_BLUE);
	draw_box(view->gfx, x2, y1, view->gfx->width-1, y2-1, COLOR_BLUE);
	draw_box(view->gfx, 0,  y2, view->gfx->width-1,
		 view->gfx->height-1, COLOR_BLUE);

	draw_box(view->gfx, x1, y1, x2-1, y2-1, COLOR_BLACK);
}

static void draw_string_box(view_info *view, string_box *sb, 
			    int xfudge, int yfudge)
{
	int x = sb->x;
	int y = sb->y + sb->font_height + 
		CALC_FUDGE(sb->font_height, yfudge);

	int x2 = sb->x + sb->width +
		 CALC_FUDGE(sb->font_width, xfudge);
	int y2 = sb->y;

	int x3 = x2 + sb->font_width;
	int y3 = y2;

	int len = strlen(sb->text);
	int color = (sb->font_style & (STYLE_BOLD | STYLE_ITALIC)) ?
			    COLOR_ORANGE : COLOR_YELLOW;

	to_pixel_coord(view, &x,  &y);
	to_pixel_coord(view, &x2, &y2);  x2--; y2--;
	to_pixel_coord(view, &x3, &y3);  x3--; y3--;

	draw_box(view->gfx, x2, y, x3, y3, COLOR_RED);
	draw_box(view->gfx, x,  y, x2, y2, COLOR_WHITE);

	if ((x2-x > 5) && (y2-y > 5)) {
		draw_line(view->gfx, x, y, x2, y2, COLOR_BLACK);
		draw_line(view->gfx, x, y2, x2, y, COLOR_BLACK);
	}

	if ((x2-x >= view->gfx->ch_width * len) &&
	    (y2-y >= view->gfx->ch_height)) {

		draw_string(view->gfx, x, y, COLOR_BLACK, color, sb->text);

	} else if ((x2-x >= view->gfx->ch_width * (len+2) / 2) &&
		   (y2-y >= view->gfx->ch_height * 2)) {

		draw_n_string(view->gfx, x, y, COLOR_BLACK, color, 
			      sb->text, (len+1)/2);
			      
		draw_string(view->gfx,
			    x2 - len * view->gfx->ch_width / 2 - 1, 
			    y+view->gfx->ch_height, COLOR_BLACK, color, 
			    sb->text + (len+1)/2);
	}
}

static void draw_bounding_box(view_info *view, int x, int y, 
				int x2, int y2, int col)
{
	to_pixel_coord(view, &x,  &y);
	to_pixel_coord(view, &x2, &y2);

	draw_line(view->gfx, x, y,  x2, y,  col);
	draw_line(view->gfx, x, y,  x, y2,  col);
	draw_line(view->gfx, x2, y, x2, y2, col);
	draw_line(view->gfx, x, y2, x2, y2, col);
}

static void draw_HUD_text(view_info *view, char *str, int line, int col)
{
	int len  = strlen(str);
	int ch_y = view->gfx->ch_height;

	int x = view->gfx->width-1 - len * view->gfx->ch_width;
	int y = line * ch_y * 2;

	draw_box(view->gfx, x-6, y, view->gfx->width-1, y+ch_y*2-1,
		 COLOR_BLUE);

	draw_string(view->gfx, x-2, y+ch_y/2, col, COLOR_BLUE, str);
}

static void draw_page_information(page_info *pg, view_info *view)
{
	char strbuf[80];

	int cur_line=0;


	sprintf(strbuf, "Page: %d", pg->page_num);
	draw_HUD_text(view, strbuf, cur_line++, COLOR_WHITE);

	sprintf(strbuf, "X-fudge: %d%%", pg->xfudge);
	draw_HUD_text(view, strbuf, cur_line++, COLOR_WHITE);

	sprintf(strbuf, "Y-fudge: %d%%", pg->yfudge);
	draw_HUD_text(view, strbuf, cur_line++, COLOR_WHITE);

	if (pg->locked) {
		draw_HUD_text(view, "LOCKED", cur_line++, COLOR_WHITE);
	}

	cur_line++;

	sprintf(strbuf, "Scale: 1:%d", view->scale);
	draw_HUD_text(view, strbuf, cur_line++, COLOR_YELLOW);

	sprintf(strbuf, "Left: %1.3f\"", 
		(float) view->xpos / (float) INCH_UNIT);
	draw_HUD_text(view, strbuf, cur_line++, COLOR_YELLOW);

	sprintf(strbuf, "Top: %1.3f\"", 
		(float) view->ypos / (float) INCH_UNIT);
	draw_HUD_text(view, strbuf, cur_line++, COLOR_YELLOW);

	if (pg->x_divide > 0) {
		sprintf(strbuf, "X-Divide: %1.3f\"", 
			(float) pg->x_divide / (float) INCH_UNIT);
		draw_HUD_text(view, strbuf, cur_line++, COLOR_YELLOW);
	}

	if (pg->y_divide > 0) {
		sprintf(strbuf, "Y-Divide: %1.3f\"", 
			(float) pg->y_divide / (float) INCH_UNIT);
		draw_HUD_text(view, strbuf, cur_line++, COLOR_YELLOW);
	}
}

static void draw_line_string(view_info *view, line_string *line,
			     int min_x, int max_x, int xfudge, int yfudge)
{
	string_box *sb;
	
	int max_height=0;
	int square;

	for (sb = (string_box *) line->words.head;
	     sb != NULL;
	     sb = (string_box *) sb->n.succ) {
	
		draw_string_box(view, sb, xfudge, yfudge);

		if (sb->font_height > max_height) {
			max_height = sb->font_height;
		}
	}

	draw_bounding_box(view, min_x, line->base_line, max_x, 
			  line->base_line, COLOR_ORANGE);

	if (! view->show_line_flags) {
		return;
	}

	square = max_height / 4;

	if (line->flags & LINEFLAG_INDENTED) {
		draw_bounding_box(view, min_x - square, 
				  line->base_line + square, min_x + square, 
				  line->base_line + square*3, COLOR_GREEN);
	}

	if (line->flags & LINEFLAG_GAP_AT_END) {
		draw_bounding_box(view, max_x - square, 
				  line->base_line + square, max_x + square, 
				  line->base_line + square*3, COLOR_YELLOW);
	}

	if (line->flags & LINEFLAG_DOTTY) {
		draw_bounding_box(view, min_x - square*4, 
			  line->base_line + square, min_x - square*2,
			  line->base_line + square*3, COLOR_RED);
	}
}

static void draw_line_box(view_info *view, line_box *lbox,
			    int xfudge, int yfudge)
{
	line_string *cur;

	for (cur = (line_string *) lbox->lines.tail;
	     cur != NULL;
	     cur = (line_string *) cur->n.pred) {
	
		draw_line_string(view, cur, lbox->outline->x,
			lbox->outline->x2, xfudge, yfudge);
	}

	draw_bounding_box(view, lbox->outline->x, lbox->outline->y,
			lbox->outline->x2, lbox->outline->y2, COLOR_CYAN);
}


/* ---------------------------------------------------------------------- */


view_info *new_view(gfx_info *gfx, int normal_width, int normal_height)
{
	view_info *result;

	result = (view_info *) safe_malloc(sizeof(view_info));

	result->gfx = gfx;

	result->scale = 1 + normal_height / result->gfx->height;

	result->xpos = normal_width / 2  - (gfx->width / 2)  * result->scale;
	result->ypos = normal_height / 2 + (gfx->height / 2) * result->scale;
	
	return result;
}

void free_view(view_info *view)
{
	safe_free(view);
}

int change_position(view_info *view, int xperc, int yperc)
{
	int xstep = xperc * view->gfx->width  * view->scale / 100;
	int ystep = yperc * view->gfx->height * view->scale / 100;

	view->xpos += xstep;
	view->ypos += ystep;

	return 0;
}

int change_scale(view_info *view, int delta)
{
	int x1 = view->gfx->width  / 2;
	int y1 = view->gfx->height / 2;
	
	int x2 = x1;
	int y2 = y1;

	int old_scale = view->scale;


	to_paper_coord(view, &x1, &y1);

	view->scale -= delta;

	if (view->scale < 1) {
		view->scale = 1;
		return (old_scale == view->scale) ? -1 : 0;
	}

	to_paper_coord(view, &x2, &y2);

	view->xpos += x1-x2;
	view->ypos += y1-y2;

	return 0;
}

void set_showflags(view_info *view, int on_off)
{
	view->show_line_flags = on_off;
}

int test_showflags(view_info *view)
{
	return view->show_line_flags;
}


/* ---------------------------------------------------------------------- */


page_info *new_page(int width, int height, int no_HA)
{
	page_info *result;

	result = (page_info *) new_node(sizeof(page_info));

	init_list(&result->boxes);
	init_list(&result->paras);
	init_list(&result->lboxes);

	result->page_num = 0;

	result->page_width  = width;
	result->page_height = height;

	result->x_divide = -1;
	result->y_divide = -1;

	result->xfudge = -90;
	result->yfudge = -90;

	result->minimum_y = -1;
	result->maximum_y = -1;

	result->locked = 0;
	result->no_horiz_aggregation = no_HA;

	return result;
}

void free_page(page_info *pg)
{
	free_list(&pg->lboxes, FREE_NODE &free_line_box);
	free_list(&pg->paras,  FREE_NODE &free_para_box);
	free_list(&pg->boxes,  FREE_NODE &free_string_box);

	free_node((Node *) pg);
}

void add_string_box(page_info *pg, string_box *sb)
{
	add_to_tail(&pg->boxes, &sb->n);
}

void change_fudging(page_info *pg, int xfudge, int yfudge)
{
	pg->xfudge = xfudge;
	pg->yfudge = yfudge;
}

void change_y_bounds(page_info *pg, int ymin, int ymax)
{
	pg->minimum_y = ymin;
	pg->maximum_y = ymax;
}

void draw_page(page_info *pg, view_info *view)
{
	string_box *cur;
	para_box *para;
	line_box *lbox;
	
	frame_hide(view->gfx);

	draw_page_outline(view, pg->page_width, pg->page_height);

	for (cur = (string_box *) pg->boxes.head;
	     cur != NULL;
	     cur = (string_box *) cur->n.succ) {
	
		draw_string_box(view, cur, pg->xfudge, pg->yfudge);
	}

	if (pg->x_divide > 0) {
		draw_bounding_box(view, pg->x_divide, 0, pg->x_divide, 
				  pg->page_height, COLOR_CYAN);
	}
	
	if (pg->y_divide > 0) {
		draw_bounding_box(view, 0, pg->y_divide, pg->page_width,
				  pg->y_divide, COLOR_YELLOW);
	}
	
	for (para = (para_box *) pg->paras.head;
	     para != NULL;
	     para = (para_box *) para->n.succ) {

		draw_bounding_box(view, para->x, para->y,
				  para->x2, para->y2, COLOR_GREEN);
	}

	for (lbox = (line_box *) pg->lboxes.head;
	     lbox != NULL;
	     lbox = (line_box *) lbox->n.succ) {

		draw_line_box(view, lbox, pg->xfudge, pg->yfudge);
	}

	draw_page_information(pg, view);

	frame_show(view->gfx);
}


/* ---------------------------------------------------------------------- */


void set_pagelock(page_info *pg, int on_off)
{
	pg->locked = on_off;
}

int test_pagelock(page_info *pg)
{
	return pg->locked;
}

void set_divider(page_info *pg, int on_off)
{
	string_box *cur;

	int x, i;

	int *vert_histogram;

	int cur_min = 0x77777777;

	int second_same = -1;


	if (! on_off) {
		pg->x_divide = -1;
		return;
	}

	
	vert_histogram = safe_malloc(sizeof(int) * pg->page_width);

	for (x=0; x < pg->page_width; x++) {
		vert_histogram[x] = 0;
	}

	for (cur = (string_box *) pg->boxes.head;
	     cur != NULL;
	     cur = (string_box *) cur->n.succ) {
	
		int x1 = cur->x;
		int x2 = x1 + cur->width;

		if (x1 < 0) x1 = 0;
		if (x2 >= pg->page_width) x2 = pg->page_width-1;

		for (x = x1; x < x2; x++) {
			vert_histogram[x] += cur->font_height;
		}
	}

	for (x=pg->page_width/4; x < pg->page_width*3/4; x++) {

		int sum = 0;

		for (i=-100; i < 100; i++) {
			sum += vert_histogram[x+i];
		}

		if (sum < cur_min) {
			cur_min = sum;
			pg->x_divide = second_same = x;
		}

		if (sum == cur_min) {
			second_same = x;
		}
	}

	pg->x_divide = (pg->x_divide + second_same) / 2;

	safe_free(vert_histogram);
}

int test_divider(page_info *pg)
{
	return (pg->x_divide > 0);
}

void set_paralist(page_info *pg, int on_off)
{
	string_box *cur;

	free_list(&pg->paras, FREE_NODE &free_para_box);

	if (! on_off) {
		return;
	}

	for (cur = (string_box *) pg->boxes.head;
	     cur != NULL;
	     cur = (string_box *) cur->n.succ) {

		add_stringbox_to_paralist(&pg->paras, cur, 
				pg->xfudge, pg->yfudge);
	}
}

int test_paralist(page_info *pg)
{
	return ! list_empty(&pg->paras);
}


/* ---------------------------------------------------------------------- */


static void update_horiz_hist(page_info *pg, int *histogram, 
			      int x1, int y1, int x2, int y2)
{
	int y;

	y1 = MAX(y1, 0);
	y2 = MIN(y2, pg->page_height);

	for (y=y1; y < y2; y++) {
		histogram[y] += (x2-x1);
	}
}

static int test_vert_break(int *histogram, int *start_y, int size)
{
	int y;
	int low_y = *start_y;

	for (y=low_y+size-1; y >= low_y; y--) {

		if (histogram[y] > 0) {
			*start_y = y+1;
			return 0;
		}
	}

	return 1;   /* yes ! */
}

void set_chapter_break(page_info *pg, int on_off)
{
	int half_way;
	int dist_needed;
	int y;
	
	int *horiz_histogram;

	para_box *para;
	line_box *lbox;


	if (! on_off) {
		pg->y_divide = -1;
		return;
	}

	if (pg->y_divide > 0) {
		return;
	}

	pg->y_divide = -1;

	half_way = (pg->maximum_y - pg->minimum_y) / 2;
	dist_needed = (half_way - pg->minimum_y) * 6 / 10;

	horiz_histogram = safe_malloc(sizeof(int) * pg->page_height);

	for (y=0; y < pg->page_height; y++) {
		horiz_histogram[y] = 0;
	}

	/* add each parabox and linebox to histogram */

	for (para = (para_box *) pg->paras.head;
	     para != NULL;
	     para = (para_box *) para->n.succ) {
	
		update_horiz_hist(pg, horiz_histogram, para->x, para->y,
				  para->x2, para->y2);
	}
	
	for (lbox = (line_box *) pg->lboxes.head;
	     lbox != NULL;
	     lbox = (line_box *) lbox->n.succ) {
	
		update_horiz_hist(pg, horiz_histogram, 
				  lbox->outline->x,  lbox->outline->y,
				  lbox->outline->x2, lbox->outline->y2);
	}
	
	/* now look for a large enough blank area */

	for (y=pg->minimum_y; y < (half_way-dist_needed); ) {

		if (test_vert_break(horiz_histogram, &y, dist_needed)) {

			pg->y_divide = y + dist_needed/2;
			break;
		}
	}

	safe_free(horiz_histogram);
}

int test_chapter_break(page_info *pg)
{
	return (pg->y_divide > 0);
}

page_info *split_page_into_two(page_info *pg)
{
	para_box *para;
	page_info *result;

	if (! test_divider(pg)) {
		set_divider(pg, 1);
	}

	if (! test_paralist(pg)) {
		set_paralist(pg, 1);
		
		if (! pg->no_horiz_aggregation) {
			horizontal_aggregation(&pg->paras, pg->x_divide);
		}
	}

	result = new_page(pg->page_width, pg->page_height,
			  pg->no_horiz_aggregation);

	result->xfudge = pg->xfudge;
	result->yfudge = pg->yfudge;

	result->minimum_y = pg->minimum_y;
	result->maximum_y = pg->maximum_y;

	/* Find all paraboxes that lie completely to the right of
	 * x_divide, and move all the string boxes from the old page
	 * `pg' to the new page `result'.
	 */

	for (para = (para_box *) pg->paras.head;
	     para != NULL;
	     para = (para_box *) para->n.succ) {

		string_box *cur;
		string_box *next;
		
		if (para->x < pg->x_divide) {
			continue;
		}

		for (cur = (string_box *) pg->boxes.head;
		     cur != NULL; cur = next) {

			next = (string_box *) cur->n.succ;
		
			if (! test_sbox_inside_parabox(para, cur)) {
				continue;
			}

			remove_node(&pg->boxes, &cur->n);

			add_string_box(result, cur);
		}
	}
	
	pg->x_divide = -1;

	free_list(&pg->paras, FREE_NODE &free_para_box);

	return result;
}

#define HEIGHT_RANGE(lo,x,hi)  (((lo) <= (x)) && ((x) <= (hi)))

void calc_relative_sizes(page_info *pg, int height_normal)
{
	int h_small = height_normal - (height_normal / 10);
	int h_large = height_normal + (height_normal / 10);

	int h_tiny  = height_normal / 2;
	int h_huge  = height_normal * 2;

	string_box *sb;


	for (sb = (string_box *) pg->boxes.head;
	     sb != NULL; 
	     sb = (string_box *) sb->n.succ) {
	
		if (HEIGHT_RANGE(h_small, sb->font_height, h_large)) {
			/* normal */
			
		} else if (HEIGHT_RANGE(h_tiny, sb->font_height, h_small)) {
			sb->font_style |= STYLE_SMALL;

		} else if (HEIGHT_RANGE(h_large, sb->font_height, h_huge)) {
			sb->font_style |= STYLE_LARGE;

		} else if (sb->font_height < h_tiny) {
			sb->font_style |= STYLE_TINY;

		} else if (sb->font_height > h_huge) {
			sb->font_style |= STYLE_HUGE;
		}
	}
}


/* ---------------------------------------------------------------------- */


static int vert_linebox_cmp(line_box *A, line_box *B)
{
	/* Test to see if the boxes span horizontally, in which case the
	 * leftmost one will come before the rightmost one.
	 */
	
	if ((A->outline->x2 < B->outline->x)  &&
	    (A->outline->y  < B->outline->y2) &&
	    (A->outline->y2 > B->outline->y)) {
		return -1;
	}
	
	if ((B->outline->x2 < A->outline->x)  &&
	    (B->outline->y  < A->outline->y2) &&
	    (B->outline->y2 > A->outline->y)) {
		return +1;
	}
	
	return B->outline->y2 - A->outline->y2;
}

void calc_lineboxes(page_info *pg)
{
	para_box *para;
	para_box *next_p;

	if (! test_paralist(pg)) {
		set_paralist(pg, 1);

		if (! pg->no_horiz_aggregation) {
			horizontal_aggregation(&pg->paras, pg->x_divide);
		}
	}

	for (para = (para_box *) pg->paras.head;
	     para != NULL;
	     para = next_p) {

		next_p = (para_box *) para->n.succ;

		remove_node(&pg->paras, &para->n);

		create_and_add_lboxes(&pg->lboxes, para, &pg->boxes);
	}

	/* sort all lineboxes into vertical order */

	merge_sort(&pg->lboxes, COMPARE_FUNC &vert_linebox_cmp);
}

void output_page(page_info *pg, output_info *outp)
{
	line_box *lbox;

	if (list_empty(&pg->lboxes)) {
		calc_lineboxes(pg);
	}

	begin_markup(outp, MARKUP_PAGE);

	for (lbox = (line_box *) pg->lboxes.head;
	     lbox != NULL;
	     lbox = (line_box *) lbox->n.succ) {

		output_paragraph(lbox, outp, pg->xfudge);
	}

	end_markup(outp, MARKUP_PAGE);

	flush_output(outp);
}
