/* ------------------------------------------------------------------------
 *	output.c  --  part of DownScript
 * ------------------------------------------------------------------------
 *
 *	Copyright (C) 1998-1999  Andrew Apted  <ajapted@netspace.net.au>
 *
 *	This program is free software; you can redistribute it and/or
 *	modify it under the terms of the GNU General Public License as
 *	published by the Free Software Foundation; either version 2, or
 *	(at your option) any later version.
 *
 *	You should have received a copy of the GNU General Public
 *	License along with this program; see the file COPYING.  If
 *	not, write to the Free Software Foundation, Inc., 59 Temple
 *	Place - Suite 330, Boston, MA 02111-1307, USA
 *
 * ------------------------------------------------------------------------
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <unistd.h>
#include <errno.h>

#include "memory.h"
#include "maths.h"
#include "chars.h"
#include "parse.h"
#include "rawout.h"
#include "debug.h"
#include "html.h"
#include "ascii.h"
#include "nroff.h"

#include "output.h"


#define CURRENT_TYPES  "HTML ASCII PLAIN DEBUG NROFF"


#define FIXUP_SPACE_PRECEDING_CHARS	".,)>@"


static lowlevel_info *new_lowlevel(FILE *fp, int width)
{
	lowlevel_info *result;

	result = (lowlevel_info *) safe_malloc(sizeof(lowlevel_info));

	result->raw  = new_rawout(fp, width);
	result->priv = NULL;

	return result;
}

static void free_lowlevel(lowlevel_info *lowlev)
{
	free_rawout(lowlev->raw);
	safe_free(lowlev);
}

char *autodetect_type(char *filename)
{
	if ((strcmp(filename, "-") == 0)      ||
	    match_extension(filename, "txt")  ||
	    match_extension(filename, "text") ||
	    match_extension(filename, "asc")  ||
	    match_extension(filename, "ascii")) {

		return "ASCII";
	}

	if (match_extension(filename, "db")  ||
	    match_extension(filename, "dbg") ||
	    match_extension(filename, "debug")) {

		return "DEBUG";
	}

	if (match_extension(filename, "htm") ||
	    match_extension(filename, "html")) {

		return "HTML";
	}

	if (match_extension(filename, "plain")) {

		return "PLAIN";
	}

	if (match_extension(filename, "nroff") ||
	    match_extension(filename, "nro")   ||
	    match_extension(filename, "troff") ||
	    match_extension(filename, "tro")   ||
	    match_extension(filename, "roff")  ||
	    match_extension(filename, "man")   ||
	    match_extension(filename, "1")     ||
	    match_extension(filename, "2")     ||
	    match_extension(filename, "3")     ||
	    match_extension(filename, "4")     ||
	    match_extension(filename, "5")     ||
	    match_extension(filename, "6")     ||
	    match_extension(filename, "7")     ||
	    match_extension(filename, "8")     ||
	    match_extension(filename, "9")) {

		return "NROFF";
	}

	if (match_extension(filename, "raw")) {

		return "RAW";
	}

	/* Unknown extension -- default to plain ascii 
	 */

	fprintf(stderr, "WARNING: Unknown output extension.\n");

	return "PLAIN";
}

static int find_table(output_info *outp, char *type)
{
	if (strcasecmp(type, "DEBUG") == 0) {

		outp->init_f  = &debug_init;
		outp->exit_f  = &debug_exit;
		outp->flush_f = &debug_flush;

		outp->begin_markup_f = &debug_begin_markup;
		outp->end_markup_f   = &debug_end_markup;
		outp->change_style_f = &debug_change_style;

		outp->output_word_f = &debug_output_word;
		outp->line_break_f  = &debug_line_break;
		outp->word_break_f  = &debug_word_break;

		return 1;
	}

	if (strcasecmp(type, "HTML") == 0) {

		outp->init_f  = &html_init;
		outp->exit_f  = &html_exit;
		outp->flush_f = &html_flush;

		outp->begin_markup_f = &html_begin_markup;
		outp->end_markup_f   = &html_end_markup;
		outp->change_style_f = &html_change_style;

		outp->output_word_f = &html_output_word;
		outp->line_break_f  = &html_line_break;
		outp->word_break_f  = &html_word_break;

		return 1;
	}

	if (strcasecmp(type, "ASCII") == 0) {

		outp->init_f  = &ascii_init;
		outp->exit_f  = &ascii_exit;
		outp->flush_f = &ascii_flush;

		outp->begin_markup_f = &ascii_begin_markup;
		outp->end_markup_f   = &ascii_end_markup;
		outp->change_style_f = &ascii_change_style;

		outp->output_word_f = &ascii_output_word;
		outp->line_break_f  = &ascii_line_break;
		outp->word_break_f  = &ascii_word_break;

		return 1;
	}

	if (strcasecmp(type, "PLAIN") == 0) {

		outp->init_f  = &plain_init;
		outp->exit_f  = &plain_exit;
		outp->flush_f = &plain_flush;

		outp->begin_markup_f = &plain_begin_markup;
		outp->end_markup_f   = &plain_end_markup;
		outp->change_style_f = &plain_change_style;

		outp->output_word_f = &plain_output_word;
		outp->line_break_f  = &plain_line_break;
		outp->word_break_f  = &plain_word_break;

		return 1;
	}

	if (strcasecmp(type, "NROFF") == 0) {

		outp->init_f  = &nroff_init;
		outp->exit_f  = &nroff_exit;
		outp->flush_f = &nroff_flush;

		outp->begin_markup_f = &nroff_begin_markup;
		outp->end_markup_f   = &nroff_end_markup;
		outp->change_style_f = &nroff_change_style;

		outp->output_word_f = &nroff_output_word;
		outp->line_break_f  = &nroff_line_break;
		outp->word_break_f  = &nroff_word_break;

		return 1;
	}

	return 0;  /* not found */
}

static void change_style(output_info *outp, int style)
{
	if (style != outp->cur_style) {

		(* outp->change_style_f)(outp->driver, style, 
					 outp->cur_style);
		outp->cur_style = style;
	}
}

static void flush_buffered_word(output_info *outp)
{
	if (outp->buffered_word == NULL) {
		return;
	}

	change_style(outp, outp->buffered_style);

	(* outp->output_word_f)(outp->driver, outp->buffered_word);

	safe_free(outp->buffered_word);

	outp->buffered_word = NULL;

	if (outp->buffered_break > 0) {
		(* outp->word_break_f)(outp->driver, outp->buffered_break);
	}
}

static void concatenate_into_buffer(output_info *outp, char *new_word)
{
	char *str;

	str = safe_strconcat(outp->buffered_word, new_word);

	safe_free(outp->buffered_word);

	outp->buffered_word = str;
}

static void handle_fixups(output_info *outp, char *new_word)
{
	int len1, len2;

	if (outp->buffered_word == NULL) {
		return;
	}

	len1 = strlen(outp->buffered_word);
	len2 = strlen(new_word);


	/* Handle the "foo- " + "bar" case, which is very common
	 * (unfortunately).  Fixing this is just a matter of removing
	 * the break.
	 */

	if ((len1 > 2) && isalpha(outp->buffered_word[len1 - 2]) &&
	    (outp->buffered_word[len1 - 1] == '-') &&
	    (outp->buffered_break > 0) && isalpha(new_word[0])) {
		
		outp->buffered_break = 0;
		return;
	}

	/* Handle cases like "foo " + "," which happens every now and
	 * then.  Again we just remove the break.
	 */

	if ((outp->buffered_break > 0) && (len2 == 1) &&
	    (strchr(FIXUP_SPACE_PRECEDING_CHARS, new_word[0]) != NULL)) {
		
		outp->buffered_break = 0;
		return;
	}

	return;
}


/* ---------------------------------------------------------------------- */


output_info *new_output(char *filename, char *type, int width)
{
	output_info *result;

	FILE *fp;


	result = (output_info *) safe_malloc(sizeof(output_info));

	result->is_stdout  = 0;
	result->cur_style  = 0;
	result->cur_markup = 0;

	result->buffered_word  = NULL;
	result->buffered_style = 0;
	result->buffered_break = 0;

	if (! find_table(result, type)) {

		fprintf(stderr, "Unknown output type '%s'.\n", type);
		fprintf(stderr, "Currently supported types: %s.\n",
			CURRENT_TYPES);
		safe_free(result);
		return NULL;
	}
	
	/* open output file */
	
	if ((filename[0] == 0) || (strcmp(filename, "-") == 0)) {

		result->is_stdout = 1;
		fp = stdout;

	} else {
		fp = fopen(filename, "w");

		if (! fp) {
			fprintf(stderr, "Unable to open file '%s': %s\n",
				filename, strerror(errno));
			safe_free(result);
			return NULL;
		}
	}

	/* initialize low-level driver */

	result->driver = new_lowlevel(fp, width);

	(* result->init_f)(result->driver);

	return result;
}

void free_output(output_info *outp)
{
	flush_buffered_word(outp);

	/* de-initialize low-level driver */

	(* outp->exit_f)(outp->driver);

	if (! outp->is_stdout) {
		fclose(outp->driver->raw->fp);
	}

	free_lowlevel(outp->driver);
	safe_free(outp);
}

void begin_markup(output_info *outp, int markup)
{
	if ((markup < 0) || (markup >= MAX_NR_MARKUP)) {
		return;
	}

	flush_buffered_word(outp);

	(* outp->begin_markup_f)(outp->driver, markup);

	outp->cur_markup |= (1 << markup);
}

void end_markup(output_info *outp, int markup)
{
	if ((markup < 0) || (markup >= MAX_NR_MARKUP)) {
		return;
	}

	flush_buffered_word(outp);
	change_style(outp, 0);

	(* outp->end_markup_f)(outp->driver, markup);

	outp->cur_markup &= ~(1 << markup);
}

void output_word(output_info *outp, char *word, int style)
{
	handle_fixups(outp, word);

	/* See if we can concatenate this word onto the buffered word 
	 */
	if ((outp->buffered_word != NULL) &&
	    (outp->buffered_style == style) &&
	    (outp->buffered_break == 0)) {
		
		/* yep */

		concatenate_into_buffer(outp, word);
		return;
	}

	flush_buffered_word(outp);

	/* now buffer word */

	outp->buffered_word  = safe_strdup(word);
	outp->buffered_style = style;
	outp->buffered_break = 0;
}

void output_word_break(output_info *outp, int size)
{
	if (size < 1) {
		/* zero means `abutting' */
		return;
	}

	if (outp->buffered_word != NULL) {
		outp->buffered_break += size;
		return;
	}
	
	(* outp->word_break_f)(outp->driver, size);
}

void output_line_break(output_info *outp)
{
	flush_buffered_word(outp);

	(* outp->line_break_f)(outp->driver);
}

void flush_output(output_info *outp)
{
	flush_buffered_word(outp);

	(* outp->flush_f)(outp->driver);
}
