/* ------------------------------------------------------------------------
 *	nroff.c  --  part of DownScript
 * ------------------------------------------------------------------------
 *
 *	Copyright (C) 1998-1999  Andrew Apted  <ajapted@netspace.net.au>
 *
 *	This program is free software; you can redistribute it and/or
 *	modify it under the terms of the GNU General Public License as
 *	published by the Free Software Foundation; either version 2, or
 *	(at your option) any later version.
 *
 *	You should have received a copy of the GNU General Public
 *	License along with this program; see the file COPYING.  If
 *	not, write to the Free Software Foundation, Inc., 59 Temple
 *	Place - Suite 330, Boston, MA 02111-1307, USA
 *
 * ------------------------------------------------------------------------
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <signal.h>
#include <sys/types.h>
#include <sys/wait.h>

#include "memory.h"
#include "chars.h"
#include "fonts.h"
#include "rawout.h"
#include "output.h"

#include "nroff.h"


#define NROFF_ROMAN	0
#define NROFF_BOLD	1
#define NROFF_ITALIC	2


static char *nroff_escapes[] =
{
	"'\\'",		/* apostrophe */
	"`\\`",		/* reverse apostrophe */
	".\\&.",	/* dot */
	"-\\-",		/* minus */
	"\\\\\\",	/* backslash */

	"\240 ",	/* no break space */
	"\251(C)",	/* copyright */
	"\265u",	/* greek u */
	"\267 *",	/* bullet */
	"\326OE",	/* O umlaut */
	"\366oe",	/* o umlaut */
	"\370o",	/* o slash */

	NULL
};

#define NROFF_BEGIN_DOCUMENT  \
	".\\\" generated by DownScript.\n"  \
	".nh \\\" turn off hyphenation\n"   \
	".ps 10 \\\" set point size\n"      \
	".vs 12 \\\" set line gap size"


/* ---------------------------------------------------------------------- */


static int to_nroff_style(int style)
{
	if (style & STYLE_BOLD) {
		return NROFF_BOLD;
	}

	if (style & (STYLE_ITALIC | STYLE_UNDERLINE)) {
		return NROFF_ITALIC;
	}

	return NROFF_ROMAN;
}


/* ---------------------------------------------------------------------- */


void nroff_init(lowlevel_info *lowlev)
{
	nroff_priv *NR;
	
	lowlev->priv = NR = (nroff_priv *) safe_malloc(sizeof(nroff_priv));

	NR->in_title = 0;
}

void nroff_exit(lowlevel_info *lowlev)
{
	safe_free(lowlev->priv);
}

void nroff_flush(lowlevel_info *lowlev)
{
	raw_flush(lowlev->raw);
}

void nroff_begin_markup(lowlevel_info *lowlev, int markup)
{
        nroff_priv *NR = NROFF_PRIV(lowlev);

	switch(markup) {

		case MARKUP_DOCUMENT:
			raw_output(lowlev->raw, NROFF_BEGIN_DOCUMENT);
			raw_output(lowlev->raw, "\n");
			break;

		case MARKUP_CHAPTER:
			/* do nothing */
			break;

		case MARKUP_PAGE:
			/* do nothing */
			break;

		case MARKUP_PARAGRAPH:
			raw_flush_word(lowlev->raw);
			raw_output(lowlev->raw, "\n");
			raw_add_token(lowlev->raw, ".PP");
			raw_output(lowlev->raw, "\n");
			break;

		case MARKUP_TITLE:
			raw_flush_word(lowlev->raw);
			raw_output(lowlev->raw, "\n");
			raw_output(lowlev->raw, ".SH ");
			NR->in_title = 1;
			break;
	}
}

void nroff_end_markup(lowlevel_info *lowlev, int markup)
{
        nroff_priv *NR = NROFF_PRIV(lowlev);

	raw_flush_word(lowlev->raw);

	switch(markup) {

		case MARKUP_DOCUMENT:
			/* do nothing */
			break;

		case MARKUP_CHAPTER:
			/* do nothing */
			break;

		case MARKUP_PAGE:
			raw_output(lowlev->raw, "\n");  /* !!! */
			raw_output(lowlev->raw, "\n");
			raw_output(lowlev->raw, "\n");
			raw_output(lowlev->raw, "\n");
			break;

		case MARKUP_PARAGRAPH:
			raw_output(lowlev->raw, "\n");
			break;

		case MARKUP_TITLE:
			raw_output(lowlev->raw, "\n");
			NR->in_title = 0;
			break;
	}
}

void nroff_change_style(lowlevel_info *lowlev, int style, int old_style)
{
        nroff_priv *NR = NROFF_PRIV(lowlev);

	if (NR->in_title) {
		return;
	}

	style     = to_nroff_style(style);
	old_style = to_nroff_style(old_style);

	if (style == old_style) {
		return;
	}

	raw_flush_word(lowlev->raw);
	raw_output(lowlev->raw, "\n");

	switch (style) {
		case NROFF_ROMAN:
			/* !! raw_output(lowlev->raw, ".R"); */
			break;
			
		case NROFF_BOLD:
			raw_output(lowlev->raw, ".B");
			break;

		case NROFF_ITALIC:
			raw_output(lowlev->raw, ".I");
			break;
	}

	raw_output(lowlev->raw, "\n");
}

static char *nroff_lookup_char(char c)
{
	static char buf[100];

	int uc = (int) c & 0xff;

	char *cur = lookup_escape_char(nroff_escapes, c);

	if (cur != NULL) {
		return cur;
	}

	if ((0x20 <= uc) && (uc <= 0x7e)) {
		
		buf[0]=c; buf[1]=0;
		return buf;
	}

	sprintf(buf, "\\%03o", uc);  /* !! wtf ? */
	return buf;
}

void nroff_output_word(lowlevel_info *lowlev, char *word)
{
	for (; *word; word++) {
		raw_add_to_word(lowlev->raw, nroff_lookup_char(*word));
	}
}

void nroff_word_break(lowlevel_info *lowlev, int size)
{
	raw_add_token(lowlev->raw, " ");

	if (size > 1) {
		raw_add_token(lowlev->raw, " ");
	}
}

void nroff_line_break(lowlevel_info *lowlev)
{
	raw_flush_word(lowlev->raw);
	raw_output(lowlev->raw, "\n");
	raw_output(lowlev->raw, ".br");
	raw_output(lowlev->raw, "\n");
}
