/* ------------------------------------------------------------------------
 *	DownScript main program
 * ------------------------------------------------------------------------
 *
 *	Copyright (c) 1998-1999  Andrew Apted  <ajapted@netspace.net.au>
 *
 *	This program is free software; you can redistribute it and/or
 *	modify it under the terms of the GNU General Public License as
 *	published by the Free Software Foundation; either version 2, or
 *	(at your option) any later version.
 *
 *	You should have received a copy of the GNU General Public
 *	License along with this program; see the file COPYING.  If not,
 *	write to the Free Software Foundation, Inc., 59 Temple Place -
 *	Suite 330, Boston, MA 02111-1307, USA
 *
 * ------------------------------------------------------------------------
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <unistd.h>
#include <errno.h>

#include <ggi/ggi.h>

#include "memory.h"
#include "list.h"
#include "parse.h"
#include "fonts.h"
#include "stringbox.h"
#include "graphics.h"
#include "page.h"
#include "document.h"
#include "input.h"
#include "output.h"


#define VERSION_STRING  "DownScript version 2.1 (c) 1998-1999 Andrew Apted."

#define MAX_LINE_BUF  1024


static char *input_str  = "-";
static char *output_str = "-";
static char *format_str = "auto";
static char *mode_str = "";
static char *target_str = NULL;

static int quit=0;
static int output_width = 78;

static int auto_mode=0;
static int double_mode=0;
static int initial_xfudge=8;
static int initial_yfudge=15;
static int no_horiz_agg=0;

static doc_info *doc;
static page_info *page;
static view_info *view;
static input_info *input;
static output_info *output;
static gfx_info *gfx;

static FILE *rawout_fp = NULL;

static int pagenum = 0;
static int first_page = 1;
static int last_page  = 999999;

static int paper_width  = 8  * INCH_UNIT;
static int paper_height = 11 * INCH_UNIT;


static void handle_user(void)
{
	int key;

	draw_page(page, view);
	
	for (;;) {
		key = get_keypress(gfx);

		switch (key) {

		case 'q': case 'Q':
		case GIIUC_Escape:
			quit=1;
			return;

		case '\014': /* CTRL-L: redraw */
			return;

		case '\010': case GIIK_PageUp:
			if (! get_prev_page(doc, page)) {
				continue;
			}
			page = get_prev_page(doc, page);
			return;

		case ' ': case GIIK_PageDown:
			if (! get_next_page(doc, page)) {
				continue;
			}
			page = get_next_page(doc, page);
			return;

		case '<': case '[': case '-':
			if (change_scale(view, -1) < 0) {
				continue;
			}
			return;

		case '>': case ']': case '+':
			if (change_scale(view, +1) < 0) {
				continue;
			}
			return;

		case GIIK_Up:
			if (change_position(view, 0, 10) < 0) {
				continue;
			}
			return;

		case GIIK_Down:
			if (change_position(view, 0, -10) < 0) {
				continue;
			}
			return;

		case GIIK_Right:
			if (change_position(view, 10, 0) < 0) {
				continue;
			}
			return;

		case GIIK_Left:
			if (change_position(view, -10, 0) < 0) {
				continue;
			}
			return;

		case 'x':
			if (change_xfudge(doc, +2) < 0) {
				continue;
			}
			return;

		case 'X':
			if (change_xfudge(doc, -2) < 0) {
				continue;
			}
			return;

		case 'y':
			if (change_yfudge(doc, +5) < 0) {
				continue;
			}
			return;

		case 'Y':
			if (change_yfudge(doc, -5) < 0) {
				continue;
			}
			return;

		case 'd':
			set_divider(page, ! test_divider(page));
			return;

		case 'p':
			set_paralist(page, ! test_paralist(page));
			return;

		case 'c':
			set_chapter_break(page, ! test_chapter_break(page));
			return;

		case 'h':
			horizontal_aggregation(&page->paras, page->x_divide);
			return;

		case 'l':
			set_pagelock(page, ! test_pagelock(page));
			return;

		case 'f':
			set_showflags(view, ! test_showflags(view));
			return;

		case 's':
		{
			page_info *new_pg = split_page_into_two(page);
			add_doc_page(doc, new_pg, page);
			return;
		}

		case 'b':
			calc_lineboxes(page);
			return;

		case 'w':
			output_page(page, output);
			return;

		/* These update the whole document (generally skipping
		 * over locked pages).
		 */
		case 'D':
		{
			page_info *cur = find_page(doc, 1);

			for (; cur != NULL; cur = get_next_page(doc, cur)) {
				if (! test_pagelock(cur)) {
					set_divider(cur, 1);
				}
			}
			return;
		}

		case 'P':
		{
			page_info *cur = find_page(doc, 1);

			for (; cur != NULL; cur = get_next_page(doc, cur)) {
				if (! test_pagelock(cur)) {
					set_paralist(cur, 1);
				}
			}
			return;
		}

		case 'C':
		{
			page_info *cur = find_page(doc, 1);

			for (; cur != NULL; cur = get_next_page(doc, cur)) {
				if (! test_pagelock(cur)) {
					set_chapter_break(cur, 1);
				}
			}
			return;
		}

		case 'H':
		{
			page_info *cur = find_page(doc, 1);

			for (; cur != NULL; cur = get_next_page(doc, cur)) {
				if (! test_pagelock(cur)) {
					horizontal_aggregation(
						&cur->paras, cur->x_divide);
				}
			}
			return;
		}

		case 'L':
		{
			page_info *cur = find_page(doc, 1);

			for (; cur != NULL; cur = get_next_page(doc, cur)) {
				set_pagelock(cur, 0);
			}
			return;
		}

		case 'B':
		{
			page_info *cur = find_page(doc, 1);

			for (; cur != NULL; cur = get_next_page(doc, cur)) {
				if (! test_pagelock(cur)) {
					calc_lineboxes(cur);
				}
			}
			return;
		}

		case 'S':
		{
			page_info *cur = find_page(doc, 1);
			page_info *next;

			for (; cur != NULL; cur = next) {

				next = get_next_page(doc, cur);

				if (! test_pagelock(cur)) {
					page_info *new_pg = 
						split_page_into_two(cur);
					add_doc_page(doc, new_pg, cur);
				}
			}
			
			return;
		}

		case 'W':
		{
			write_document(doc, output);
			return;
		}

		/* End of whole-document changers */

		}
	}
}

static void do_automatic(void)
{
	if (double_mode) {

		page_info *cur = find_page(doc, 1);
		page_info *next;

		for (; cur != NULL; cur = next) {

			next = get_next_page(doc, cur);

			add_doc_page(doc, split_page_into_two(cur), cur);
		}
	}
	
	write_document(doc, output);
}


/* ---------------------------------------------------------------------- */


static void handle_string(char *line)
{
	int x, y, width, len;
	
	char strbuf[MAX_LINE_BUF];

	string_box *sbox;

	line = parse_number(&width,
		parse_string(strbuf, MAX_LINE_BUF,
		 parse_number(&y,
		  parse_number(&x, line))));
	
	/* Check if string is merely all whitespace 
	 */

	len = strlen(strbuf);

	for (; (len > 0) && isspace(strbuf[len-1]); len--) {
	}

	if (len == 0) {
		return;
	}
	
	sbox = new_string_box(x, y, width, doc->font_width, doc->font_height,
				doc->font_style, strbuf);

	add_string_box(page, sbox);

	add_to_histogram(doc, sbox);
}

static void handle_font(char *line)
{
	int height, width;

	char fontbuf[MAX_LINE_BUF];

	line = parse_string(fontbuf, MAX_LINE_BUF,
		parse_number(&width,
		 parse_number(&height, line)));

	if (height == 0) {
		height = DEFAULT_FONT_HEIGHT;
	}
	if (width == 0) {
		width = DEFAULT_FONT_WIDTH;
	}

	doc->font_height = height;
	doc->font_width  = width;
	doc->font_style  = font_name_to_style(fontbuf, width, height);
}

static void handle_rectangle(char *line)
{
}

static void handle_image(char *line)
{
}

static void handle_line(char *line)
{
	switch (line[0]) {
		
		case 'S': handle_string(line+1);
			  break;

		case 'F': handle_font(line+1);
			  break;

		case 'R': handle_rectangle(line+1);
			  break;

		case 'I': handle_image(line+1);
			  break;
	}
}

static int handle_one_page(void)
{
	char linebuf[MAX_LINE_BUF];

	int read_eof;
	int page_empty=1;
	int page_in_range;

	pagenum++;

	page_in_range = (first_page <= pagenum) && (pagenum <= last_page);
	
	if (! rawout_fp) {
		page = new_page(paper_width, paper_height, no_horiz_agg);
	}

	for (;;) {
		read_eof = read_one_line(input->fp, linebuf, MAX_LINE_BUF);

		if ((read_eof < 0) || (linebuf[0] == 'P')) {
			break;
		}
		
		if (! page_in_range) {
			continue;
		}

		if (! rawout_fp) {

			handle_line(linebuf);
			page_empty = 0;
		} else {
			fprintf(rawout_fp, "%s\n", linebuf);
		}
	}

	if (! rawout_fp) {

		if ((! read_eof || ! page_empty) && page_in_range) {
			add_doc_page(doc, page, NULL);   /* add to tail */
		} else {
			free_page(page);
		}
	} else {
		if ((! read_eof || ! page_empty) && page_in_range) {
			fprintf(rawout_fp, "P\n");
		}
	}

	if (! read_eof || ! page_empty) {
		fprintf(stderr, "PAGE %3d  ", pagenum);
	}

	if (read_eof) {
		fprintf(stderr, "\n");
		return -1;
	}

	return (pagenum > last_page) ? -1 : 0;
}


/* ---------------------------------------------------------------------- */


static void usage(void)
{
	fprintf(stderr,
		"\nUSAGE:  downscript  [OPTIONS...]  [in-file]\n\n"
		"OPTIONS:\n"
		"	-m --mode     <mode spec>\n"
		"	-t --target   <target spec>\n"
		"	-o --output   <file>\n"
		"	-i --input    <file>\n"
		"	-f --format   <format name>\n"
		"	-p --pages    <first>,<last>\n"
		"	-x --xfudge   <percent>\n"
		"	-y --yfudge   <percent>\n"
		"	-d --double\n"
		"	-n --nohag\n"
		"	-a --auto\n"
		"	-h --help\n"
		"	-V --version\n\n");
}

static int handle_args(int argc, char **argv)
{
	argc--; 
	argv++;

	#define CMP_OPT(x, short, long)  \
		((strcmp((x), short) == 0) || (strcmp((x), long) == 0))
	
	while (argc > 0) {

		if (CMP_OPT(argv[0], "-h", "--help")) {
			usage();
			return -2;
		}

		if (CMP_OPT(argv[0], "-V", "--version")) {
			fprintf(stderr, VERSION_STRING "\n");
			return -2;
		}

		if ((argc > 1) && CMP_OPT(argv[0], "-m", "--mode")) {
			mode_str = argv[1];
			argc -= 2; argv += 2;
			continue;
		}

		if ((argc > 1) && CMP_OPT(argv[0], "-t", "--target")) {
			target_str = argv[1];
			argc -= 2; argv += 2;
			continue;
		}

		if ((argc > 1) && CMP_OPT(argv[0], "-o", "--output")) {
			output_str = argv[1];
			argc -= 2; argv += 2;
			continue;
		}

		if ((argc > 1) && CMP_OPT(argv[0], "-i", "--input")) {
			input_str = argv[1];
			argc -= 2; argv += 2;
			continue;
		}

		if ((argc > 1) && CMP_OPT(argv[0], "-f", "--format")) {
			format_str = argv[1];
			argc -= 2; argv += 2;
			continue;
		}

		if ((argc > 1) && CMP_OPT(argv[0], "-p", "--pages")) {
			if ((sscanf(argv[1], "%d,%d", &first_page, 
			    &last_page) != 2) || (first_page > last_page)) {

				fprintf(stderr, "Bad page range.\n");
				return -1;
			}
			argc -= 2; argv += 2;
			continue;
		}

		if (CMP_OPT(argv[0], "-a", "--auto")) {
			auto_mode = 1;
			argc--; argv++;
			continue;
		}

		if (CMP_OPT(argv[0], "-d", "--double")) {
			double_mode = 1;
			argc--; argv++;
			continue;
		}

		if (CMP_OPT(argv[0], "-n", "--nohag")) {
			no_horiz_agg = 1;
			argc--; argv++;
			continue;
		}

		if ((argc > 1) && CMP_OPT(argv[0], "-x", "--xfudge")) {
			initial_xfudge = atoi(argv[1]);
			argc -= 2; argv += 2;
			continue;
		}

		if ((argc > 1) && CMP_OPT(argv[0], "-y", "--yfudge")) {
			initial_yfudge = atoi(argv[1]);
			argc -= 2; argv += 2;
			continue;
		}

		if (argv[0][0] == '-') {

			fprintf(stderr, "\ndownscript: Unknown option '%s'"
				".\n(use --help for usage summary).\n\n",
				argv[0]);

			return -1;
		}

		input_str = argv[0];
		argc--; argv++;
	}
	
	#undef CMP_OPT

	return 0;
}

void main(int argc, char **argv, char **envp)
{
	init_memory();

	if (handle_args(argc, argv) < 0) {
		exit(1);
	}

	input = new_input(input_str);

	if (! input) {
		exit(2);
	}
	
	if (strcasecmp(format_str, "auto") == 0) {
		format_str = autodetect_type(output_str);
	}
	
	if (strcasecmp(format_str, "raw") == 0) {
		
		rawout_fp = fopen(output_str, "w");

		if (! rawout_fp) {
			perror("Unable to open rawout file: ");
			exit(3);
		}
	} else {
		output = new_output(output_str, format_str, output_width);

		if (! output) {
			free_input(input);
			exit(4);
		}
	}
	
	doc  = new_document(first_page, initial_xfudge, initial_yfudge);

	while (! handle_one_page()) {
	}

	free_input(input);

	analyse_histogram(doc);
	update_y_bounds(doc);

	if (rawout_fp) {

		fclose(rawout_fp);
		
	} else if (auto_mode) {

		do_automatic();
	} else {

		usleep(500 * 1000);

		gfx = init_graphics(target_str, mode_str);

		if (! gfx) {
			exit(5);
		}

		view = new_view(gfx, paper_width, paper_height);
		page = find_page(doc, 1);

		if (page == NULL) {
			fprintf(stderr, "\nNO PAGES !\n\n");
		} else {
			while (! quit) {

				handle_user();
			}
		}

		free_view(view);

		exit_graphics(gfx);
	}

	free_document(doc);

	if (! rawout_fp) {
		free_output(output);
	}

	exit_memory();

	exit(0);
}
