/* ------------------------------------------------------------------------
 *	list.c  -  part of DownScript
 * ------------------------------------------------------------------------
 *
 *	Copyright (c) 1998-1999  Andrew Apted  <ajapted@netspace.net.au>
 *
 *	This program is free software; you can redistribute it and/or
 *	modify it under the terms of the GNU General Public License as
 *	published by the Free Software Foundation; either version 2, or
 *	(at your option) any later version.
 *
 *	You should have received a copy of the GNU General Public
 *	License along with this program; see the file COPYING.  If
 *	not, write to the Free Software Foundation, Inc., 59 Temple
 *	Place - Suite 330, Boston, MA 02111-1307, USA
 *
 * ------------------------------------------------------------------------
 */

#include <stdlib.h>
#include <string.h>

#include "memory.h"
#include "list.h"


void init_list(List *L)
{
	L->head = NULL;
	L->tail = NULL;
}

Node *new_node(int size)
{
	Node *N;

	N = safe_malloc(size);

	N->succ = NULL;
	N->pred = NULL;

	return N;
}

void free_node(Node *N)
{
	safe_free(N);
}

void add_to_head(List *L, Node *N)
{
	N->succ = L->head;
	N->pred = NULL;

	if (L->head != NULL) {
		L->head->pred = N;
	} else {
		L->tail = N;
	}

	L->head = N;
}

void add_to_tail(List *L, Node *N)
{
	N->succ = NULL;
	N->pred = L->tail;

	if (L->tail != NULL) {
		L->tail->succ = N;
	} else {
		L->head = N;
	}

	L->tail = N;
}

Node *remove_node(List *L, Node *N)
{
	if (N->pred != NULL) {
		N->pred->succ = N->succ;
	} else {
		L->head = N->succ;
	}

	if (N->succ != NULL) {
		N->succ->pred = N->pred;
	} else {
		L->tail = N->pred;
	}

	N->succ = NULL;
	N->pred = NULL;

	return N;
}

void insert_node_before(List *L, Node *N, Node *next)
{
	if (next->pred != NULL) {
		next->pred->succ = N;
	} else {
		L->head = N;
	}

	N->succ = next;
	N->pred = next->pred;

	next->pred = N;
}

void insert_node_after(List *L, Node *N, Node *previous)
{
	if (previous->succ != NULL) {
		previous->succ->pred = N;
	} else {
		L->tail = N;
	}

	N->succ = previous->succ;
	N->pred = previous;

	previous->succ = N;
}

void concat_list_after(List *L, List *old)
{
	if (old->head == NULL) {
		return;
	}

	old->head->pred = L->tail;

	if (L->tail != NULL) {
		L->tail->succ = old->head;
	} else {
		L->head = old->head;
	}

	L->tail = old->tail;

	init_list(old);
}

void concat_list_before(List *L, List *old)
{
	if (old->tail == NULL) {
		return;
	}

	old->tail->succ = L->head;

	if (L->head != NULL) {
		L->head->pred = old->tail;
	} else {
		L->tail = old->tail;
	}

	L->head = old->head;

	init_list(old);
}

void free_list(List *L, void (* freefunc)(Node *))
{
	Node *N;

	while (L->head) {

		N = remove_node(L, L->head);
		
		(* freefunc)(N);
	}
}

Node *find_node(List *L, void *data, 
		int (* searchfunc)(Node *N, void *data))
{
     Node *N;

     for (N=L->head; N != NULL; N=N->succ) {

	     if ((* searchfunc)(N, data)) {
		     return N;   
	     }
     }

     return NULL;
}

Node *find_nth_node(List *L, int n)
{
	Node *N;

	if (n <= 0) {
		return NULL;
	}

	for (N=L->head; N != NULL; N=N->succ, n--) {

		if (n == 1) {
			return N;
		}
	}

	return NULL;
}

void update_list(List *L, void *data,
		 void (* updatefunc)(Node *N, void *data))
{
	Node *N;
	Node *next = NULL;

	for (N=L->head; N != NULL; N=next) {

		next = N->succ;

		(* updatefunc)(N, data);
	}
}

int list_empty(List *L)
{
	return (L->head == NULL);
}

int count_nodes(List *L)
{
	Node *N;

	int result=0;

	for (N=L->head; N != NULL; N=N->succ) {
		result++;
	}

	return result;
}
