/* ------------------------------------------------------------------------
 *	linebox.c  --  part of DownScript
 * ------------------------------------------------------------------------
 *
 *	Copyright (C) 1998-1999  Andrew Apted  <ajapted@netspace.net.au>
 *
 *	This program is free software; you can redistribute it and/or
 *	modify it under the terms of the GNU General Public License as
 *	published by the Free Software Foundation; either version 2, or
 *	(at your option) any later version.
 *
 *	You should have received a copy of the GNU General Public
 *	License along with this program; see the file COPYING.  If
 *	not, write to the Free Software Foundation, Inc., 59 Temple
 *	Place - Suite 330, Boston, MA 02111-1307, USA
 *
 * ------------------------------------------------------------------------
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>

#include "memory.h"
#include "maths.h"
#include "list.h"
#include "sort.h"
#include "stringbox.h"
#include "parabox.h"
#include "linebox.h"


#define DOTTY_CHARS  ".:"

#define MINIMUM_NR_DOTS  4

#define MAX_WORD_SIZE  1024


static line_string *new_line_string(void)
{
	line_string *result;

	result = (line_string *) new_node(sizeof(line_string));

	init_list(&result->words);
	
	result->flags = 0;
	result->base_line = -1;

	return result;
}

void free_line_string(line_string *line)
{
	free_list(&line->words, FREE_NODE &free_string_box);
	free_node((Node *) line);
}

static void calc_base_line(line_string *line)
{
	int sum   = 0;
	int count = 0;

	string_box *sb;

	for (sb = (string_box *) line->words.head;
	     sb != NULL;
	     sb = (string_box *) sb->n.succ) {
		
		sum   += sb->y * sb->width;
		count += sb->width;
	}

	if (count > 0) {
		line->base_line = (sum + count - 1) / count;
	}
}

static void linestring_update_sboxes(line_string *line, int xfudge)
{
	string_box *sb;
	string_box *next;

	for (sb = (string_box *) line->words.head; sb != NULL; sb = next) {

		next = (string_box *) sb->n.succ;

		set_super_and_subscript(sb, line->base_line);

		set_word_gap(sb, next, xfudge);
	}
}


/* ---------------------------------------------------------------------- */


line_box *new_line_box(para_box *para)
{
	line_box *result;

	result = (line_box *) new_node(sizeof(line_box));

	init_list(&result->lines);
	
	result->outline = para;
	result->paragraph_type = PARAGRAPH_SHORT;

	return result;
}

void free_line_box(line_box *lbox)
{
	free_para_box(lbox->outline);
	free_list(&lbox->lines, FREE_NODE &free_line_string);
	free_node((Node *) lbox);
}


/* ---------------------------------------------------------------------- */


int linebox_is_title(line_box *lbox)
{
	line_string *line;
	string_box *sbox;

	int style;

	if (count_nodes(&lbox->lines) != 1) {
		return 0;
	}

	line = (line_string *) lbox->lines.head;

	if (line->flags & LINEFLAG_DOTTY) {
		return 0;
	}

	/* check whether the style is `title-ish' */

	sbox = (string_box *) line->words.head;
	
	if (sbox == NULL) {
		return 0;
	}

	style = sbox->font_style;

	if (style & (STYLE_SMALL | STYLE_TINY | STYLE_COMPRESSED |
		     STYLE_SUPERSCRIPT | STYLE_SUBSCRIPT)) {
		return 0;
	}
	
	if (! (style & (STYLE_BOLD | STYLE_ITALIC | STYLE_UNDERLINE |
			STYLE_LARGE | STYLE_HUGE | STYLE_EXPANDED))) {
		
		/* Style is normal, so let's check if the first
		 * character is a digit (signifying a section heading).
		 */

		if (! isdigit(sbox->text[0])) {
			return 0;
		}
	}
	
	/* check if all string_boxes in the line have the same style */

	for (sbox = (string_box *) sbox->n.succ;
	     sbox != NULL;
	     sbox = (string_box *) sbox->n.succ) {
	
		if (sbox->font_style != style) {
			return 0;
		}
	}

	return 1;
}

static char *concat_line_string(line_string *line)
{
	int length=2;  /* trailing `nul' + safety margin */

	string_box *sbox;

	char *result;


	/* First calculate maximum length */

	for (sbox = (string_box *) line->words.head;
	     sbox != NULL;
	     sbox = (string_box *) sbox->n.succ) {
	
		length += strlen(sbox->text);
	}

	/* Now do the concatenation.
	 * While we're at it, remove all spaces.
	 */

	result = safe_malloc(length);
	length = 0;
	
	for (sbox = (string_box *) line->words.head;
	     sbox != NULL;
	     sbox = (string_box *) sbox->n.succ) {
	
		char *s = sbox->text;

		for (; *s; s++) {
			if (! isspace(*s)) {
				result[length++] = *s;
			}
		}
	}

	result[length] = 0;

	return result;
}

static int linestring_is_dotty(line_string *line)
{
	char *str;
	char *s;

	int found=0;

	s = str = concat_line_string(line);

	/* Look for a sequence of dots ('.') or colons (':'), possibly
	 * interspersed with spaces.  All spaces have been removed from
	 * the string by concat_line_string(), which simplifies things
	 * for us a lot.
	 */

	for (; *s && ! found; s++) {

		int count;
		
		if (strchr(DOTTY_CHARS, s[0]) == NULL) {
			continue;
		}

		count = 1;

		while ((s[count] == s[0]) && (count < MINIMUM_NR_DOTS)) {
			count++;
		}

		if (count >= MINIMUM_NR_DOTS) {
			found = 1;
		}
	}

	safe_free(str);

	return found;
}

static int linestring_is_indented(line_box *lbox, line_string *line)
{
	string_box *first;

	first = (string_box *) line->words.head;

	if (first == NULL) {
		return 0;
	}

	if (first->x < lbox->outline->x + first->font_width / 2) {
		return 0;
	}

	return 1;
}

static int linestring_has_gap_at_end(line_box *lbox, line_string *line)
{
	string_box *last;

	int x_threshold;

	last = (string_box *) line->words.tail;

	if (last == NULL) {
		return 0;
	}

	x_threshold = lbox->outline->x2 - last->font_width * 2;

	if (last->x + last->width > x_threshold) {
		return 0;
	}

	return 1;
}

static void linestring_set_flags(line_box *lbox, line_string *line)
{
	line->flags &= ~LINEFLAG_INDENTED;
	line->flags &= ~LINEFLAG_GAP_AT_END;
	line->flags &= ~LINEFLAG_DOTTY;

	if (linestring_is_indented(lbox, line)) {
		line->flags |= LINEFLAG_INDENTED;
	}

	if (linestring_has_gap_at_end(lbox, line)) {
		line->flags |= LINEFLAG_GAP_AT_END;
	}
	
	if (linestring_is_dotty(line)) {
		line->flags |= LINEFLAG_DOTTY;
	}
}


/* ---------------------------------------------------------------------- */


static int vert_para_cmp(para_box *A, para_box *B)
{
	/* We want higher paraboxes (which have higher y values) to come
	 * first.
	 */

	return B->y - A->y;
}

static int horiz_sbox_cmp(string_box *A, string_box *B)
{
	return A->x - B->x;
}

static void fill_line_box(line_box *lbox, List /* of string_box */ *sboxes)
{
	List line_paras;

	string_box *cur;

	para_box *para;
	para_box *next_p;

	int num_lines=0;
	int num_dots=0;

	int left_gaps=0;
	int right_gaps=0;


	/* Create a set of paraboxes by adding each string-box with a
	 * yfudge of -50%, meaning each string box is half height.  This
	 * should leave enough gap between each line (even, hopefully,
	 * with superscripts and subscripts) that horizontal aggregation
	 * can be used to form each line.
	 */
	
	init_list(&line_paras);

	for (cur = (string_box *) sboxes->head;
	     cur != NULL;
	     cur = (string_box *) cur->n.succ) {
	
		if (! test_sbox_inside_parabox(lbox->outline, cur)) {
			continue;
		}

		add_stringbox_to_paralist(&line_paras, cur, 0, -50);
	}

	horizontal_aggregation(&line_paras, -1);

	
	/* sort paraboxes vertically */

	merge_sort(&line_paras, COMPARE_FUNC &vert_para_cmp);


	/* Now for each parabox in the line_paras list, create a
	 * line_string object, and then attach all stringboxes that lie
	 * within that parabox to that line.
	 */

	for (para = (para_box *) line_paras.head;
	     para != NULL;
	     para = next_p) {

		line_string *line = new_line_string();

		string_box *next_sb;

		next_p = (para_box *) para->n.succ;
	
		for (cur = (string_box *) sboxes->head;
		     cur != NULL;
		     cur = next_sb) {

			next_sb = (string_box *) cur->n.succ;

			if (! test_sbox_inside_parabox(lbox->outline, cur)) {
				continue;
			}

			if (test_sbox_inside_parabox(para, cur)) {

				/* Okidoke, we've found a stringbox that
				 * belongs to this parabox / linestring.
				 * Remove it from the sboxes list and
				 * add it to the linestring's words list.
				 */

				add_to_tail(&line->words,
					remove_node(sboxes, &cur->n));
			}
		}

		/* sort words horizontally */

		merge_sort(&line->words, COMPARE_FUNC &horiz_sbox_cmp);

		/* calculate stuff */

		calc_base_line(line);
		linestring_set_flags(lbox, line);

		if (line->flags & LINEFLAG_DOTTY) {
			num_dots++;
		}
		if (line->flags & LINEFLAG_INDENTED) {
			left_gaps++;
		}
		if (line->flags & LINEFLAG_GAP_AT_END) {
			right_gaps++;
		}

		/* add line_string to line_box */

		add_to_tail(&lbox->lines, &line->n);

		num_lines++;

		/* remove para_box and free it */

		remove_node(&line_paras, &para->n);
		free_para_box(para);
	}

	/* Now see if this linebox looks `normal', by counting the
	 * number of gaps at the left or right of each line.  Non-normal
	 * lineboxes are treated differently (e.g. indentation isn't 
	 * used to indicate where a new sub-paragraph starts).
	 */
	
	if (num_lines < 4) {

		lbox->paragraph_type = PARAGRAPH_SHORT;
		
	} else if (right_gaps >= (num_lines*2+1) / 3) {

		lbox->paragraph_type = PARAGRAPH_UNEVEN_ENDED;
	    
	} else if ((num_dots > 0) ||
	         (left_gaps  >= (num_lines+4) / 3) ||
		 (right_gaps >= (num_lines+4) / 3)) {

		lbox->paragraph_type = PARAGRAPH_ABNORMAL;

	} else {
		lbox->paragraph_type = PARAGRAPH_NORMAL;
	}
}

void create_and_add_lboxes(List /* of line_box */ *lboxes, 
			   para_box *para, 
			   List /* of string_box */ *sboxes)
{
	line_box *lbox;

	lbox = new_line_box(para);

	fill_line_box(lbox, sboxes);

	add_to_tail(lboxes, &lbox->n);
}


/* ---------------------------------------------------------------------- */


static void break_one_sbox(output_info *outp, string_box *sb)
{
	char wordbuf[MAX_WORD_SIZE];

	char *s = sb->text;

	int first=1;


	for (;;) {

		char *t;

		while (*s && isspace(*s)) {
			s++;
		}

		if (*s == 0) {
			return;
		}

		t = wordbuf;

		while (*s && ! isspace(*s)) {
			*t++ = *s++;
		}

		*t = 0;

		if (! first) {
			output_word_break(outp, 1);
		}

		output_word(outp, wordbuf, sb->font_style);

		first = 0;
	}
}

static void output_line(line_string *line, output_info *outp, int xfudge)
{
	string_box *sb;
	string_box *next;

	linestring_update_sboxes(line, xfudge);

	for (sb = (string_box *) line->words.head; sb != NULL; sb = next) {

		next = (string_box *) sb->n.succ;

 		break_one_sbox(outp, sb);

		if (! word_abuts(sb, next, xfudge)) {
			output_word_break(outp, MAX(1, sb->dist_to_next));
		}
	}
}

void output_paragraph(line_box *lbox, output_info *outp, int xfudge)
{
	line_string *line;
	line_string *next;
	
	int in_para=0;

	/* Title */

	if (linebox_is_title(lbox)) {

		begin_markup(outp, MARKUP_TITLE);

		for (line = (line_string *) lbox->lines.head;
		     line != NULL;
		     line = (line_string *) line->n.succ) {
		
			output_line(line, outp, xfudge);
		}

		end_markup(outp, MARKUP_TITLE);

		return;
	}

	/* Paragraph */

	for (line = (line_string *) lbox->lines.head;
	     line != NULL;
	     line = next) {
	
		next = (line_string *) line->n.succ;

		if (! in_para) {
			begin_markup(outp, MARKUP_PARAGRAPH);
			in_para = 1;
		}

		output_line(line, outp, xfudge);

		/* Check if there seems to be a paragraph break between
		 * the line just written, and the following line.  Also
		 * check for a `soft' break (the equivalent of HTML's
		 * <BR> tag).
		 */
		if ((lbox->paragraph_type == PARAGRAPH_NORMAL) && 
		    next && (next->flags & LINEFLAG_INDENTED)) {
		     
			end_markup(outp, MARKUP_PARAGRAPH);
			in_para = 0;

		} else if ((lbox->paragraph_type == PARAGRAPH_UNEVEN_ENDED) 
			   || (line->flags & LINEFLAG_DOTTY) ||
			   ((lbox->paragraph_type == PARAGRAPH_NORMAL) && 
			    next && (line->flags & LINEFLAG_GAP_AT_END))) {
		     
			output_line_break(outp);
		}
	}

	if (in_para) {
		end_markup(outp, MARKUP_PARAGRAPH);
	}
}
