/* ------------------------------------------------------------------------
 *	input.c  --  part of DownScript
 * ------------------------------------------------------------------------
 *
 *	Copyright (C) 1998-1999  Andrew Apted  <ajapted@netspace.net.au>
 *
 *	This program is free software; you can redistribute it and/or
 *	modify it under the terms of the GNU General Public License as
 *	published by the Free Software Foundation; either version 2, or
 *	(at your option) any later version.
 *
 *	You should have received a copy of the GNU General Public
 *	License along with this program; see the file COPYING.  If
 *	not, write to the Free Software Foundation, Inc., 59 Temple
 *	Place - Suite 330, Boston, MA 02111-1307, USA
 *
 * ------------------------------------------------------------------------
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <signal.h>
#include <errno.h>

#include <sys/types.h>
#include <sys/wait.h>

#include "memory.h"
#include "parse.h"
#include "input.h"


#define PIPE_R  0
#define PIPE_W  1

#define IN_FD   0
#define OUT_FD  1
#define ERR_FD  2


static int spawn_ps2ascii(input_info *inp, char *filename, int compressed)
{
	int gs_pipe[2];
	int gzip_pipe[2];


	if (pipe(gs_pipe) < 0) {
		perror("Unable to create pipe");
		return -1;
	}

	if (compressed && (pipe(gzip_pipe) < 0)) {
		perror("Unable to create pipe");
		return -1;
	}

	
	/* spawn Ghostscript process */
	
	fprintf(stderr, "Invoking Ghostscript...\n");

	inp->gs_pid = fork();

	if (inp->gs_pid < 0) {
		perror("Unable to fork");
		return -1;
	}
	
	if (inp->gs_pid == 0) {

		/* we are the child */

		char *argv[] = { "gs", "-q", "-dNODISPLAY", "-dNOBIND",
			"-dWRITESYSTEMDICT", "-dCOMPLEX", "-c", "save",
			"-f", "ps2ascii.ps", "-", "-c", "quit", NULL };

		if (! compressed && ! inp->is_stdin) {
			argv[10] = filename;
		}

		close(OUT_FD);		/* stdout -> gs_pipe */
		dup2(gs_pipe[PIPE_W], OUT_FD);

		close(gs_pipe[PIPE_R]);
		close(gs_pipe[PIPE_W]);

		if (compressed) {

			close(IN_FD);		/* stdin <- gzip_pipe */
			dup2(gzip_pipe[PIPE_R], IN_FD);

			close(gzip_pipe[PIPE_R]);
			close(gzip_pipe[PIPE_W]);
		}
		
		if (execvp(argv[0], argv) < 0) {
			perror("Unable to run ghostscript");
			exit(1);
		}

		exit(0);
	}

	close(gs_pipe[PIPE_W]);	

	inp->gs_fd = gs_pipe[PIPE_R];

	inp->fp = fdopen(inp->gs_fd, "r");
	
	if (inp->fp == NULL) {
		perror("Unable to fdopen pipe");
		kill(inp->gs_pid, SIGKILL);
		waitpid(inp->gs_pid, NULL, 0);
		return -1;
	}

	if (! compressed) {
		return 0;
	}


	/* spawn GZIP process */

	inp->gzip_pid = fork();

	if (inp->gzip_pid < 0) {
		perror("Unable to fork");
		return -1;
	}
	
	if (inp->gzip_pid == 0) {

		/* we are the child */

		char *argv[] = { "gzip", "-d", "-c", "-", NULL };

		argv[3] = filename;

		close(OUT_FD);		/* stdout -> gzip_pipe */
		dup2(gzip_pipe[PIPE_W], OUT_FD);

		close(gzip_pipe[PIPE_R]);
		close(gzip_pipe[PIPE_W]);

		if (execvp(argv[0], argv) < 0) {
			perror("Unable to run gzip");
			exit(1);
		}

		exit(0);
	}

	close(gzip_pipe[PIPE_R]);	
	close(gzip_pipe[PIPE_W]);	

	return 0;
}

void reap_ps2ascii(input_info *inp)
{
	if (inp->gzip_pid >= 0) {
		kill(inp->gzip_pid, SIGKILL);
		waitpid(inp->gzip_pid, NULL, 0);
	}

	kill(inp->gs_pid, SIGKILL);
	waitpid(inp->gs_pid, NULL, 0);

	/* remove some ugly side-effects of ps2ascii.ps */

	unlink("_temp_.out");
	unlink("_temp_.err");
	
	fclose(inp->fp);
	close(inp->gs_fd);
}


/* ---------------------------------------------------------------------- */


input_info *new_input(char *filename)
{
	input_info *result;

	int compressed = 0;

	
	result = (input_info *) safe_malloc(sizeof(input_info));

	result->fp = NULL;

	result->is_stdin = 0;

	result->gs_pid   = -1;
	result->gs_fd    = -1;
	result->gzip_pid = -1;

	if (strcmp(filename, "-") == 0) {
		
		result->is_stdin = 1;
	}
	
	if (match_extension(filename, ".raw")) {

		result->fp = fopen(filename, "r");

		if (! result->fp) {
	
			fprintf(stderr, "Unable to open file '%s': %s\n",
				filename, strerror(errno));
			safe_free(result);
			return NULL;
		}

		return result;
	}

	compressed = match_extension(filename, ".ps.z")  ||
		     match_extension(filename, ".ps.Z")  ||
		     match_extension(filename, ".ps.gz") ||
		     match_extension(filename, ".pdf.z") ||
		     match_extension(filename, ".pdf.Z") ||
		     match_extension(filename, ".pdf.gz");
	
	if (match_extension(filename, ".ps")  ||
	    match_extension(filename, ".pdf") || 
	    compressed || result->is_stdin) {
		
		if (spawn_ps2ascii(result, filename, compressed) < 0) {
			
			/* error occurred */

			safe_free(result);
			return NULL;
		}

		return result;
	}

	fprintf(stderr, "Unknown file extension with `%s'.\n", filename);

	safe_free(result);
	return NULL;
}

void free_input(input_info *inp)
{
	if (inp->gs_pid >= 0) {
		reap_ps2ascii(inp);
	} else {
		fclose(inp->fp);
	}

	safe_free(inp);
}
