/* ------------------------------------------------------------------------
 *	html.c  --  part of DownScript
 * ------------------------------------------------------------------------
 *
 *	Copyright (C) 1998-1999  Andrew Apted  <ajapted@netspace.net.au>
 *
 *	This program is free software; you can redistribute it and/or
 *	modify it under the terms of the GNU General Public License as
 *	published by the Free Software Foundation; either version 2, or
 *	(at your option) any later version.
 *
 *	You should have received a copy of the GNU General Public
 *	License along with this program; see the file COPYING.  If
 *	not, write to the Free Software Foundation, Inc., 59 Temple
 *	Place - Suite 330, Boston, MA 02111-1307, USA
 *
 * ------------------------------------------------------------------------
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <signal.h>
#include <sys/types.h>
#include <sys/wait.h>

#include "memory.h"
#include "chars.h"
#include "output.h"

#include "html.h"


typedef struct markup_entry
{
	char *begin_str;
	char *end_str;

} markup_entry;


#define HTML_DOC_START  \
	"<!DOCTYPE HTML PUBLIC \"-//IETF//DTD HTML 3.2//EN\">\n"  \
	"<!-- generated by DownScript -->\n<HTML>\n<HEAD>\n"	  \
	"<TITLE>Unknown</TITLE>\n</HEAD>\n<BODY>\n"

#define HTML_DOC_END  \
	"</BODY>\n</HTML>\n"


static char *html_escapes[] =
{
	"<&lt;", 
	">&gt;", 
	"&&amp;",
	  
	"\240&nbsp;",		/* no break space */
	"\251(C)",		/* copyright */
	"\265&micro;",		/* greek u */
	"\267&nbsp;&middot;",	/* bullet */
	"\326&Ouml;",		/* O umlaut */
	"\366&ouml;",		/* o umlaut */
	"\370&oslash;",		/* o slash */

	NULL
};

static markup_entry html_markups[MAX_NR_MARKUP] =
{
	{ HTML_DOC_START,	HTML_DOC_END },	/* DOCUMENT */
	{ "",			"" },		/* CHAPTER */
	{ "",			"<HR><BR>\n" },	/* PAGE */
	{ "<P>",		"</P>\n" },	/* PARAGRAPH */
	{ "<H2 ALIGN=LEFT>",	"</H2>\n" },	/* TITLE */
	{ "",			"" },		/* place holder  5 */
	{ "",			"" },		/* place holder  6 */
	{ "",			"" },		/* place holder  7 */
	{ "",			"" },		/* place holder  8 */
	{ "",			"" },		/* place holder  9 */
	{ "",			"" },		/* place holder 10 */
	{ "",			"" },		/* place holder 11 */
	{ "",			"" },		/* place holder 12 */
	{ "",			"" },		/* place holder 13 */
	{ "",			"" },		/* place holder 14 */
	{ "",			"" }		/* place holder 15 */
};

static markup_entry html_styles[MAX_NR_STYLE] =
{
	{ "<B>",		"</B>" },	/* BOLD */
	{ "<I>",		"</I>" },	/* ITALIC */
	{ "<U>",		"</U>" },	/* UNDERLINE */
	{ "",			"" },		/* COMPRESSED */
	{ "",			"" },		/* EXPANDED */
	{ "<FONT SIZE=-2>",	"</FONT>" },	/* TINY */
	{ "<FONT SIZE=-1>",	"</FONT>" },	/* SMALL */
	{ "<FONT SIZE=+2>",	"</FONT>" },	/* LARGE */
	{ "<FONT SIZE=+4>",	"</FONT>" },	/* HUGE */
	{ "<SUP>",		"</SUP>" },	/* SUPERSCRIPT */
	{ "<SUB>",		"</SUB>" },	/* SUBSCRIPT */
	{ "",			"" },		/* TIMES */
	{ "<TT>",		"</TT>" },	/* COURIER */
	{ "",			"" },		/* HELVETICA */
	{ "",			"" },		/* SYMBOL */
	{ "",			"" },		/* place holder 15 */
	{ "",			"" },		/* place holder 16 */
	{ "",			"" },		/* place holder 17 */
	{ "",			"" },		/* place holder 18 */
	{ "",			"" },		/* place holder 19 */
	{ "",			"" },		/* place holder 20 */
	{ "",			"" },		/* place holder 21 */
	{ "",			"" },		/* place holder 22 */
	{ "",			"" }		/* place holder 23 */
};


/* ---------------------------------------------------------------------- */


void html_init(lowlevel_info *lowlev)
{
}

void html_exit(lowlevel_info *lowlev)
{
}

void html_flush(lowlevel_info *lowlev)
{
	raw_flush(lowlev->raw);
}

void html_begin_markup(lowlevel_info *lowlev, int markup)
{
	char token_buf[400];

	if (html_markups[markup].begin_str[0]) {

		sprintf(token_buf, "%s", html_markups[markup].begin_str);
		raw_add_token(lowlev->raw, token_buf);
	}
}

void html_end_markup(lowlevel_info *lowlev, int markup)
{
	char token_buf[400];

	if (html_markups[markup].end_str[0]) {

		sprintf(token_buf, "%s", html_markups[markup].end_str);
		raw_add_token(lowlev->raw, token_buf);
	}
}

void html_change_style(lowlevel_info *lowlev, int style, int old_style)
{
	int i;
	int changers = old_style ^ style;
	int stayers  = old_style & style;

	char token_buf[100];


	if (changers == 0) {
		return;
	}
	
	/* Determine which stayers are not affected by higher priority
	 * changers.  These stayers don't need to be included as an
	 * </XX> ... <XX> pair.  Stayers that *are* affected by higher
	 * priority changers need to be explicitly unset and reset, to
	 * honor the HTML nesting rules.
	 */

	for (i=changers; i > 0; i >>= 1) {
		stayers &= ~i;
	}
	
	/* Turn off everything currently on, except for the remaining
	 * stayers.  The order (lowest priority to highest) is
	 * significant.
	 */
	
	for (i=0; i < MAX_NR_STYLE; i++) {

		int mask = 1 << i;

		if ((old_style & mask) && ! (stayers & mask)) {
			
			sprintf(token_buf, "%s", html_styles[i].end_str);
			raw_add_token(lowlev->raw, token_buf);
		}
	}

	/* Turn on everything in the current style, except for the
	 * remaining stayers.  The order (hightest priority to lowest)
	 * is significant.
	 */
	
	for (i=MAX_NR_STYLE-1; i >= 0; i--) {

		int mask = 1 << i;

		if ((style & mask) && ! (stayers & mask)) {
			
			sprintf(token_buf, "%s", html_styles[i].begin_str);
			raw_add_token(lowlev->raw, token_buf);
		}
	}
}

static char *html_lookup_char(char c)
{
	static char buf[100];

	int uc = (int) c & 0xff;

	char *cur;

	/* look in escape table */

	cur = lookup_escape_char(html_escapes, c);

	if (cur != NULL) {
		return cur;
	}

	/* otherwise construct it */

	if ((0x20 <= uc) && (uc <= 0x7e)) {
		
		buf[0] = c; buf[1] = 0;
		return buf;
	}

	sprintf(buf, "&#%d;", uc);
	return buf;
}

void html_output_word(lowlevel_info *lowlev, char *word)
{
	for (; *word; word++) {
		raw_add_to_word(lowlev->raw, html_lookup_char(*word));
	}
}

void html_word_break(lowlevel_info *lowlev, int size)
{
	if (size == 1) {
		raw_add_token(lowlev->raw, " ");
	} else {
		raw_add_token(lowlev->raw, "&nbsp; ");
	}
}

void html_line_break(lowlevel_info *lowlev)
{
	raw_add_token(lowlev->raw, "<BR>\n");
}
