/* ------------------------------------------------------------------------
 *	graphics.c  --  part of DownScript
 * ------------------------------------------------------------------------
 *
 *	Copyright (c) 1998-1999  Andrew Apted  <ajapted@netspace.net.au>
 *
 *	This program is free software; you can redistribute it and/or
 *	modify it under the terms of the GNU General Public License as
 *	published by the Free Software Foundation; either version 2, or
 *	(at your option) any later version.
 *
 *	You should have received a copy of the GNU General Public
 *	License along with this program; see the file COPYING.  If not,
 *	write to the Free Software Foundation, Inc., 59 Temple Place -
 *	Suite 330, Boston, MA 02111-1307, USA
 *
 * ------------------------------------------------------------------------
 */

#include <stdio.h>
#include <stdlib.h>

#include <ggi/ggi.h>

#include "memory.h"
#include "chars.h"
#include "graphics.h"


gfx_info *init_graphics(char *targetstr, char *modestr)
{
	gfx_info *result;

	ggi_color palette[8] = 
	{
		{0x0000, 0x0000, 0x0000},  /* COLOR_BLACK */
		{0xdddd, 0xdddd, 0xdddd},  /* COLOR_WHITE */
		{0x0000, 0x0000, 0x7777},  /* COLOR_BLUE */
		{0xffff, 0x0000, 0x0000},  /* COLOR_RED */
		{0xffff, 0xffff, 0x0000},  /* COLOR_YELLOW */
		{0x0000, 0xffff, 0x0000},  /* COLOR_GREEN */
		{0x0000, 0xaaaa, 0xffff},  /* COLOR_CYAN */
		{0xffff, 0xaaaa, 0x0000}   /* COLOR_ORANGE */
	};


	if (ggiInit() != 0) {
		return NULL;
	}

	result = (gfx_info *) safe_malloc(sizeof(gfx_info));

	if ((result->vis = ggiOpen(targetstr, NULL)) == NULL) {
		ggiExit();
		safe_free(result);
		return NULL;
	}

	ggiParseMode(modestr, &result->mode);

	ggiCheckMode(result->vis, &result->mode);

	fprintf(stderr, "Using mode ");
	ggiFPrintMode(stderr, &result->mode);
	fprintf(stderr, ".\n");
	
	if (ggiSetMode(result->vis, &result->mode)) {
		ggiClose(result->vis);
		ggiExit();
		safe_free(result);
		return NULL;
	}

	result->width  = result->mode.virt.x;
	result->height = result->mode.virt.y;

	if (GT_SCHEME(result->mode.graphtype) == GT_PALETTE) {
		ggiSetColorfulPalette(result->vis);
	}
		
	result->colors[COLOR_BLACK]  = ggiMapColor(result->vis, palette+0);
	result->colors[COLOR_WHITE]  = ggiMapColor(result->vis, palette+1);
	result->colors[COLOR_BLUE]   = ggiMapColor(result->vis, palette+2);
	result->colors[COLOR_RED]    = ggiMapColor(result->vis, palette+3);
	result->colors[COLOR_YELLOW] = ggiMapColor(result->vis, palette+4);
	result->colors[COLOR_GREEN]  = ggiMapColor(result->vis, palette+5);
	result->colors[COLOR_CYAN]   = ggiMapColor(result->vis, palette+6);
	result->colors[COLOR_ORANGE] = ggiMapColor(result->vis, palette+7);

	ggiSetGCForeground(result->vis, result->colors[COLOR_CYAN]);
	ggiFillscreen(result->vis);

	ggiGetCharSize(result->vis, &result->ch_width, &result->ch_height);
	result->ch_width--;  /* we'll overlap them by 1 pixel */

	return result;
}

void exit_graphics(gfx_info *gfx)
{
	ggiSetDisplayFrame(gfx->vis, 0);
	ggiClose(gfx->vis);
	ggiExit();

	safe_free(gfx);
}

void draw_box(gfx_info *gfx, int x, int y, int x2, int y2, int col)
{
	if ((x2 < 0) || (x >= gfx->mode.virt.x) || 
	    (y2 < 0) || (y >= gfx->mode.virt.y)) {
	    	return;
	}
	
	if (x < 0) x = 0;
	if (y < 0) y = 0;

	if (x2 >= gfx->mode.virt.x) x2 = gfx->mode.virt.x-1;
	if (y2 >= gfx->mode.virt.y) y2 = gfx->mode.virt.y-1;
	
	ggiSetGCForeground(gfx->vis, gfx->colors[col]);
	ggiDrawBox(gfx->vis, x, y, x2-x+1, y2-y+1);
}

void draw_line(gfx_info *gfx, int x, int y, int x2, int y2, int col)
{
	if (((x < 0) && (x2 < 0)) || ((y < 0) && (y2 < 0)) ||
	    ((x >= gfx->mode.virt.x) && (x2 >= gfx->mode.virt.x)) ||
	    ((y >= gfx->mode.virt.y) && (y2 >= gfx->mode.virt.y))) {
	    	return;
	}
	
	ggiSetGCForeground(gfx->vis, gfx->colors[col]);
	ggiDrawLine(gfx->vis, x, y, x2, y2);
}

void draw_n_string(gfx_info *gfx, int x, int y, int fg, int bg, 
		   char *str, int max)
{
	if ((y >= gfx->mode.virt.y) || (y < -gfx->ch_height) ||
	    (x >= gfx->mode.virt.x)) {
	    	return;
	}
	
	ggiSetGCBackground(gfx->vis, gfx->colors[bg]);
	ggiSetGCForeground(gfx->vis, gfx->colors[fg]);

	for (; *str && (max > 0); str++, max--) {

		int ch = *str;

		/* handle non-ascii chars */

		if (ch == '\267') {	/* bullet */
			ch = '*';
		} else if ((ch < 0x20) || (ch > 0x7e)) {
			ch = '?';
		}

		ggiPutc(gfx->vis, x, y, ch);

		x += gfx->ch_width;
	}
}

void draw_string(gfx_info *gfx, int x, int y, int fg, int bg, char *str)
{
	draw_n_string(gfx, x, y, fg, bg, str, 999999);
}

void frame_hide(gfx_info *gfx)
{
	int cur_draw = ggiGetWriteFrame(gfx->vis);
	cur_draw = (cur_draw+1) % gfx->mode.frames;

	ggiSetWriteFrame(gfx->vis, cur_draw);
	ggiSetReadFrame(gfx->vis,  cur_draw);
}

void frame_show(gfx_info *gfx)
{
	ggiSetDisplayFrame(gfx->vis, ggiGetWriteFrame(gfx->vis));
}

unsigned int get_keypress(gfx_info *gfx)
{
	ggi_event ev;

	for (;;) {
		ggiEventRead(gfx->vis, &ev, emKey | emExpose);

		if ((ev.any.type == evKeyPress) ||
		    (ev.any.type == evKeyRepeat)) {
			return ev.key.sym;
		}

		if (ev.any.type == evExpose) {
			/* Return the code for CTRL-L.  It's a bit of a
			 * hack, but it works.
			 */
			return '\014';
		}
	}
}
