/* ------------------------------------------------------------------------
 *	document.c  --  part of DownScript
 * ------------------------------------------------------------------------
 *
 *	Copyright (C) 1998-1999  Andrew Apted  <ajapted@netspace.net.au>
 *
 *	This program is free software; you can redistribute it and/or
 *	modify it under the terms of the GNU General Public License as
 *	published by the Free Software Foundation; either version 2, or
 *	(at your option) any later version.
 *
 *	You should have received a copy of the GNU General Public
 *	License along with this program; see the file COPYING.  If
 *	not, write to the Free Software Foundation, Inc., 59 Temple
 *	Place - Suite 330, Boston, MA 02111-1307, USA
 *
 * ------------------------------------------------------------------------
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "memory.h"
#include "list.h"
#include "fonts.h"
#include "stringbox.h"
#include "page.h"
#include "document.h"


static void clear_histogram(doc_info *doc)
{
	int h;

	for (h=0; h < MAX_FONT_HEIGHT; h++) {
		doc->font_histogram[h] = 0;
	}
}

static void renumber_pages(doc_info *doc)
{
	page_info *cur;

	int cur_num = doc->first_page;
	
	for (cur = (page_info *) doc->pages.head;
	     cur != NULL;
	     cur = (page_info *) cur->n.succ) {
	
		cur->page_num = cur_num;
		cur_num++;
	}
}


/* ---------------------------------------------------------------------- */


doc_info *new_document(int first_page, int xfudge, int yfudge)
{
	doc_info *result;

	result = (doc_info *) safe_malloc(sizeof(doc_info));

	init_list(&result->pages);
	
	result->num_pages = 0;

	result->xfudge = xfudge;
	result->yfudge = yfudge;

	result->font_height = DEFAULT_FONT_HEIGHT;
	result->font_width  = DEFAULT_FONT_WIDTH;
	result->font_style  = DEFAULT_FONT_STYLE;

	clear_histogram(result);

	result->first_page = first_page;

	result->minimum_y = 999999;
	result->maximum_y = 0;

	return result;
}

void free_document(doc_info *doc)
{
	free_list(&doc->pages, FREE_NODE &free_page);
	safe_free(doc);
}

void add_doc_page(doc_info *doc, page_info *pg, page_info *after)
{
	change_fudging(pg, doc->xfudge, doc->yfudge);

	if (! after) {
		add_to_tail(&doc->pages, &pg->n);
		
		pg->page_num = doc->first_page + doc->num_pages;
	} else {
		insert_node_after(&doc->pages, &pg->n, &after->n);

		renumber_pages(doc);
	}

	doc->num_pages++;
}

void add_to_histogram(doc_info *doc, string_box *sb)
{
	int height = sb->font_height;

	if (height >= MAX_FONT_HEIGHT) {
		height = MAX_FONT_HEIGHT-1;
	}

	doc->font_histogram[height] += sb->width;

	/* calculate vertical bounds */

	if (sb->y < doc->minimum_y) {
		doc->minimum_y = sb->y;
	}

	if (sb->y+sb->font_height > doc->maximum_y) {
		doc->maximum_y = sb->y+sb->font_height;
	}
}

page_info *find_page(doc_info *doc, int page_num)
{
	page_info *result = (page_info *) doc->pages.head;

	if (page_num < 1) {
		return NULL;
	}

	for (; (page_num > 1) && result; page_num--) {
		result = (page_info *) result->n.succ;
	}

	return result;
}

page_info *find_last_page(doc_info *doc)
{
	return (page_info *) doc->pages.tail;
}

page_info *get_next_page(doc_info *doc, page_info *pg)
{
	return (page_info *) pg->n.succ;
}

page_info *get_prev_page(doc_info *doc, page_info *pg)
{
	return (page_info *) pg->n.pred;
}

int change_xfudge(doc_info *doc, int delta) 
{
	page_info *cur;

	if (doc->xfudge+delta < -90) {
		return -1;
	}
	
	doc->xfudge += delta; 

	/* update all pages */

	for (cur = (page_info *) doc->pages.head;
	     cur != NULL;
	     cur = (page_info *) cur->n.succ) {
	
		change_fudging(cur, doc->xfudge, doc->yfudge);
	}

	return 0;
}

int change_yfudge(doc_info *doc, int delta) 
{
	page_info *cur;

	if (doc->yfudge+delta < -90) {
		return -1;
	}
	
	doc->yfudge += delta; 

	/* update all pages */

	for (cur = (page_info *) doc->pages.head;
	     cur != NULL;
	     cur = (page_info *) cur->n.succ) {
	
		change_fudging(cur, doc->xfudge, doc->yfudge);
	}

	return 0;
}

/* ---------------------------------------------------------------------- */


void analyse_histogram(doc_info *doc)
{
	int h, i;
	int max=0;

	page_info *cur;


	doc->height_normal = DEFAULT_FONT_HEIGHT;
	
	for (h=3; h < MAX_FONT_HEIGHT-3; h++) {

		int sum = 0;

		for (i=(h - 3); i <= (h + 3); i++) {
			sum += doc->font_histogram[i] / 7;
		}

		if (sum >= max) {
			max = sum;
			doc->height_normal = h;
		}
	}

	for (cur = (page_info *) doc->pages.head;
	     cur != NULL;
	     cur = (page_info *) cur->n.succ) {
	
		calc_relative_sizes(cur, doc->height_normal);
	}
}

void update_y_bounds(doc_info *doc)
{
	page_info *cur;

	for (cur = (page_info *) doc->pages.head;
	     cur != NULL;
	     cur = (page_info *) cur->n.succ) {
	
		change_y_bounds(cur, doc->minimum_y, doc->maximum_y);
	}
}

void write_document(doc_info *doc, output_info *outp)
{
	page_info *cur = find_page(doc, 1);

	int chapter=0;

	begin_markup(outp, MARKUP_DOCUMENT);

	for (; cur != NULL; cur = get_next_page(doc, cur)) {

		if (test_pagelock(cur)) {
			continue;
		}

		if (! chapter) {
			begin_markup(outp, MARKUP_CHAPTER);
			chapter=1;
		}

		output_page(cur, outp);

		set_chapter_break(cur, 1);

		if (test_chapter_break(cur)) {
			end_markup(outp, MARKUP_CHAPTER);
			chapter=0;
		}
	}
	
	if (chapter) {
		end_markup(outp, MARKUP_CHAPTER);
	}
	end_markup(outp, MARKUP_DOCUMENT);

	flush_output(outp);
}
