/* ------------------------------------------------------------------------
 *	debug.c  --  part of DownScript
 * ------------------------------------------------------------------------
 *
 *	Copyright (C) 1998-1999  Andrew Apted  <ajapted@netspace.net.au>
 *
 *	This program is free software; you can redistribute it and/or
 *	modify it under the terms of the GNU General Public License as
 *	published by the Free Software Foundation; either version 2, or
 *	(at your option) any later version.
 *
 *	You should have received a copy of the GNU General Public
 *	License along with this program; see the file COPYING.  If
 *	not, write to the Free Software Foundation, Inc., 59 Temple
 *	Place - Suite 330, Boston, MA 02111-1307, USA
 *
 * ------------------------------------------------------------------------
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <signal.h>
#include <sys/types.h>
#include <sys/wait.h>

#include "memory.h"
#include "chars.h"
#include "fonts.h"
#include "rawout.h"
#include "output.h"

#include "debug.h"


static char *debug_markups[MAX_NR_MARKUP] =
{
	"DOCUMENT",	/* DOCUMENT */
	"CHAPTER",	/* CHAPTER */
	"PAGE",		/* PAGE */
	"PARAGRAPH",	/* PARAGRAPH */
	"TITLE",	/* TITLE */
	"MARKUP_5",	/* place holder  5 */
	"MARKUP_6",	/* place holder  6 */
	"MARKUP_7",	/* place holder  7 */
	"MARKUP_8",	/* place holder  8 */
	"MARKUP_9",	/* place holder  9 */
	"MARKUP_10",	/* place holder 10 */
	"MARKUP_11",	/* place holder 11 */
	"MARKUP_12",	/* place holder 12 */
	"MARKUP_13",	/* place holder 13 */
	"MARKUP_14",	/* place holder 14 */
	"MARKUP_15"	/* place holder 15 */
};

static char *debug_styles[MAX_NR_STYLE] =
{
	"BOLD",		/* BOLD */
	"ITALIC",	/* ITALIC */
	"UNDERLINE",	/* UNDERLINE */
	"COMPRESSED",	/* COMPRESSED */
	"EXPANDED",	/* EXPANDED */
	"TINY",		/* TINY */
	"SMALL",	/* SMALL */
	"LARGE",	/* LARGE */
	"HUGE",		/* HUGE */
	"SUPERSCRIPT",	/* SUPERSCRIPT */
	"SUBSCRIPT",	/* SUBSCRIPT */
	"TIMES",	/* TIMES */
	"COURIER",	/* COURIER */
	"HELVETICA",	/* HELVETICA */
	"SYMBOL",	/* SYMBOL */
	"STYLE_15",	/* place holder 15 */
	"STYLE_16",	/* place holder 16 */
	"STYLE_17",	/* place holder 17 */
	"STYLE_18",	/* place holder 18 */
	"STYLE_19",	/* place holder 19 */
	"STYLE_20",	/* place holder 20 */
	"STYLE_21",	/* place holder 21 */
	"STYLE_22",	/* place holder 22 */
	"STYLE_23"	/* place holder 23 */
};


/* ---------------------------------------------------------------------- */


void debug_init(lowlevel_info *lowlev)
{
}

void debug_exit(lowlevel_info *lowlev)
{
}

void debug_flush(lowlevel_info *lowlev)
{
	raw_flush(lowlev->raw);
}

void debug_begin_markup(lowlevel_info *lowlev, int markup)
{
	char token_buf[100];

	sprintf(token_buf, "[%s]", debug_markups[markup]);
	raw_add_token(lowlev->raw, token_buf);
}

void debug_end_markup(lowlevel_info *lowlev, int markup)
{
	char token_buf[100];

	sprintf(token_buf, "[/%s]\n", debug_markups[markup]);

	raw_add_token(lowlev->raw, token_buf);
}

void debug_change_style(lowlevel_info *lowlev, int style, int old_style)
{
	int i;
	int changed = old_style ^ style;

	char token_buf[100];

	for (i=MAX_NR_STYLE-1; i >= 0; i--) {

		int mask = 1 << i;

		if (changed & mask) {

			sprintf(token_buf, "[%s%s]", (style & mask) ? "" :
				"/", debug_styles[i]);
			raw_add_token(lowlev->raw, token_buf);
		}
	}
}

static char *debug_lookup_char(char c)
{
	static char buf[100];

	int uc = (int) c & 0xff;

	if (c == '\\') {
		return "\\\\";
	}

	if ((0x20 <= uc) && (uc <= 0x7e)) {
		
		buf[0]=c; buf[1]=0;
		return buf;
	}

	sprintf(buf, "\\%03o", uc);
	return buf;
}

void debug_output_word(lowlevel_info *lowlev, char *word)
{
	for (; *word; word++) {
		raw_add_to_word(lowlev->raw, debug_lookup_char(*word));
	}
}

void debug_word_break(lowlevel_info *lowlev, int size)
{
	char token_buf[100];

	if (size == 1) {
		raw_add_token(lowlev->raw, " ");
	} else {
		sprintf(token_buf, "[WORDBREAK_%d] ", size);
		raw_add_token(lowlev->raw, token_buf);
	}
}

void debug_line_break(lowlevel_info *lowlev)
{
	raw_add_token(lowlev->raw, "[LINEBREAK]\n");
}
