/* ------------------------------------------------------------------------
 *	ascii.c  --  part of DownScript
 * ------------------------------------------------------------------------
 *
 *	Copyright (C) 1998-1999  Andrew Apted  <ajapted@netspace.net.au>
 *
 *	This program is free software; you can redistribute it and/or
 *	modify it under the terms of the GNU General Public License as
 *	published by the Free Software Foundation; either version 2, or
 *	(at your option) any later version.
 *
 *	You should have received a copy of the GNU General Public
 *	License along with this program; see the file COPYING.  If
 *	not, write to the Free Software Foundation, Inc., 59 Temple
 *	Place - Suite 330, Boston, MA 02111-1307, USA
 *
 * ------------------------------------------------------------------------
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <signal.h>
#include <sys/types.h>
#include <sys/wait.h>

#include "memory.h"
#include "chars.h"
#include "fonts.h"
#include "rawout.h"
#include "output.h"

#include "ascii.h"


static char *ascii_escapes[] =
{
	"\\\\\\",	/* backslash */
	"\240 ",	/* no break space */
	"\251(C)",	/* copyright */
	"\265u",	/* greek u */
	"\267 *",	/* bullet */
	"\326OE",	/* O umlaut */
	"\366oe",	/* o umlaut */
	"\370o",	/* o slash */

	NULL
};


/* ---------------------------------------------------------------------- */


static void ascii_push_word(lowlevel_info *lowlev, char *s)
{
	ascii_priv *AS = ASCII_PRIV(lowlev);

	char buf[20];
	
	int underline = AS->style & (STYLE_UNDERLINE | STYLE_ITALIC);

	for (; *s; s++) {

		buf[0]=*s; buf[1]=0;

		raw_output(lowlev->raw, buf);

		if (AS->style & STYLE_BOLD) {
			raw_output(lowlev->raw, "\b");
			raw_output(lowlev->raw, buf);
		}

		if (underline) {
			raw_output(lowlev->raw, "\b_");
		}

		if (AS->style & STYLE_EXPANDED) {
			raw_output(lowlev->raw, underline ? "_" : " ");
		}
	}
}

static void ascii_flush_word(lowlevel_info *lowlev)
{
	if (lowlev->raw->x_pos + lowlev->raw->word_len + 1 >= 
	    lowlev->raw->chop_pos) {
		raw_output(lowlev->raw, "\n");
	}
	
	if (lowlev->raw->word_len > 0) {
		ascii_push_word(lowlev, lowlev->raw->word_buf);
	}

	lowlev->raw->word_len = 0;
}

static void ascii_push_break(lowlevel_info *lowlev)
{
	ascii_priv *AS = ASCII_PRIV(lowlev);

	ascii_flush_word(lowlev);

	if (lowlev->raw->x_pos + 2 >= lowlev->raw->chop_pos) {
		raw_output(lowlev->raw, "\n");
		return;
	}
		
	if (AS->style & STYLE_EXPANDED) {
		raw_output(lowlev->raw, "  ");
	} else {
		raw_output(lowlev->raw, " ");
	}
}


/* ---------------------------------------------------------------------- */


void ascii_init(lowlevel_info *lowlev)
{
	ascii_priv *AS;
	
	lowlev->priv = AS = (ascii_priv *) safe_malloc(sizeof(ascii_priv));

	AS->style = 0;
}

void ascii_exit(lowlevel_info *lowlev)
{
	ascii_flush_word(lowlev);

	safe_free(lowlev->priv);
}

void ascii_flush(lowlevel_info *lowlev)
{
	ascii_flush_word(lowlev);

	raw_flush(lowlev->raw);
}

void ascii_begin_markup(lowlevel_info *lowlev, int markup)
{
	switch(markup) {

		case MARKUP_DOCUMENT:
			/* do nothing */
			break;

		case MARKUP_CHAPTER:
			/* do nothing */
			break;

		case MARKUP_PAGE:
			/* do nothing */
			break;

		case MARKUP_PARAGRAPH:
			/* do nothing */
			break;

		case MARKUP_TITLE:
			/* do nothing */
			break;
	}
}

void ascii_end_markup(lowlevel_info *lowlev, int markup)
{
	int i;

	ascii_flush_word(lowlev);

	switch(markup) {

		case MARKUP_DOCUMENT:
			break;

		case MARKUP_CHAPTER:
			break;

		case MARKUP_PAGE:
			for (i=0; i < lowlev->raw->chop_pos - 2; i++) {
				raw_output(lowlev->raw, "-");
			}
			break;

		case MARKUP_PARAGRAPH:
			break;

		case MARKUP_TITLE:
			break;
	}

	raw_output(lowlev->raw, "\n\n");
}

void ascii_change_style(lowlevel_info *lowlev, int style, int old_style)
{
	ascii_priv *AS = ASCII_PRIV(lowlev);

	ascii_flush_word(lowlev);

	AS->style = style;
}

static char *ascii_lookup_char(char c)
{
	static char buf[100];

	int uc = (int) c & 0xff;

	char *cur = lookup_escape_char(ascii_escapes, c);

	if (cur != NULL) {
		return cur;
	}

	if ((0x20 <= uc) && (uc <= 0x7e)) {
		
		buf[0]=c; buf[1]=0;
		return buf;
	}

	sprintf(buf, "\\%03o", uc);
	return buf;
}

void ascii_output_word(lowlevel_info *lowlev, char *word)
{
	for (; *word; word++) {
		raw_add_to_word(lowlev->raw, ascii_lookup_char(*word));
	}
}

void ascii_word_break(lowlevel_info *lowlev, int size)
{
	ascii_push_break(lowlev);

	if (size > 1) {
		ascii_push_break(lowlev);
	}
}

void ascii_line_break(lowlevel_info *lowlev)
{
	ascii_flush_word(lowlev);

	raw_output(lowlev->raw, "\n");
}


/* ---------------------------------------------------------------------- */


void plain_init(lowlevel_info *lowlev)
{
}

void plain_exit(lowlevel_info *lowlev)
{
}

void plain_flush(lowlevel_info *lowlev)
{
	raw_flush(lowlev->raw);
}

void plain_begin_markup(lowlevel_info *lowlev, int markup)
{
	switch(markup) {

		case MARKUP_DOCUMENT:
			/* do nothing */
			break;

		case MARKUP_CHAPTER:
			/* do nothing */
			break;

		case MARKUP_PAGE:
			/* do nothing */
			break;

		case MARKUP_PARAGRAPH:
			/* do nothing */
			break;

		case MARKUP_TITLE:
			/* do nothing */
			break;
	}
}

void plain_end_markup(lowlevel_info *lowlev, int markup)
{
	int i;

	raw_flush_word(lowlev->raw);

	switch(markup) {

		case MARKUP_DOCUMENT:
			break;

		case MARKUP_CHAPTER:
			break;

		case MARKUP_PAGE:
			for (i=0; i < lowlev->raw->chop_pos - 2; i++) {
				raw_output(lowlev->raw, "-");
			}
			break;

		case MARKUP_PARAGRAPH:
			break;

		case MARKUP_TITLE:
			break;
	}

	raw_output(lowlev->raw, "\n\n");
}

void plain_change_style(lowlevel_info *lowlev, int style, int old_style)
{
}

void plain_output_word(lowlevel_info *lowlev, char *word)
{
	for (; *word; word++) {
		raw_add_to_word(lowlev->raw, ascii_lookup_char(*word));
	}
}

void plain_word_break(lowlevel_info *lowlev, int size)
{
	raw_add_token(lowlev->raw, " ");

	if (size > 1) {
		raw_add_token(lowlev->raw, " ");
	}
}

void plain_line_break(lowlevel_info *lowlev)
{
	raw_flush_word(lowlev->raw);
	raw_output(lowlev->raw, "\n");
}
