/* Copyright (C) 1999 Jan Kneschke
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * You should have received a copy of the GNU General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <string.h>

#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <unistd.h>

#include <ggi/ggi.h>

#include "soundbase.h"
#include "soundfft.h"

saftSound *sound ;
saftFFT *fft;

ggi_visual_t vis = NULL;
ggi_mode mode;

ggi_pixel white,black,red,blue,green;

signed short null=0;

int triggpos, mousex = 0,mousey = 0;
float xdiv=0.005,ydiv=256;

void calibrate() {
	float fnull = 0;
	int i;
// null-linie ermitteln
	for (i=0;i<sound->getLength();i++) {
		fnull += (float)sound->getData(i)/sound->getLength();
	}
	null = (short int)rint(fnull);
}

void draw_grid(ggi_visual_t vis) {
	float i = 0;
// hintergrund weiss
	ggiSetGCForeground(vis, white);
	ggiDrawBox(vis,0,0,10,mode.virt.y);
	ggiDrawBox(vis,0,0,mode.virt.x,10);
	
	ggiSetGCForeground(vis, black);
	ggiDrawBox(vis,10,10,mode.virt.x-20,mode.virt.y-20);

	ggiSetGCForeground(vis, green);
// umrandung zeichen	
	ggiDrawHLine(vis, 0+10, 0+10, mode.virt.x-20);
	ggiDrawHLine(vis, 0+10, mode.virt.y-10, mode.virt.x-20);
	ggiDrawVLine(vis, 0+10, 0+10, mode.virt.y-20);
	ggiDrawVLine(vis, mode.virt.x-10, 0+10, mode.virt.y-20);
}

void draw_status(ggi_visual_t vis, uint xpos, uint ypos, uint width, uint height) {
	char str[128];
	int	my = 0,i,j;
	
	ggiSetGCForeground(vis, red);
	
	for (i = 0;i < fft->getMaxAmplitudeLength();i++ ) {
//		if (fft->getAmplitude(fft->getMaxAmplitudeIdx(i)) > 10) {
			sprintf (str, "%5.0f - %4.2f ",fft->getFrequency(fft->getMaxAmplitudeIdx(i)),fft->getAmplitude(fft->getMaxAmplitudeIdx(i)));
			ggiPuts(vis, xpos, ypos+(8*i), str);
//		}
	}
}

void draw_scope(ggi_visual_t vis, uint xpos, uint ypos, uint width, uint height) {
	int x;
	int trigp1,trigp2;
	int i = 0;
	int my = 0;
	int lastpos;
	char str[256];
	
/* Oscilloscope */
// scope gitter zeichnen
	ggiSetGCForeground(vis, green);
	ggiDrawHLine(vis, xpos, ypos, width);
	for (i = xpos; i < xpos + width; i += sound->getSpeed()*xdiv) {
		ggiDrawVLine(vis, i, ypos, height);
	}
/*
searching the triggerpoints
*/
	i = 0;

	if (triggpos) {
		while (i < sound->getLength() && sound->getData(i) > null) i++;
		while (i < sound->getLength() && sound->getData(i) <= null) i++;
	} else {
		while (i < sound->getLength() && sound->getData(i) < null) i++;
		while (i < sound->getLength() && sound->getData(i) >= null) i++;
	}
	
	if (i < sound->getLength()) {
//		printf("1: %i -> %i\n",i,sound->getData(i));
		trigp1 = i;
	} else {
		trigp1 = -1;
		i = 0;
	}

	my = ( ypos+ (height / 2) );
	
	ggiSetGCForeground(vis, blue);

/*
drawing the graph of the oscilloscope 
*/	
	lastpos = my-(sound->getData(i)/ydiv/2);
	for (x = 1;x < width; x++) {
		int nextpos = my-(sound->getData(x+i)/ydiv/2);
		ggiDrawLine(vis,x-1+10,lastpos,x+10,nextpos);
		lastpos = nextpos;
	}
/*
next triggerpoint for automatic frequency measurement
*/
	if (triggpos) {
		while (i < sound->getLength() && sound->getData(i) > null) i++;
		while (i < sound->getLength() && sound->getData(i) <= null) i++;
	} else {
		while (i < sound->getLength() && sound->getData(i) < null) i++;
		while (i < sound->getLength() && sound->getData(i) >= null) i++;
	}

	if (i < sound->getLength()) {
		trigp2 = i;
//		printf("2: %i -> %i\n",i,sound->getData(i));
	} else {
		trigp2 = -1;
	}

	if ((trigp2-trigp1) > 0) {
		
		ggiSetGCForeground(vis, red);
		ggiDrawVLine(vis,trigp2-trigp1+xpos,my-(sound->getData(trigp2)/ydiv)-10,20);

// statusline setzen		
		ggiSetGCForeground(vis, black);
		ggiSetGCBackground(vis, white);
		
		sprintf (str,"fscop: %8.2fHz | %s trigg",
			sound->getSpeed()/(float)(trigp2-trigp1),
			triggpos ? "pos" : "neg");
		ggiPuts(vis, xpos, ypos + height - 10, str);
		ggiSetGCBackground(vis, black);
	}
	
// triggerpunkt einzeichnen
	if (xpos > 5 && (my - (null/ydiv) < (ypos + height))) {
		ggiDrawHLine(vis, xpos - 5, my - (null/ydiv), 5);
	}
}

float lm = 0;
void draw_fft(ggi_visual_t vis) {
	int x;
	int i = 0, mi = 0;
	int my = 0;
	float  scale_log;
	
	int skip_below_10_hz, lasty, lastx;
	
// Spectrum-Analyser
		
	for (i = 5; i < sound->getLength(); i++ ) {
		if (fft->getAmplitude(i) > lm) {
			if (finite(fft->getAmplitude(i)))
				lm = fft->getAmplitude(i);
		}
	}
	i = 0;
	
/* Spectrumanalyser zeichnen */
	ggiSetGCForeground(vis, green);
// spec gitter zeichnen
//	ggiDrawHLine(vis, 0+10, mode.virt.y/2, mode.virt.x-20);

// logarithmic scaling-multiplier (10Hz - [sound->getSpeed()]Hz)
	scale_log = mode.virt.x/(log10(sound->getSpeed()/2)-1);
	
	for (i = 10;i<mode.virt.x-10;i += (float)scale_log) {
		ggiDrawVLine(vis, i, 0+10, (mode.virt.y-20)/2);
	}
	
	skip_below_10_hz = (int)(10 * sound->getLength() / sound->getSpeed());
	
/* getting the maximun values */
	my = 0;
	ggiSetGCForeground(vis, white);
	lasty = (mode.virt.y*(float)(fft->getAmplitude( 0 )/(float)(lm)))/2;
	lastx = 10;
	
	for (x = 1 + skip_below_10_hz; x < sound->getLength(); x++) {
		int ll = (mode.virt.y*(float) ( fft->getAmplitude( x )/ (float)(lm) ))/2;
		int kk = (log10( x * (sound->getSpeed()/(sound->getLength()-skip_below_10_hz))/2 ) * scale_log)+10-scale_log;
		ggiDrawVLine(vis,kk,(mode.virt.y / 2)-ll, ll );
		ggiDrawLine(vis,lastx,(mode.virt.y / 2)-lasty,kk,(mode.virt.y / 2)-ll);
		lasty = ll;
		lastx = kk;
	}
	
/* drawing the maximum values */
	ggiSetGCForeground(vis, red);
	for (i=0; i < fft->getMaxAmplitudeLength();i++) {
		int ll = (mode.virt.y*(float) ( fft->getAmplitude( fft->getMaxAmplitudeIdx (i) )/ (float)(lm) ))/2;
		int kk = (log10( fft->getMaxAmplitudeIdx (i) * (sound->getSpeed()/(sound->getLength()-skip_below_10_hz))/2 ) * scale_log)+10-scale_log;
		ggiDrawVLine(vis,kk,(mode.virt.y / 2)-ll-5, 10 );
	}
}





void show_data() {
	char *notes;
	draw_grid(vis);
	draw_fft(vis);
	draw_scope(vis, 10, mode.virt.y/2, mode.virt.x-200, mode.virt.y/2-10);
	draw_status(vis, 10 + mode.virt.x - 200, mode.virt.y/2, 190, mode.virt.y/2-10);
	
	// mouse-line zeichen
	ggiDrawVLine(vis,mousex > 0+10 ? mousex : 0+10,mousey > mode.virt.y/2 ? mode.virt.y/2 : 0+10,mode.virt.y/2-10);

	ggiFlush(vis);
	
//	if ((notes = get_notes()) != NULL) {
//		get_chord(notes);
//		free(notes);
//	}
}

void display_data() {
	if (mousex > 10 && mousey < mode.virt.y/2) {
		printf ("DUMMY: %f\n",fft->getFrequency(mousex-10));
		printf ("F-CORR: %f\n",(float)sound->getSpeed()/sound->getLength());
	}
}

int main(int argc, char *argv[])
{
	ggi_color map;
	int key = ' ', showdata = !0, frame = 0,i;
	struct timeval tv = {0,0}, tp, tc, t1, t2, t3;

// grafik einstellen	
	ggiInit();
	
	if ((vis = ggiOpen(NULL)) == 0) {
		ggiExit();
		perror("Can't open visual");
		exit(-1);
	}
	
	ggiSetFlags(vis,GGIFLAG_ASYNC);
	
	ggiParseMode("", &mode);
// double buffering
	mode.frames = 2;
	
	if (ggiSetMode(vis,&mode)) {
		ggiExit();
		perror("Can't set default mode");
		exit(-1);
	}
	
	ggiGetMode(vis,&mode);

// farben definieren	
	map.r= map.g = map.b = 0xFFFF;
	white=ggiMapColor(vis, &map);
	
	map.r= map.g= map.b= 0x0;
	black=ggiMapColor(vis, &map);
	
	map.r=0xFFFF;
	red=ggiMapColor(vis, &map);
	
	map.b=0xFFFF;
	map.r=0x0;
	blue=ggiMapColor(vis, &map);
	
	map.g=0xFFFF;
	map.b=0x0;
	green=ggiMapColor(vis, &map);
		
	ggiSetGCBackground(vis, black);
	
// null einstellen
//	calibrate();
	sound = new saftSound();
	fft = new saftFFT(sound);
	
//	init_chords();
	
// positive triggern
	triggpos = !0;
	
// los geht's
	i = 0;
	gettimeofday(&tp, NULL);
	while (key != 'q' && key != 'Q') {
		key = ggiEventPoll(vis,emKeyPress | emPtrAbsolute, &tv);
		if (key & emKeyPress) {
			key = ggiGetc(vis);
			switch(key) {
				case 'p':
				case 'P':
					showdata = !showdata;
					break;
				case 8:
					triggpos = !triggpos;
					break;
				case 'd':
				case 'D':
					display_data();
					break;
				case 'k':
				case 'K':
					calibrate();
					break;
#if 0
				case '+':
					ydiv = ydiv * 2;
					break;
				case '-':
					ydiv = ydiv / 2;
					break;
#endif
				default:
					break;
			}
#if 1
		} else if (key & emPtrAbsolute) {
			ggi_event ev;
			ggiEventRead(vis, &ev, emPtrAbsolute);
			if (ev.any.type == evPtrAbsolute) {
				mousex = ev.pmove.x;
				mousey = ev.pmove.y;
			}
#endif			
		}
		
		if (showdata) {
			if (mode.frames > 1) {
				ggiSetDisplayFrame(vis,frame);
				frame++;
				if (frame == mode.frames) frame=0;
				ggiSetWriteFrame(vis,frame);
			}
			
			gettimeofday(&t1, NULL);
			sound->sampleData();
			fft->calc();
			gettimeofday(&t2, NULL);
			show_data();
			gettimeofday(&t3, NULL);
#if 1
			printf ("fft: %fms -> display: %fms\n",
				(float)((t2.tv_sec  - t1.tv_sec)  * 1000 + (t2.tv_usec - t1.tv_usec) / 1000),
				(float)((t3.tv_sec  - t2.tv_sec)  * 1000 + (t3.tv_usec - t2.tv_usec) / 1000));
#endif				
			i++;
		}
	}
	gettimeofday(&tc, NULL);
	printf ("%f updates/s\n",i*1000/(float)((tc.tv_sec  - tp.tv_sec)  * 1000 + (tc.tv_usec - tp.tv_usec) / 1000));
	
	delete fft;
	delete sound;
	
	ggiClose(vis);
	ggiExit();
	return 0;
}
