
// The GGI graphics module (Sam Lantinga's VGA framebuf, stripped, then added GGI
// stuff from my (=W.H.Scholten) own C++ framebuffer class)
//

#if defined(USE_GGI)
#include <sys/types.h>
#include <sys/time.h>
#include <sys/ioctl.h>
#include <fcntl.h>
#include <stdio.h>
#include <unistd.h>
#include <ctype.h>
#include <errno.h>
#include <stdlib.h>

#include "ggi_framebuf.h"
#include "cursor_sprite.h"
#include "keyboard.h"
#include "vga_keys.h"

#if 1
#define rgb_triple ggi_color
#else
typedef struct rgb_triple {
//	unsigned char red;
//	unsigned char green;
//	unsigned char blue;
        unsigned short red;
	unsigned short green;
	unsigned short blue;
	};
#endif

/* The following are almost IBM standard color codes */
static rgb_triple vga_colors[] = {
	{0x00,0x00,0x00},{0x18,0x18,0xB2},{0x18,0xB2,0x18},{0x18,0xB2,0xB2},
	{0xB2,0x18,0x18},{0xB2,0x18,0xB2},{0xB2,0x68,0x18},{0xB2,0xB2,0xB2},
	{0x68,0x68,0x68},{0x54,0x54,0xFF},{0x54,0xFF,0x54},{0x54,0xFF,0xFF},
	{0xFF,0x54,0x54},{0xFF,0x54,0xFF},{0xFF,0xFF,0x54},{0xFF,0xFF,0xFF}
};


static GGI_FrameBuf *THIS=NULL;


GGI_FrameBuf:: GGI_FrameBuf(unsigned short width, unsigned short height, 
					int bpp) : FrameBuf(width, height, bpp)
{
	int    i;

	/* Make sure we only have one GGI framebuffer per process */
	if ( THIS ) {
		error("GGI_FrameBuf: GGI frame buffer already running!\n");
		exit(255);
	}

	/* Verify the width and height */
	if ( (width != 640) || (height != 480) ) {
		error("GGI_FrameBuf only supports 640x480 resolution!\n");
		exit(255);
	}


	for ( i=0; i<256; ++i )
		pressed_keys[i] = 0;


        if (ggiInit()) {
	    fprintf(stderr, "GGI: libggi error\n");
	    return;
	}
	ggierror=1;
//	DEB1(prt("GGI: ggiInit OK\n"));


#if 0
	visual=ggiOpen("display-KGI:/dev/graphic", NULL);
	if (visual==NULL) {
	    fprintf(stderr, "GGI: failed to create visual\n");
	    return;
	}
#else
	if ( !(visual = ggiOpen (NULL)) ) { //automatisch op X of KGI
	    fprintf(stderr, "GGI: failed to create visual\n");
	    return;
	}
#endif

	ggierror=2;

	int ggi_error;
	/* Open our DISPLAY */
	ggi_error=ggiSetGraphMode(visual, width, height, width, height, GT_8BIT);
//	DEB1(prt("modeswitch to %dx%d (error= %i)\n", full_width, full_height, ggi_error));
	if (ggi_error) {
	    fprintf(stderr, "GGI: Can't set screen mode..\n");
	    return;
	}
	ggierror=3;

	/* Direct display buffer */
	dbuf = NULL;
	if ( (ggi_error = ggiDBGetBuffer (visual, &dbuf)) ) {
	    fprintf(stderr, "GGI: couldn't get framebuffer access\n");
	    return;
	}
	if (ggiDBGetLayout (dbuf) != blPixelLinearBuffer) {
	    fprintf(stderr, "GGI: non linear framebuffer not supported\n");
	    return;
	}
	if ( !(plb = ggiDBGetPLB (dbuf)) ) {
	    fprintf(stderr, "GGI: GetPLB Failed\n");
	    return;
	}



#ifdef LOW_RES
	/* Allocate virtual context and associated buffer */
//	low_res_mem = new unsigned char[320*240];
//	gl_setcontextvirtual(320, 240, 1, 8, low_res_mem);
#endif /* LOW_RES */


#ifdef DOUBLE_BUFFER
	/* Enable page flipping for smooth animation on screen */
//	gl_enablepageflipping(screen);
#endif
	/* Set basic color palette */
//	gl_setrgbpalette();

	/* Get a black pixel and fill our pixel array with it */
	ggi_color colour={ 0,0,0 };
	Black = ggiMapColor(visual, &colour);

	for ( i=0; i<NUM_COLORS; ++i )
		Pixel_colors[i] = Black;

	/* Allocate a screen pixmap, and set it black */
	shared_len = (width * height);	// 8 bits per pixel
	backbuf = new unsigned char[shared_len];
	memset(backbuf, Black, shared_len);
	shared_mem = new unsigned char[shared_len];
	memset(shared_mem, Black, shared_len);

	/* Initialize Event Queues */
	Events = new Queue<XEvent>(ChunkSize);
	key_state = 0;
	Reefer = new Stack<Area>(ChunkSize);

	/* Set mouse handler */
	mouseSprite.width = *((unsigned short *)cursor_sprite);
	mouseSprite.height = *((unsigned short *)&cursor_sprite[2]);
	mouseSprite.pixels = (unsigned char *)&cursor_sprite[4];
	mouseSprite.mask = (unsigned char *)&cursor_sprite[
				(mouseSprite.width*mouseSprite.height)+4];
	behind_mouse = new unsigned char
				[mouseSprite.width*mouseSprite.height];
#ifdef CENTER_MOUSE
	mouseX = WIDTH/2;
	mouseY = HEIGHT/2;
#else
	mouseX = 0;
	mouseY = 0;
#endif
	mouseButtons = 0;
	mouse_accel = 1.0;
	THIS = this;

	/* Set the cursor */
	hidden_cursor = 1;
	Show_Cursor();

}

GGI_FrameBuf:: ~GGI_FrameBuf()
{

	/* Free the video memory */
#ifdef LOW_RES
	delete[] low_res_mem;
#endif
	delete[] backbuf;
	delete[] shared_mem;
	delete   Reefer;
	delete   Events;

	THIS = NULL;
}

int
GGI_FrameBuf:: Alloc_Cmap(Color Cmap[NUM_COLORS])
{
	return(Alloc_Private_Cmap(Cmap));
}

/* Allocate a private 256 color colormap :) */
/* Note: The orig_colors array is here to keep a record of the _actual_
   contents of the palette.  This is used during the Fade, and
   differs from Color_Map because the Trident SVGA card requires the
   zero'th element of the VGA palette to be the background color, black.
*/
static rgb_triple orig_colors[NUM_COLORS];
int
GGI_FrameBuf:: Alloc_Private_Cmap(Color Cmap[NUM_COLORS])
{
	int        c;

#ifdef VGA_16COLOR
	/* 16 color VGA? - Ick! */
	/* Most of the mapping code is adapted from 'xv' */
	int ri, gi, bi;			/* The color shades we want */
	int rd, gd, bd;			/* The color shades we have */
	int mdist, close, i, d;		/* Distance and closest pixel */

	/* Map the whole colormap */
	for ( i=0; i<NUM_COLORS; ++i ) {
		mdist = 100000; close=0;

		ri = (Cmap[i].r >> 8);
		gi = (Cmap[i].g >> 8);
		bi = (Cmap[i].b >> 8);


		/* Cycle through the colors we have */
		for ( c=0; c<16; ++c ) {
			rd = ri - vga_colors[c].r;
			gd = gi - vga_colors[c].g;
			bd = bi - vga_colors[c].b;

			d = rd*rd + gd*gd + bd*bd;

			if ( d < mdist ) {
				mdist = d;
				close = c;
			}
		}
#ifdef DEBUG
error("Desired RGB (%d,%d,%d) mapped to existing RGB (%d,%d,%d)\n",
				ri, gi, bi, vga_colors[close].red,
			vga_colors[close].green, vga_colors[close].blue);
#endif
		Pixel_colors[i] = close;
		Color_Map[i].r  = (vga_colors[close].r<<8);
		Color_Map[i].g  = (vga_colors[close].g<<8);
		Color_Map[i].b  = (vga_colors[close].b<<8);
	}
	return(16);
#endif /* VGA_16COLOR */

	// Gently refresh a black screen in the new colormap
	memset(orig_colors, 0, NUM_COLORS*sizeof(rgb_triple));
//	gl_setpalette(orig_colors);
	ggiSetPaletteVec(visual, 0, 256, orig_colors);
	for ( c=0; c<NUM_COLORS; ++c ) {
		Color_Map[c].red   = Cmap[c].red<<8;
		Color_Map[c].green = Cmap[c].green<<8;
		Color_Map[c].blue  = Cmap[c].blue<<8;
		orig_colors[c].r    = Cmap[c].red <<8;
		orig_colors[c].g    = Cmap[c].green <<8;
		orig_colors[c].b    = Cmap[c].blue <<8;
		Pixel_colors[c]    = c;

		if ( !Cmap[c].red && !Cmap[c].green && !Cmap[c].blue )
			Black = c;
	}
	orig_colors[Black].r = orig_colors[0].r;
	orig_colors[Black].g = orig_colors[0].g;
	orig_colors[Black].b = orig_colors[0].b;
	orig_colors[0].r = orig_colors[0].g = orig_colors[0].b = 0;
	Pixel_colors[Black] = 0;
	Pixel_colors[0] = Black;
	Black = 0;
	memset(backbuf, Black, shared_len);
	memset(shared_mem, Black, shared_len);
	Refresh();
//	gl_setpalette(orig_colors);
	ggiSetPaletteVec(visual, 0, 256, orig_colors);

	return(NUM_COLORS);
}

void
GGI_FrameBuf:: Hide_Cursor(void)
{
	if ( hidden_cursor ==  0 ) {
		showmouse = 0;
		EraseMouse();
		hidden_cursor = 1;
	}
}

void
GGI_FrameBuf:: Show_Cursor(void)
{
	if ( hidden_cursor == 1 ) {
		DrawMouse();
		showmouse = 1;
		hidden_cursor = 0;
	}
}

/* This function was adapted from 'xscreensaver', by Jamie Zawinski 

	-- Thanks! :)
*/
void
GGI_FrameBuf:: Fade(int steps)
{
	static int state = XFADE_IN;
	rgb_triple Colors[NUM_COLORS];
	int        i, c;

	/* Find out the state of the fade */
	state = ( (state == XFADE_IN) ? XFADE_OUT : XFADE_IN );

	/* Do we really fade? */
	switch (DoFade) {
		case FADE_FAKE:
			/* Do a pixellated fade */
			Pixel_Fade(state);
			
		case FADE_NONE:
			/* We're done */
			return;

		case FADE_REAL:
#ifdef VGA_16COLOR
			/* Do a pixellated fade */
			Pixel_Fade(state);
			return;
#else
			/* Continue... */
			break;
#endif
	}

	if ( state == XFADE_OUT ) {
		for ( i=steps; i > 0; i-- ) {
			for (c = 0; c < NUM_COLORS; c++) {
				Colors[c].r   =
					((orig_colors[c].r   * i) / steps);
				Colors[c].g =
					((orig_colors[c].g * i) / steps);
				Colors[c].b  =
					((orig_colors[c].b  * i) / steps);
			}
//			gl_setpalette(Colors);
			ggiSetPaletteVec(visual, 0, 256, orig_colors);
			select_usleep(5);
		}
	} else {
		Refresh();
		for ( i=0; i < steps; i++ ) {
			for (c = 0; c < NUM_COLORS; c++) {
				Colors[c].r   =
					((orig_colors[c].r   * i) / steps);
				Colors[c].g =
					((orig_colors[c].g * i) / steps);
				Colors[c].b  =
					((orig_colors[c].b  * i) / steps);
			}
//			gl_setpalette(Colors);
			ggiSetPaletteVec(visual, 0, 256, orig_colors);
			select_usleep(5);
		}
	}
	
	if ( state == XFADE_IN ) {
		/* Restore the original colormap */
		for (c = 0; c < NUM_COLORS; c++) {
			Colors[c].r   = orig_colors[c].r;
			Colors[c].g   = orig_colors[c].g;
			Colors[c].b   = orig_colors[c].b;
		}
//		gl_setpalette(Colors);
		ggiSetPaletteVec(visual, 0, 256, orig_colors);
	}
}
	
/* BTW, you _can_ pass negative x0 and y0 values to this function */
void
GGI_FrameBuf:: RefreshArea(int x0, int y0, int width, int height)
{
	Area area;

	/* Do bounds checking */
	if ( y0 < 0 ) {
		if ( (height += y0) <= 0 )
			return;
		y0 = 0;
	}
	if ( (y0 + height) >= HEIGHT ) {
		if ( y0 > HEIGHT )
			return;
		height = (HEIGHT-y0);
	}
	if ( x0 < 0 ) {
		if ( (width += x0) <= 0 )
			return;
		x0 = 0;
	}
	if ( (x0 + width) >= WIDTH ) {
		if ( x0 > WIDTH )
			return;
		width = (WIDTH-x0);
	}
	/* Set up the area and queue it */
	area.x = x0;
	area.y = y0;
	area.width = width;
	area.height = height;
	Reefer->Add(&area);
}

#ifdef LOW_RES
void
GGI_FrameBuf:: shrinkwrap(void)
{
	unsigned char *from = shared_mem;
	unsigned char *to   = low_res_mem;
	int i, j;

	for ( i=HEIGHT/2; i; --i ) {
		for ( j=WIDTH; j; j -= 2 ) {
			*(to++) = *from;
			from += 2;
		}
		from += WIDTH;
	}
}
void
GGI_FrameBuf:: shrinkwrap(int x, int y, int w, int h, unsigned char *data)
{
	unsigned char *to = low_res_mem;
	int i, j;

	/* Adjust entry offsets */
	if ( y%2 ) {
		data += w;
		++y;
	}
	if ( x%2 ) {
		++data;
		++x;
	}
	to += (((y/2)*(WIDTH/2))+(x/2));

	for ( i=h/2; i; --i ) {
		for ( j=w/2; j; --j ) {
			*(to++) = *data;
			data += 2;
		}
		to += (WIDTH/2)-(w/2);
		data += w;
	}
}
#endif /* LOW_RES */

void
GGI_FrameBuf:: Refresh(void)
{
	/* Don't refresh during fake pixel fade */
	if ( faded ) return;

	Reefer->Flush();

#ifdef LOW_RES
	shrinkwrap();
	if ( showmouse )
		DrawMouse();
	else
//		gl_copyscreen(screen);
#else
//	gl_putbox(0, 0, WIDTH, HEIGHT, shared_mem);
	ggiPutBox(visual, 0, 0, WIDTH, HEIGHT, shared_mem);
	if ( showmouse )
		DrawMouse();
#ifdef VGA_16COLOR
//	gl_copyscreen(screen);
#endif
#endif /* LOW_RES */
}

void
GGI_FrameBuf:: FlushEvents(void)
{
	int    nevents;
	XEvent event;

	for ( nevents=NumEvents(); nevents; --nevents )
		GetEvent(&event);
}



/* Does the nasty key decoding stuff (WHS)
 */
static int ggi_key(int c, int code) {
    int result;


    switch(KTYP(U(c))) {

    case KT_LATIN:
    case KT_LETTER:
	case KT_META:
		result =KVAL(U(c));
		if (result<32) {
//	    result |=0xFF00;
			if (result == 9) result=XK_Tab;
			else if (result == 8)  result=XK_BackSpace;
			else if (result == 27) result=XK_Escape;
			else if (result == 28) result=XK_Print;
			else {
					/* here I'm assuming CTRL+A-Z was pressed etc */
				result +='A'-1;
//				DEB1(prt("unhandled latin sym <32 (assume CTRL+letter)%d\n", result));
			}
		}
		else if (result==127)
			result=XK_Delete;
	else {
	    result=tolower(result);
	}
	return result;
    default:
	switch(U(c)) {

	case K_P0: return XK_KP_0;
	case K_P1: return XK_KP_1;
	case K_P2: return XK_KP_2;
	case K_P3: return XK_KP_3;
	case K_P4: return XK_KP_4;
	case K_P5: return XK_KP_5;
	case K_P6: return XK_KP_6;
	case K_P7: return XK_KP_7;
	case K_P8: return XK_KP_8;
	case K_P9: return XK_KP_9;

	case K_PPLUS: return XK_KP_Add;
	case K_PMINUS: return XK_KP_Subtract;
	case K_PSTAR: return XK_KP_Multiply;
	case K_PSLASH: return XK_KP_Divide;
	case K_PENTER: return XK_KP_Enter;
//	case K_PCOMMA: return KEY_PAD;
	case K_PDOT: return XK_KP_Decimal;
//	case K_PPLUSMINUS: return KEY_PAD;
//	case K_PPARENL: return KEY_PAD;
//	case K_PPARENR: return KEY_PAD;
//	case K_P: return KEY_PAD;

//Not def'd: (why?)
//	case K_HOME: return XK_Home;
        case K_FIND: return XK_Home;

//	case K_END: return XK_End;
	case K_SELECT: return XK_End;

	case K_UP: return XK_Up;
	case K_DOWN: return XK_Down;
	case K_LEFT: return XK_Left;
	case K_RIGHT: return XK_Right;

	    //Should choose the next two depending on keycode & keysym:
	case K_NORMAL_SHIFT:
	    if (code==54) return XK_Shift_R;
	    return XK_Shift_L;
	case K_NORMAL_CTRL:
	    if (code==97) return XK_Control_R;
	    return XK_Control_L;
	case K_NORMAL_ALT: return XK_Alt_L;
	case K_NORMAL_ALTGR: return XK_Alt_R;
	case K_NORMAL_SHIFTL: return XK_Shift_L;
	case K_NORMAL_SHIFTR: return XK_Shift_R;
	case K_NORMAL_CTRLL: return XK_Control_L;
	case K_NORMAL_CTRLR: return XK_Control_R;

	case K_CAPS: return XK_Caps_Lock;

	case K_ENTER: return XK_Return;

	case K_PGUP: return XK_Page_Up;
	case K_PGDN: return XK_Page_Down;

	case K_INSERT: return XK_Insert;
	case K_REMOVE: return XK_Delete;

	case K_PAUSE: return XK_Pause;
	case K_HOLD: return XK_Scroll_Lock;

//Not defined = latin 9.
//	case K_TAB: return XK_Tab;
	case K_F1: return XK_F1;
	case K_F2: return XK_F2;
	case K_F3: return XK_F3;
	case K_F4: return XK_F4;
	case K_F5: return XK_F5;
	case K_F6: return XK_F6;
	case K_F7: return XK_F7;
	case K_F8: return XK_F8;
	case K_F9: return XK_F9;
	case K_F10: return XK_F10;
	case K_F11: return XK_F11;
	case K_F12: return XK_F12;

	default:
//	    DEB1(prt("   GGI undhandled key event, keycode=%d (keysym=%d)\n", code, c));
	    return 0;
	}
    }
    return 0;
}


/* WHS: my own version of the kbd decoding (X emulation), 
 * defines are almost the same as Sam's version
 */
#define SHIFT_R_BIT	0x01
#define SHIFT_L_BIT	0x02
#define CTRL_R_BIT	0x04
#define CTRL_L_BIT	0x08
#define ALT_R_BIT	0x10
#define ALT_L_BIT	0x20
#define CAPSLOCK_BIT	0x40

/* To check keyboard state (is either SHIFT key pressed?, ditto CTRL, ALT) */
#define SHIFT_STATE(X)	( (X & CAPSLOCK_BIT) ? (!(X & (SHIFT_R_BIT | SHIFT_L_BIT))) : (X & (SHIFT_R_BIT | SHIFT_L_BIT)) )
#define CTRL_STATE(X)	(X & (CTRL_R_BIT | CTRL_L_BIT))
#define ALT_STATE(X)	(X & (ALT_R_BIT | ALT_L_BIT))


#define UP_BIT	0x80

int
GGI_FrameBuf:: NumEvents(void)
{
    XEvent xevent;
    struct timeval tv;
    int  key;

    tv.tv_sec = 0;
    tv.tv_usec = 0;

    ggi_event event;
    ggi_event_mask event_mask;
//    ggi_event_mask event_return;
    int c, keycode;



    event_mask= (ggi_event_mask) (emKeyPress | emKeyRelease | emPtrRelative | emPtrButtonPress | emPtrButtonRelease);
    if (ggi_key_repeat)  event_mask |=emKeyRepeat;

    while (ggiEventPoll(visual, event_mask, &tv)) {
//	event_return= ggiEventPoll(visual, event_mask, tv);
	ggiEventRead(visual, &event, event_mask);
	switch(event.any.type) {
	case evKeyRepeat:
	case evKeyPress:
	    keycode=event.key.code;
	    c=event.key.sym;
//	    DEB1(prt("ggi key press %d (sym=%d)\n", keycode, c));
	    break;
	case evKeyRelease:
	    keycode=event.key.code;
	    c=event.key.sym;
//	    DEB1(prt("ggi key release %d (sym=%d)\n", keycode, c));
	    break;
	case evPtrRelative:	/* pointer movements reported relative	*/
//	    DEB1(prt("GGI event: pointer rel motion dx=%d, dy=%d\n", event.pmove.x, event.pmove.y));

	    xevent.type=MotionNotify;
#if 1
//This replaces MoveMouse:
	    EraseMouse();
	    mouseX=mouseX+event.pmove.x; // *factor?
	    mouseY=mouseY+event.pmove.y; // *factor?
	    xevent.xmotion.x_root=mouseX;
	    xevent.xmotion.y_root=mouseY;
	    DrawMouse();
#else
	    xevent.xmotion.x_root=event.pmove.x;
	    xevent.xmotion.y_root=event.pmove.y;
#endif
	    Events->Push(&xevent);
	    break;
//	evPtrAbsolute:	/* pointer movements reported absolute	*/
//	    break;
	case evPtrButtonPress:	/* pointer button pressed	*/
//	    DEB1(prt("GGI event: button press (%d)\n", event.pbutton.button));
	    xevent.type=ButtonPress;
//	    xevent.xbutton.button=event.pbutton.button;
	    {int button=event.pbutton.button;
	    xevent.xbutton.x = xevent.xbutton.x_root = (int)mouseX;
	    xevent.xbutton.y = xevent.xbutton.y_root = (int)mouseY;
//	    for (int i=0; i<3; i++) {
//		if (button & (1<<i)) {
//		    xevent.xbutton.button=xButtons[i];
//		    Events->Push(&xevent);
//	        }
//	    }
	    if (button & 1) {
		xevent.xbutton.button=Button1;
		Events->Push(&xevent);
	    }
	    if (button & 4) {
		xevent.xbutton.button=Button2;
		Events->Push(&xevent);
	    }
	    if (button & 2) {
		xevent.xbutton.button=Button3;
		Events->Push(&xevent);
	    }
	    }
	    break;
	case evPtrButtonRelease:
//	    DEB1(prt("GGI event: button release (%d)\n", event.pbutton.button));
	    xevent.type=ButtonRelease;
//	    xevent.xbutton.button=event.pbutton.button;
	    {int button=event.pbutton.button;

	    xevent.xbutton.x = xevent.xbutton.x_root = (int)mouseX;
	    xevent.xbutton.y = xevent.xbutton.y_root = (int)mouseY;
	    if (button & 1) {
		xevent.xbutton.button=Button1;
		Events->Push(&xevent);
	    }
	    if (button & 4) {
		xevent.xbutton.button=Button2;
		Events->Push(&xevent);
	    }
	    if (button & 2) {
		xevent.xbutton.button=Button3;
		Events->Push(&xevent);
	    }
	    }
	    break;
	default: continue;
	}

 	if (event.any.type==evKeyRelease) {
	    key=ggi_key(c, keycode);
//	    DEB1(prt("keycode -> XKeysym=%d\n", key));

	    xevent.type = KeyRelease;
	    switch (key) {
	    case XK_Caps_Lock:
		key_state &= ~CAPSLOCK_BIT;
		break;
	    case XK_Shift_L:
		key_state &= ~SHIFT_L_BIT;
		break;
	    case XK_Shift_R:
		key_state &= ~SHIFT_R_BIT;
		break;
	    case XK_Control_L:
		key_state &= ~CTRL_L_BIT;
		break;
	    case XK_Control_R:
		key_state &= ~CTRL_R_BIT;
		break;
	    case XK_Alt_L:
		key_state &= ~ALT_L_BIT;
		break;
	    case XK_Alt_R:
		key_state &= ~ALT_R_BIT;
		break;
	    default:
		break;
	    }
	    xevent.xkey.state = key_state;
//	    xevent.xkey.keycode = keycode;
//	    xevent.xkey.keycode = c;
	    xevent.xkey.keycode = key;

	    Events->Push(&xevent);
	    pressed_keys[keycode] = 0;
	}

 	if (event.any.type==evKeyPress || event.any.type==evKeyRepeat ) {
	    key=ggi_key(c, keycode);
//	    DEB1(prt("keycode -> XKeysym=%d\n", key));

	    xevent.type = KeyPress;
//	    switch (ggi_keys[key]) {
	    switch (key) {
	    case XK_Caps_Lock:
		key_state |= CAPSLOCK_BIT;
		break;
	    case XK_Shift_L:
		key_state |= SHIFT_L_BIT;
		break;
	    case XK_Shift_R:
		key_state |= SHIFT_R_BIT;
		break;
	    case XK_Control_L:
		key_state |= CTRL_L_BIT;
		break;
	    case XK_Control_R:
		key_state |= CTRL_R_BIT;
		break;
	    case XK_Alt_L:
		key_state |= ALT_L_BIT;
		break;
	    case XK_Alt_R:
		key_state |= ALT_R_BIT;
		break;
	    default:
		break;
	    }
	    xevent.xkey.state = key_state;
//	    xevent.xkey.keycode = keycode;
//	    xevent.xkey.keycode = c;
	    xevent.xkey.keycode = key;
	    Events->Push(&xevent);
	    pressed_keys[keycode] = 1;
	}
//	Events->Push(&xevent);
    }

//    DEB1(if (Events->Size()) printf("GGI: Num events=%d\n", Events->Size()));
    return(Events->Size());
}

void
GGI_FrameBuf:: GetEvent(XEvent *event)
{
	Flush(0);
	while ( ! NumEvents() )
		;
	Events->Pull(event);
}

/* Macro to 'controlify' a lower case letter */
#define toctrl(X)	(toupper(X)-'@')

int
GGI_FrameBuf:: KeyToAscii(XEvent *event, char *buf, int buflen, KeySym *key)
{
    // WHS: based on my own GGI frame buffer code (which is very similar to
    // Maelstrom's VGA buffer code as I emulate X in the GGI framebuffer) :

    *key = event->xkey.keycode;
    for ( int i=0; keycodes[i].name; ++i ) {
	//I (whs) could replace the ggi_keys by altering slightly the keycodes list, and then this would be simpler
	if ( *key == keycodes[i].key ) {
	    if ( (strlen(keycodes[i].ascii)+1) >= buflen ) {
		strncpy(buf, keycodes[i].ascii, buflen-1);
		buf[buflen-1] = 0;
	    } else
		strcpy(buf, keycodes[i].ascii);

	    /* Function strings start with ESC (std convention, see defs in ggi_keys.h) */
	    if ( CTRL_STATE(event->xkey.state) ) {
		if ( isalpha(*buf) )
		    *buf = toupper(*buf)-64; // A-> ctrl-A etc.
	    } else if ( SHIFT_STATE(event->xkey.state) ) {
		if ( isalnum(*buf) )
		    *buf = toupper(*buf);
	    }
	    return(strlen(buf));
	}
    }
	return(0);
}

/* Sorting routine for refresh area stack */
int sort_areas(GGI_FrameBuf::Area *item1, GGI_FrameBuf::Area *item2) {
	return(item1->y - item2->y);
}

void
GGI_FrameBuf:: Flush(int sync)
{
	int nreefs;
	Area *area;

#if defined(LOW_RES) || defined(VGA_16COLOR)
	Refresh();
	return;
#endif
#ifdef SORT_REFRESH
	Reefer->Sort(sort_areas);
#endif
	for ( nreefs = Reefer->Size(); nreefs; --nreefs ) {
		area = Reefer->Pop();	// This should never be NULL
//		gl_putboxpart(area->x, area->y, area->width, area->height,
//				WIDTH, HEIGHT, shared_mem, area->x, area->y);

		//some init's so I wouldn't need to rewrite the code below:
		int x0=area->x; int y0=area->y;
		int width=area->width+1;  int height=area->height+1; //other Maelstrom buffers use pixel - pixel+width instead of the natural pixel - pixel +width -1, so adjust here.
		int win_x_offset=0; int win_y_offset=0;


		byte *frame_mem=shared_mem;

		//Set the graphics offset
		byte *memaddr = (byte *) frame_mem +((y0*WIDTH)+x0);

		{
		    int y;
		    byte *src,*dst;
		    src = (byte *) frame_mem +((y0*WIDTH)+x0);
		    dst = ((byte *) plb->write) + (plb->stride * y0)+x0;
//	    dst = ((byte *) plb->write) + (plb->stride * y1)+x1;

		    for (y = 0; y < height; y++) {
			memcpy (dst, src, width);
			src += WIDTH;
			dst += plb->stride;
		    }
		}

	}
	ggiFlush(visual);

	if ( showmouse )
		DrawMouse();
	Unused(sync);
}



// Miscellaneous function:
/* Tell whether we are on the GGI console, not implemented */
int On_GGI_Console(void)
{
    return 1;
}
#else
int On_GGI_Console(void)
{
	return(0);
}
#endif /* USE_GGI */

