// Emacs style mode select   -*- C -*-
//-----------------------------------------------------------------------------
//
// $Id: i_svga.c,v 1.5 1998/08/31 22:05:41 chris Exp $
//
// Copyright (C) 1993-1996 by id Software, Inc.
//
// This source is available for distribution and/or modification
// only under the terms of the DOOM Source Code License as
// published by id Software. All rights reserved.
//
// The source is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// FITNESS FOR A PARTICULAR PURPOSE. See the DOOM Source Code License
// for more details.
//
// $Log: i_svga.c,v $
// Revision 1.5  1998/08/31 22:05:41  chris
// added InitGraphLib() to initialize SVGALib as soon as possible
//
// Revision 1.4  1998/08/31 16:20:06  chris
// removed I_ReadScreen() (now in v_video.h)
//
// Revision 1.3  1998/07/31 00:41:39  chris
// included change from rafaj@cedric.vabo.cz (Jan Rafaj):
// implemented vga_runinbackground_version() and vga_runinbackground()
// functions to allow non-suspended background run of sdoom on
// non-active virtual console
//
// Revision 1.2  1998/06/08 23:42:43  chris
// minor cleanups; fixed I_ReadScreen(), now the status bar works in
// non-standard resolutions
//
// Revision 1.1  1998/06/08 22:08:20  chris
// Initial revision
//
//
// DESCRIPTION:
//	Linux SVGA graphics IO
//
//-----------------------------------------------------------------------------

#include <stdarg.h>
#include <stdio.h>
#include <stdlib.h>
#include <signal.h>

#include <vga.h>
#include <vgakeyboard.h>
#include <vgamouse.h>

#include "doomdef.h"
#include "doomstat.h"
#include "i_system.h"
#include "m_argv.h"
#include "d_main.h"
#include "d_net.h"
#include "v_video.h"
#include "g_game.h"

int           svgalib_inited=0;
extern char   *mousedev;
extern char   *mousetype;
extern int    usemouse;
int           mouserate = MOUSE_DEFAULTSAMPLERATE;
int           UseDisplay = 1;
int           UseKeyboard = 1;
char          *framebuffer_ptr;

static vga_modeinfo *mode_info;
static int mode;                 /* svga mode to use */
static void (* put_screen)(void)=NULL;


int con_translate(int ch)
{
    int key=ch&0x7f;  /* strip bit 7 (press/release) */

    switch (key)
    {
        case 42 : ch=KEY_RSHIFT;           break;
        case 54 : ch=KEY_RSHIFT;           break;
        case 72 : ch=KEY_UPARROW;          break;
        case 80 : ch=KEY_DOWNARROW;        break;
        case 75 : ch=KEY_LEFTARROW;        break;
        case 77 : ch=KEY_RIGHTARROW;       break;
        case 103 : ch=KEY_UPARROW;         break;
        case 108 : ch=KEY_DOWNARROW;       break;
        case 105 : ch=KEY_LEFTARROW;       break;
        case 106 : ch=KEY_RIGHTARROW;      break;
        case 29 : ch=KEY_RCTRL;            break;
        case 56 : ch=KEY_LALT;             break;
        case 125 : ch=KEY_LALT;            break;
        case 97 : ch=KEY_RCTRL;            break;
        case 100 : ch=KEY_RALT;            break;
        case 119 : ch=KEY_PAUSE;           break;
        /*
        case 58 : ch=JK_CAPS;              break;
        case 69 : ch=JK_NUM_LOCK;          break;
        case 71 : ch=JK_HOME;              break;
        case 79 : ch=JK_END;               break;
        case 83 : ch=JK_DEL;               break;
        */
        case 57 : ch=' ';                  break;
        case 1  : ch=KEY_ESCAPE;           break;
        case 28  : ch=KEY_ENTER;           break;
        case 15  : ch=KEY_TAB;             break;
        case 14  : ch=KEY_BACKSPACE;       break;
        case 2  :  ch= '1';                break;
        case 3  :  ch= '2';                break;
        case 4  :  ch= '3';                break;
        case 5  :  ch= '4';                break;
        case 6  :  ch= '5';                break;
        case 7  :  ch= '6';                break;
        case 8  :  ch= '7';                break;
        case 9  :  ch= '8';                break;
        /*
        case 10  :  ch= '9';               break;
        */
        case 10  :  ch= KEY_EQUALS;        break;
        case 11  :  ch= '0';               break;
        case 12  :  ch= KEY_MINUS;         break;
        case 74  :  ch= KEY_MINUS;         break;
        case 13  :  ch= KEY_EQUALS;        break;
        case 78  :  ch= KEY_EQUALS;        break;
        case 41  :  ch= '`';               break;
        case 26  :  ch= '[';               break;
        case 27  :  ch= ']';               break;
        case 39  :  ch= ';';               break;
        case 40  :  ch= '\'';              break;
        case 51  :  ch= ',';               break;
        case 52  :  ch= '.';               break;
        case 53  :  ch= '/';               break;
        case 43  :  ch= '\\';              break;
        case 59 :  ch=KEY_F1;              break;
        case 60 :  ch=KEY_F2;              break;
        case 61 :  ch=KEY_F3;              break;
        case 62 :  ch=KEY_F4;              break;
        case 63 :  ch=KEY_F5;              break;
        case 64 :  ch=KEY_F6;              break;
        case 65 :  ch=KEY_F7;              break;
        case 66 :  ch=KEY_F8;              break;
        case 67 :  ch=KEY_F9;              break;
        case 68 :  ch=KEY_F10;             break;
        case 87 :  ch=KEY_F11;             break;
        case 88 :  ch=KEY_F12;             break;

        default :
            switch (key)
            {
                case 30 : ch='a';       break;
                case 48 : ch='b';       break;
                case 46 : ch='c';       break;
                case 32 : ch='d';       break;
                case 18 : ch='e';       break;
                case 33 : ch='f';       break;
                case 34 : ch='g';       break;
                case 35 : ch='h';       break;
                case 23 : ch='i';       break;
                case 36 : ch='j';       break;
                case 37 : ch='k';       break;
                case 38 : ch='l';       break;
                case 50 : ch='m';       break;
                case 49 : ch='n';       break;
                case 24 : ch='o';       break;
                case 25 : ch='p';       break;
                case 16 : ch='q';       break;
                case 19 : ch='r';       break;
                case 31 : ch='s';       break;
                case 20 : ch='t';       break;
                case 22 : ch='u';       break;
                case 47 : ch='v';       break;
                case 17 : ch='w';       break;
                case 45 : ch='x';       break;
                case 21 : ch='y';       break;
                case 44 : ch='z';       break;
                default : ch='a';       break;
            }
    }
    return ch;
}

void keyhandler(int scancode, int state)
{
    event_t ev;

    ev.type = state == KEY_EVENTPRESS ? ev_keydown : ev_keyup;
    ev.data1 = con_translate(scancode);

    D_PostEvent(&ev);
}

void mousehandler(int button, int dx, int dy)
{
    event_t ev;

    ev.type = ev_mouse;
    ev.data1 = button;
    ev.data2 = dx * 3;
    ev.data3 = -dy * 3;

    D_PostEvent(&ev);
}

void I_StartTic(void)
{
    if (!svgalib_inited)
        return;

    if (UseKeyboard)
        while (keyboard_update());
    if (usemouse)
        while (mouse_update());
}

void I_ShutdownGraphics(void)
{
    if (!svgalib_inited) return;

    fprintf(stderr,"i_svga: shutdown graphics called\n");
    if (UseKeyboard)
        keyboard_close();
    if (usemouse)
        mouse_close();
    if (UseDisplay)
        vga_setmode(TEXT);
    fprintf(stderr,"i_svga: shutdown graphics finished\n");
}

int I_getmode()
{
    if (screenwidth >= 1280 || screenheight >= 1024) return(G1280x1024x256);
    if (screenwidth >= 1024 || screenheight >= 768) return(G1024x768x256);
    if (screenwidth >= 800 || screenheight >= 600) return(G800x600x256);
    if (screenwidth >= 640 && screenheight >= 480) return(G640x480x256);

    if (screenwidth >= 360 || screenheight >= 480) return(G360x480x256);
    if (screenwidth >= 320 && screenheight >= 400) return(G320x400x256);
    if (screenwidth >= 320 && screenheight >= 240) return(G320x240x256);

    return(G320x200x256);
}

/* frame buffer update routines */
/* ---------------------------- */

void put_screen_linear(void)  /* linear screen buffer */
{
    memcpy(framebuffer_ptr, screens[0], screenwidth*screenheight);
}

void put_screen_def(void)  /* default (paged) screen buffer */
{
    int sizetocopy;
    int stc,page;
    unsigned char *screen_ptr;

    sizetocopy=screenwidth*screenheight;
    page=0;
    screen_ptr = screens[0];
    while (sizetocopy) {
        vga_setpage(page);
        stc = sizetocopy>0x10000 ? 0x10000 : sizetocopy;
        memcpy(framebuffer_ptr, screen_ptr, stc);
        page ++;
        screen_ptr += stc;
        sizetocopy -= stc;
    }
}

void put_screen_modex(void)
{
    int sw=screenwidth;
    int i;

    for (i=0; i<screenheight; i++) {
        vga_drawscanline(i,screens[0]+i*sw);
    }

#if 0  /* doesn't work: why?? */
    int sizetocopy,page,i;

    sizetocopy=screenwidth*screenheight/4;
    for (page=0; page<4; page++) {
        vga_setpage(page);
        for (i=0; i<sizetocopy; i++) {
            *(framebuffer_ptr+i)=*(screens[0]+(i<<2)+page);
        }
    }
#endif
}

/* gets called soon
 * after startup to
 * revoke root rights
 */
void InitGraphLib(void)
{
    if (vga_init()) {
        fprintf(stderr,"I_InitGraphics: cannot initialize svgalib\n");
        I_Quit();
    }
}


void I_InitGraphics(void)
{
    int mtype;
    int linear=FALSE,modex=FALSE;

    fprintf(stderr,"I_InitGraphics: Init SVGALib-visual.\n");

    if (svgalib_inited)  /* can this happen? */
	return;

    signal(SIGHUP, (void (*)(int)) I_Quit);
    signal(SIGINT, (void (*)(int)) I_Quit);
    signal(SIGKILL, (void (*)(int)) I_Quit);
    signal(SIGTERM, (void (*)(int)) I_Quit);
    signal(SIGSTOP, (void (*)(int)) I_Quit);
    signal(SIGQUIT, (void (*)(int)) I_Quit);

    if (UseDisplay) {
        /*
	if (vga_init()) {
            fprintf(stderr,"I_InitGraphics: cannot initialize svgalib\n");
            I_Quit();
        }
        */
	if (vga_runinbackground_version() == 1)
		vga_runinbackground(1);
	vga_setmode(mode);
	vga_setpage(0);
        if (mode==G320x200x256 || vga_setlinearaddressing() != -1) {
          linear:   /* 320x200 is implicitly linear */
            put_screen=put_screen_linear;   /* set put function */
            linear=TRUE;
        }
        else {
            mode_info=vga_getmodeinfo(mode);
            if (!mode_info) { /* failed!? */
                vga_setmode(TEXT);
                fprintf(stderr,"i_svga: cannot get info for mode %d\n",mode);
                I_Quit();
            }
            if (mode_info->flags & IS_LINEAR) { /* shouldn't go to else block... */
                goto linear;
            }
            else if (mode_info->flags & IS_MODEX) {
                put_screen=put_screen_modex;
                modex=TRUE;
            }
            else { /* packed/banked mode */
                put_screen=put_screen_def;
            }
        }
	framebuffer_ptr = (char *) vga_getgraphmem();
        if (linear) fprintf(stderr,"Linear screen addressing enabled: %p\n",framebuffer_ptr);
        else if (modex) fprintf(stderr,"ModeX enabled; frame buffer at %p\n",framebuffer_ptr);
        else fprintf(stderr,"Frame buffer at %p\n",framebuffer_ptr);
        fprintf(stderr,"Mode: %d, Screen width: %d, height: %d\n",mode,screenwidth,screenheight);
    }

    if (UseKeyboard)
    {
	if (keyboard_init())
	{
	    fprintf(stderr,"keyboard_init() failed\n");
	    if (UseDisplay)
		vga_setmode(TEXT);
	    I_Quit();
	}
	keyboard_seteventhandler(keyhandler);
    }

    if (usemouse)
    {
	if (!strcasecmp(mousetype, "microsoft"))
	    mtype = MOUSE_MICROSOFT;
	if (!strcasecmp(mousetype, "mousesystems"))
	    mtype = MOUSE_MOUSESYSTEMS;
	if (!strcasecmp(mousetype, "mmseries"))
	    mtype = MOUSE_MMSERIES;
	if (!strcasecmp(mousetype, "logitech"))
	    mtype = MOUSE_LOGITECH;
	if (!strcasecmp(mousetype, "busmouse"))
	    mtype = MOUSE_BUSMOUSE;
	if (!strcasecmp(mousetype, "ps2"))
	    mtype = MOUSE_PS2;

	if (mouse_init(mousedev, mtype, mouserate))
	{
	    fprintf(stderr,"mouse_init() failed\n");

	    if (UseKeyboard)
		keyboard_close();
	    if (UseDisplay)
		vga_setmode(TEXT);
	    I_Quit();
	}

	mouse_seteventhandler(mousehandler);
    }

    svgalib_inited=1;
}

void I_CheckRes()
{
    int res=M_CheckParm("-vgamode");

    if (res && screenwidth != 320 && screenheight != 200) {
        I_Error("Do not mix -mode and -vgamode parameters!");
    }

    if (!res)
        mode=I_getmode();      /* determine SVGA mode to use */
    else
        mode=atoi(myargv[res+1]);  /* get user specified mode */

    if (!vga_hasmode(mode))
    {
        fprintf(stderr,"Cannot support selected video mode (%d)!\n",mode);
        I_Quit();
    }
    mode_info=vga_getmodeinfo(mode);
    if (!mode_info) { /* failed!? */
        I_Error("i_svga: cannot get info for mode %d\n",mode);
    }
    screenwidth=mode_info->width;
    screenheight=mode_info->height & ~7;  /* must be multiple of 8 */
    if (screenwidth<320 || screenheight<200) {
        I_Error("Resolution too low!\n");
    }
    if (screenwidth>MAXSCREENWIDTH || screenheight>MAXSCREENHEIGHT) {
        I_Error("Resolution too high!\n"
                "Please change MAXSCREEN* in doomdef.h\n");
    }
    if (mode_info->colors != 256) { /* no-no */
        I_Error("sdoom currently only supports 256-color modes");
    }
    return;
}

void I_StartFrame(void)
{
}

void I_UpdateNoBlit(void)
{
}

void I_FinishUpdate(void)
{
    static int lasttic;
    int tics, i;

    if (!svgalib_inited)
        return;

    /* draws little dots on the bottom of the screen */
    if (devparm)
    {
        i = I_GetTime();
        tics = i - lasttic;
        lasttic = i;
        if (tics > 20) tics = 20;

        for (i=0 ; i<tics*2 ; i+=2)
            screens[0][ (screenheight-1)*screenwidth + i] = 0xff;
        for ( ; i<20*2 ; i+=2)
            screens[0][ (screenheight-1)*screenwidth + i] = 0x0;
    }

    if (UseDisplay) {
        put_screen();
    }
}

void I_SetPalette(byte *palette)
{
    static int tmppal[256*3];
    int *tp;
    int i;

    if (!svgalib_inited)
        return;

    tp = tmppal;
    for (i=256*3 ; i ; i--)
        *(tp++) = gammatable[usegamma][*(palette++)] >> 2;

    if (UseDisplay)
        vga_setpalvec(0, 256, tmppal);
}

/*
 * Local variables:
 * c-basic-offset: 4
 * c-file-style: "cpg"
 * compile-command: "make svga"
 * End:
 */
