// Copyright (C) 1999 Dominic Letourneau
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2, or (at your option)
// any later version.
//
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this file.  If not, write to the Free Software Foundation,
// 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.


#include "PathList.h"
#include "Object.h"
#include "ObjectRef.h"
#include <stdio.h>
#include <sys/types.h>
#include <dirent.h>

DECLARE_NODE(PathList)
/*Node
 *
 * @name PathList
 * @category IO
 * @description Creates a list of input files from a specified path
 *
 * @input_name PATH
 * @input_description The path where to look for files
 * @input_type string
 *
 * @input_name EXTENSION
 * @input_description The extention of the files
 * @input_type string
 *
 * @output_name FULLNAME_OUT
 * @output_description The full path of the files found
 * @output_type string
 *
 * @output_name FILENAME_OUT
 * @output_description The file name(s) only of the files found
 * @output_type string
 *
END*/


PathList::PathList(string nodeName, ParameterSet params) 
   : Node(nodeName, params) 
   , path (object_cast<String>(parameters.get("PATH")))
   , extension (object_cast<String>(parameters.get("EXTENSION"))) {

   try {
      fullPathID = addOutput("FULLNAME_OUT");
      fileNameID = addOutput("FILENAME_OUT");
      //Let's read the path
      ReadPath();
   } 
   catch (BaseException *e) {
      //e->print();
      throw e->add(new NodeException (NULL,"Caught Exception in PathList Constructor",__FILE__,__LINE__));
   }
}

ObjectRef PathList::getOutput (int output_id, int count) {
   
   
   if (!hasOutput(output_id)) throw new NodeException (this, "Cannot getOutput id",__FILE__,__LINE__);

   if (count != processCount) {
      //We are updating our output only if needed
      try {
                  
         if (fileQueue.empty()) {
            //returning an invalid object
            fullName = ObjectRef(new Object(Object::nil));
            shortName = ObjectRef (new Object(Object::nil));
         }
         else {
            //returning the file
            fullName = ObjectRef (new String(fileQueue.front()));
            shortName = ObjectRef(new String(removePath(fileQueue.front())));
            fileQueue.pop();

         }

         //updating processCount
         processCount = count;                 

      } //end of the try block

      catch (GenericCastException *e) {
         //We had a problem casting, our inputs are invalid?
         e->print();
         fullName = ObjectRef(new Object(Object::nil));
         shortName = ObjectRef(new Object(Object::nil));
      }

      catch (BaseException *e) {
         //Something weird happened
         //e->print();
         throw e->add(new NodeException (this,string("Error!!! "), __FILE__,__LINE__));
      }
      
   }
   
   if (output_id == fileNameID) {
      return shortName;
   }
   else {
      return fullName;
   }
}

void PathList::ReadPath() {

   DIR *my_directory;
   struct dirent *current_entry;

   my_directory = opendir (path.c_str());
   
   if (my_directory == NULL) {
      throw new NodeException(this,string("Invalid Path : ") + path,__FILE__,__LINE__);
   }

   for (current_entry = readdir(my_directory); 
        current_entry != NULL; current_entry = readdir(my_directory)) {
      
      if (string(current_entry->d_name).find(extension) != string::npos) {
         fileQueue.push(path + string(current_entry->d_name));
      }
   }
   closedir(my_directory);
}

string PathList::removePath(const string &fullName) {

   string output;

   for (int i = fullName.size() - 1; i >= 0; i--) {
      
      if (fullName[i] != '/') {
         //inserting at the beginning
         output.insert(output.begin(),fullName[i]);
      }
      else {
         break;
      }
   }

   return output;
}

