// Copyright (C) 1999 Dominic Letourneau
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2, or (at your option)
// any later version.
//
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this file.  If not, write to the Free Software Foundation,
// 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.

#include "NetExec.h"
#include "Object.h"
#include "ObjectRef.h"
#include <stdio.h>
#include <errno.h>
#include <sys/wait.h>

DECLARE_NODE(NetExec)
/*Node
 *
 * @name NetExec
 * @category IO
 * @description Executes an external program and get the output from stdout
 *
 * @input_name CMDSTRING_IN
 * @input_description The command
 * @input_type string
 *
 * @input_name ARGS_IN
 * @input_description The command args
 * @input_type string
 *
 * @output_name STRING_OUT
 * @output_description The command result from stdout
 * @output_type string
 *
 * @output_name STATUS_OUT
 * @output_description The return status of the command
 * @output_type int
 *
END*/


NetExec::NetExec(string nodeName, ParameterSet params) 
   : Node(nodeName, params)
   , status (new Object(Object::nil))
   , output (new Object(Object::nil)) {
   CmdInputID = addInput("CMDSTRING_IN");
   ArgsInputID = addInput("ARGS_IN");
   StringOutID = addOutput("STRING_OUT");
   StatusOutID = addOutput("STATUS_OUT");
}

ObjectRef NetExec::getOutput (int output_id, int count) {
     
   if (!hasOutput(output_id)) throw new NodeException (this, "Cannot getOutput id",__FILE__,__LINE__);

   if (count != processCount) {
      //We are updating our output only if needed
      try {
         
         int my_status;

         //getting all data from our inputs.
         int cmdOutputID = inputs[CmdInputID].outputID;
         int argsOutputID = inputs[ArgsInputID].outputID;

         ObjectRef cmd_ref (inputs[CmdInputID].node->getOutput(cmdOutputID, count));
         ObjectRef arg_ref (inputs[ArgsInputID].node->getOutput(argsOutputID, count));

         String cmdstring = object_cast<String>(cmd_ref);
         String args = object_cast<String>(arg_ref);


         string command = cmdstring + string(" ") + args;
 
         //computing
         FILE *file_desc = popen (command.c_str(), "r");
        
         if (!file_desc) {
            throw new NodeException(this,string("Invalid command :") + cmdstring, __FILE__,__LINE__);
         }
         
         string my_output = string("");
         char line[1024];

         while (fgets(line, 1024, file_desc) != NULL) {
            my_output += string(line);
         }
         
         output = ObjectRef(new String(my_output));

         //closing the file descriptor
         pclose(file_desc);
      
         wait(&my_status);

         status = ObjectRef(new Int(my_status));

         //updating processCount
         processCount = count;                 

      } //end of the try block

      catch (GenericCastException *e) {
         //We had a problem casting, our inputs are invalid?
         e->print();
         output = ObjectRef(new Object(Object::nil));
         status = ObjectRef(new Object(Object::nil));
      }

      catch (BaseException *e) {
         //Something weird happened
         //e->print();
         throw e->add(new NodeException (this,string("Cannot get CMDSTRING_IN from ") + 
                              inputs[CmdInputID].node->getName() + string (" OR ") +
                              string("Cannot get ARGS_IN from ") + 
                              inputs[ArgsInputID].node->getName()
                              , __FILE__,__LINE__));
      }
      
   }
   
   if (StatusOutID == output_id) {
      return status;
   }
   else {
      return output;
   }
}
