// Copyright (C) 1999 Jean-Marc Valin
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2, or (at your option)
// any later version.
//
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this file.  If not, write to the Free Software Foundation,
// 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.

#include "Node.h"
#include "GrowingFrameBuffer.h"

class FrameAccumulator;

DECLARE_NODE(FrameAccumulator)
/*Node
 *
 * @name FrameAccumulator
 * @category Signal:Manip
 * @description Accumulation of frames into a buffer
 *
 * @input_name INPUT
 * @input_description The input vector
 * @input_type Vector
 *
 * @output_name OUTPUT
 * @output_description The acculmulation of frames
 * @output_type Buffer
 *
 * @parameter_name LENGTH
 * @parameter_description The length of the frames
 * @parameter_type int
 *
END*/

class FrameAccumulator : public Node {
protected:
   int inputID;
   int outputID;
   int length;
   ObjectRef buf;
   int processCount;
public:
   FrameAccumulator (string nodeName, const ParameterSet &params)
      : Node(nodeName, params)
   {
      inputID = addInput("INPUT");
      outputID = addOutput("OUTPUT");
      length = dereference_cast<int> (parameters.get("LENGTH"));
      
   }

   void specificInitialize()
   {
      processCount = -1;
      buf = ObjectRef(new GrowingFrameBuffer<float> (length, 1));
      Node::specificInitialize();
   }

   void reset()
   {
      processCount = -1;
      buf = ObjectRef(new GrowingFrameBuffer<float> (length, 1));
      Node::reset();
   }

   ObjectRef getOutput(int output_id, int count)
   {
      int i,j;
      for (i=processCount+1;i<=count;i++)
      {
	 ObjectRef inputValue = getInput(inputID,i);
	 if (inputValue->status == Object::valid)
	 {
	    const Vector<float> &inputFrame = object_cast<Vector<float> > (inputValue);
	    
	    GrowingFrameBuffer<float> &out = object_cast<GrowingFrameBuffer<float> > (buf);     
	    Vector<float> &outputFrame = object_cast<Vector<float> > (out[i]);
	    
	    for (j=0;j<length;j++)
	       outputFrame[j]=inputFrame[j];
	 }
	 
      }
      processCount = count;
      return buf;
   }


};
