// Copyright (C) 1999 Jean-Marc Valin
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2, or (at your option)
// any later version.
//
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this file.  If not, write to the Free Software Foundation,
// 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.

#ifndef GROWING_BUFFER_H
#define GROWING_BUFFER_H

#include "Object.h"
#include "ObjectRef.h"
#include "Exception.h"
#include <typeinfo>
//#include "Buffer.h"
#include <vector>

/**A growing buffer implementation.
   This buffer keeps the last N lines (frames) it has*/
class GrowingBuffer : public Object {
protected:
   ///Pointer to raw float
   mutable vector<ObjectRef> data;

   ///The number N of lines kept
   mutable int bufferLength;

   mutable int currentPos;

public:
   ///Constructor, requires the vector length (vLength) and the buffer length (bLength)
   GrowingBuffer(int bLength)
      : data(bLength)
   {
      for (int i=0;i<bLength;i++) 
         data[i]=ObjectRef(new Object(Object::nil));
      currentPos=-1;
   }

   ///Copy constructor (not implemented, do we need one?)
   GrowingBuffer(const GrowingBuffer&);

   ///Indexing operator, also sets the indexed frame as being the current frame
   virtual ObjectRef operator[] (int ind) const;
   
   ///Indexing operator, also sets the indexed frame as being the current frame
   virtual ObjectRef & operator[] (int ind) ;
   
   ///Print
   virtual void printOn(ostream &out = cout) const;

   ///Returns the current position in the matrix (the last valid line)
   virtual int getCurrentPos() const {return currentPos;}

};


class GrowingBufferException : public BaseException {
public:
   ///The constructor with the parameters
   GrowingBufferException(const GrowingBuffer *_buffer, string _message, int _element) 
      : buffer (_buffer)
      , message(_message)
      , element(_element)
   {}

   ///The print method
   virtual void print(ostream &out = cerr) 
   {
      out<< typeid(GrowingBuffer).name() << " error: "<< message << ".\nElement " << element << endl;
      out << "Buffer is: \n";
      out << *buffer;
   }
protected:
   ///the buffer that generated the error
   const GrowingBuffer *buffer;

   ///the error message
   string message;

   //The element for which the error occured
   int element;
};


inline ObjectRef & GrowingBuffer::operator[] (int ind) 
{
   if (ind > currentPos)
   {
      currentPos = ind;
      data.resize(ind+1,Object::nilObject);
   }
      if (ind < 0)
   {
      //cerr << "ind = " << ind << endl;
      throw new GrowingBufferException (this, "trying to access non-existing element",ind);
   }
   return data[ind];
}


inline ObjectRef GrowingBuffer::operator[] (int ind) const
{
   if (ind > currentPos) 
   {
      currentPos = ind;
      data.resize(ind+1,Object::nilObject);
   }
   if (ind < 0)
   {
      throw new GrowingBufferException (this, "trying to access non-existing element",ind);
   }
   return data[ind];
}


inline void GrowingBuffer::printOn(ostream &out) const
{
   int i;
   out << "<GrowingBuffer" << endl;
   for (i=0;i<=currentPos;i++)
   {
      out << "< " << i << " < ";
      out << *((*this)[i]);
      out << " > > ";
      out << endl;
   }
   out << " > " << endl;
}


inline GrowingBuffer::GrowingBuffer(const GrowingBuffer&) 
{throw new GrowingBufferException(NULL,"use an ObjectRef instead",0);}


#endif
