{ *************************************************************************** }
{                                                                             }
{ This file is part of the LinLocalize project                                }
{                                                                             }
{ Copyright (c) 2003                                                          }
{ Jens Khner <kuehner@users.sourceforge.net>                                 }
{                                                                             }
{ This program is free software; you can redistribute it and/or               }
{ modify it under the terms of the GNU General Public                         }
{ License as published by the Free Software Foundation; either                }
{ version 2 of the License, or (at your option) any later version.            }
{                                                                             }
{ This program is distributed in the hope that it will be useful,             }
{ but WITHOUT ANY WARRANTY; without even the implied warranty of              }
{ MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU           }
{ General Public License for more details.                                    }
{                                                                             }
{ You should have received a copy of the GNU General Public License           }
{ along with this program; see the file COPYING.  If not, write to            }
{ the Free Software Foundation, Inc., 59 Temple Place - Suite 330,            }
{ Boston, MA 02111-1307, USA.                                                 }
{                                                                             }
{ *************************************************************************** }

unit uTransListFm;

interface

uses
  uTextLists, uTransFilterDlg,
  SysUtils, Types, Classes, Variants, QTypes, QGraphics, QControls, QForms,
  QDialogs, QStdCtrls, uCustomListFm, QMenus, QImgList, QActnList,
  QComCtrls, QGrids, QExtCtrls, uLLStringGrid, QStdActns;

type
  TTransListFm = class(TCustomListFm)
    MainMenu1: TMainMenu;
    Edit1: TMenuItem;
    actProps1: TMenuItem;
    View1: TMenuItem;
    OpenResource2: TMenuItem;
    DisplayallStrings1: TMenuItem;
    N2: TMenuItem;
    DefineFilter1: TMenuItem;
    FilterOff1: TMenuItem;
    N3: TMenuItem;
    actOutput1: TMenuItem;
    StringList1: TMenuItem;
    Update1: TMenuItem;
    N1: TMenuItem;
    Goto1: TMenuItem;
    actNextBookmark: TAction;
    actPrevBookmark: TAction;
    NextBookmark1: TMenuItem;
    PreviousBookmark1: TMenuItem;
    actToggleBookmark: TAction;
    actClearAllBookmarks: TAction;
    ToggleBookmark1: TMenuItem;
    ClearallBookmarks1: TMenuItem;
    Translation1: TMenuItem;
    actGenerateTarget: TAction;
    GenerateTargetFile1: TMenuItem;
    actValidateTranslation: TAction;
    ValidateTranslation1: TMenuItem;
    N4: TMenuItem;
    actUntranslate: TAction;
    Untranslate1: TMenuItem;
    ValidateTranslation2: TMenuItem;
    Untranslate2: TMenuItem;
    StringNumber1: TMenuItem;
    N6: TMenuItem;
    actTransComment: TAction;
    EditTranslationComment1: TMenuItem;
    actStatistics: TAction;
    N7: TMenuItem;
    Statistics1: TMenuItem;
    actAutoTranslateString: TAction;
    AutoTranslateString1: TMenuItem;
    AutoTranslateString2: TMenuItem;
    ToolButton11: TToolButton;
    actAutoTranslate: TAction;
    N8: TMenuItem;
    AutoTranslate1: TMenuItem;
    actCheckAllTranslations: TAction;
    ToolButton12: TToolButton;
    CheckallTranslations1: TMenuItem;
    actNextCheckErr: TAction;
    actPrevCheckErr: TAction;
    ImportDlg: TOpenDialog;
    N9: TMenuItem;
    NextError1: TMenuItem;
    PreviousError1: TMenuItem;
    N10: TMenuItem;
    Find1: TMenuItem;
    Findnext1: TMenuItem;
    Findprevious1: TMenuItem;
    N11: TMenuItem;
    actScanTargetFile: TAction;
    ScanTargetFile1: TMenuItem;
    ToolButton13: TToolButton;
    ToolButton14: TToolButton;
    actExportTransList: TAction;
    procedure FormShow(Sender: TObject);
    procedure FormDestroy(Sender: TObject);
    procedure gridTextesDblClick(Sender: TObject);
    function gridTextesTextColor(ACol, ARow: Integer; ARect: TRect;
      AState: TGridDrawState): TColor;
    procedure actPropsExecute(Sender: TObject);
    procedure actGenerateTargetExecute(Sender: TObject);
    procedure actUpdateStringListExecute(Sender: TObject);
    procedure HasProjectUpdate(Sender: TObject);
    procedure actSaveExecute(Sender: TObject);
    procedure actDefineFilterExecute(Sender: TObject);
    procedure actNextBookmarkExecute(Sender: TObject);
    procedure actPrevBookmarkExecute(Sender: TObject);
    procedure actToggleBookmarkExecute(Sender: TObject);
    procedure actClearAllBookmarksExecute(Sender: TObject);
    procedure actOpenResourceExecute(Sender: TObject);
    procedure actValidateTranslationExecute(Sender: TObject);
    procedure actValidateTranslationUpdate(Sender: TObject);
    procedure actUntranslateExecute(Sender: TObject);
    procedure actUntranslateUpdate(Sender: TObject);
    procedure actGotoStringNoExecute(Sender: TObject);
    procedure actTransCommentExecute(Sender: TObject);
    procedure actStatisticsExecute(Sender: TObject);
    procedure actAutoTranslateStringExecute(Sender: TObject);
    procedure actAutoTranslateStringUpdate(Sender: TObject);
    procedure actAutoTranslateExecute(Sender: TObject);
    procedure actCheckAllTranslationsExecute(Sender: TObject);
    procedure actNextCheckErrExecute(Sender: TObject);
    procedure actPrevCheckErrExecute(Sender: TObject);
    procedure ImportDlgFilterChange(Sender: TObject; NewIndex: Integer);
    procedure actScanTargetFileExecute(Sender: TObject);
    procedure actExportTransListExecute(Sender: TObject);
  private
    { Private-Deklarationen }
    m_TransTextList : TTransTextList;
    m_Filter : TFilter;
    function GetCurTransItem : TTransitem;
  protected
    procedure UpdateTextListLine( iRow : integer ); override;
    function CanAdd( item : TObject ) : boolean; override;
    procedure SaveTextList; override;
  public
    { Public-Deklarationen }
    constructor Create( AOwner : TComponent; TransTextList : TTransTextList ); reintroduce;
    procedure Import;
    procedure Export;
    property TransTextList : TTransTextList read m_TransTextList;
  end;

var
  TransListFm: TTransListFm;

implementation

{$R *.xfm}

uses
  uAutoTranslationDlg, uExportTransListDlg, uScanTargetDlg,
  uEditTransItemFm, uLanguage, uMainFm, uApp, uGlobal, uTransCommentDlg, uGotoStringNoDlg,
  uStatisticsDlg;

constructor TTransListFm.Create( AOwner : TComponent; TransTextList : TTransTextList );
begin
   inherited Create( AOwner );
   m_TransTextList := TransTextList;
   m_ResItems      := TransTextList.items;
   m_Filter        := [tfUnTranslated];
end;

procedure TTransListFm.FormShow(Sender: TObject);
resourcestring
  rsNo = 'No.';
  rsID = 'ID';
  rsComment = 'Comment';
begin
   inherited;
    caption := caption + ' - ';
    if assigned(m_TransTextList.ParentTranslation) then
       caption := caption + m_TransTextList.ParentTranslation.ParentSource.Title + ' - ';
    caption := caption + GetLangID(m_TransTextList.SourceLangNo) + ' -> '+
                         GetLangID(m_TransTextList.TransLangNo);
    gridTextes.colcount := 6;
    gridTextes.cells[0,0] := 'B';
    gridTextes.cells[1,0] := rsNo;
    gridTextes.cells[2,0] := rsID;
    gridTextes.cells[3,0] := GetLangID(m_TransTextList.SourceLangNo);
    gridTextes.cells[4,0] := GetLangID(m_TransTextList.TransLangNo);
    gridTextes.cells[5,0] := rsComment;
    gridTextes.ColWidths[0] := 12;
    gridTextes.ColWidths[1] := 32;
    gridTextes.ColWidths[2] := 120;
    gridTextes.ColWidths[3] := 150;
    gridTextes.ColWidths[4] := 150;
    gridTextes.ColWidths[5] := 100;
    FillResourceTree;
    actDisplayAllExecute(nil);
end;

procedure TTransListFm.FormDestroy(Sender: TObject);
begin
  inherited;
  m_TransTextList.free;
end;

procedure TTransListFm.SaveTextList;
begin
  m_TransTextList.WriteToFile;
  setModified(false);
end;

function TTransListFm.GetCurTransItem : TTransitem;
begin
   result := TTransItem(gridTextes.Objects[0, gridTextes.row]);
end;

procedure TTransListFm.UpdateTextListLine( iRow : integer );
var
  TransItem : TTransItem;
  strComment : string;
begin
   TransItem := TTransItem(gridTextes.Objects[0,iRow]);
   if assigned(TransItem) then begin
     if TransItem.Bookmark then
       gridTextes.Cells[0,iRow] := 'b'
     else
       gridTextes.Cells[0,iRow] := '';
     gridTextes.Cells[1,iRow] := inttostr(TransItem.No);
     gridTextes.Cells[2,iRow] := stripId(TransItem.ID);
     gridTextes.Cells[3,iRow] := trimSysChars(TransItem.SourceText);
     gridTextes.Cells[4,iRow] := trimSysChars(TransItem.TransText);
     strComment := trim(trimSysChars(TransItem.SourceComment) + ' ' + trimSysChars(TransItem.TransComment));
     gridTextes.Cells[5,iRow] := strComment;
   end else begin
     gridTextes.Cells[0,iRow] := '';
     gridTextes.Cells[1,iRow] := '';
     gridTextes.Cells[2,iRow] := '';
     gridTextes.Cells[3,iRow] := '';
     gridTextes.Cells[4,iRow] := '';
     gridTextes.Cells[5,iRow] := '';
   end;
end;

function TTransListFm.gridTextesTextColor(ACol, ARow: Integer;
  ARect: TRect; AState: TGridDrawState): TColor;
var
  TransItem : TTransItem;
begin
  inherited;
  result := clBlack;
  TransItem := TTransItem(gridTextes.objects[0, ARow]);
  if assigned(TransItem) then begin
    if not TransItem.Translated then
        result := clRed;
    if TransItem.Review then
        result := clBlue;
    if TransItem.ReadOnly then
        result := clSilver;
  end;
end;

procedure TTransListFm.gridTextesDblClick(Sender: TObject);
begin
  inherited;
  actPropsExecute(nil);
end;

procedure TTransListFm.actPropsExecute(Sender: TObject);
begin
  inherited;
  actPropsUpdate(nil);
  if actProps.Enabled then begin
      with TEditTransItemFm.create(self, m_TransTextList, gridTextes) do begin
        try
         showmodal;
        finally
         free;
        end;
      end;
   end;
end;

procedure TTransListFm.actGenerateTargetExecute(Sender: TObject);
begin
  inherited;
  MainFm.memOutput.lines.clear;
  screen.Cursor := crHourGlass;
  try
   m_TransTextList.GenerateTarget(g_app.ResbindLocation, MainFm.memOutput.lines);
  finally
   screen.Cursor := crDefault;
  end;
end;

procedure TTransListFm.actUpdateStringListExecute(Sender: TObject);
begin
  inherited;
  MainFm.memOutput.lines.clear;
  screen.Cursor := crHourGlass;
  try
   if fileexists(m_TransTextList.ListFileName) then
     m_TransTextList.WriteToFile;
   m_TransTextList.CreateUpdate( MainFm.memOutput.lines );
   SetModified(false);
   FillResourceTree;
   actDisplayAllExecute(nil);
  finally
    screen.Cursor := crDefault;
  end;
end;

procedure TTransListFm.HasProjectUpdate(Sender: TObject);
begin
  inherited;
  (sender as TAction).Enabled := assigned(m_TransTextList.ParentTranslation);
end;

procedure TTransListFm.actSaveExecute(Sender: TObject);
begin
  inherited;
  m_TransTextList.WriteToFile;
  setModified(false);
end;

function TTransListFm.CanAdd( item : TObject ) : boolean;
var
  TransItem : TTransItem;
begin
  TransItem := TTransItem(item);
  result := not TransItem.Hidden;
  if m_bFilter and not TransItem.Hidden then begin
     result := false;
     if not result and (tfTranslated in m_Filter) and TransItem.Translated and not TransItem.readonly then
       result := true;
     if not result and (tfUntranslated in m_Filter) and not TransItem.Translated then
       result := true;
     if not result and (tfReview in m_Filter) and TransItem.Review then
       result := true;
     if not result and (tfReadonly in m_Filter) and TransItem.ReadOnly then
       result := true;
  end;
end;

procedure TTransListFm.actDefineFilterExecute(Sender: TObject);
begin
  inherited;
  with TTransFilterDlg.Create(self) do begin
    try
     Filter := m_Filter;
     case showmodal of
       mrOK: begin
          m_Filter  := Filter;
          m_bFilter := true;
          FillTextList;
       end;
       mrNo: begin
          m_bFilter := false;
          FillTextList;
       end;
     end;
    finally
     free;
    end;
  end;
end;

procedure TTransListFm.actNextBookmarkExecute(Sender: TObject);
var
  TransItem : TTransItem;
  iRow : integer;
begin
  inherited;
  for iRow := gridTextes.Row to gridTextes.rowcount-1 do begin
    TransItem := TTransItem(gridTextes.objects[0, iRow]);
    if assigned(TransItem) and TransItem.Bookmark then begin
       gridTextes.Row := iRow;
       break;
    end;
  end;
end;

procedure TTransListFm.actPrevBookmarkExecute(Sender: TObject);
var
  TransItem : TTransItem;
  iRow : integer;
begin
  inherited;
  for iRow := gridTextes.Row  downto 1 do begin
    TransItem := TTransItem(gridTextes.objects[0, iRow]);
    if assigned(TransItem) and TransItem.Bookmark then begin
       gridTextes.Row := iRow;
       break;
    end;
  end;
end;

procedure TTransListFm.actToggleBookmarkExecute(Sender: TObject);
var
  i : integer;
  TransItem : TTransitem;
  bBookmark : boolean;
begin
  inherited;
  if GetCurTransItem=nil then exit;
  bBookMark := not GetCurTransItem.Bookmark;
  for i := gridTextes.Selection.Top to gridTextes.selection.Bottom do begin
    TransItem := TTransItem(gridTextes.objects[0, i]);
    if assigned(TransItem) then begin
      Transitem.Bookmark := bBookMark;
      UpdateTextListLine(i);
    end;
  end;
  setModified(true);
end;

procedure TTransListFm.actClearAllBookmarksExecute(Sender: TObject);
begin
  inherited;
  m_TransTextList.ClearAllBookmarks;
  setModified(true);
  UpdateTextListLines;
end;

procedure TTransListFm.actOpenResourceExecute(Sender: TObject);
begin
  inherited;
  if activeControl = gridTextes then begin
      Openresource(GetCurTransItem.ParentTransResItem, getCurTransItem);
  end;
end;

procedure TTransListFm.actValidateTranslationExecute(Sender: TObject);
var
  i : integer;
  TransItem : TTransitem;
begin
  inherited;
  for i := gridTextes.Selection.Top to gridTextes.selection.Bottom do begin
    TransItem := TTransItem(gridTextes.objects[0, i]);
    Transitem.Translated := true;
    Transitem.Review     := false;
    UpdateTextListLine(i);
  end;
  setModified(true);
end;

procedure TTransListFm.actValidateTranslationUpdate(Sender: TObject);
var
  i : integer;
  bEnabled : boolean;
  TransItem : TTransitem;
begin
  inherited;
  bEnabled := false;
  for i := gridTextes.Selection.Top to gridTextes.selection.Bottom do begin
    TransItem := TTransItem(gridTextes.objects[0, i]);
    if assigned(TransItem) and not TransItem.ReadOnly and (not Transitem.Translated or getCurTransitem.Review) then begin
      bEnabled := true;
      break;
    end;
  end;
  actValidateTranslation.Visible := bEnabled;
end;

procedure TTransListFm.actUntranslateExecute(Sender: TObject);
var
  i : integer;
  TransItem : TTransitem;
begin
  inherited;
  for i := gridTextes.Selection.Top to gridTextes.selection.Bottom do begin
    TransItem := TTransItem(gridTextes.objects[0, i]);
    Transitem.Translated := false;
    Transitem.Review     := false;
    Transitem.TransText := Transitem.SourceText;
    UpdateTextListLine(i);
  end;
  setModified(true);
end;

procedure TTransListFm.actUntranslateUpdate(Sender: TObject);
var
  i : integer;
  bEnabled : boolean;
  TransItem : TTransitem;
begin
  inherited;
  bEnabled := false;
  for i := gridTextes.Selection.Top to gridTextes.selection.Bottom do begin
    TransItem := TTransItem(gridTextes.objects[0, i]);
    if assigned(TransItem) and not TransItem.ReadOnly and Transitem.Translated then begin
      bEnabled := true;
      break;
    end;
  end;
  actUntranslate.Visible := bEnabled;
end;

procedure TTransListFm.actAutoTranslateStringExecute(Sender: TObject);
var
  i : integer;
  TransItem : TTransitem;
begin
  inherited;
  for i := gridTextes.Selection.Top to gridTextes.selection.Bottom do begin
    TransItem := TTransItem(gridTextes.objects[0, i]);
    if not TransItem.ReadOnly then
      if g_app.Translate(TransItem) then begin
        Transitem.Translated := true;
        Transitem.Review     := false;
        UpdateTextListLine(i);
      end;
  end;
  setModified(true);
end;

procedure TTransListFm.actAutoTranslateStringUpdate(Sender: TObject);
var
  i : integer;
  bEnabled : boolean;
  TransItem : TTransitem;
begin
  inherited;
  bEnabled := false;
  for i := gridTextes.Selection.Top to gridTextes.selection.Bottom do begin
    TransItem := TTransItem(gridTextes.objects[0, i]);
    if assigned(TransItem) and not TransItem.ReadOnly then begin
      bEnabled := true;
      break;
    end;
  end;
  actAutoTranslateString.Visible := bEnabled;
end;

procedure TTransListFm.actGotoStringNoExecute(Sender: TObject);
var
  TransItem : TTransItem;
begin
   inherited;
   with TGotoStringNoDlg.create(self) do begin
     if showModal = mrOK then begin
        TransItem := m_TransTextList.GetItemByNumber(spinNo.Value);
        if assigned(TransItem) then begin
           OpenResource(TransItem.ParentTransResItem, TransItem);
        end;
     end;
   end;
end;

procedure TTransListFm.actTransCommentExecute(Sender: TObject);
resourcestring
  rsCaption = 'Translation-String %d - String %s';
begin
   inherited;
   with TTransCommentDlg.create(self) do begin
      lblCaption.Caption := format(rsCaption,
                                    [GetCurTransItem.No, stripID(GetCurTransItem.ID)]);
      memComment.Text    := GetCurTransItem.TransComment;
      if showmodal = mrOk then begin
          if GetCurTransItem.TransComment <> memComment.Text then begin
            GetCurTransItem.TransComment := memComment.Text;
            setModified(true);
          end;
      end;
   end;
end;

procedure TTransListFm.actStatisticsExecute(Sender: TObject);
begin
  inherited;
  with TStatisticsDlg.Create(self, m_TransTextList) do begin
     showmodal;
  end;
end;

procedure TTransListFm.actAutoTranslateExecute(Sender: TObject);
var
  TransItem : TTransitem;
  iRes, i : integer;
  TransResItem : TTransResItem;
  iChecked, iTranslated : integer;
resourcestring
  rsCaption = 'Automatic translation of';
  rsStatusOk = 'Checked strings: %d, Translated strings: %d.';
begin
  inherited;
  with TAutoTranslationDlg.Create(self) do begin
   try
    rbCurrent.Enabled := not m_bDisplayAll;
    if rbCurrent.Enabled then
       rbCurrent.Checked := true
    else
       rbAll.Checked := true;
    if showmodal=mrOK then begin
       iChecked    := 0;
       iTranslated := 0;
       MainFm.memOutput.lines.clear;
       MainFm.memOutput.lines.add(rsCaption+' '+self.caption);
       screen.Cursor := crHourGlass;
       try
       if rbAll.checked then begin
          for iRes := 0 to m_ResItems.Count-1 do begin
              TransResItem := m_TransTextList.items[iRes] as TTransResItem;
              for i := 0 to TransResItem.Count-1 do begin
                TransItem := TransResItem.Transitems[i];
                if TransItem.Editable and not TransItem.translated then begin
                  inc(iChecked);
                  if g_app.Translate(TransItem) then begin
                        setModified(true);
                        inc(iTranslated);
                  end;
                end;
              end;//for i
          end;//for iRes
       end;
       //current view
       if rbCurrent.checked then begin
          for i := 1 to gridTextes.RowCount-1 do begin
             TransItem := TTransItem(gridTextes.objects[0, i]);
             if assigned(TransItem) and TransItem.Editable and not TransItem.translated then begin
                inc(iChecked);
                if g_app.Translate(TransItem) then begin
                   setModified(true);
                   inc(iTranslated);
                end;
             end;
          end;
       end;
       UpdateTextListLines;
       MainFm.memOutput.lines.add(format(rsStatusOk, [iChecked, iTranslated]));
       finally
         screen.Cursor := crDefault;
       end;
    end;
   finally
    free;
   end;
  end;
end;

procedure TTransListFm.actCheckAllTranslationsExecute(Sender: TObject);
var
  i, iErr, iErrCount : integer;
  TransItem : TTransitem;
resourcestring
  rsChecking = 'Checking Translation-List - %s -> %s';
  rsResult = '%d error(s)';
begin
  inherited;
  MainFm.memOutput.lines.clear;
  MainFm.memOutput.lines.add( format(rsChecking,
                                     [getLangId(m_TransTextList.SourceLangNo), getLangID(m_TransTextList.TransLangNo)] ));
  application.ProcessMessages;
  iErrCount := 0;
  for i := 1 to gridTextes.RowCount-1 do begin
     TransItem := TTransItem(gridTextes.objects[0, i]);
     if assigned(TransItem) then begin
        iErr := g_app.CheckTranslation(TransItem);
        if iErr > 0 then begin
           MainFm.memOutput.lines.add( g_app.FormatCheckErr(TransItem, iErr) );
           inc(iErrCount);
        end;
     end;
  end;
  MainFm.memOutput.lines.add(format(rsResult, [iErrCount]));
end;

procedure TTransListFm.actNextCheckErrExecute(Sender: TObject);
var
  i : integer;
  TransItem : TTransItem;
  iErr : integer;
begin
  inherited;
  for i := gridTextes.row+1 to gridTextes.RowCount-1 do begin
     TransItem := TTransItem(gridTextes.objects[0, i]);
     if assigned(TransItem) then begin
        iErr := g_app.CheckTranslation(TransItem);
        if iErr > 0 then begin
          MainFm.memOutput.lines.clear;
          MainFm.memOutput.lines.add( g_app.FormatCheckErr(TransItem, iErr));
          gridTextes.row := i;
          break;
        end;
     end;
  end;
end;

procedure TTransListFm.actPrevCheckErrExecute(Sender: TObject);
var
  i : integer;
  TransItem : TTransItem;
  iErr : integer;
begin
  inherited;
  for i := gridTextes.row-1 downto 1 do begin
     TransItem := TTransItem(gridTextes.objects[0, i]);
     if assigned(TransItem) then begin
        iErr := g_app.CheckTranslation(TransItem);
        if iErr > 0 then begin
          MainFm.memOutput.lines.clear;
          MainFm.memOutput.lines.add( g_app.FormatCheckErr(TransItem, iErr));
          gridTextes.row := i;
          break;
        end;
     end;
  end;
end;

procedure TTransListFm.Export;
begin
  with TExportTransListDlg.create(self, m_TransTextList) do begin
     showmodal;
  end;
end;

procedure TTransListFm.Import;
begin
  ImportDlgFilterChange(self, ImportDlg.FilterIndex);
  if ImportDlg.Execute then begin
    case ImportDlg.FilterIndex of
      1: m_TranstextList.ReadFromFile(ImportDlg.filename);
      2: m_TranstextList.ReadFromTextFile(ImportDlg.filename);
    end;
    setModified(true);
    FillResourceTree;
    actDisplayAllExecute(nil);
  end;
end;

procedure TTransListFm.ImportDlgFilterChange(Sender: TObject;
  NewIndex: Integer);
begin
  inherited;
  case NewIndex of
     1: ImportDlg.DefaultExt := '.trl';
     2: ImportDlg.DefaultExt := '.trltxt';
  end;
end;

procedure TTransListFm.actScanTargetFileExecute(Sender: TObject);
begin
  inherited;
  with TScanTargetDlg.create(self) do begin
     try
      lblLanguage.Caption := getLang( m_TranstextList.TransLangNo );
      if showmodal = mrOk then begin
        MainFm.memOutput.lines.clear;
        screen.Cursor := crHourGlass;
        try
          m_TransTextList.ScanTarget(mainfm.memoutput.lines, efTargetFilename.text,
                                     g_app.resbindlocation,
                                     cbTranslated.checked, cbReview.checked );
          UpdateTextListLines;
          setModified(true);
        finally
          screen.cursor := crDefault;
        end;
      end;
     finally
      free;
     end;
  end;
end;

procedure TTransListFm.actExportTransListExecute(Sender: TObject);
begin
  inherited;
  Export;
end;

end.
