{ *************************************************************************** }

{                                                                             }
{ This file is part of the LinLocalize project                                }
{                                                                             }
{ Copyright (c) 2003                                                          }
{ Jens Khner <kuehner@users.sourceforge.net>                                 }
{                                                                             }
{ This program is free software; you can redistribute it and/or               }
{ modify it under the terms of the GNU General Public                         }
{ License as published by the Free Software Foundation; either                }
{ version 2 of the License, or (at your option) any later version.            }
{                                                                             }
{ This program is distributed in the hope that it will be useful,             }
{ but WITHOUT ANY WARRANTY; without even the implied warranty of              }
{ MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU           }
{ General Public License for more details.                                    }
{                                                                             }
{ You should have received a copy of the GNU General Public License           }
{ along with this program; see the file COPYING.  If not, write to            }
{ the Free Software Foundation, Inc., 59 Temple Place - Suite 330,            }
{ Boston, MA 02111-1307, USA.                                                 }
{                                                                             }
{ *************************************************************************** }


unit uTextLists;

interface

uses
  Contnrs, uProject, Classes;

type
  TFindDef = class
     public
       m_strFindWhat : string;
       m_lstHistory : TStringlist;
       m_bMatchCase : boolean;
       m_bMatchWholeWord : boolean;
       m_bPassEOF : boolean;
       m_bIgnoreAmpersand : boolean;
       m_bDirDown : boolean;
       m_bAllResources : boolean;
       m_bLookInAllFields : boolean;
       m_bLookInText : boolean;
       m_bLookInOldTextSourceText : boolean;
       m_bLookInComment : boolean;
       constructor create;
       destructor destroy; override;
  end;
  TSourceResItem = class;
  TTransResItem = class;
  TCustomtextList = class;
  TCustomResItem = class;
  TResType = (rtResourceString, rtForm);
  TUpdateState = (usNone, usNew, usChanged);
  TCustomSourceTransItem = class
    protected
     m_ParentCustomResItem : TCustomResItem;
     m_iNo         : integer;
     m_strID       : string;
    public
     function Find( findef : TFindDef ) : boolean; virtual; abstract;
     property ParentCustomResItem : TCustomResItem read m_ParentCustomResItem;
     property No : integer read m_iNo;
     property ID : string read m_strID;
  end;
  TSourceItem = class(TCustomSourceTransItem)
    private
      m_strComment  : string;
      m_strText     : string;
      m_strOldText  : string;
      m_UpdateState : TUpdateState;
      //flags
      m_bReadOnly   : boolean;
      m_bHidden     : boolean;
      m_bCorrection : boolean;
      function GetParentSourceResItem : TSourceResItem;
    public
     constructor Create(ParentResItem : TSourceResItem; const strID, strText : string); overload;
     constructor Create(ParentResItem : TSourceResItem; r : TReader ); overload;
     procedure WriteToStream( w : TWriter );
     procedure ReadFromStream( r : TReader );
     function Find( finddef : TFindDef ) : boolean; override;
     function UpdateFromOld( OldSourceItem : TSourceItem ) : boolean;
     property ParentSourceResItem : TSourceResItem read GetParentSourceResItem;
     property Text : string read m_strText;
     property OldText : string read m_strOldtext;
     property Comment : string read m_strComment write m_strComment;
     property UpdateState : TUpdateState read m_UpdateState write m_UpdateState;
     property ReadOnly : boolean read m_bReadOnly write m_bReadOnly;
     property Hidden : boolean read m_bHidden write m_bHidden;
     property Correction : boolean read m_bCorrection write m_bCorrection;
     property No : integer read m_iNo write m_iNo;
  end;
  TTransItem = class(TCustomSourceTransItem)
    private
     m_bHidden       : boolean;
     m_bReadOnly     : boolean;
     m_bBookmark     : boolean;
     m_bTranslated   : boolean;
     m_bReview       : boolean;
     m_strOldText       : string;
     m_strSourceText    : string;
     m_strTransText     : string;
     m_strSourceComment : string;
     m_strTransComment  : string;
     function GetParentTransResItem : TTransResItem;
     function GetEditable : boolean;
    public
     constructor Create( ParentResItem : TTransResItem; sourceItem : TSourceItem ); overload;
     constructor Create( ParentResItem : TTransResItem; r : TReader ); overload;
     procedure WriteToStream( w : TWriter );
     procedure ReadFromStream( r : TReader );
     function UpdateFromOld( OldTransItem : TTransItem; bCorrection : boolean) : boolean;
     function UpdateFlagsFromSourceItem( sourceItem : TSourceItem ) : boolean;
     function Find( finddef : TFindDef ) : boolean; override;
     property ParentTransResItem : TTransResItem read GetParentTransResItem;
     property Hidden : boolean read m_bHidden;
     property ReadOnly : boolean read m_bReadOnly;
     property Bookmark : boolean read m_bBookmark write m_bBookmark;
     property Translated : boolean read m_bTranslated write m_bTranslated;
     property Review : boolean read m_bReview write m_bReview;
     property OldText : string read m_strOldText;
     property SourceText : string read m_strSourceText write m_strSourceText;
     property TransText : string read m_strTransText write m_strTransText;
     property SourceComment : string read m_strSourceComment;
     property TransComment : string read m_strTransComment write m_strTransComment;
     property Editable : boolean read GetEditable;
  end;
  TCustomResItem = class
    private
     m_ParentList : TCustomtextList;
     m_strResName  : string;
     m_lstItems    : TObjectList;
     m_resType     : TResType;
     m_UpdateState : TUpdateState;
     function getCount : integer;
     function GetItem( i : integer ) : TCustomSourceTransItem;
    public
     constructor Create( parentList : TCustomtextList ); overload;
     constructor Create( parentList : TCustomtextList; r : TReader ); overload;
     constructor Create( parentList : TCustomtextList; resitem : TCustomResItem ); overload;
     destructor destroy; override;
     procedure AddItem( item : TCustomSourceTransItem );
     procedure ReadFromStream( r : TReader ); virtual; abstract;
     property UpdateState : TUpdateState read m_UpdateState write m_UpdateState;
     property ResType : TResType read m_ResType write m_ResType;
     property ResName : string read m_strResname write m_strResName;
     property ParentList : TCustomtextList read m_ParentList;
     property Count : integer read GetCount;
     property CustomItems[i:integer] : TCustomSourceTransItem read GetItem;
  end;
  TSourceResItem = class(TCustomResItem)
    private
     function GetItem( i : integer ) : TSourceItem;
    public
     procedure WriteToStream( w : TWriter );
     procedure ReadFromStream( r : TReader ); override;
     property SourceItems[i:integer] : TSourceItem read GetItem;
  end;
  TTransResItem = class(TCustomResItem)
    private
     function GetItem( i : integer ) : TTransItem;
    public
     procedure WriteToStream( w : TWriter; bOnlyBookmarked : boolean);
     procedure ReadFromStream( r : TReader ); override;
     property TransItems[i:integer] : TTransItem read GetItem;
  end;
  TCustomTextList = class
  end;
  TSourceTextList = class( TCustomtextList)
    private
     m_ParentSource : TSource;
     m_lstItems     : TObjectList; //TSourceResItem
     procedure ReadFromResFile( out iNewStrings : integer );
     function GetResItem( ResType : TResType; const ResName : string ) : TSourceResItem;
     function GetSourceItem( ResType : TResType; const ResName, strID : string ) : TSourceItem;
    public
     constructor Create( source : TSource );
     destructor destroy; override;
     procedure CreateUpdate( const strResbind : string; slOutput : TStrings );
     procedure ReadFromFile;
     procedure WriteToFile;
     procedure ExportToASCII( const strFileName : string );
     procedure ResetChangedFlags;
     function GetItemByNumber( iNo : integer ) : TSourceItem;
     property ParentSource : TSource read m_ParentSource;
     property Items : TObjectList read m_lstItems;
  end;
  TTransTextList = class( TCustomtextList)
    private
     m_ParentTranslation : TTranslation;
     m_lstItems          : TObjectList; //TTranslationResItem
     m_strListFileName   : string;
     m_iSourceLangNo     : integer;
     m_iTransLangNo      : integer;
     function GetResItem( ResType : TResType; const ResName : string ) : TTransResItem;
     function GetTransItem( ResType : TResType; const ResName, strID : string ) : TTransItem;
     function GetTransLationResItemByName( const strName : string ) : TTransResItem;
     procedure UpdateFromSourceTextList( out iNewResources, iDeletedResources,
                                             iNewStrings, iChangedStrings, iDeletedStrings : integer );
     procedure WriteToResFile;
    public
     constructor Create( translation : TTranslation ); overload;
     constructor Create( const strFileName : string ); overload;
     destructor destroy; override;
     procedure CreateUpdate( slOutput : TStrings );
     procedure GenerateTarget( const strResBind : string; slOutput : TStrings );
     procedure ScanTarget( slOutput : TStrings; const strFileName : string;
                           const strResBind : string;
                           bTranslated, bReview : boolean );
     procedure ReadFromFile( const strFilename : string = '');
     procedure WriteToFile( const strFilename : string = ''; bOnlyBookmarked : boolean = false );
     procedure ReadFromTextFile( const strFilename : string );
     procedure WriteToTextFile( const strFilename : string; bOnlyBookmarked : boolean );
     procedure ClearAllBookmarks;
     function Translate( TransItemIn : TTransItem; const strSource : string; out strTrans : string ) : integer;
     function TranslateHint( TransItemIn : TTransItem; const strSource : string; out strTrans : string ) : boolean;
     procedure GetStatistics( out iStringCount, iWordCount, iTransRate, iLengthChange : integer );
     function GetItemByNumber( iNo : integer ) : TTransItem;
     property SourceLangNo : integer read m_iSourceLangNo write m_iSourceLangNo;
     property TransLangNo : integer read m_iTransLangNo write m_iTransLangNo;
     property ParentTranslation : TTranslation read m_ParentTranslation write m_ParentTranslation;
     property ListFileName : string read m_strListFileName;
     property Items : TObjectList read m_lstItems;
  end;

function StripID( const strID : string ) : string;

implementation

uses
  uResIO, uResFile, SysUtils, uLanguage, uGlobal, QForms;

resourcestring
  rsTransWithoutProject  = 'Translation-list has no project!';
  rsSourceWithoutProject = 'Source-list has no project!';
  rsSourceList = 'Source-List';
  rsTransList = 'Translation-List';
  rsUpdating = 'Updating';
  rsCreating = 'Creating';
  rsErrCannotExec = 'Error: Cannot execute %s';
  rsErrNoSourceFile = 'Error: Source-File %s does not exist!';
  rsOKNoChanges = 'OK - No changes';
  rsOKStatus = 'OK - Resources: %d new, %d deleted / Strings: %d new, %d changed, %d deleted';
  rsError = 'Error';

function StripID( const strID : string ) : string;
var
  strTemp : string;
begin
  result  := strID;
  strTemp := uppercase(strID);
  if pos('.CAPTION', strTemp) = length(strTemp)-(length('.CAPTION')-1) then
     result := copy(result, 1, pos('.CAPTION', strTemp)-1 );
end;

//**************** TFindDef *******************
constructor TFindDef.create;
begin
  m_bMatchCase               := false;
  m_bMatchWholeWord          := false;
  m_bPassEOF                 := true;
  m_bIgnoreAmpersand         := true;
  m_bDirDown                 := true;
  m_bAllResources            := false;
  m_bLookInAllFields         := true;
  m_bLookInText              := false;
  m_bLookInOldTextSourceText := false;
  m_bLookInComment           := false;
  m_lstHistory               := TStringList.create;
end;

destructor TFinddef.destroy;
begin
  m_lstHistory.free;
  inherited;
end;
//**************** SourceItem *******************
constructor TSourceItem.Create( ParentResItem : TSourceResItem; const strID, strText : string);
begin
  inherited Create;
  m_ParentCustomResItem := ParentResItem;
  m_strID   := strID;
  m_strText := strText;
end;

constructor TSourceItem.Create( ParentResItem : TSourceResItem; r : TReader );
begin
  inherited Create;
  m_ParentCustomResItem := ParentResItem;
  ReadFromStream( r );
end;

function TSourceItem.GetParentSourceResItem : TSourceResItem;
begin
  result := m_ParentCustomResItem as TSourceResItem;
end;

function TSourceItem.UpdateFromOld( OldSourceItem : TSourceItem ) : boolean;
begin
 result := false;
 m_strOldText := OldSourceItem.Text;
 m_bReadOnly  := OldSourceItem.ReadOnly;
 m_bHidden    := OldSourceItem.Hidden;
 m_strComment := OldSourceItem.Comment;
 if m_strText <> OldSourceItem.Text then begin
     m_UpdateState := usChanged;
     result := true;
 end;
end;

procedure TSourceItem.WriteToStream( w : TWriter );
begin
  w.writeInteger( m_iNo );
  w.writeString( m_strID );
  w.writeinteger( ord(m_UpdateState) );
  w.WriteBoolean( m_bReadOnly );
  w.WriteBoolean( m_bHidden );
  w.WriteBoolean( m_bCorrection );
  w.writestring( m_strOldText );
  w.writestring( m_strText );
  w.writestring( m_strComment );
end;

procedure TSourceItem.ReadFromStream( r : TReader );
begin
  m_iNo         := r.readinteger;
  m_strID       := r.readstring;
  m_UpdateState := TUpdateState(r.readinteger);
  m_bReadOnly   := r.ReadBoolean;
  m_bHidden     := r.ReadBoolean;
  m_bCorrection := r.ReadBoolean;
  m_strOldText  := r.readstring;
  m_strText     := r.readstring;
  m_strComment  := r.readstring;
end;

function TSourceItem.Find( finddef : TFindDef ) : boolean;
begin
   result := false;
   if not result and (finddef.m_bLookInAllFields or finddef.m_bLookInText) then
      result := Compare(finddef.m_strFindWhat, m_strText, finddef.m_bMatchCase, finddef.m_bMatchWholeWord, finddef.m_bIgnoreAmpersand );
   if not result and (finddef.m_bLookInAllFields or finddef.m_bLookInOldTextSourceText) then
      result := Compare(finddef.m_strFindWhat, m_strOldText, finddef.m_bMatchCase, finddef.m_bMatchWholeWord, finddef.m_bIgnoreAmpersand );
   if not result and (finddef.m_bLookInAllFields or finddef.m_bLookInComment) then
      result := Compare(finddef.m_strFindWhat, m_strComment, finddef.m_bMatchCase, finddef.m_bMatchWholeWord, finddef.m_bIgnoreAmpersand );
end;
//****************** TranslationItem **********************
constructor TTransItem.Create( ParentResItem : TTransResItem; sourceItem : TSourceItem );
begin
  inherited Create;
  m_ParentCustomResItem := ParentResItem;
  m_iNo              := SourceItem.No;
  m_strID            := SourceItem.ID;
  m_strSourceText    := SourceItem.Text;
  m_strTransText     := SourceItem.Text;
  m_strSourceComment := SourceItem.Comment;
end;

constructor TTransItem.Create( ParentResItem : TTransResItem; r : TReader );
begin
   inherited create;
   m_ParentCustomResItem := ParentResItem;
   readfromStream(r);
end;

function TTransItem.GetParentTransResItem : TTransResItem;
begin
  result := m_ParentCustomResItem as TTransResItem;
end;

function TTransItem.UpdateFromOld( OldTransItem : TTransItem; bCorrection : boolean ) : boolean;
begin
  result := false;
  m_bReadOnly   := OldTransItem.ReadOnly;
  m_bHidden     := OldTransItem.Hidden;
  m_bBookmark   := OldTransItem.Bookmark;
  m_bTranslated := OldTransItem.Translated;
  m_bReview     := OldTransItem.Review;
  if m_bTranslated then
     m_strTransText  := OldTransItem.TransText;
  if (OldTransItem.SourceText <> m_strSourceText) and not bCorrection then begin
    m_bTranslated := false;
    m_bReview     := false;
    result        := true;
  end;
end;

function TTransItem.UpdateFlagsFromSourceItem( sourceItem : TSourceItem ) : boolean;
begin
   result := (m_bReadOnly <> SourceItem.ReadOnly) or (m_bHidden <> SourceItem.Hidden);
   if SourceItem.ReadOnly then begin
    m_bReadOnly   := true;
    m_bTranslated := true;
   end;
   m_bHidden := SourceItem.Hidden;
end;

procedure TTransItem.WriteToStream( w : TWriter );
begin
  w.writeInteger( m_iNo );
  w.writeString( m_strID );
  //flags
  w.writeboolean(m_bHidden);
  w.writeboolean(m_bReadOnly);
  w.writeboolean(m_bBookmark);
  w.writeboolean(m_bTranslated);
  w.writeboolean(m_bReview);
  //
  w.writestring( m_strOldText );
  w.writestring( m_strSourceText );
  w.writestring( m_strTransText );
  w.writestring( m_strSourceComment );
  w.writestring( m_strTransComment );
end;

procedure TTransItem.ReadFromStream( r : TReader );
begin
  m_iNo          := r.readinteger;
  m_strID        := r.readString;
  //flags
  m_bHidden      := r.ReadBoolean;
  m_bReadOnly    := r.ReadBoolean;
  m_bBookmark    := r.ReadBoolean;
  m_bTranslated  := r.ReadBoolean;
  m_bReview      := r.ReadBoolean;
  //
  m_strOldText       := r.readString;
  m_strSourceText    := r.readString;
  m_strTransText     := r.readString;
  m_strSourceComment := r.readString;
  m_strTransComment  := r.readString;
end;

function TTransItem.Find( finddef : TFindDef ) : boolean;
begin
   result := false;
   if not result and (finddef.m_bLookInAllFields or finddef.m_bLookInText) then
      result := Compare(finddef.m_strFindWhat, m_strTransText, finddef.m_bMatchCase, finddef.m_bMatchWholeWord, finddef.m_bIgnoreAmpersand );
   if not result and (finddef.m_bLookInAllFields or finddef.m_bLookInOldTextSourceText) then
      result := Compare(finddef.m_strFindWhat, m_strSourceText, finddef.m_bMatchCase, finddef.m_bMatchWholeWord, finddef.m_bIgnoreAmpersand );
   if not result and (finddef.m_bLookInAllFields or finddef.m_bLookInComment) then
      result := Compare(finddef.m_strFindWhat, m_strSourceComment, finddef.m_bMatchCase, finddef.m_bMatchWholeWord, finddef.m_bIgnoreAmpersand );
end;

function TTransItem.GetEditable : boolean;
begin
  result := not m_bHidden and not m_bReadonly;
end;
//***************** CustomResItem **********************
constructor TCustomResItem.Create( parentList : TCustomtextList );
begin
  m_lstItems := TObjectList.create;
  m_ParentList := parentlist;
  inherited Create;
end;

constructor TCustomResItem.Create( parentList : TCustomtextList; r : TReader );
begin
   create( parentlist );
   ReadFromStream( r );
end;

constructor TCustomResItem.Create( parentList : TCustomtextList; resitem : TCustomResItem );
begin
   create( parentlist );
   m_strResName := resitem.ResName;
   m_resType    := resitem.ResType;
end;

destructor TCustomResItem.destroy;
begin
  m_lstItems.free;
  inherited;
end;

function TCustomResItem.GetCount : integer;
begin
  result := m_lstItems.count;
end;

function TCustomResItem.GetItem( i : integer ) : TCustomSourceTransItem;
begin
  result := TCustomSourceTransItem(m_lstItems[i]);
end;

procedure TCustomResItem.AddItem( item : TCustomSourceTransItem );
begin
  m_lstItems.add( item );
end;

procedure TSourceResItem.WriteToStream( w : TWriter );
var
  i : integer;
begin
    w.writestring(m_strResName);
    w.writeinteger( ord(m_resType) );
    w.WriteInteger( ord(m_UpdateState) );

    w.writeinteger(m_lstItems.Count);
    for i := 0 to m_lstItems.Count-1 do
      (m_lstItems[i] as TSourceItem).WriteToStream(w);
end;

procedure TSourceResItem.ReadFromStream( r : TReader );
var
  i : integer;
  iCount : integer;
begin
    m_strResName  := r.readString;
    m_resType     := TResType(r.readinteger);
    m_UpdateState := TUpdateState( r.ReadInteger );

    iCount := r.readInteger;
    for i := 0 to iCount-1 do begin
      m_lstItems.Add( TSourceItem.Create(self, r) )
    end;
end;

function TSourceResItem.GetItem( i : integer ) : TSourceItem;
begin
  result := TSourceItem(m_lstItems[i]);
end;

procedure TTransResItem.WriteToStream( w : TWriter; bOnlyBookmarked : boolean);
var
  i : integer;
  TransItem : TTransItem;
begin
    w.writestring(m_strResName);
    w.writeinteger( ord(m_resType) );
    w.WriteInteger( ord(m_UpdateState) );

    w.writeinteger(m_lstItems.Count);
    for i := 0 to m_lstItems.Count-1 do begin
      TransItem := m_lstItems[i] as TTransItem;
      if not bOnlyBookmarked or (bOnlyBookmarked and TransItem.Bookmark) then
         TransItem.WriteToStream(w);
    end;
end;

procedure TTransResItem.ReadFromStream( r : TReader );
var
  i : integer;
  iCount : integer;
begin
    m_strResName := r.readString;
    m_resType := TresType(r.readinteger);
    m_UpdateState := TUpdateState( r.ReadInteger );

    iCount := r.readInteger;
    for i := 0 to iCount-1 do begin
      m_lstItems.Add( TTransItem.Create(self, r) )
    end;
end;

function TTransResItem.GetItem( i : integer ) : TTransItem;
begin
  result := TTransItem(m_lstItems[i]);
end;
//******************** SourceTextList *********************
constructor TSourceTextList.Create( source : TSource );
begin
  m_ParentSource := source;
  m_lstItems := TObjectList.create;
  inherited Create;
end;

destructor TSourceTextList.destroy;
begin
  m_lstItems.free;
  inherited;
end;

procedure TSourceTextList.ReadFromFile;
var
  s : TFileStream;
  r : TReader;
  i, iCount : integer;
begin
  s := TFileStream.Create(m_ParentSource.ListFileName, fmOpenRead);
  r := TReader.Create(s, 4096);
  try
   r.ReadString;
   r.readinteger;
   iCount := r.readinteger;
   m_lstItems.Clear;
   m_lstItems.Capacity := iCount;//speeds up memory-management
   for i := 0 to iCount-1 do begin
         m_lstItems.Add( TSourceResItem.Create(self, r) );
   end;
  finally
   r.free;
   s.Free;
  end;
end;

procedure TSourceTextList.WriteToFile;
var
  s : TFileStream;
  w : TWriter;
  i : integer;
begin
  s := TFileStream.Create(m_ParentSource.ListFileName, fmCreate or fmOpenWrite);
  w := TWriter.Create(s, 4096);
  try
   w.writestring('SRC');
   w.writeinteger(0);
   w.writeinteger(m_lstItems.Count);
   for i := 0 to m_lstItems.Count-1 do begin
       (m_lstItems[i] as TSourceResItem).WriteToStream(w);
   end;
  finally
   w.free;
   s.Free;
  end;
end;

procedure TSourceTextList.ExportToASCII( const strFileName : string );
var
  i, iRes : integer;
  sourceResItem : TSourceResItem;
  sourceItem : TSourceItem;
  f : TextFile;
begin
  assignfile( f, strFileName );
  rewrite(f);
  for iRes := 0 to m_lstItems.Count-1 do begin
      sourceResItem := m_lstItems[iRes] as TSourceResItem;
      for i := 0 to sourceResItem.count-1 do begin
        sourceItem := sourceResItem.sourceitems[i];
        if not sourceItem.Hidden then begin
          writeln(f, sourceItem.ID, '=', TrimSysChars(sourceItem.Text) );
        end;
      end;
  end;
  closefile(f);
end;

procedure TSourceTextList.ResetChangedFlags;
var
  i, iRes : integer;
  sourceResItem : TSourceResItem;
  sourceItem : TSourceItem;
begin
  for iRes := 0 to m_lstItems.Count-1 do begin
      sourceResItem := m_lstItems[iRes] as TSourceResItem;
      sourceResItem.UpdateState := usNone;
      for i := 0 to sourceResItem.count-1 do begin
        sourceItem := sourceResItem.sourceitems[i];
        sourceItem.UpdateState := usNone;
      end;
  end;
end;

function TSourceTextList.GetItemByNumber( iNo : integer ) : TSourceItem;
var
  i, t : integer;
  SourceResItem : TSourceResItem;
begin
  result := nil;
  for i := 0 to m_lstItems.Count-1 do begin
      SourceResItem := m_lstItems[i] as TSourceResItem;
      for t := 0 to SourceResItem.Count-1 do begin
          if SourceResItem.SourceItems[t].No = iNo then begin
             result := SourceResItem.SourceItems[t];
             exit;
          end;
      end;
  end;
end;

function TSourceTextList.GetResItem( ResType : TResType; const ResName : string ) : TSourceResItem;
var
  i : integer;
  sourceResItem : TSourceResItem;
begin
  result := nil;
  for i := 0 to m_lstItems.Count-1 do begin
      sourceResItem := m_lstItems[i] as TSourceResItem;
      if (sourceResItem.ResType = resType) and (sourceResItem.resName = resName) then begin
         result := m_lstItems[i] as TSourceResItem;
         break;
      end;
  end;
end;

function TSourceTextList.GetSourceItem( ResType : TResType; const ResName, strID : string ) : TSourceItem;
var
  SourceResItem : TSourceResItem;
  SourceItem : TSourceItem;
  i : integer;
begin
   result := nil;
   SourceResItem := GetResItem( resType, ResName);
   if assigned(SourceResItem) then begin
     for i := 0 to SourceResItem.count-1 do begin
         SourceItem := SourceResItem.sourceitems[i];
         if sourceItem.ID = strID then
           result := sourceItem;
     end;
   end;
end;

procedure TSourceTextList.ReadFromResFile( out iNewStrings : integer );
var
  resFile : TResourceFile;
  i, iRes : integer;
  entry : TResourceEntry;
  sl : TStringList;
  ResourcestringItem : TSourceResItem;
  FormItem : TSourceResItem;
  SourceResItem : TSourceResItem;
  iNo : integer;
begin
  iNewStrings := 0;
  resFile := TResourceFile.Create;
  sl := TStringList.create;
  try
   resfile.loadfromfile(m_ParentSource.ResFileName);
   m_lstItems.Clear;
   ResourcestringItem         := TSourceResItem.Create(self);
   ResourcestringItem.ResType := rtResourceString;
   m_lstItems.Add(ResourcestringItem);
   sl.Sorted := true;
   for iRes := 0 to resfile.resources.Count-1 do begin
      entry := resfile.resources[iRes] as TResourceEntry;
      GetRes( entry, sl );
      if entry.IsResourceString then begin
         for i := 0 to sl.Count-1 do begin
           ResourcestringItem.AddItem( TSourceItem.Create(ResourcestringItem, sl.Names[i], sl.ValueFromIndex[i]) );
           inc(iNewStrings);
         end;
      end;
      if entry.IsForm then begin
         FormItem := TSourceResItem.Create(self);
         FormItem.ResType := rtForm;
         FormItem.resName := entry.sResourcename;
         m_lstItems.Add(FormItem);
         for i := 0 to sl.Count-1 do begin
           FormItem.AddItem( TSourceItem.Create(FormItem, sl.Names[i], sl.ValueFromIndex[i]) );
           inc(iNewStrings);
         end;
      end;
   end;
  finally
   sl.free;
   resFile.Free;
  end;

  //numbering
  iNo := 1;
  for iRes := 0 to m_lstItems.Count-1 do begin
     SourceResItem := m_lstItems[iRes] as TSourceResItem;
     for i := 0 to SourceResItem.count-1 do begin
       SourceResItem.SourceItems[i].No := iNo;
       inc(iNo);
     end;
  end;
end;

procedure TSourceTextList.CreateUpdate( const strResbind : string; slOutput : TStrings );
var
  iRes, iResult, i : integer;
  strOutput : string;
  strLine : string;
  OldSourceTextList : TSourceTextList;
  SourceResItem : TSourceResItem;
  SourceItem, OldSourceItem : TSourceItem;
  bUpdating : boolean;
  iNewResources, iDeletedResources, iNewStrings, iChangedStrings, iDeletedStrings : integer;
begin
  bUpdating := fileexists(m_ParentSource.ListFileName);
  if bUpdating then
    strLine := rsUpdating
  else
    strLine := rsCreating;
  strLine := strLine + ' '+rsSourceList+' - ' + m_ParentSource.Title;
  slOutput.add(strLine);
  application.ProcessMessages;
  try
   if not fileexists(strResbind) then begin
     slOutput.add(format(rsErrCannotExec, [strResbind]));
     exit;
   end;
   if not fileexists(m_ParentSource.ExeFileName) then begin
     slOutput.add(format(rsErrNoSourceFile, [m_ParentSource.ExeFileName]));
     exit;
   end;

   forceDirectories( m_ParentSource.BinPath );
   iResult := startapp(strResBind, [m_ParentSource.ExeFileName, '-r', m_ParentSource.ResFileName], strOutput);
   if strOutput <> '' then
     slOutput.Append(strOutput);

   if iResult = 0 then begin
     iChangedStrings   := 0;
     ideletedResources := 0;
     iDeletedStrings   := 0;
     ReadFromResFile( iNewStrings );
     iNewResources := m_lstItems.Count;
     //check update
     if bUpdating then begin
        iNewStrings   := 0;
        iNewResources := 0;
        OldSourceTextList := TSourceTextList.Create(m_ParentSource);
        try
         OldSourceTextList.ReadFromFile;
         for iRes := 0 to m_lstItems.Count-1 do begin
           sourceResItem := (m_lstItems[iRes] as TSourceresItem);
           if OldSourceTextList.GetResItem( sourceResItem.ResType, sourceResItem.ResName ) <> nil then begin
              for i := 0 to sourceResItem.count-1 do begin
                 SourceItem := sourceResItem.sourceitems[i];
                 OldSourceItem := OldSourceTextList.GetSourceItem(sourceResItem.ResType,
                                                                  sourceResItem.ResName,
                                                                  sourceItem.ID);
                 if assigned(OldSourceItem) then begin
                    if SourceItem.UpdateFromOld(OldSourceItem) then
                       inc(iChangedStrings);
                 end else begin
                    SourceItem.UpdateState := usNew;
                    inc(iNewStrings);
                 end;
              end;
           end else begin
              inc(iNewResources);
              sourceResItem.UpdateState := usNew;
           end;
         end;//for

         //check deleted
         if assigned(OldSourceTextList) then
           for iRes := 0 to OldSourceTextList.Items.Count-1 do begin
             SourceResItem  := OldSourceTextList.Items[iRes] as TSourceResItem;
             if not assigned(GetResItem(SourceResItem.resType, SourceResItem.ResName)) then
               inc(iDeletedResources);
             for i := 0 to SourceResItem.Count-1 do begin
               SourceItem := SourceResItem.SourceItems[i];
               if not assigned(GetSourceItem(SourceResItem.resType, SourceResItem.ResName, SourceItem.ID )) then
                 inc(iDeletedStrings);
             end;//for i
           end;//for iRes
        finally
         OldSourceTextList.free;
        end;
     end;//if bUpdating
     m_ParentSource.LastUpdate := now;
     writeToFile;
     if (iNewResources=0) and (iDeletedResources=0) and
        (iNewStrings=0) and (iChangedStrings=0) and (iDeletedStrings=0)
     then
        slOutput.add( rsOKNoChanges )
     else
        slOutput.add( format(rsOKStatus, [iNewResources, iDeletedResources,
                                          iNewStrings, iChangedStrings, iDeletedStrings]) )
   end else
      slOutput.add(rsError+' '+inttostr(iResult) );
  except
    on e : exception do
      slOutput.add( rsError+': '+ e.Message );
  end;
end;

//******************** TranslationTextList *********************
constructor TTransTextList.Create( const strFileName : string );
begin
   m_lstItems := TObjectList.create;
   m_strListFileName := strFileName;
   inherited Create;
end;

constructor TTransTextList.Create( Translation : TTranslation );
begin
  m_ParentTranslation := translation;
  m_iSourceLangNo := translation.ParentSource.LangNo;
  m_iTransLangNo  := translation.LangNo;
  Create( Translation.ListFileName );
end;

destructor TTransTextList.destroy;
begin
  m_lstItems.free;
  inherited;
end;

procedure TTransTextList.ReadFromFile( const strFilename : string );
var
  s : TFileStream;
  r : TReader;
  i, iCount : integer;
  strTmpFileName : string;
begin
  if strFileName = '' then
    strTmpFileName := m_strListFileName
  else
    strTmpFileName := strFileName;

  s := TFileStream.Create(m_strListFileName, fmOpenRead);
  r := TReader.Create(s, 4096);
  try
   r.ReadString;
   r.readinteger;
   m_iSourceLangNo := GetLangNoFromID(r.Readstring);
   m_iTransLangNo  := GetLangNoFromID(r.Readstring);
   iCount := r.readinteger;
   m_lstItems.Clear;
   m_lstItems.capacity := iCount;//speeds up memory-management
   for i := 0 to iCount-1 do begin
         m_lstItems.Add( TTransResItem.Create(self, r) );
   end;
  finally
   r.free;
   s.Free;
  end;
end;

procedure TTransTextList.WriteToFile( const strFileName : string; bOnlyBookmarked : boolean );
var
  s : TFileStream;
  w : TWriter;
  strTmpFileName : string;
  i : integer;
begin
  if strFileName = '' then
    strTmpFileName := m_strListFileName
  else
    strTmpFileName := strFileName;
  s := TFileStream.Create(strTmpFileName, fmCreate or fmOpenWrite);
  w := TWriter.Create(s, 4096);
  try
   w.writestring('TRL');
   w.writeinteger(0);
   w.writestring(getlangID(m_iSourceLangNo));
   w.writestring(getlangID(m_iTransLangNo));
   w.writeinteger(m_lstItems.Count);
   for i := 0 to m_lstItems.Count-1 do begin
        (m_lstItems[i] as TTransResItem).WriteToStream(w, bOnlyBookmarked);
   end;
  finally
   w.free;
   s.Free;
  end;
end;

procedure TTransTextList.WriteToTextFile( const strFilename : string; bOnlyBookmarked : boolean );
var
  f : TextFile;
  i, iRes, t : integer;
  TransResItem : TTransResItem;
  TransItem : TTransItem;
  sl : Tstringlist;
begin
  assignfile(f, strFilename );
  rewrite(f);
  sl := TStringList.create;
  try
  writeln(f, '@ LinLocalize Text-Translation-File');
  writeln(f, '@ '+ getLangID(m_iSourceLangNo) + ' -> ' + GetLangID(m_iTransLangNo) );
  write(f, '@ Translation-List - ');
  if assigned(m_ParentTranslation) then
     write(f, m_ParentTranslation.ParentSource.title+', ');
  writeln(f, GetLang(m_iSourceLangNo)+' -> '+GetLang(m_iTransLangNo) );
  writeln(f, '@ Dont modify lines starting with ''@''!');
  writeln(f, '@ For translation change lines after "="-line');

  for iRes := 0 to m_lstItems.Count-1 do begin
      TransResItem := m_lstItems[iRes] as TTransResItem;
      for i := 0 to TransResItem.count-1 do begin
        TransItem := TransResItem.transitems[i];
        if transitem.Editable and not TransItem.Translated then begin
           if not bOnlyBookmarked or (bOnlyBookmarked and TransItem.Bookmark) then begin
              writeln(f);
              write(f, format('@%d  String %s', [TransItem.No, stripID(TransItem.ID)]));
              if TransResItem.ResName <> '' then
                 write(f, ' in ' + uppercase(TransResItem.ResName) );
              writeln(f, '--------------------------------');
              sl.CommaText := TransItem.SourceComment;
              for t := 0 to sl.count-1 do begin
                writeln(f, '@ '+sl[t]);
              end;
              writeln(f, TransItem.SourceText);
              writeln(f, '=');
              writeln(f, TransItem.SourceText);
           end;
        end;
      end;
  end;
  finally
  closefile(f);
  end;
end;

procedure TTransTextList.ReadFromTextFile( const strFilename : string );
var
  f : TextFile;
  strLine : string;
  strSource, strTrans : string;
  iNo, iCode : integer;
  TransItem : TTransItem;
begin
  assignfile(f, strFilename );
  reset(f);
  try
//    assert(false, 'No implemented yet!');
    readln(f, strLine);
    while not eof(f) do begin
      if (length(strLine) >= 2) and (strLine[1]='@') and (strLine[2] in ['0'..'9']) then begin
         val(copy(strLine, 2, length(strLine)), iNo, iCode);
         readLn(f, strLine);
         //read comment
         while (strLine<>'') and (strLine[1]='@') do
           readLn(f, strLine);
         //read source-text
         strSource := '';
         while not eof(f) and (strLine<>'=') do begin
           if strSource <> '' then
             strSource := strSource + #10;
           strSource := strSource + strLine;
           readln(f, strLine);
         end;
         if not eof(f) then begin
             //read trans-text
             strTrans := '';
             strLine  := '';
             while not eof(f) and not ( (length(strLine)>=2) and (strLine[1]='@') and (strLine[2] in ['0'..'9'])) do begin
                if strTrans <> '' then
                   strTrans := strTrans + #10;
                strTrans := strTrans + strLine;
                readln(f, strLine);
             end;
             if eof(f) then
               strTrans := strTrans + strLine;
             if (length(strTrans) > 0) and (strTrans[length(strTrans)]=#10) then
               setLength(strTrans, length(strTrans)-1);

             TransItem := GetItemByNumber(iNo);
             if assigned(TransItem) and Transitem.Editable then begin
                if TransItem.SourceText <> strSource then begin
                   TransItem.SourceText := strSource;
                   if TransItem.Translated then
                     TransItem.Review := true;
                end;
                if TransItem.TransText <> strTrans then begin
                  TransItem.TransText := strTrans;
                  TransItem.Translated  := true;
                  TransItem.Review      := true;
                end;
             end;
         end;
      end else
        readln(f, strLine);
    end;
  finally
    closefile(f);
  end;
end;

function TTransTextList.Translate( TransItemIn : TTransItem; const strSource : string; out strTrans : string) : integer;
var
  i, iRes : integer;
  TransResItem : TTransResItem;
  TransItem : TTransItem;
begin
  result := 0;
  strTrans := '';
  for iRes := 0 to m_lstItems.Count-1 do begin
      TransResItem := m_lstItems[iRes] as TTransResItem;
      for i := 0 to TransResItem.count-1 do begin
        TransItem := TransResItem.transitems[i];
        if (TransItem<>TransItemIn) and TransItem.Translated and
           CompareNoEllipse(TransItem.SourceText, strSource)
        then begin
           if result = 0 then begin
             strTrans := TransItem.TransText;
             if hasEllipse(strSource) then
               strTrans := IncludeEllipse(strTrans)
             else
               strTrans := ExcludeEllipse(strTrans)
           end;
           inc(result);
        end;
      end;
  end;
end;

function TTransTextList.TranslateHint( TransItemIn : TTransItem; const strSource : string; out strTrans : string ) : boolean;
var
  iPos : integer;
  i, iRes : integer;
  TransResItem : TTransResItem;
  TransItem : TTransItem;
  strSource1, strSource2 : string;
  strTrans1, strTrans2 : string;
  b1, b2 : boolean;
begin
  result := false;
  strTrans := '';
  iPos := pos('|', strSource);
  if iPos=0 then exit;

  strSource1 := copy(strSource, 1, ipos-1);
  strSource2 := copy(strSource, iPos+1, length(strSource) );
  strTrans1 := strSource1;
  strTrans2 := strSource2;
  b1 := false;
  b2 := false;
  for iRes := 0 to m_lstItems.Count-1 do begin
      TransResItem := m_lstItems[iRes] as TTransResItem;
      for i := 0 to TransResItem.count-1 do begin
        TransItem := TransResItem.transitems[i];
        if (TransItem<>TransItemIn) and TransItem.Translated then begin
             if not b1 and compareNoEllipse(strSource1, TransItem.SourceText) then begin
               strTrans1 := TransItem.TransText;
               if hasEllipse(strSource1) then
                 strTrans1 := IncludeEllipse(strTrans1)
               else
                 strTrans1 := ExcludeEllipse(strTrans1);
               b1 := true;
             end;
             if not b2 and compareNoEllipse(strSource2, TransItem.SourceText) then begin
               strTrans2 := TransItem.TransText;
               if hasEllipse(strSource2) then
                 strTrans2 := IncludeEllipse(strTrans2)
               else
                 strTrans2 := ExcludeEllipse(strTrans2);
               b2 := true;
             end;
             if b1 and b2 then break;
        end;//if
        if b1 and b2 then break;
      end;//for
  end;//for
  strTrans := strTrans1 + '|' + strTrans2;
  result := b1 or b2;
end;

procedure TTransTextList.ClearAllBookmarks;
var
  i, iRes : integer;
  TransResItem : TTransResItem;
begin
  for iRes := 0 to m_lstItems.Count-1 do begin
      TransResItem := m_lstItems[iRes] as TTransResItem;
      for i := 0 to TransResItem.count-1 do begin
        TransResItem.Transitems[i].Bookmark := false;
      end;
  end;
end;

procedure TTransTextList.GetStatistics( out iStringCount, iWordCount, iTransRate, iLengthChange : integer );
var
  i, t : integer;
  TransResItem : TTransResItem;
  TransItem : TTransItem;
  iTransCount, iSourceLen, iTransLen : integer;
begin
  iStringCount := 0;
  iWordCount   := 0;
  iTransCount  := 0;
  iSourceLen   := 0;
  iTransLen    := 0;
  for i := 0 to m_lstItems.Count-1 do begin
      TransResItem := m_lstItems[i] as TTransResItem;
      for t := 0 to TransResItem.Count-1 do begin
          TransItem := TransResItem.TransItems[t];
          if transitem.Editable then begin
            inc(iStringCount);
            if TransItem.Translated then begin
              inc(iTransCount);
              iSourceLen := iSourceLen + GetCharCount(TransItem.SourceText);
              iTransLen  := iTransLen + GetCharCount(TransItem.TransText);
            end;
            iWordCount := iWordCount + GetWordCount(TransItem.SourceText);
          end;
      end;
  end;
  iTransRate    := 0;
  iLengthChange := 0;
  if iStringCount > 0 then
    iTransRate    := round(iTransCount / iStringCount * 100);
  if iSourceLen > 0 then
    iLengthChange := round(iTransLen / iSourceLen * 100);
end;

function TTransTextList.GetItemByNumber( iNo : integer ) : TTransItem;
var
  i, t : integer;
  TransResItem : TTransResItem;
begin
  result := nil;
  for i := 0 to m_lstItems.Count-1 do begin
      TransResItem := m_lstItems[i] as TTransResItem;
      for t := 0 to TransResItem.Count-1 do begin
          if TransResItem.TransItems[t].No = iNo then begin
             result := TransResItem.TransItems[t];
             exit;
          end;
      end;
  end;
end;

function TTransTextList.GetResItem( ResType : TResType; const ResName : string ) : TTransResItem;
var
  i : integer;
  TransResItem : TTransResItem;
begin
  result := nil;
  for i := 0 to m_lstItems.Count-1 do begin
      TransResItem := m_lstItems[i] as TTransResItem;
      if (TransResItem.ResType = resType) and (TransResItem.resName = resName) then begin
         result := m_lstItems[i] as TTransResItem;
         break;
      end;
  end;
end;

function TTransTextList.GetTransItem( ResType : TResType; const ResName, strID : string ) : TTransItem;
var
  TransResItem : TTransResItem;
  TransItem : TTransItem;
  i : integer;
begin
   result := nil;
   TransResItem := GetResItem( resType, ResName);
   if assigned(TransResItem) then begin
     for i := 0 to TransResItem.count-1 do begin
         TransItem := TransResItem.Transitems[i];
         if TransItem.ID = strID then
           result := TransItem;
     end;
   end;
end;

function TTransTextList.GetTransLationResItemByName( const strName : string ) : TTransResItem;
var
  i : integer;
begin
  result := nil;
  for i := 0 to m_lstItems.Count-1 do begin
     if (m_lstItems[i] as TTransResItem).ResName = strName then begin
         result := m_lstItems[i] as TTransResItem;
         exit;
     end;
  end;
end;

procedure TTransTextList.WriteToResFile;
var
  sourceResFile : TResourceFile;
  resEntry : TResourceEntry;
  TransResItem : TTransResItem;
  TransItem : TTransItem;
  iResEntry, i : integer;
  sl : TStringList;
  iStrNo : integer;
begin
  sourceResFile := TResourceFile.Create;
  try
   sourceResFile.LoadFromFile(m_ParentTranslation.ParentSource.ResFileName);
   sl := TStringList.Create;
   try
    for iResEntry := 0 to sourceResFile.resources.count-1 do begin
      resEntry := sourceResFile.resources[iResEntry] as TResourceEntry;
      if resEntry.IsForm then begin
         TransResItem := GetTransLationResItemByName( resEntry.sResourcename );
         if assigned( TransResItem ) then begin
            sl.Clear;
            for i := 0 to TransResItem.count-1 do begin
               TransItem := TransResItem.Transitems[i];
               sl.Add(TransItem.ID+'='+TransItem.TransText);
            end;
            SetRes(resEntry, sl);
         end;
      end;//form
      if resEntry.IsResourceString then begin
         TransResItem := m_lstItems[0] as TTransResItem;
         sl.Clear;
         for i := 0 to TransResItem.count-1 do begin
             TransItem := TransResItem.Transitems[i];
            iStrNo := strtoint(TransItem.ID);
            if (iStrNo >= resEntry.wResourcename*16) and (iStrNo <= resEntry.wResourcename*16+16-1) then begin
                sl.Add(TransItem.ID+'='+TransItem.TransText);
                if sl.Count = 16 then break;
            end;
         end;
         SetRes(resEntry, sl);
      end;//resourcestring
    end;//for
   finally
     sl.Free;
   end;
   sourceResFile.SaveToFile(m_ParentTranslation.ResFileName);
  finally
   sourceResFile.free;
  end;
end;

procedure TTransTextList.UpdateFromSourceTextList( out iNewResources, iDeletedResources,
                                                       iNewStrings, iChangedStrings, iDeletedStrings : integer );
var
  SourceTextList     : TSourceTextList;
  SourceResItem      : TSourceResItem;
  TransResItem       : TTransResItem;
  SourceItem         : TSourceItem;
  OldTransTextList   : TTransTextList;
  TransItem, OldTransItem : TTransItem;
  iRes, i : integer;
  bChanged : boolean;
begin
  OldTransTextList := nil;
  sourceTextList   := TSourceTextList.Create(m_ParentTransLation.ParentSource);
  try
  sourceTextList.ReadFromFile;
  if fileexists(m_strListFileName) then begin
    OldTransTextList := TTransTextList.create(m_strListFileName);
    OldTransTextList.ReadFromFile;
  end;

  iNewResources   := 0;
  iNewStrings     := 0;
  iChangedStrings := 0;
  m_lstItems.Clear;
  for iRes := 0 to sourcetextList.Items.Count-1 do begin
     SourceResItem := sourcetextList.Items[iRes] as TSourceResItem;
     TransResItem  := TTransResItem.Create( self, sourceResItem );
     if assigned(OldTransTextList) then begin
        if not assigned(OldTransTextList.GetResItem(TransResItem.resType, TransResItem.ResName)) then begin
          TransResItem.UpdateState := usNew;
          inc(iNewResources);
        end;
     end else begin
        inc(iNewResources);
     end;
     for i := 0 to sourceResItem.Count-1 do begin
         SourceItem := sourceResItem.sourceItems[i];
         TransItem  := TTransItem.Create(TransResItem, SourceItem);
         bChanged := false;
         if assigned(OldTransTextList) then begin
           OldTransItem := OldTransTextList.GetTransItem(TransResItem.resType,
                                                         TransResItem.ResName,
                                                         TransItem.ID );
           if assigned(OldTransItem) then begin
               if TransItem.updateFromOld( oldTransItem, sourceItem.Correction) then
                  bChanged := true;
           end else begin
              inc(iNewStrings);
           end;
         end else begin
           inc(iNewStrings);
         end;
         if TransItem.UpdateFlagsFromSourceItem(SourceItem) then
           bChanged := true;
         if bChanged then inc(iChangedStrings);
         TransResItem.AddItem(transItem);
     end;
     m_lstItems.Add(TransResItem);
  end;//for ires

  //check deleted
  ideletedResources := 0;
  iDeletedStrings   := 0;
  if assigned(OldTransTextList) then
    for iRes := 0 to OldTransTextList.Items.Count-1 do begin
       TransResItem  := OldTransTextList.Items[iRes] as TTransResItem;
       if not assigned(GetResItem(TransResItem.resType, TransResItem.ResName)) then
         inc(iDeletedResources);
       for i := 0 to TransResItem.Count-1 do begin
           TransItem := TransResItem.TransItems[i];
           if not assigned(GetTransItem(TransResItem.resType, TransResItem.ResName, TransItem.ID )) then
             inc(iDeletedStrings);
       end;//for i
    end;//for iRes

  m_ParentTransLation.lastUpdate := now;
  finally
  OldTransTextList.free;
  sourceTextList.free;
  end;
end;

procedure TTransTextList.CreateUpdate( slOutput : TStrings );
var
  strLine : string;
  iNewResources, iDeletedResources, iNewStrings, iChangedStrings, iDeletedStrings : integer;
resourcestring
 rsErrNoSourceList = 'Error: Source-List %s does not exist!';
begin
  if not assigned(m_ParentTranslation) then
    raise Exception.Create(rsTransWithoutProject);
  forceDirectories( m_ParentTranslation.ParentSource.BinPath );
  if fileexists(m_strListFileName) then
     strLine := rsUpdating
  else
     strLine := rsCreating;
  strLine := strLine + format(' '+rsTransList+' - %s, %s -> %s',
                             [m_ParentTranslation.ParentSource.Title, GetLangID(m_iSourceLangNo), GetLangID(m_iTransLangNo)]);
  slOutput.add(strLine);
  application.ProcessMessages;

  try
   if not fileexists(m_ParentTranslation.ParentSource.ListFileName) then begin
       slOutput.add(format(rsErrNoSourceList, [m_ParentTranslation.ParentSource.ListFileName]));
       exit;
   end;

   UpdateFromSourcetextList( iNewResources, iDeletedResources, iNewStrings, iChangedStrings, iDeletedStrings );
   writeToFile;

   if (iNewResources=0) and (iDeletedResources=0) and
      (iNewStrings=0) and (iChangedStrings=0) and (iDeletedStrings=0)
   then
     slOutput.add( rsOKNoChanges )
   else
     slOutput.add( format(rsOkStatus, [iNewResources, iDeletedResources,
                                       iNewStrings, iChangedStrings, iDeletedStrings]) );
  except
    on e : exception do
      slOutput.add( rsError+': '+ e.Message );
  end;
end;

procedure TTransTextList.GenerateTarget( const strResBind : string; slOutput : TStrings );
var
  iRes : integer;
  strOutput : string;
  strFlag : string;
  strFileFormat : string;
resourcestring
  rsSharedResources = 'Shared Resources';
  rsExecutable = 'Executable';
  rsGenerating = 'Generating "%s" version of %s';
  rsErrNoTransList = 'Error: Translation-List %s does not exist!';
  rsOKWritten = 'OK - %s written to %s';
begin
  if not assigned(m_ParentTranslation) then
    raise Exception.Create(rsTransWithoutProject);

  case m_ParentTranslation.Fileformat of
    tfShared: begin
       strFlag := '-s';
       strFileFormat := rsSharedResources;
    end;
    tfExe: begin
       strFlag := '-o';
       strFileFormat := rsExecutable;
    end;
  end;

  slOutput.add(format(rsGenerating,
                     [getLangID(m_iTransLangNo),
                      m_ParentTranslation.parentsource.Title]));
  application.ProcessMessages;
  try
   if not fileexists(strResbind) then begin
       slOutput.add(format(rsErrCannotExec,[strResbind]));
       exit;
   end;
   if not fileexists(m_ParentTranslation.ParentSource.ExeFileName) then begin
       slOutput.add(format(rsErrNoSourceFile, [m_ParentTranslation.ParentSource.ExeFileName]));
       exit;
   end;
   if not fileexists(m_ParentTranslation.ListFileName) then begin
       slOutput.add(format(rsErrNoTransList, [m_ParentTranslation.ListFileName]));
       exit;
   end;

   writeToResFile;
   iRes := startapp(strResBind, [strFlag, m_ParentTranslation.TargetFileName, m_ParentTranslation.ParentSource.ExeFileName,
                                          m_ParentTranslation.ResFileName], strOutput);
   if strOutput <> '' then
       slOutput.Append(strOutput);
   if iRes = 0 then begin
       slOutput.add(format(rsOkWritten,[strFileFormat, m_ParentTranslation.TargetFileName]));
       m_ParentTranslation.LastTargetGeneration := now;
   end else
      slOutput.add(rsError+' '+inttostr(iRes) );
  except
    on e : exception do
      slOutput.add( rsError+': '+ e.Message );
  end;
end;

procedure TTransTextList.ScanTarget( slOutput : TStrings; const strFileName : string;
                                     const strResBind : string;
                                     bTranslated, bReview : boolean );
var
  iResult : integer;
  strOutput : string;
  iNewStrings, iTranslated : integer;
  strResFileName : string;
  sourceTextList : TSourceTextList;
  i, t : integer;
  TransResItem : TTransResItem;
  TransItem : TTransItem;
  SourceItem : TSourceItem;
resourcestring
  rsOKStatus = 'OK - %d strings have been translated.';
begin
  slOutput.add('Scanning');
  application.ProcessMessages;
  strResFileName := changefileext(m_strListFileName, '.res');
  try
   try
    iResult := startapp(strResBind, [strFileName, '-r', strResFileName], strOutput);
    if strOutput <> '' then
       slOutput.Append(strOutput);
    if iResult = 0 then begin
       iTranslated := 0;
       sourceTextList := TSourceTextList.Create(m_ParentTransLation.ParentSource);
       try
        sourceTextList.ReadFromResFile(iNewStrings);
        for i := 0 to m_lstItems.Count-1 do begin
         TransResItem := m_lstItems[i] as TTransResItem;
         for t := 0 to TransResItem.Count-1 do begin
           TransItem := TransResItem.TransItems[t];
           if Transitem.Editable and (not transitem.Translated or bTranslated) then begin
              sourceItem := SourcetextList.GetSourceItem(TransResItem.ResType, TransResItem.ResName, TransItem.ID);
              if assigned(sourceItem) then begin
                 //found
                 if transitem.transText <> sourceitem.Text then begin
                    //different
                    transitem.transText := sourceitem.Text;
                    transitem.translated := true;
                    inc(iTranslated);
                    if bReview then
                      transitem.review := true;
                 end;
              end;
           end;
         end;
        end;
       finally
        sourceTextList.free;
       end;
       slOutput.add( format(rsOKStatus, [iTranslated]) );
    end else
      slOutput.add(rsError+' '+inttostr(iResult) );
   finally
    deletefile(strResFileName);
   end;
  except
   on e : exception do
     slOutput.add( rsError+': '+ e.Message );
  end;
end;

end.
