{ *************************************************************************** }
{                                                                             }
{ This file is part of the LinLocalize project                                }
{                                                                             }
{ Copyright (c) 2003                                                          }
{ Jens Khner <kuehner@users.sourceforge.net>                                 }
{                                                                             }
{ This program is free software; you can redistribute it and/or               }
{ modify it under the terms of the GNU General Public                         }
{ License as published by the Free Software Foundation; either                }
{ version 2 of the License, or (at your option) any later version.            }
{                                                                             }
{ This program is distributed in the hope that it will be useful,             }
{ but WITHOUT ANY WARRANTY; without even the implied warranty of              }
{ MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU           }
{ General Public License for more details.                                    }
{                                                                             }
{ You should have received a copy of the GNU General Public License           }
{ along with this program; see the file COPYING.  If not, write to            }
{ the Free Software Foundation, Inc., 59 Temple Place - Suite 330,            }
{ Boston, MA 02111-1307, USA.                                                 }
{                                                                             }
{ *************************************************************************** }

unit uSourceListFm;

interface

uses
  uTextLists, uSourceFilterDlg,
  SysUtils, Types, Classes, Variants, QTypes, QGraphics, QControls, QForms,
  QDialogs, QStdCtrls, uCustomListFm, QMenus, QImgList, QActnList,
  QComCtrls, QGrids, QExtCtrls, uLLStringGrid;

type
  TSourceListFm = class(TCustomListFm)
    MainMenu1: TMainMenu;
    Edit1: TMenuItem;
    actProps1: TMenuItem;
    View1: TMenuItem;
    OpenResource2: TMenuItem;
    DisplayallStrings1: TMenuItem;
    N2: TMenuItem;
    DefineFilter1: TMenuItem;
    FilterOff1: TMenuItem;
    N3: TMenuItem;
    actOutput1: TMenuItem;
    StringList1: TMenuItem;
    Update1: TMenuItem;
    actSwapReadOnly: TAction;
    actSwapHidden: TAction;
    actResetChanged: TAction;
    N1: TMenuItem;
    SwapreadonlyFlag1: TMenuItem;
    SwaphiddenFlag1: TMenuItem;
    ResetchangedFlags1: TMenuItem;
    SwapreadonlyFlag2: TMenuItem;
    SwaphiddenFlag2: TMenuItem;
    N4: TMenuItem;
    N6: TMenuItem;
    GotoStringNumber1: TMenuItem;
    ExportSourceDlg: TSaveDialog;
    N7: TMenuItem;
    Find1: TMenuItem;
    Findnext1: TMenuItem;
    Findprevious1: TMenuItem;
    procedure FormShow(Sender: TObject);
    procedure FormDestroy(Sender: TObject);
    procedure actSwapReadonlyExecute(Sender: TObject);
    procedure actSwapHiddenFlagExecute(Sender: TObject);
    function gridTextesTextColor(ACol, ARow: Integer; ARect: TRect;
      AState: TGridDrawState): TColor;
    procedure actResetChangedFlagsExecute(Sender: TObject);
    procedure actPropsExecute(Sender: TObject);
    procedure gridTextesDblClick(Sender: TObject);
    procedure actUpdateStringListExecute(Sender: TObject);
    procedure actSaveExecute(Sender: TObject);
    procedure actDefineFilterExecute(Sender: TObject);
    procedure actOpenResourceExecute(Sender: TObject);
    procedure actGotoStringNoExecute(Sender: TObject);
  private
    { Private-Deklarationen }
    m_SourceTextList : TSourceTextList;
    m_Filter : TFilter;
    function GetCurSourceItem : TSourceItem;
  protected
    procedure UpdateTextListLine( iRow : integer ); override;
    function CanAdd( item : TObject ) : boolean; override;
    procedure SaveTextList; override;
  public
    { Public-Deklarationen }
    constructor Create( AOwner : TComponent; SourceTextList : TSourceTextList ); reintroduce;
    procedure Export;
    property SourceTextList : TSourceTextList read m_SourceTextList;
  end;

var
  SourceListFm: TSourceListFm;

implementation

{$R *.xfm}

uses
  uLanguage,
  uEditSourceItemFm, uApp, uMainFm, uGlobal, uGotoStringNoDlg;

constructor TSourceListFm.Create( AOwner : TComponent; SourceTextList : TSourceTextList );
begin
   inherited Create( AOwner );
   m_SourceTextList := SourceTextList;
   m_ResItems := SourceTextList.Items;
   m_Filter := [sfModified, sfNew];
end;

procedure TSourceListFm.FormShow(Sender: TObject);
resourcestring
  rsNo = 'No.';
  rsID = 'ID';
  rsState = 'State';
  rsText = 'Text';
  rsComment = 'Comment';
begin
    inherited;
    caption := caption + ' - ' + m_SourceTextList.ParentSource.Title;
    gridTextes.colcount := 5;
    gridTextes.cells[0,0] := rsNo;
    gridTextes.cells[1,0] := rsID;
    gridTextes.cells[2,0] := rsState;
    gridTextes.cells[3,0] := rsText;
    gridTextes.cells[4,0] := rsComment;
    gridTextes.ColWidths[0] := 32;
    gridTextes.ColWidths[1] := 120;
    gridTextes.ColWidths[2] := 35;
    gridTextes.ColWidths[3] := 150;
    gridTextes.ColWidths[4] := 100;
    FillResourceTree;
    actDisplayAllExecute(nil);
end;

procedure TSourceListFm.FormDestroy(Sender: TObject);
begin
  inherited;
  m_SourceTextList.free;
end;

procedure TSourceListFm.SaveTextList;
begin
  m_SourceTextList.WriteToFile;
  setModified(false);
end;

function TSourceListFm.GetCurSourceItem : TSourceItem;
begin
   result := TSourceItem(gridTextes.Objects[0, gridTextes.row]);
end;

procedure TSourceListFm.UpdateTextListLine( iRow : integer );
var
  sourceitem : TSourceItem;
  strState : string;
begin
   sourceItem := TSourceItem(gridTextes.Objects[0,iRow]);
   if assigned(sourceItem) then begin;
     gridTextes.Cells[0,iRow] := inttostr(sourceItem.No);
     gridTextes.Cells[1,iRow] := stripId(sourceItem.ID);
     case sourceItem.UpdateState of
        usNew:     strState := 'n';
        usChanged: strState := 'c';
        else       strState := '';
     end;
     if sourceItem.ReadOnly then
       strState := strState + 'r';
     if sourceItem.Hidden then
       strState := strState + 'h';
     if sourceItem.Correction then
       strState := strState + 'o';
     gridTextes.Cells[2,iRow] := strState;
     gridTextes.Cells[3,iRow] := TrimSysChars(sourceItem.Text);
     gridTextes.Cells[4,iRow] := TrimSysChars(sourceItem.Comment);
   end else begin
     gridTextes.Cells[0,iRow] := '';
     gridTextes.Cells[1,iRow] := '';
     gridTextes.Cells[2,iRow] := '';
     gridTextes.Cells[3,iRow] := '';
     gridTextes.Cells[4,iRow] := '';
   end;
end;

procedure TSourceListFm.actSwapReadonlyExecute(Sender: TObject);
var
  i : integer;
  sourceitem : TSourceItem;
  bReadOnly : boolean;
begin
  inherited;
  if GetCurSourceItem=nil then exit;
  bReadOnly := not GetCurSourceItem.ReadOnly;
  for i := gridTextes.Selection.Top to gridTextes.selection.Bottom do begin
    SourceItem := TSourceItem(gridTextes.objects[0, i]);
    if assigned(SourceItem) then begin
      SourceItem.ReadOnly := bReadOnly;
      UpdateTextListLine(i);
    end;
  end;
  setModified(true);
end;

procedure TSourceListFm.actSwapHiddenFlagExecute(Sender: TObject);
var
  i : integer;
  sourceitem : TSourceItem;
  bHidden : boolean;
begin
  inherited;
  if GetCurSourceItem=nil then exit;
  bHidden := not GetCurSourceItem.Hidden;
  for i := gridTextes.Selection.Top to gridTextes.selection.Bottom do begin
    SourceItem := TSourceItem(gridTextes.objects[0, i]);
    if assigned(SourceItem) then begin
      SourceItem.Hidden := bHidden;
      UpdateTextListLine(i);
    end;
  end;
  setModified(true);
end;

function TSourceListFm.gridTextesTextColor(ACol, ARow: Integer;
  ARect: TRect; AState: TGridDrawState): TColor;
var
  SourceItem : TSourceItem;
begin
  inherited;
  result := clBlack;
  SourceItem := TSourceItem(gridTextes.objects[0, ARow]);
  if assigned(SourceItem) then begin
    if SourceItem.UpdateState <> usNone then
        result := clBlue;
  end;
end;

procedure TSourceListFm.actResetChangedFlagsExecute(Sender: TObject);
begin
  inherited;
  m_SourceTextList.ResetChangedFlags;
  setModified(true);
  UpdateTextListLines;
end;

procedure TSourceListFm.gridTextesDblClick(Sender: TObject);
begin
  inherited;
  actPropsExecute(nil);
end;

procedure TSourceListFm.actPropsExecute(Sender: TObject);
begin
  inherited;
  actPropsUpdate(nil);
  if actProps.Enabled then begin
      with TEditSourceItemFm.create(self, m_SourceTextList, gridTextes) do begin
        try
         showmodal;
        finally
         free;
        end;
      end;
  end;
end;

procedure TSourceListFm.actUpdateStringListExecute(Sender: TObject);
begin
  inherited;
  MainFm.memOutput.lines.clear;
  screen.cursor := crHourglass;
  try
    if fileexists(m_SourceTextList.parentsource.listFilename) then
       m_SourceTextList.WriteToFile;
    m_SourceTextList.CreateUpdate( g_app.ResbindLocation, MainFm.memOutput.lines);
    setModified(false);
    FillResourceTree;
    actDisplayAllExecute(nil);
  finally
    screen.cursor := crDefault;
  end;
end;

procedure TSourceListFm.actSaveExecute(Sender: TObject);
begin
  inherited;
  m_SourceTextList.WriteToFile;
  setModified(false);
end;

function TSourceListFm.CanAdd( item : TObject ) : boolean;
var
  SourceItem : TSourceItem;
begin
  SourceItem := TSourceItem(item);
  result := true;
  if m_bFilter then begin
     result := false;
     if not result and (sfUnModified in m_Filter) and (SourceItem.UpdateState = usNone) then
       result := true;
     if not result and (sfModified in m_Filter) and (SourceItem.UpdateState = usChanged) then
       result := true;
     if not result and (sfNew in m_Filter) and (SourceItem.UpdateState = usNew) then
       result := true;
  end;
end;

procedure TSourceListFm.actDefineFilterExecute(Sender: TObject);
begin
  inherited;
  with TSourceFilterDlg.Create(self) do begin
    try
     Filter := m_Filter;
     case showmodal of
       mrOK: begin
          m_Filter  := Filter;
          m_bFilter := true;
          FillTextList;
       end;
       mrNo: begin
          m_bFilter := false;
          FillTextList;
       end;
     end;
    finally
     free;
    end;
  end;
end;

procedure TSourceListFm.actOpenResourceExecute(Sender: TObject);
var
  sourceItem : TSourceItem;
begin
  inherited;
  if activeControl = gridTextes then begin
      SourceItem := gridTextes.objects[0, gridTextes.row] as TSourceItem;
      Openresource(SourceItem.ParentSourceResItem, SourceItem);
  end;
end;

procedure TSourceListFm.actGotoStringNoExecute(Sender: TObject);
var
  SourceItem : TSourceItem;
begin
   inherited;
   with TGotoStringNoDlg.create(self) do begin
     if showModal = mrOK then begin
        SourceItem := m_SourceTextList.GetItemByNumber(spinNo.Value);
        if assigned(SourceItem) then begin
           OpenResource(SourceItem.ParentSourceResItem, SourceItem);
        end;
     end;
   end;
end;

procedure TSourceListFm.export;
begin
  exportSourceDlg.filename := changeFileExt(extractFileName(m_SourceTextList.ParentSource.ListFileName), '')
                              + '.'+getLangId(m_SourceTextList.ParentSource.LangNo)+'.srctxt';
  if exportSourceDlg.execute then begin
     m_SourceTextList.ExportToAscii( exportSourceDlg.Filename );
  end;
end;


end.
