{ *************************************************************************** }
{                                                                             }
{ This file is part of the LinLocalize project                                }
{                                                                             }
{ Copyright (c) 2003                                                          }
{ Jens Khner <kuehner@users.sourceforge.net>                                 }
{                                                                             }
{ This program is free software; you can redistribute it and/or               }
{ modify it under the terms of the GNU General Public                         }
{ License as published by the Free Software Foundation; either                }
{ version 2 of the License, or (at your option) any later version.            }
{                                                                             }
{ This program is distributed in the hope that it will be useful,             }
{ but WITHOUT ANY WARRANTY; without even the implied warranty of              }
{ MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU           }
{ General Public License for more details.                                    }
{                                                                             }
{ You should have received a copy of the GNU General Public License           }
{ along with this program; see the file COPYING.  If not, write to            }
{ the Free Software Foundation, Inc., 59 Temple Place - Suite 330,            }
{ Boston, MA 02111-1307, USA.                                                 }
{                                                                             }
{ *************************************************************************** }

unit uProjectSetupDlg;

interface

uses
  uProject,
  SysUtils, Types, Classes, Variants, QTypes, QGraphics, QControls, QForms,
  QDialogs, QStdCtrls, QActnList;

type
  TProjectSetupDlg = class(TForm)
    Label1: TLabel;
    Label2: TLabel;
    efName: TEdit;
    efLocation: TEdit;
    Label3: TLabel;
    lbSources: TListBox;
    lbLanguages: TListBox;
    Label4: TLabel;
    btnBrowseLocation: TButton;
    btnAddSource: TButton;
    btnAddLanguage: TButton;
    btnOK: TButton;
    btnCancel: TButton;
    AddSourceDlg: TOpenDialog;
    btnDeleteSource: TButton;
    ActionList1: TActionList;
    actDeleteSource: TAction;
    actAddSource: TAction;
    actAddLang: TAction;
    Button1: TButton;
    actDeleteLang: TAction;
    procedure FormShow(Sender: TObject);

    procedure btnBrowseLocationClick(Sender: TObject);
    procedure efNameChange(Sender: TObject);
    procedure efLocationChange(Sender: TObject);
    procedure btnOKClick(Sender: TObject);
    procedure actAddSourceExecute(Sender: TObject);
    procedure actAddLangExecute(Sender: TObject);
    procedure actDeleteSourceExecute(Sender: TObject);
    procedure actDeleteSourceUpdate(Sender: TObject);
    procedure actDeleteLangExecute(Sender: TObject);
    procedure actDeleteLangUpdate(Sender: TObject);
    procedure FormDestroy(Sender: TObject);
  private
    { Private-Deklarationen }
    m_Project : TProject;
    m_bNew : boolean;
    procedure UpdateOperations;
  public
    { Public-Deklarationen }
    constructor Create( AOwner : TComponent; project : TProject; bNew : boolean ); reintroduce;
  end;

var
  ProjectSetupDlg: TProjectSetupDlg;

implementation

{$R *.xfm}

uses
  uSelectLanguageDlg, uLanguage, uSourceListPropsDlg, uGlobal;

type
  TSourceItem = class
    m_strFileName : string;
    m_iLangNo : integer;
  end;

constructor TProjectSetupDlg.Create( AOwner : TComponent; project : TProject; bNew : boolean );
begin
  inherited Create(AOwner);
  m_Project := project;
  m_bNew    := bNew;
end;

procedure TProjectSetupDlg.FormShow(Sender: TObject);
var
  i : integer;
  source : TSource;
  sourceItem : TSourceItem;
begin
   for i := 0 to m_Project.sources.Count-1 do begin
      source := m_project.Sources[i] as TSource;
      sourceitem := TSourceItem.create;
      sourceitem.m_strFileName := source.ExeFileName;
      sourceitem.m_iLangNo     := source.LangNo;
      lbSources.Items.AddObject( source.ExeFileName + ' - ' + GetLangID(source.LangNo),
                                 sourceitem );
   end;
   for i := 0 to m_Project.languages.Count-1 do begin
      lbLanguages.Items.AddObject( GetLang(integer(m_Project.languages[i])), pointer(m_Project.languages[i]) );
   end;
   efName.text     := changeFileExt(extractFileName(m_project.filename),'');
   if m_bNew then begin
     efLocation.text := ExcludeTrailingPathDelimiter(getHomeDir)
   end else
     efLocation.text := ExcludeTrailingPathDelimiter(extractFilePath(m_project.filename));
   efName.Enabled     := m_bNew;
   efLocation.enabled := m_bNew;
   btnBrowseLocation.Visible := m_bNew;
   UpdateOperations;
end;

procedure TProjectSetupDlg.FormDestroy(Sender: TObject);
var
 i : integer;
begin
  for i := 0 to lbSources.items.Count-1 do
    lbSources.items.objects[i].free;
end;

procedure TProjectSetupDlg.btnBrowseLocationClick(Sender: TObject);
var
  strDir : Widestring;
resourcestring
  rsTitle = 'Select location of project';
begin
  if SelectDirectory(rsTitle, '', strDir) then begin
    efLocation.Text := strDir;
    UpdateOperations;
  end;
end;

procedure TProjectSetupDlg.actAddSourceExecute(Sender: TObject);
var
  sourceItem : TSourceItem;
begin
   if AddSourceDlg.Execute then begin
       with TSourceListPropsDlg.create(self, nil) do begin
        try
         //preset last language
         if lbSources.ItemIndex <> -1 then begin
           sourceitem := lbSources.items.objects[lbSources.ItemIndex] as TSourceItem;
           comboLanguage.itemindex := sourceitem.m_iLangNo;
         end;
         efFileName.Text := AddSourceDlg.FileName;
         if showmodal = mrOk then begin
             sourceitem := TSourceItem.create;
             sourceitem.m_strFileName := efFileName.Text;
             sourceitem.m_iLangNo     := comboLanguage.itemindex;
             lbSources.ItemIndex := lbSources.Items.addobject( efFileName.Text + ' - ' + GetLangID(comboLanguage.itemindex), sourceitem );
             UpdateOperations;
         end;
        finally
         free;
        end;
       end;//with
   end;//if
end;

procedure TProjectSetupDlg.actAddLangExecute(Sender: TObject);
begin
  with TSelectLanguageDlg.create(self) do begin
    try
     if showmodal = mrOK then begin
         if lbLanguages.Items.IndexOfObject(pointer(comboLanguage.itemindex)) = -1 then begin
           lbLanguages.Items.AddObject( GetLang(comboLanguage.itemindex), pointer(comboLanguage.itemindex) );
           UpdateOperations;
         end;
     end;
    finally
     free;
    end;
  end;
end;

procedure TProjectSetupDlg.UpdateOperations;
begin
  btnOK.Enabled := (efName.Text<>'') and directoryexists(efLocation.Text);
end;

procedure TProjectSetupDlg.efNameChange(Sender: TObject);
begin
   UpdateOperations;
end;

procedure TProjectSetupDlg.efLocationChange(Sender: TObject);
begin
   UpdateOperations;
end;

procedure TProjectSetupDlg.btnOKClick(Sender: TObject);
var
  i, t : integer;
  sl : TStringList;
  source : TSource;
  sourceItem : TSourceItem;
  iLangNo : integer;
  bFound : boolean;
begin
  m_project.FileName := changeFileExt(includeTrailingPathDelimiter(efLocation.text) + efName.Text, '.llp');
  //delete languages
  for i := m_Project.languages.Count-1 downto 0 do begin
      iLangNo := integer(m_Project.languages[i]);
      if lbLanguages.Items.indexofObject(pointer(iLangNo)) = -1 then begin
         m_Project.RemoveLanguageAndFiles(iLangNo);
      end;
  end;
  //add new languages
  for i := 0 to lbLanguages.items.count-1 do begin
      iLangNo := integer(lbLanguages.Items.objects[i]);
      if m_Project.Languages.IndexOf(pointer(iLangNo)) = -1 then
         m_Project.AddLanguage(iLangNo);
  end;

  //delete sources
  for i := m_Project.sources.Count-1 downto 0 do begin
      source := (m_Project.sources[i] as TSource);
      bFound := false;
      for t := 0 to lbSources.Items.count-1 do begin
        if (lbSources.Items.objects[t] as TSourceItem).m_strFilename = source.ExeFilename then begin
           bFound := true;
           break;
        end;
      end;
      if not bFound then begin
        source.deletefiles;
        m_Project.Sources.Delete(i);
      end;
  end;
  //sources of project to stringlist
  sl := TStringList.Create;
  try
   for i := 0 to m_Project.sources.Count-1 do begin
      sl.Add( (m_project.sources[i] as TSource).ExeFilename );
   end;
   //add new sources
   for i := 0 to lbSources.items.count-1 do begin
      if sl.IndexOf( (lbSources.Items.objects[i] as TSourceItem).m_strFilename ) = -1 then begin
         sourceitem := ((lbSources.Items.objects[i]) as TSourceitem);
         source := TSource.Create( m_Project,
                                  sourceitem.m_iLangNo,
                                  sourceitem.m_strFileName );
         m_Project.AddSource(source);
      end;
   end;
  finally
   sl.free;
  end;
end;

procedure TProjectSetupDlg.actDeleteSourceExecute(Sender: TObject);
begin
  lbSources.items.objects[lbSources.ItemIndex].free;
  lbSources.items.delete(lbSources.ItemIndex);
  UpdateOperations;
end;

procedure TProjectSetupDlg.actDeleteSourceUpdate(Sender: TObject);
begin
   actDeleteSource.Enabled := lbSources.ItemIndex >= 0;
end;

procedure TProjectSetupDlg.actDeleteLangExecute(Sender: TObject);
begin
  lbLanguages.items.delete(lbLanguages.ItemIndex);
  UpdateOperations;
end;

procedure TProjectSetupDlg.actDeleteLangUpdate(Sender: TObject);
begin
   actDeleteLang.Enabled := lbLanguages.ItemIndex >= 0;
end;


end.
