{ *************************************************************************** }

{                                                                             }
{ This file is part of the LinLocalize project                                }
{                                                                             }
{ Copyright (c) 2003                                                          }
{ Jens Khner <kuehner@users.sourceforge.net>                                 }
{                                                                             }
{ This program is free software; you can redistribute it and/or               }
{ modify it under the terms of the GNU General Public                         }
{ License as published by the Free Software Foundation; either                }
{ version 2 of the License, or (at your option) any later version.            }
{                                                                             }
{ This program is distributed in the hope that it will be useful,             }
{ but WITHOUT ANY WARRANTY; without even the implied warranty of              }
{ MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU           }
{ General Public License for more details.                                    }
{                                                                             }
{ You should have received a copy of the GNU General Public License           }
{ along with this program; see the file COPYING.  If not, write to            }
{ the Free Software Foundation, Inc., 59 Temple Place - Suite 330,            }
{ Boston, MA 02111-1307, USA.                                                 }
{                                                                             }
{ *************************************************************************** }


unit uProject;

interface

uses
  contnrs, classes;

type
  TSource  = class;
  TProject = class;
  TTargetFileFormat = (tfShared, tfExe);
  TTranslation = class
    private
      m_ParentSource : TSource;
      m_iLangNo : integer;
      m_strTargetFileName : string;
      m_FileFormat : TTargetFileFormat;
      m_LastTargetGeneration : TDateTime;
      m_LastUpdate : TDateTime;
      m_bLocked    : boolean;
     function GetListFileName : string;
      function GetDefaultTargetFileName : string;
      function GetResFileName : string;
    public
      constructor Create( Source : TSource; iLangNo : integer = -1 );
      destructor destroy; override;
      procedure WriteToStream( w : TWriter );
      procedure ReadFromStream( r : TReader );
      procedure deleteFiles;
      property ParentSource : TSource read m_ParentSource;
      property LangNo : integer read m_iLangNo write m_iLangNo;
      property ListFileName : string read GetListFileName;
      property TargetFileName : string read m_strTargetFileName write m_strTargetFileName;
      property ResFileName : string read GetResFileName;
      property FileFormat : TTargetFileFormat read m_FileFormat write m_FileFormat;
      property LastTargetGeneration : TDateTime read m_LastTargetGeneration write m_LastTargetGeneration;
      property LastUpdate : TDateTime read m_LastUpdate write m_LastUpdate;
      property Locked : boolean read m_bLocked write m_bLocked;
  end;
  TSource = class
    private
      m_ParentProject : TProject;
      m_strExeFileName : string;
      m_iLangNo : integer;
      m_lstTranslations : TObjectList;
      m_LastUpdate : TDateTime;
      function GetTitle : string;
      function GetBinPath : string;
      function GetResFileName : string;
      function GetListFileName : string;
    public
      constructor Create( Project : TProject; iLangNo : integer = -1; const strFileName : string = '' );
      destructor destroy; override;
      procedure WriteToStream( w : TWriter );
      procedure ReadFromStream( r : TReader );
      function GetTranslationByLanguageNo( iLangNo : integer ) : TTranslation;
      procedure deleteFiles;
      property ParentProject : TProject read m_ParentProject;
      property ExeFileName : string read m_strExeFileName write m_strExeFileName;
      property Title : string read GetTitle;
      property ResFileName : string read GetResFileName;
      property ListFileName : string read GetListFileName;
      property BinPath : string read GetBinPath;
      property LangNo : integer read m_iLangNo write m_iLangNo;
      property Translations : TObjectList read m_lstTranslations;
      property LastUpdate : TDateTime read m_LastUpdate write m_LastUpdate;
  end;
  TProject = class
    private
      m_strFileName : string;
      m_lstSources : TObjectList;
      m_lstLanguages : TList;
      m_lstGlossaries : TObjectList;
      function GetTokenPath : string;
    public
      constructor Create;
      destructor destroy; override;
      procedure ReadFromFile(const strFileName : string);
      procedure WriteToFile;
      procedure AddSource( source : TSource );
      procedure AddLanguage( iLangNo : integer );
      function RemoveLanguageAndFiles( iLangNo : integer ) : integer;
      property FileName : string read m_strFileName write m_strFileName;
      property TokenPath : string read GetTokenPath;
      property Sources : TObjectList read m_lstSources;
      property Languages : TList read m_lstLanguages;
      property Glossaries : TObjectList read m_lstGlossaries;
  end;

implementation

uses
  SysUtils, uLanguage, uGlossary;

//***************** Translation *******************************
constructor TTranslation.Create( Source : TSource; iLangNo : integer );
begin
  m_ParentSource      := source;
  m_iLangNo           := iLangNo;
  m_strTargetFileName := GetDefaultTargetFilename;
  inherited Create;
end;

destructor TTranslation.destroy;
begin
  inherited;
end;

procedure TTranslation.deleteFiles;
begin
   deletefile(GetListFileName);
   deletefile(GetDefaultTargetFileName);
   deletefile(GetResFileName);
end;

procedure TTranslation.WriteToStream( w : TWriter );
begin
  w.writeString( GetLangID(m_iLangNo) );
  w.writeString(m_strTargetFilename);
  w.writeInteger( ord(m_FileFormat) );
  w.writeDate( m_LastTargetGeneration );
  w.writeDate( m_LastUpdate );
  w.writeboolean(m_bLocked);
end;

procedure TTranslation.ReadFromStream( r : TReader );
begin
  m_iLangNo              := GetLangNoFromID(r.readstring);
  m_strTargetFilename    := r.readString;
  m_FileFormat           := TTargetFileFormat(r.readinteger);
  m_LastTargetGeneration := r.readDate;
  m_LastUpdate           := r.readDate;
  m_bLocked              := r.readboolean;
end;

function TTranslation.GetListFileName : string;
begin
  result := m_ParentSource.BinPath + extractFilename(m_ParentSource.ExeFileName);
  result := result + '.' + GetLangID(m_iLangNo) + '.trl';
end;

function TTranslation.GetResFileName : string;
begin
  result := m_ParentSource.BinPath + extractFilename(m_ParentSource.ExeFileName);
  result := result + '.' + GetLangID(m_iLangNo) + '.res';
end;

function TTranslation.GetDefaultTargetFileName : string;
begin
  result := m_ParentSource.m_strExeFileName+'.'+GetLangID(m_iLangNo);
end;
//***************** Source *******************************
constructor TSource.Create( Project : TProject; iLangNo : integer; const strFileName : string );
begin
  m_ParentProject      := Project;
  m_lstTranslations    := TObjectList.create;
  m_iLangNo            := iLangNo;
  m_strExeFileName     := strFileName;
  inherited create;
end;

destructor TSource.destroy;
begin
  m_lstTranslations.Free;
  inherited;
end;

procedure TSource.deleteFiles;
var
  i : integer;
begin
   deleteFile(GetResFileName);
   deletefile(GetListFileName);
   for i := 0 to m_lstTranslations.Count-1 do begin
     (m_lstTranslations[i] as TTranslation).deletefiles;
   end;
   removedir(GetBinPath);
end;

procedure TSource.WriteToStream( w : TWriter );
var
  i : integer;
begin
  w.writestring(m_strExeFileName);
  w.writestring( getLangID(m_iLangNo) );
  w.WriteInteger(m_lstTranslations.Count);
  for i := 0 to m_lstTranslations.Count-1 do
    (m_lstTranslations[i] as TTranslation).WriteToStream(w);
  w.writedate(m_LastUpdate);
end;

procedure TSource.ReadFromStream( r : TReader );
var
  i : integer;
  Trans : TTranslation;
  iCount : integer;
begin
  m_strExeFileName := r.readstring;
  m_iLangNo := GetLangNofromID( r.readstring );
  iCount := r.ReadInteger;
  for i := 0 to iCount-1 do begin
    trans := TTranslation.create(self);
    trans.ReadFromStream(r);
    m_lstTranslations.Add( trans );
  end;
  m_LastUpdate := r.ReadDate;
end;

function TSource.GetBinPath : string;
begin
  result := IncludeTrailingPathDelimiter(m_ParentProject.TokenPath+extractFileName(m_strExeFilename+'.bin'));
end;

function TSource.GetTitle : string;
begin
  result := extractFilename(m_strExeFileName);
end;

function TSource.GetResFileName : string;
begin
  result := GetBinPath + extractFilename(m_strExeFileName)+'.res';
end;

function TSource.GetListFileName : string;
begin
  result := GetBinPath + extractFilename(m_strExeFileName)+'.src';
end;

function TSource.GetTranslationByLanguageNo( iLangNo : integer ) : TTranslation;
var
  i : integer;
begin
   result := nil;
   for i := 0 to m_lstTranslations.Count-1 do begin
       if (m_lstTranslations[i] as TTranslation).LangNo = iLangNo then begin
          result := m_lstTranslations[i] as TTranslation;
          exit;
       end;
   end;
end;

//***************** Project *******************************
constructor TProject.Create;
begin
  inherited;
  m_lstSources    := TObjectList.Create;
  m_lstLanguages  := TList.Create;
  m_lstGlossaries := TObjectList.create;
end;

destructor TProject.destroy;
begin
  m_lstLanguages.free;
  m_lstSources.free;
  m_lstGlossaries.free;
  inherited;
end;

procedure TProject.ReadFromFile( const strFileName : string );
var
  s : TFileStream;
  r : TReader;
  i : integer;
  iCount : integer;
  source : TSource;
  str : string;
  bEnabled : boolean;
  Glossary : TGlossary;
begin
  m_strFileName := strFileName;
  s := TFileStream.Create(m_strFileName, fmOpenRead);
  r := TReader.Create(s, 4096);
  try
  r.ReadString;
  r.readinteger;

  iCount := r.ReadInteger;
  for i := 0 to iCount-1 do
    m_lstLanguages.Add( pointer( GetLangNoFromID(r.readstring) ) );
  iCount := r.ReadInteger;
  for i := 0 to iCount-1 do begin
    source := TSource.create(self);
    source.ReadFromStream(r);
//    AddSource( source );
    m_lstSources.add( source );
  end;

  iCount := r.ReadInteger;
  for i := 0 to iCount-1 do begin
     str := r.readString;
     bEnabled := r.readboolean;
     if fileexists(str) then begin
         Glossary := TGlossary.create;
         try
          Glossary.readFromFile( str );
          Glossary.enabled := bEnabled;
          m_lstGlossaries.Add(Glossary);
         except
           Glossary.free;
         end;
      end;
  end;
  finally
  r.free;
  s.Free;
  end;
end;

procedure TProject.WriteToFile;
var
  s : TFileStream;
  w : TWriter;
  i : integer;
begin
  s := TFileStream.Create(m_strFileName, fmCreate or fmOpenWrite);
  w := TWriter.Create(s, 4096);
  try
  w.WriteString('LLP');
  w.writeinteger(0);

  w.WriteInteger(m_lstLanguages.Count);
  for i := 0 to m_lstLanguages.Count-1 do
    w.writestring( GetLangID(integer(m_LstLanguages[i])) );
  w.WriteInteger(m_lstSources.Count);
  for i := 0 to m_lstSources.Count-1 do
    (m_lstSources[i] as TSource).WriteToStream(w);

  w.writeinteger(m_lstGlossaries.Count);
  for i := 0 to m_lstGlossaries.Count-1 do begin
     w.writestring( (m_lstGlossaries[i] as TGlossary).filename );
     w.writeBoolean( (m_lstGlossaries[i] as TGlossary).enabled );
  end;
  finally
  w.free;
  s.Free;
  end;
end;

function TProject.GetTokenPath : string;
begin
  result := IncludeTrailingPathDelimiter( changeFileExt(m_strFileName,'.tok') );
end;

procedure TProject.AddSource( source : TSource );
var
  i : integer;
begin
  m_lstSources.Add(source);
  for i := 0 to m_lstLanguages.count-1 do begin
    source.Translations.Add( TTranslation.Create(source, integer(m_lstLanguages[i])) );
  end;
end;

procedure TProject.AddLanguage( iLangNo : integer );
var
  iSource : integer;
  source : TSource;
begin
  m_lstLanguages.add(pointer(iLangNo));
  for iSource := 0 to m_lstSources.Count-1 do begin
     source := m_lstSources[iSource] as TSource;
     if source.GetTranslationByLanguageNo(iLangNo) = nil then
        source.translations.add( TTranslation.Create(source, iLangNo) );
  end;
end;

function TProject.RemoveLanguageAndFiles( iLangNo : integer ) : integer;
var
  iSource : integer;
  source : TSource;
  trans : TTranslation;
begin
  result := -1;
  m_lstLanguages.Remove(pointer(iLangNo));
  for iSource := 0 to m_lstSources.Count-1 do begin
     source := m_lstSources[iSource] as TSource;
     Trans := source.GetTranslationByLanguageNo(iLangNo);
     Trans.deleteFiles;
     result := source.translations.Remove( Trans );
  end;
end;

end.
