{ *************************************************************************** }
{                                                                             }
{ This file is part of the LinLocalize project                                }
{                                                                             }
{ Copyright (c) 2003                                                          }
{ Jens Khner <kuehner@users.sourceforge.net>                                 }
{                                                                             }
{ This program is free software; you can redistribute it and/or               }
{ modify it under the terms of the GNU General Public                         }
{ License as published by the Free Software Foundation; either                }
{ version 2 of the License, or (at your option) any later version.            }
{                                                                             }
{ This program is distributed in the hope that it will be useful,             }
{ but WITHOUT ANY WARRANTY; without even the implied warranty of              }
{ MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU           }
{ General Public License for more details.                                    }
{                                                                             }
{ You should have received a copy of the GNU General Public License           }
{ along with this program; see the file COPYING.  If not, write to            }
{ the Free Software Foundation, Inc., 59 Temple Place - Suite 330,            }
{ Boston, MA 02111-1307, USA.                                                 }
{                                                                             }
{ *************************************************************************** }

unit uGlossaryDlg;

interface

uses
  uProject, ContNrs, uLLStringGrid,
  SysUtils, Types, Classes, Variants, QTypes, QGraphics, QControls, QForms, 
  QDialogs, QStdCtrls, QGrids, QActnList;

type
  TItem = class
     public
       m_strFileName : string;
       m_iSize : integer;
       m_strLanguages : string;
       m_bEnabled : boolean;
  end;
  TGlossaryDlg = class(TForm)
    btnOK: TButton;
    btnCancel: TButton;
    gridGlossary: TLLStringGrid;
    btnAdd: TButton;
    btnDelete: TButton;
    btnMoveUp: TButton;
    btnMoveDown: TButton;
    btnDisable: TButton;
    ActionList1: TActionList;
    actAdd: TAction;
    actDelete: TAction;
    actMoveUp: TAction;
    actMoveDown: TAction;
    actDisable: TAction;
    AddGlossaryDlg: TOpenDialog;
    procedure actAddExecute(Sender: TObject);
    procedure FormShow(Sender: TObject);
    procedure FormDestroy(Sender: TObject);
    procedure gridGlossarySelectCell(Sender: TObject; ACol, ARow: Integer;
      var CanSelect: Boolean);
    procedure actDeleteExecute(Sender: TObject);
    procedure actItemSelectedUpdate(Sender: TObject);
    procedure actMoveUpExecute(Sender: TObject);
    procedure actMoveUpUpdate(Sender: TObject);
    procedure actMoveDownExecute(Sender: TObject);
    procedure actMoveDownUpdate(Sender: TObject);
    function gridGlossaryTextColor(ACol, ARow: Integer; ARect: TRect;
      AState: TGridDrawState): TColor;
    procedure actDisableExecute(Sender: TObject);
    procedure actDisableUpdate(Sender: TObject);
    procedure btnOKClick(Sender: TObject);
  private
    { Private-Deklarationen }
    m_lstGlossaries : TObjectList;
    m_lstItems : TObjectList;
    m_Project : TProject;
    m_iItemIndex : integer;
    procedure RebuildList;
  public
    { Public-Deklarationen }
    constructor Create( AOwner : TComponent; lstGlossaries : TObjectList; Project : TProject ); reintroduce;
  end;

var
  GlossaryDlg: TGlossaryDlg;

implementation

{$R *.xfm}

uses
  uGlossary, math, uApp;

constructor TGlossaryDlg.Create( AOwner : TComponent; lstGlossaries : TObjectList; project : TProject );
begin
  inherited Create(AOwner);
  m_lstGlossaries := lstGlossaries;
  m_lstItems      := TObjectList.Create;
  m_Project       := Project;
end;

procedure TGlossaryDlg.FormDestroy(Sender: TObject);
begin
  m_lstItems.Free;
end;

procedure TGlossaryDlg.FormShow(Sender: TObject);
var
  i : integer;
  item : TItem;
  glossary : TGlossary;
resourcestring
  rsFile = 'File';
  rsSize = 'Size';
  rsLanguages = 'Languages';
  rsProjectCaption = 'Glossaries for project %s';
begin
   if assigned(m_project) then
      caption := format(rsProjectCaption, [extractFileName(m_Project.FileName)]);
   gridGlossary.Cells[0, 0]  := rsFile;
   gridGlossary.Cells[1, 0]  := rsSize;
   gridGlossary.Cells[2, 0]  := rsLanguages;
   gridGlossary.ColWidths[0] := 400;
   gridGlossary.ColWidths[1] := 35;
   gridGlossary.ColWidths[2] := 200;
   for i := 0 to m_lstGlossaries.count-1 do begin
     Glossary := m_lstGlossaries[i] as TGlossary;
     item := TItem.Create;
     item.m_bEnabled     := Glossary.Enabled;
     item.m_strFileName  := Glossary.FileName;
     item.m_strLanguages := Glossary.LanguagesAsString;
     item.m_iSize        := Glossary.Count;
     m_lstItems.Add(item);
   end;
   m_iItemIndex := 0;
   RebuildList;
end;

procedure TGlossaryDlg.RebuildList;
var
  i : integer;
begin
  gridGlossary.rowCount := max(m_lstItems.count+1, 2);
  gridGlossary.cells[0, 1] := '';
  gridGlossary.cells[1, 1] := '';
  gridGlossary.cells[2, 1] := '';
  for i := 0 to m_lstItems.count-1 do begin
    gridGlossary.cells[0, i+1] := (m_lstItems[i] as TItem).m_strFileName;
    gridGlossary.cells[1, i+1] := inttostr((m_lstItems[i] as TItem).m_iSize);
    gridGlossary.cells[2, i+1] := (m_lstItems[i] as TItem).m_strLanguages;
  end;
  m_iItemIndex := min(m_lstItems.Count-1, m_iItemIndex);
  gridGlossary.Row := max(m_iItemIndex+1, 1);
end;

procedure TGlossaryDlg.actAddExecute(Sender: TObject);
var
  item : TItem;
begin
   if assigned(m_project) then
      AddGlossaryDlg.InitialDir := extractFilePath(m_Project.FileName)
   else
      AddGlossaryDlg.InitialDir := g_app.GlossaryDir;
  if AddGlossaryDlg.Execute then begin
     gridGlossary.rowcount := gridGlossary.rowcount+1;
     gridGlossary.cells[0, gridGlossary.rowcount-1] := AddGlossaryDlg.filename;
     with TGlossary.create do begin
       try
        readFromFile(AddGlossaryDlg.filename);
        item := TItem.Create;
        item.m_strFileName  := AddGlossaryDlg.filename;
        item.m_bEnabled     := true;
        item.m_iSize        := Count;
        item.m_strLanguages := LanguagesAsString;
        m_iItemIndex := m_lstItems.Add( item );
        RebuildList;
       finally
        free;
       end;
     end;
  end;
end;

procedure TGlossaryDlg.gridGlossarySelectCell(Sender: TObject; ACol,
  ARow: Integer; var CanSelect: Boolean);
begin
    CanSelect := ARow-1 <= m_lstItems.count-1;
    if Canselect then
      m_iItemIndex := ARow-1;
end;

procedure TGlossaryDlg.actDeleteExecute(Sender: TObject);
begin
    m_lstItems.Delete(m_iItemIndex);
    RebuildList;
end;

procedure TGlossaryDlg.actItemSelectedUpdate(Sender: TObject);
begin
  actDelete.Enabled := m_iItemIndex <> -1;
end;

procedure TGlossaryDlg.actMoveUpExecute(Sender: TObject);
begin
  m_lstItems.Move(m_iItemIndex, m_iItemIndex-1);
  m_iItemIndex := m_iItemIndex-1;
  RebuildList;
end;

procedure TGlossaryDlg.actMoveUpUpdate(Sender: TObject);
begin
   actMoveUp.Enabled := m_iItemIndex > 0;
end;

procedure TGlossaryDlg.actMoveDownExecute(Sender: TObject);
begin
  m_lstItems.Move(m_iItemIndex, m_iItemIndex+1);
  m_iItemIndex := m_iItemIndex+1;
  RebuildList;
end;

procedure TGlossaryDlg.actMoveDownUpdate(Sender: TObject);
begin
   actMoveDown.Enabled := (m_iItemIndex < m_lstItems.count-1) and (m_iItemIndex>=0);
end;

function TGlossaryDlg.gridGlossaryTextColor(ACol, ARow: Integer;
  ARect: TRect; AState: TGridDrawState): TColor;
begin
  result := clBlack;
  if (ARow-1 >= 0) and (ARow-1 <= m_lstItems.count-1) then begin
    if not (m_lstItems[ARow-1] as TItem).m_bEnabled then
      result := clRed;
  end;
end;

procedure TGlossaryDlg.actDisableExecute(Sender: TObject);
begin
  (m_lstItems[m_iItemIndex] as TItem).m_bEnabled := not (m_lstItems[m_iItemIndex] as TItem).m_bEnabled;
  RebuildList;
end;

procedure TGlossaryDlg.actDisableUpdate(Sender: TObject);
resourceString
  rsDisable = 'Disable';
  rsEnable = 'Enable';
begin
  actDisable.Enabled := m_iItemIndex <> -1;
  if actDisable.Enabled and not (m_lstItems[m_iItemIndex] as TItem).m_bEnabled then
    actDisable.Caption := rsEnable
  else
    actDisable.Caption := rsDisable;
end;

procedure TGlossaryDlg.btnOKClick(Sender: TObject);
var
  i, t : integer;
  glossary : TGlossary;
  iIndex : integer;
  item : Titem;
begin
  //delete
  for i := m_lstGlossaries.Count-1 downto 0 do begin
      Glossary := m_lstGlossaries[i] as TGlossary;
      iIndex := -1;
      for t := 0 to m_lstItems.Count-1 do begin
        if (m_lstItems[t] as TItem).m_strFileName = Glossary.FileName then begin
          iIndex := t;
          break;
        end;
      end;
      if iIndex = -1 then
        m_lstGlossaries.Delete(i);
  end;
  //add
  for i := 0 to m_lstItems.count-1 do begin
     item := m_lstItems[i] as TItem;
     iIndex := -1;
     for t := 0 to m_lstGlossaries.count-1 do begin
        if (m_lstGlossaries[t] as TGlossary).FileName = item.m_strFileName then begin
          iIndex := t;
          break;
        end;
     end;
     if iIndex = -1 then begin
       Glossary := TGlossary.Create;
       Glossary.ReadFromFile(item.m_strFileName);
       Glossary.Enabled := item.m_bEnabled;
       m_lstGlossaries.Add(Glossary);
     end;
  end;
  //sort
  for i := 0 to m_lstItems.count-1 do begin
     item := m_lstItems[i] as TItem;
     iIndex := -1;
     for t := 0 to m_lstGlossaries.count-1 do begin
        Glossary := m_lstGlossaries[t] as TGlossary;
        if Glossary.FileName = item.m_strFileName then begin
          iIndex := t;
          Glossary.Enabled := item.m_bEnabled;
          break;
        end;
     end;
     m_lstGlossaries.Move(iIndex, i);
  end;
end;

end.
