{ *************************************************************************** }
{                                                                             }
{ This file is part of the LinLocalize project                                }
{                                                                             }
{ Copyright (c) 2003                                                          }
{ Jens Khner <kuehner@users.sourceforge.net>                                 }
{                                                                             }
{ This program is free software; you can redistribute it and/or               }
{ modify it under the terms of the GNU General Public                         }
{ License as published by the Free Software Foundation; either                }
{ version 2 of the License, or (at your option) any later version.            }
{                                                                             }
{ This program is distributed in the hope that it will be useful,             }
{ but WITHOUT ANY WARRANTY; without even the implied warranty of              }
{ MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU           }
{ General Public License for more details.                                    }
{                                                                             }
{ You should have received a copy of the GNU General Public License           }
{ along with this program; see the file COPYING.  If not, write to            }
{ the Free Software Foundation, Inc., 59 Temple Place - Suite 330,            }
{ Boston, MA 02111-1307, USA.                                                 }
{                                                                             }
{ *************************************************************************** }

unit uGlossary;

interface

uses
  uTextLists, Classes, Contnrs;

type
 TGlossary = class
   private
     m_bEnabled : boolean;
     m_strFileName : string;
     m_lstLanguages : TList;
     m_lstTextes : TObjectList;
     m_strCreator : string;
     function GetCount : integer;
   public
     constructor Create; overload;
     constructor Create( TransTextList : TTransTextList; bOnlyBookmarked : boolean ); overload;
     destructor destroy; override;
     procedure AddLanguage( iLangNo : integer );
     function IndexOfLang( iLangNo : integer ) : integer;
     function LanguagesAsString : string;
     procedure ReadFromFile( const strFileName : string );
     procedure WriteToFile( const strFileName : string );
     procedure Add( sl : TStrings );
     function Translate( iSourceLangNo : integer; const strSource : string; iTransLangNo : integer; out strTrans : string) : integer;
     property Enabled : boolean read m_bEnabled write m_bEnabled;
     property Count : integer read GetCount;
     property FileName : string read m_strFileName;
     property Creator : string read m_strCreator write m_strCreator;
 end;

implementation

uses
  SysUtils, uLanguage, uGlobal;

constructor TGlossary.Create;
begin
  inherited;
  m_bEnabled     := true;
  m_lstTextes    := TObjectList.create;
  m_lstLanguages := TList.create;
end;

constructor TGlossary.Create( TransTextList : TTransTextList; bOnlyBookmarked : boolean );
var
  i, iRes : integer;
  TransResItem : TTransResItem;
  TransItem : TTransItem;
begin
  Create;
  AddLanguage( TransTextList.SourceLangNo );
  AddLanguage( TransTextList.TransLangNo );
  for iRes := 0 to TransTextList.Items.Count-1 do begin
      TransResItem := TransTextList.Items[iRes] as TTransResItem;
      for i := 0 to TransResItem.count-1 do begin
        TransItem := TransResItem.Transitems[i];
        if TransItem.Editable and TransItem.Translated then begin
          if not bOnlyBookmarked or (bOnlyBookmarked and TransItem.Bookmark) then begin
            TStringList(m_lstTextes[0]).add(TransItem.SourceText);
            TStringList(m_lstTextes[1]).add(TransItem.TransText);
          end;
        end;
      end;
  end;
end;

destructor TGlossary.destroy;
begin
  m_lstTextes.free;
  m_lstLanguages.free;
end;

procedure TGlossary.AddLanguage( iLangNo : integer );
begin
  m_lstLanguages.Add( pointer(iLangNo) );
  m_lstTextes.Add( TStringList.Create );
end;

procedure TGlossary.Add( sl : TStrings );
var
  i : integer;
begin
  assert( sl.count = m_lstTextes.Count );
  for i := 0 to m_lstTextes.Count-1 do begin
      (m_lstTextes[i] as TStringList).Add(sl[i]);
  end;
end;

{procedure TGlossary.ReadFromFile( const strFileName : string );
var
  i : integer;
  iLangCount : integer;
  f : TextFile;
  sl : TStringlist;
  strLine : string;
begin
  m_strFileName := strFileName;
  assignfile(f, m_strFilename);
  reset(f);
  sl := TStringList.Create;
  sl.DelimitedText := #9;
  sl.QuoteChar     := #0;
  try
   m_strCreator := '';
   m_lstLanguages.clear;

   readln(f, strLine);
   sl.DelimitedText := strLine;
   iLangCount := sl.count;
   for i := 0 to iLangCount-1 do begin
      AddLanguage( GetLangNoFromID(sl[i]) );
   end;
   for i := 0 to iLangCount-1 do begin
     (m_lstTextes[i] as TStringList).clear;
   end;
   readln(f, strLine);
   while not eof(f) do begin
      sl.DelimitedText := strLine;
      for i := 0 to iLangCount-1 do begin
        writeln( addsyschars(sl[i]) );
        (m_lstTextes[i] as TStringList).Add( addsyschars(sl[i]) )
      end;
      readln(f, strLine);
   end;
  finally
    sl.free;
    closefile(f);
  end;
end;

procedure TGlossary.WriteToFile( const strFileName : string );
var
  i, t : integer;
  f : TextFile;
begin
  m_strFileName := strFileName;
  assignfile(f, m_strFilename);
  rewrite(f);
  try
   for i := 0 to m_lstLanguages.count-1 do begin
     if i > 0 then write(f, #9);
     write(f, getLangID( integer(m_lstLanguages[i])) );
   end;
   writeln(f);

   for i := 0 to Getcount-1 do begin
      for t := 0 to m_lstTextes.Count-1 do begin
         if t > 0 then write(f, #9);
         write(f, trimsysChars((m_lstTextes[t] as TStringList)[i]) );
      end;
      writeln(f);
   end;
  finally
    closefile(f);
  end;
end;}

procedure TGlossary.ReadFromFile( const strFileName : string );
var
  s : TFileStream;
  r : TReader;
  i, t : integer;
  iLangCount : integer;
  iStringCount : integer;
begin
  m_strFileName := strFileName;
  s := TFileStream.Create(m_strFileName, fmOpenRead);
  r := TReader.Create(s, 4096);
  try
   r.readstring;
   r.readinteger;
   m_strCreator := r.readstring;

   m_lstLanguages.clear;
   iLangCount := r.readinteger;
   for i := 0 to iLangCount-1 do begin
      AddLanguage( GetLangNoFromID(r.readstring) );
   end;

   iStringCount := r.readinteger;
   for t := 0 to iLangCount-1 do begin
     (m_lstTextes[t] as TStringList).clear;
     (m_lstTextes[t] as TStringList).capacity := iStringCount;//speeds up memory-management
   end;
   for i := 0 to iStringCount-1 do begin
      for t := 0 to iLangCount-1 do begin
        (m_lstTextes[t] as TStringList).Add(r.readstring)
      end;
   end;
  finally
   r.free;
   s.Free;
  end;
end;

procedure TGlossary.WriteToFile( const strFileName : string );
var
  s : TFileStream;
  w : TWriter;
  i, t : integer;
begin
  m_strFileName := strFileName;
  s := TFileStream.Create(m_strFileName, fmCreate or fmOpenWrite);
  w := TWriter.Create(s, 4096);
  try
   w.WriteString('GLO');
   w.writeinteger(0);
   w.writestring(m_strCreator);

   w.writeinteger(m_lstLanguages.count);
   for i := 0 to m_lstLanguages.count-1 do
     w.writestring( getLangID( integer(m_lstLanguages[i]) ) );

   w.writeinteger( getCount );
   for i := 0 to Getcount-1 do begin
      for t := 0 to m_lstTextes.Count-1 do begin
         w.WriteString( (m_lstTextes[t] as TStringList)[i] );
      end;
   end;
  finally
   w.free;
   s.Free;
  end;
end;

function TGlossary.GetCount : integer;
begin
  result := (m_lstTextes[0] as TStringList).count;
end;

function TGlossary.IndexOfLang( iLangNo : integer ) : integer;
var
 i : integer;
 strID : string;
begin
  result := m_lstLanguages.indexof( pointer(iLangNo) );
  if result = -1 then begin
    strID := stripSubLangFromID(getLangID(iLangNo));
    for i := 0 to m_lstLanguages.count-1 do begin
      if strID = stripSubLangFromID(getLangID(integer(m_lstLanguages[i]))) then begin
        result := i;
        break;
      end;
    end;
  end;
end;

function TGlossary.LanguagesAsString : string;
var
  i : integer;
begin
  result := '';
  for i := 0 to m_lstLanguages.count-1 do begin
    if i > 0 then
      result := result + ', ';
    result := result + getLangID( integer(m_lstLanguages[i]) );
  end;
end;

function TGlossary.Translate( iSourceLangNo : integer; const strSource : string; iTransLangNo : integer; out strTrans : string) : integer;
var
  iSourceIndex, iTransIndex : integer;
  lstSource, lstTrans : TStringList;
  i : integer;
begin
  result       := 0;
  strTrans     := '';
  iSourceIndex := IndexOflang( iSourcelangNo );
  iTransIndex  := IndexOfLang( iTransLangNo );
  if (iSourceIndex<>-1) and (iTransIndex<>-1) then begin
    lstSource := m_lstTextes[iSourceIndex] as TStringList;
    lstTrans  := m_lstTextes[iTransIndex] as TStringList;
    for i := 0 to getcount-1 do begin
       if compareNoEllipse(lstSource[i], strSource) then begin
         if result = 0 then
           strTrans := lstTrans[i];
         if hasEllipse(strSource) then
            strTrans := IncludeEllipse(strTrans)
         else
            strTrans := ExcludeEllipse(strTrans);
         inc(result);
       end;
    end;
  end;
end;

end.
